!!****f* ABINIT/contstr21
!! NAME
!! contstr21
!!
!! FUNCTION
!! Carries out specialized metric tensor operations needed for contraction
!! of the 2nd strain derivative of the l=0,1,2,3 nonlocal Kleinman-Bylander
!! pseudopotential operation.  Derivatives are wrt a pair of cartesian
!! strain components.
!! Full advantage is taken of the full permutational symmetry of these
!! tensors.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2022 ABINIT group (DRH)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! INPUTS
!!  istr1=1,...6 specifies cartesian strain component 11,22,33,32,31,21
!!  istr2=seccond strain component
!!  rank=angular momentum
!!  gm(3,3)=metric tensor (array is symmetric but stored as 3x3)
!!  gprimd(3,3)=reciprocal space dimensional primitive translations
!!  aa(2,*)=unique elements of complex right-hand tensor
!!  bb(2,*)=unique elements of complex left-hand tensor
!!
!! OUTPUT
!!  e2nl=contraction for nonlocal 2nd-order strain derivative energy
!!
!! NOTES
!! All tensors are stored in a compressed storage mode defined below;
!! input and output conform to this scheme.
!! When tensor elements occur repeatedly due to symmetry, the
!! WEIGHT IS INCLUDED in the output tensor element to simplify later
!! contractions with other tensors of the same rank and form, i.e. the
!! next contraction is then simply a dot product over the unique elements.
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"


module m_contstr21
contains
!!**



subroutine contstr21(istr1,istr2,rank,gm,gprimd,e2nl,aa,bb)

 use defs_basis
 use m_abicore
 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: istr1,istr2,rank
 real(dp),intent(out) :: e2nl
!arrays
 real(dp),intent(in) :: aa(2,((rank+1)*(rank+2))/2),bb(2,((rank+6)*(rank+5))/2)
 real(dp),intent(in) :: gm(3,3),gprimd(3,3)

!Local variables-------------------------------
!scalars
 integer,parameter :: mrank=3
 integer :: ii,jj,ka,kb,kd,kg
!arrays
 integer,save :: idx(12)=(/1,1,2,2,3,3,3,2,3,1,2,1/)
 real(dp) :: d2gm(3,3),dgm01(3,3),dgm10(3,3),tmp(2)
 real(dp),allocatable :: cm(:,:)

! *************************************************************************
 ABI_MALLOC(cm,(((mrank+1)*(mrank+2))/2,((mrank+5)*(mrank+6))/2))

 ka=idx(2*istr1-1);kb=idx(2*istr1);kg=idx(2*istr2-1);kd=idx(2*istr2)

 do ii = 1,3
   dgm01(:,ii)=-(gprimd(ka,:)*gprimd(kb,ii)+gprimd(kb,:)*gprimd(ka,ii))
   dgm10(:,ii)=-(gprimd(kg,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(kg,ii))
 end do

 d2gm(:,:)=0.d0
 do ii = 1,3
   if(ka==kg) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(kb,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(kb,ii)
   if(ka==kd) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(kb,:)*gprimd(kg,ii)+gprimd(kg,:)*gprimd(kb,ii)
   if(kb==kg) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(ka,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(ka,ii)
   if(kb==kd) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(ka,:)*gprimd(kg,ii)+gprimd(kg,:)*gprimd(ka,ii)
 end do
 d2gm(:,:)=0.5d0*d2gm(:,:)

!
!The code below was written by a Mathematica program and formatted by
!a combination of editing scripts.  It is not intended to be read
!by human beings, and certainly not to be modified by one.  Conceivably
!it could be shortened somewhat by identifying common subexpressions.
!

 if(rank==0)then
   cm(1,1)=dgm01(1,1)*dgm10(1,1)
   cm(1,2)=dgm01(2,2)*dgm10(1,1)+4*dgm01(1,2)*dgm10(1,2)+dgm01(1,1)&
&   *dgm10(2,2)
   cm(1,3)=dgm01(3,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)+dgm01(1,1)&
&   *dgm10(3,3)
   cm(1,4)=2*(dgm01(2,3)*dgm10(1,1)+2*dgm01(1,3)*dgm10(1,2)+2*dgm01(1,2)&
&   *dgm10(1,3)+dgm01(1,1)*dgm10(2,3))
   cm(1,5)=2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))
   cm(1,6)=2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))
   cm(1,7)=2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))
   cm(1,8)=2*(dgm01(3,3)*dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)+2*dgm01(1,3)&
&   *dgm10(2,3)+dgm01(1,2)*dgm10(3,3))
   cm(1,9)=2*(2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3))
   cm(1,10)=2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))
   cm(1,11)=dgm01(2,2)*dgm10(2,2)
   cm(1,12)=dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)+dgm01(2,2)&
&   *dgm10(3,3)
   cm(1,13)=2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))
   cm(1,14)=2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))
   cm(1,15)=dgm01(3,3)*dgm10(3,3)
 elseif(rank==1)then
   cm(1,1)=gm(1,1)*dgm01(1,1)*dgm10(1,1)
   cm(2,1)=gm(1,2)*dgm01(1,1)*dgm10(1,1)
   cm(3,1)=gm(1,3)*dgm01(1,1)*dgm10(1,1)
   cm(1,2)=2*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&   +gm(1,1)*(dgm01(2,2)*dgm10(1,1)+4*dgm01(1,2)*dgm10(1,2)+dgm01(1,1)&
&   *dgm10(2,2))
   cm(2,2)=2*gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&   +gm(1,2)*(dgm01(2,2)*dgm10(1,1)+4*dgm01(1,2)*dgm10(1,2)+dgm01(1,1)&
&   *dgm10(2,2))
   cm(3,2)=2*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&   +gm(1,3)*(dgm01(2,2)*dgm10(1,1)+4*dgm01(1,2)*dgm10(1,2)+dgm01(1,1)&
&   *dgm10(2,2))
   cm(1,3)=2*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,1)*(dgm01(3,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)+dgm01(1,1)&
&   *dgm10(3,3))
   cm(2,3)=2*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,2)*(dgm01(3,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)+dgm01(1,1)&
&   *dgm10(3,3))
   cm(3,3)=2*gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,3)*(dgm01(3,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)+dgm01(1,1)&
&   *dgm10(3,3))
   cm(1,4)=2*((gm(1,3)*dgm01(1,2)+gm(1,2)*dgm01(1,3)+gm(1,1)*dgm01(2,3))&
&   *dgm10(1,1)+(gm(1,3)*dgm01(1,1)+2*gm(1,1)*dgm01(1,3))*dgm10(1,2)&
&   +(gm(1,2)*dgm01(1,1)+2*gm(1,1)*dgm01(1,2))*dgm10(1,3)+gm(1,1)&
&   *dgm01(1,1)*dgm10(2,3))
   cm(2,4)=2*((gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3)+gm(1,2)*dgm01(2,3))&
&   *dgm10(1,1)+(gm(2,3)*dgm01(1,1)+2*gm(1,2)*dgm01(1,3))*dgm10(1,2)&
&   +(gm(2,2)*dgm01(1,1)+2*gm(1,2)*dgm01(1,2))*dgm10(1,3)+gm(1,2)&
&   *dgm01(1,1)*dgm10(2,3))
   cm(3,4)=2*((gm(3,3)*dgm01(1,2)+gm(2,3)*dgm01(1,3)+gm(1,3)*dgm01(2,3))&
&   *dgm10(1,1)+(gm(3,3)*dgm01(1,1)+2*gm(1,3)*dgm01(1,3))*dgm10(1,2)&
&   +(gm(2,3)*dgm01(1,1)+2*gm(1,3)*dgm01(1,2))*dgm10(1,3)+gm(1,3)&
&   *dgm01(1,1)*dgm10(2,3))
   cm(1,5)=gm(1,3)*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))
   cm(2,5)=gm(2,3)*dgm01(1,1)*dgm10(1,1)+2*gm(1,2)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))
   cm(3,5)=gm(3,3)*dgm01(1,1)*dgm10(1,1)+2*gm(1,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))
   cm(1,6)=gm(1,2)*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))
   cm(2,6)=gm(2,2)*dgm01(1,1)*dgm10(1,1)+2*gm(1,2)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))
   cm(3,6)=gm(2,3)*dgm01(1,1)*dgm10(1,1)+2*gm(1,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))
   cm(1,7)=gm(1,2)*(dgm01(2,2)*dgm10(1,1)+4*dgm01(1,2)*dgm10(1,2)&
&   +dgm01(1,1)*dgm10(2,2))+2*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))
   cm(2,7)=gm(2,2)*(dgm01(2,2)*dgm10(1,1)+4*dgm01(1,2)*dgm10(1,2)&
&   +dgm01(1,1)*dgm10(2,2))+2*gm(1,2)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))
   cm(3,7)=gm(2,3)*(dgm01(2,2)*dgm10(1,1)+4*dgm01(1,2)*dgm10(1,2)&
&   +dgm01(1,1)*dgm10(2,2))+2*gm(1,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))
   cm(1,8)=2*gm(1,3)*dgm01(2,3)*dgm10(1,1)+gm(1,2)*dgm01(3,3)*dgm10(1,1)&
&   +2*(2*gm(1,3)*dgm01(1,3)+gm(1,1)*dgm01(3,3))*dgm10(1,2)+4*(gm(1,3)&
&   *dgm01(1,2)+gm(1,2)*dgm01(1,3)+gm(1,1)*dgm01(2,3))*dgm10(1,3)&
&   +2*(gm(1,3)*dgm01(1,1)+2*gm(1,1)*dgm01(1,3))*dgm10(2,3)+(gm(1,2)&
&   *dgm01(1,1)+2*gm(1,1)*dgm01(1,2))*dgm10(3,3)
   cm(2,8)=2*gm(2,3)*dgm01(2,3)*dgm10(1,1)+gm(2,2)*dgm01(3,3)*dgm10(1,1)&
&   +2*(2*gm(2,3)*dgm01(1,3)+gm(1,2)*dgm01(3,3))*dgm10(1,2)+4*(gm(2,3)&
&   *dgm01(1,2)+gm(2,2)*dgm01(1,3)+gm(1,2)*dgm01(2,3))*dgm10(1,3)&
&   +2*(gm(2,3)*dgm01(1,1)+2*gm(1,2)*dgm01(1,3))*dgm10(2,3)+(gm(2,2)&
&   *dgm01(1,1)+2*gm(1,2)*dgm01(1,2))*dgm10(3,3)
   cm(3,8)=2*gm(3,3)*dgm01(2,3)*dgm10(1,1)+gm(2,3)*dgm01(3,3)*dgm10(1,1)&
&   +2*(2*gm(3,3)*dgm01(1,3)+gm(1,3)*dgm01(3,3))*dgm10(1,2)+4*(gm(3,3)&
&   *dgm01(1,2)+gm(2,3)*dgm01(1,3)+gm(1,3)*dgm01(2,3))*dgm10(1,3)&
&   +2*(gm(3,3)*dgm01(1,1)+2*gm(1,3)*dgm01(1,3))*dgm10(2,3)+(gm(2,3)&
&   *dgm01(1,1)+2*gm(1,3)*dgm01(1,2))*dgm10(3,3)
   cm(1,9)=gm(1,3)*dgm01(2,2)*dgm10(1,1)+2*gm(1,2)*dgm01(2,3)*dgm10(1,1)&
&   +4*(gm(1,3)*dgm01(1,2)+gm(1,2)*dgm01(1,3)+gm(1,1)*dgm01(2,3))&
&   *dgm10(1,2)+2*(2*gm(1,2)*dgm01(1,2)+gm(1,1)*dgm01(2,2))*dgm10(1,3)&
&   +(gm(1,3)*dgm01(1,1)+2*gm(1,1)*dgm01(1,3))*dgm10(2,2)+2*(gm(1,2)&
&   *dgm01(1,1)+2*gm(1,1)*dgm01(1,2))*dgm10(2,3)
   cm(2,9)=gm(2,3)*dgm01(2,2)*dgm10(1,1)+2*gm(2,2)*dgm01(2,3)*dgm10(1,1)&
&   +4*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3)+gm(1,2)*dgm01(2,3))&
&   *dgm10(1,2)+2*(2*gm(2,2)*dgm01(1,2)+gm(1,2)*dgm01(2,2))*dgm10(1,3)&
&   +(gm(2,3)*dgm01(1,1)+2*gm(1,2)*dgm01(1,3))*dgm10(2,2)+2*(gm(2,2)&
&   *dgm01(1,1)+2*gm(1,2)*dgm01(1,2))*dgm10(2,3)
   cm(3,9)=gm(3,3)*dgm01(2,2)*dgm10(1,1)+2*gm(2,3)*dgm01(2,3)*dgm10(1,1)&
&   +4*(gm(3,3)*dgm01(1,2)+gm(2,3)*dgm01(1,3)+gm(1,3)*dgm01(2,3))&
&   *dgm10(1,2)+2*(2*gm(2,3)*dgm01(1,2)+gm(1,3)*dgm01(2,2))*dgm10(1,3)&
&   +(gm(3,3)*dgm01(1,1)+2*gm(1,3)*dgm01(1,3))*dgm10(2,2)+2*(gm(2,3)&
&   *dgm01(1,1)+2*gm(1,3)*dgm01(1,2))*dgm10(2,3)
   cm(1,10)=gm(1,3)*(dgm01(3,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)&
&   +dgm01(1,1)*dgm10(3,3))+2*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))
   cm(2,10)=gm(2,3)*(dgm01(3,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)&
&   +dgm01(1,1)*dgm10(3,3))+2*gm(1,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))
   cm(3,10)=gm(3,3)*(dgm01(3,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)&
&   +dgm01(1,1)*dgm10(3,3))+2*gm(1,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))
   cm(1,11)=gm(1,1)*dgm01(2,2)*dgm10(2,2)+2*gm(1,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))
   cm(2,11)=gm(1,2)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))
   cm(3,11)=gm(1,3)*dgm01(2,2)*dgm10(2,2)+2*gm(2,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))
   cm(1,12)=2*gm(1,3)*(2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3))+2*gm(1,2)*(dgm01(3,3)&
&   *dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)+2*dgm01(1,3)*dgm10(2,3)+dgm01(1,2)&
&   *dgm10(3,3))+gm(1,1)*(dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)&
&   +dgm01(2,2)*dgm10(3,3))
   cm(2,12)=2*gm(2,3)*(2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3))+2*gm(2,2)*(dgm01(3,3)&
&   *dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)+2*dgm01(1,3)*dgm10(2,3)+dgm01(1,2)&
&   *dgm10(3,3))+gm(1,2)*(dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)&
&   +dgm01(2,2)*dgm10(3,3))
   cm(3,12)=2*gm(3,3)*(2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3))+2*gm(2,3)*(dgm01(3,3)&
&   *dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)+2*dgm01(1,3)*dgm10(2,3)+dgm01(1,2)&
&   *dgm10(3,3))+gm(1,3)*(dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)&
&   +dgm01(2,2)*dgm10(3,3))
   cm(1,13)=2*(gm(1,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +gm(1,2)*(2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3))+gm(1,1)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)))
   cm(2,13)=2*(gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +gm(2,2)*(2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3))+gm(1,2)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)))
   cm(3,13)=2*(gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +gm(2,3)*(2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3))+gm(1,3)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)))
   cm(1,14)=2*(gm(1,3)*(dgm01(3,3)*dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)&
&   +2*dgm01(1,3)*dgm10(2,3)+dgm01(1,2)*dgm10(3,3))+gm(1,2)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(1,1)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3)))
   cm(2,14)=2*(gm(2,3)*(dgm01(3,3)*dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)&
&   +2*dgm01(1,3)*dgm10(2,3)+dgm01(1,2)*dgm10(3,3))+gm(2,2)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(1,2)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3)))
   cm(3,14)=2*(gm(3,3)*(dgm01(3,3)*dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)&
&   +2*dgm01(1,3)*dgm10(2,3)+dgm01(1,2)*dgm10(3,3))+gm(2,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(1,3)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3)))
   cm(1,15)=gm(1,1)*dgm01(3,3)*dgm10(3,3)+2*gm(1,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))
   cm(2,15)=gm(1,2)*dgm01(3,3)*dgm10(3,3)+2*gm(2,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))
   cm(3,15)=gm(1,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))
   cm(1,16)=gm(1,2)*dgm01(2,2)*dgm10(2,2)
   cm(2,16)=gm(2,2)*dgm01(2,2)*dgm10(2,2)
   cm(3,16)=gm(2,3)*dgm01(2,2)*dgm10(2,2)
   cm(1,17)=2*gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(1,2)*(dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)+dgm01(2,2)&
&   *dgm10(3,3))
   cm(2,17)=2*gm(2,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(2,2)*(dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)+dgm01(2,2)&
&   *dgm10(3,3))
   cm(3,17)=2*gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(2,3)*(dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)+dgm01(2,2)&
&   *dgm10(3,3))
   cm(1,18)=gm(1,3)*dgm01(2,2)*dgm10(2,2)+2*gm(1,2)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))
   cm(2,18)=gm(2,3)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))
   cm(3,18)=gm(3,3)*dgm01(2,2)*dgm10(2,2)+2*gm(2,3)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))
   cm(1,19)=gm(1,3)*(dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)&
&   +dgm01(2,2)*dgm10(3,3))+2*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))
   cm(2,19)=gm(2,3)*(dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)&
&   +dgm01(2,2)*dgm10(3,3))+2*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))
   cm(3,19)=gm(3,3)*(dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)&
&   +dgm01(2,2)*dgm10(3,3))+2*gm(2,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))
   cm(1,20)=gm(1,2)*dgm01(3,3)*dgm10(3,3)+2*gm(1,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))
   cm(2,20)=gm(2,2)*dgm01(3,3)*dgm10(3,3)+2*gm(2,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))
   cm(3,20)=gm(2,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))
   cm(1,21)=gm(1,3)*dgm01(3,3)*dgm10(3,3)
   cm(2,21)=gm(2,3)*dgm01(3,3)*dgm10(3,3)
   cm(3,21)=gm(3,3)*dgm01(3,3)*dgm10(3,3)
 elseif(rank==2)then
   cm(1,1)=gm(1,1)**2*dgm01(1,1)*dgm10(1,1)
   cm(2,1)=((6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,1)*dgm10(1,1))&
&   /4.d0
   cm(3,1)=((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,1)*dgm10(1,1))&
&   /4.d0
   cm(4,1)=((6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,1)*dgm10(1,1))&
&   /2.d0
   cm(5,1)=2*gm(1,1)*gm(1,3)*dgm01(1,1)*dgm10(1,1)
   cm(6,1)=2*gm(1,1)*gm(1,2)*dgm01(1,1)*dgm10(1,1)
   cm(1,2)=(12*(2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,1)+32*gm(1,1)&
&   *gm(1,2)*dgm01(1,2)+8*gm(1,1)**2*dgm01(2,2))*dgm10(1,1)+384*gm(1,1)&
&   *(gm(1,2)*dgm01(1,1)+gm(1,1)*dgm01(1,2))*dgm10(1,2)+96*gm(1,1)&
&   **2*dgm01(1,1)*dgm10(2,2))/96.d0
   cm(2,2)=(3*(8*gm(2,2)**2*dgm01(1,1)+32*gm(1,2)*gm(2,2)*dgm01(1,2)&
&   +2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,2))*dgm10(1,1)+4*(24*gm(1,2)&
&   *gm(2,2)*dgm01(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,2))&
&   *dgm10(1,2)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,1)*dgm10(2,2))&
&   /24.d0
   cm(3,2)=(6*((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(6*gm(1,3)&
&   *gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,2)+(6*gm(1,3)**2-2*gm(1,1)&
&   *gm(3,3))*dgm01(2,2))*dgm10(1,1)+4*(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)&
&   *gm(3,3))*dgm01(1,1)+6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,2))&
&   *dgm10(1,2)+6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,1)*dgm10(2,2))&
&   /24.d0
   cm(4,2)=(3*(8*gm(2,2)*gm(2,3)*dgm01(1,1)+8*(3*gm(1,3)*gm(2,2)&
&   +gm(1,2)*gm(2,3))*dgm01(1,2)+2*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(2,2))*dgm10(1,1)+4*(6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))&
&   *dgm01(1,1)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,2))&
&   *dgm10(1,2)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,1)&
&   *dgm10(2,2))/12.d0
   cm(5,2)=(12*(gm(2,3)*(12*gm(1,2)*dgm01(1,1)+24*gm(1,1)*dgm01(1,2))&
&   +gm(1,3)*(-4*gm(2,2)*dgm01(1,1)+8*(gm(1,2)*dgm01(1,2)+gm(1,1)&
&   *dgm01(2,2))))*dgm10(1,1)+16*(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(1,1)+24*gm(1,1)*gm(1,3)*dgm01(1,2))*dgm10(1,2)&
&   +96*gm(1,1)*gm(1,3)*dgm01(1,1)*dgm10(2,2))/48.d0
   cm(6,2)=(12*(8*gm(1,2)**2*dgm01(1,2)+24*gm(1,1)*gm(2,2)*dgm01(1,2)&
&   +8*gm(1,2)*(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2)))*dgm10(1,1)&
&   +16*(6*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(1,1)+24*gm(1,1)&
&   *gm(1,2)*dgm01(1,2))*dgm10(1,2)+96*gm(1,1)*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,2))/48.d0
   cm(1,3)=(12*(2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,1)+32*gm(1,1)&
&   *gm(1,3)*dgm01(1,3)+8*gm(1,1)**2*dgm01(3,3))*dgm10(1,1)+384*gm(1,1)&
&   *(gm(1,3)*dgm01(1,1)+gm(1,1)*dgm01(1,3))*dgm10(1,3)+96*gm(1,1)&
&   **2*dgm01(1,1)*dgm10(3,3))/96.d0
   cm(2,3)=(6*((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,3)+(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,1)+4*(6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*dgm01(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,3))&
&   *dgm10(1,3)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,1)*dgm10(3,3))&
&   /24.d0
   cm(3,3)=(3*(8*gm(3,3)**2*dgm01(1,1)+32*gm(1,3)*gm(3,3)*dgm01(1,3)&
&   +2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(3,3))*dgm10(1,1)+4*(24*gm(1,3)&
&   *gm(3,3)*dgm01(1,1)+6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3))&
&   *dgm10(1,3)+6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,1)*dgm10(3,3))&
&   /24.d0
   cm(4,3)=(3*(gm(2,3)*(8*gm(3,3)*dgm01(1,1)+8*gm(1,3)*dgm01(1,3)&
&   -4*gm(1,1)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(1,3)+12*gm(1,3)&
&   *dgm01(3,3)))*dgm10(1,1)+4*(6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))&
&   *dgm01(1,1)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,3))&
&   *dgm10(1,3)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,1)&
&   *dgm10(3,3))/12.d0
   cm(5,3)=(12*(8*gm(1,3)**2*dgm01(1,3)+24*gm(1,1)*gm(3,3)*dgm01(1,3)&
&   +8*gm(1,3)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))*dgm10(1,1)&
&   +16*(6*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(1,1)+24*gm(1,1)&
&   *gm(1,3)*dgm01(1,3))*dgm10(1,3)+96*gm(1,1)*gm(1,3)*dgm01(1,1)&
&   *dgm10(3,3))/48.d0
   cm(6,3)=(12*(2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)&
&   +8*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,3)+8*gm(1,1)&
&   *gm(1,2)*dgm01(3,3))*dgm10(1,1)+16*(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(1,1)+24*gm(1,1)*gm(1,2)*dgm01(1,3))*dgm10(1,3)&
&   +96*gm(1,1)*gm(1,2)*dgm01(1,1)*dgm10(3,3))/48.d0
   cm(1,4)=(12*(gm(1,2)*(12*gm(1,3)*dgm01(1,1)+16*gm(1,1)*dgm01(1,3))&
&   +gm(1,1)*(-4*gm(2,3)*dgm01(1,1)+16*gm(1,3)*dgm01(1,2)+8*gm(1,1)&
&   *dgm01(2,3)))*dgm10(1,1)+192*gm(1,1)*(gm(1,3)*dgm01(1,1)+gm(1,1)&
&   *dgm01(1,3))*dgm10(1,2)+192*gm(1,1)*(gm(1,2)*dgm01(1,1)+gm(1,1)&
&   *dgm01(1,2))*dgm10(1,3)+96*gm(1,1)**2*dgm01(1,1)*dgm10(2,3))&
&   /48.d0
   cm(2,4)=(3*(gm(2,2)*(8*gm(2,3)*dgm01(1,1)-8*gm(1,3)*dgm01(1,2)&
&   +16*gm(1,2)*dgm01(1,3)-4*gm(1,1)*dgm01(2,3))+gm(1,2)*(24*gm(2,3)&
&   *dgm01(1,2)+12*gm(1,2)*dgm01(2,3)))*dgm10(1,1)+2*(6*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,3))*dgm10(1,2)+2*(24*gm(1,2)*gm(2,2)*dgm01(1,1)&
&   +6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,2))*dgm10(1,3)+6*(6*gm(1,2)&
&   **2-2*gm(1,1)*gm(2,2))*dgm01(1,1)*dgm10(2,3))/12.d0
   cm(3,4)=(3*(8*gm(2,3)*gm(3,3)*dgm01(1,1)+16*gm(1,3)*gm(3,3)&
&   *dgm01(1,2)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   +2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,3))*dgm10(1,1)+2*(24*gm(1,3)&
&   *gm(3,3)*dgm01(1,1)+6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3))&
&   *dgm10(1,2)+2*(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)&
&   +6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,2))*dgm10(1,3)+6*(6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*dgm01(1,1)*dgm10(2,3))/12.d0
   cm(4,4)=(3*(2*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(3*gm(1,3)*gm(2,2)+gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+2*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(2,3))*dgm10(1,1)+2*(6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))&
&   *dgm01(1,1)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,3))&
&   *dgm10(1,2)+2*(6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,1)&
&   +6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,2))*dgm10(1,3)&
&   +6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,1)*dgm10(2,3))&
&   /6.d0
   cm(5,4)=(12*(2*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,1)&
&   +4*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(1,2)+4*(1*gm(1,2)*gm(1,3)&
&   +3*gm(1,1)*gm(2,3))*dgm01(1,3)+8*gm(1,1)*gm(1,3)*dgm01(2,3))&
&   *dgm10(1,1)+8*(6*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(1,1)&
&   +24*gm(1,1)*gm(1,3)*dgm01(1,3))*dgm10(1,2)+8*(6*(1*gm(1,2)*gm(1,3)&
&   +3*gm(1,1)*gm(2,3))*dgm01(1,1)+24*gm(1,1)*gm(1,3)*dgm01(1,2))&
&   *dgm10(1,3)+96*gm(1,1)*gm(1,3)*dgm01(1,1)*dgm10(2,3))/24.d0
   cm(6,4)=(12*(2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,1)&
&   +4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,2)+4*(1*gm(1,2)&
&   **2+3*gm(1,1)*gm(2,2))*dgm01(1,3)+8*gm(1,1)*gm(1,2)*dgm01(2,3))&
&   *dgm10(1,1)+8*(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,1)&
&   +24*gm(1,1)*gm(1,2)*dgm01(1,3))*dgm10(1,2)+8*(6*(1*gm(1,2)**2+3*gm(1,1)&
&   *gm(2,2))*dgm01(1,1)+24*gm(1,1)*gm(1,2)*dgm01(1,2))*dgm10(1,3)&
&   +96*gm(1,1)*gm(1,2)*dgm01(1,1)*dgm10(2,3))/24.d0
   cm(1,5)=2*gm(1,1)*(gm(1,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))
   cm(2,5)=((6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,1)&
&   +6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,3))*dgm10(1,1)+6*(6*gm(1,2)&
&   **2-2*gm(1,1)*gm(2,2))*dgm01(1,1)*dgm10(1,3))/12.d0
   cm(3,5)=((24*gm(1,3)*gm(3,3)*dgm01(1,1)+6*(6*gm(1,3)**2-2*gm(1,1)&
&   *gm(3,3))*dgm01(1,3))*dgm10(1,1)+6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))&
&   *dgm01(1,1)*dgm10(1,3))/12.d0
   cm(4,5)=((6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,1)&
&   +6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,3))*dgm10(1,1)&
&   +6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,1)*dgm10(1,3))&
&   /6.d0
   cm(5,5)=(20*(6*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(1,1)+24*gm(1,1)&
&   *gm(1,3)*dgm01(1,3))*dgm10(1,1)+480*gm(1,1)*gm(1,3)*dgm01(1,1)&
&   *dgm10(1,3))/120.d0
   cm(6,5)=(20*(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,1)&
&   +24*gm(1,1)*gm(1,2)*dgm01(1,3))*dgm10(1,1)+480*gm(1,1)*gm(1,2)&
&   *dgm01(1,1)*dgm10(1,3))/120.d0
   cm(1,6)=2*gm(1,1)*(gm(1,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)))
   cm(2,6)=((24*gm(1,2)*gm(2,2)*dgm01(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,2))*dgm10(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))&
&   *dgm01(1,1)*dgm10(1,2))/12.d0
   cm(3,6)=((6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)&
&   +6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,2))*dgm10(1,1)+6*(6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*dgm01(1,1)*dgm10(1,2))/12.d0
   cm(4,6)=((6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,1)+6*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,2))*dgm10(1,1)+6*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,1)*dgm10(1,2))/6.d0
   cm(5,6)=(20*(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,1)&
&   +24*gm(1,1)*gm(1,3)*dgm01(1,2))*dgm10(1,1)+480*gm(1,1)*gm(1,3)&
&   *dgm01(1,1)*dgm10(1,2))/120.d0
   cm(6,6)=(20*(6*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(1,1)+24*gm(1,1)&
&   *gm(1,2)*dgm01(1,2))*dgm10(1,1)+480*gm(1,1)*gm(1,2)*dgm01(1,1)&
&   *dgm10(1,2))/120.d0
   cm(1,7)=((6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,2)+24*gm(1,1)&
&   *gm(1,2)*dgm01(2,2))*dgm10(1,1)+3*(2*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,1)+32*gm(1,1)*gm(1,2)*dgm01(1,2)+8*gm(1,1)&
&   **2*dgm01(2,2))*dgm10(1,2)+24*gm(1,1)*(gm(1,2)*dgm01(1,1)+gm(1,1)&
&   *dgm01(1,2))*dgm10(2,2))/12.d0
   cm(2,7)=(24*gm(2,2)*(gm(2,2)*dgm01(1,2)+gm(1,2)*dgm01(2,2))&
&   *dgm10(1,1)+3*(8*gm(2,2)**2*dgm01(1,1)+32*gm(1,2)*gm(2,2)*dgm01(1,2)&
&   +2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,2))*dgm10(1,2)+(24*gm(1,2)&
&   *gm(2,2)*dgm01(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,2))&
&   *dgm10(2,2))/12.d0
   cm(3,7)=((6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,2)+6*(6*gm(1,3)&
&   *gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,2))*dgm10(1,1)+6*((6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)&
&   *gm(3,3))*dgm01(1,2)+(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,2))&
&   *dgm10(1,2)+(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)&
&   +6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,2))*dgm10(2,2))/12.d0
   cm(4,7)=((24*gm(2,2)*gm(2,3)*dgm01(1,2)+6*(3*gm(1,3)*gm(2,2)&
&   +gm(1,2)*gm(2,3))*dgm01(2,2))*dgm10(1,1)+3*(8*gm(2,2)*gm(2,3)&
&   *dgm01(1,1)+8*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,2)&
&   +2*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,2))*dgm10(1,2)&
&   +(6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,1)+6*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,2))*dgm10(2,2))/6.d0
   cm(5,7)=((6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,2)&
&   +6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,2))*dgm10(1,1)&
&   +3*(gm(2,3)*(12*gm(1,2)*dgm01(1,1)+24*gm(1,1)*dgm01(1,2))+gm(1,3)&
&   *(-4*gm(2,2)*dgm01(1,1)+8*(gm(1,2)*dgm01(1,2)+gm(1,1)*dgm01(2,2))))&
&   *dgm10(1,2)+(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,1)&
&   +24*gm(1,1)*gm(1,3)*dgm01(1,2))*dgm10(2,2))/6.d0
   cm(6,7)=((24*gm(1,2)*gm(2,2)*dgm01(1,2)+6*(1*gm(1,2)**2+3*gm(1,1)&
&   *gm(2,2))*dgm01(2,2))*dgm10(1,1)+3*(8*gm(1,2)**2*dgm01(1,2)+24*gm(1,1)&
&   *gm(2,2)*dgm01(1,2)+8*gm(1,2)*(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2)))&
&   *dgm10(1,2)+(6*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(1,1)+24*gm(1,1)&
&   *gm(1,2)*dgm01(1,2))*dgm10(2,2))/6.d0
   cm(1,8)=(6*(2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,2)+4*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,3)+16*gm(1,1)*gm(1,3)*dgm01(2,3)&
&   +8*gm(1,1)*gm(1,2)*dgm01(3,3))*dgm10(1,1)+6*(2*(6*gm(1,3)**2-2*gm(1,1)&
&   *gm(3,3))*dgm01(1,1)+32*gm(1,1)*gm(1,3)*dgm01(1,3)+8*gm(1,1)&
&   **2*dgm01(3,3))*dgm10(1,2)+12*(gm(1,2)*(12*gm(1,3)*dgm01(1,1)&
&   +16*gm(1,1)*dgm01(1,3))+gm(1,1)*(-4*gm(2,3)*dgm01(1,1)+16*gm(1,3)&
&   *dgm01(1,2)+8*gm(1,1)*dgm01(2,3)))*dgm10(1,3)+96*gm(1,1)*(gm(1,3)&
&   *dgm01(1,1)+gm(1,1)*dgm01(1,3))*dgm10(2,3)+48*gm(1,1)*(gm(1,2)&
&   *dgm01(1,1)+gm(1,1)*dgm01(1,2))*dgm10(3,3))/24.d0
   cm(2,8)=(3*(2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,2)+16*gm(2,2)&
&   *gm(2,3)*dgm01(1,3)+4*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))&
&   *dgm01(2,3)+8*gm(1,2)*gm(2,2)*dgm01(3,3))*dgm10(1,1)+6*((6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(1,2)+6*(gm(2,2)*(8*gm(2,3)*dgm01(1,1)-8*gm(1,3)*dgm01(1,2)&
&   +16*gm(1,2)*dgm01(1,3)-4*gm(1,1)*dgm01(2,3))+gm(1,2)*(24*gm(2,3)&
&   *dgm01(1,2)+12*gm(1,2)*dgm01(2,3)))*dgm10(1,3)+2*(6*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,3))*dgm10(2,3)+(24*gm(1,2)*gm(2,2)*dgm01(1,1)&
&   +6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,2))*dgm10(3,3))/12.d0
   cm(3,8)=(6*(8*gm(3,3)**2*dgm01(1,2)+12*gm(1,3)*gm(2,3)*dgm01(3,3)&
&   +gm(3,3)*(16*gm(2,3)*dgm01(1,3)+16*gm(1,3)*dgm01(2,3)-4*gm(1,2)&
&   *dgm01(3,3)))*dgm10(1,1)+6*(8*gm(3,3)**2*dgm01(1,1)+32*gm(1,3)&
&   *gm(3,3)*dgm01(1,3)+2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(3,3))&
&   *dgm10(1,2)+12*(8*gm(2,3)*gm(3,3)*dgm01(1,1)+16*gm(1,3)*gm(3,3)&
&   *dgm01(1,2)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   +2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,3))*dgm10(1,3)+4*(24*gm(1,3)&
&   *gm(3,3)*dgm01(1,1)+6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3))&
&   *dgm10(2,3)+2*(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)&
&   +6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,2))*dgm10(3,3))/24.d0
   cm(4,8)=(6*(8*gm(2,3)*gm(3,3)*dgm01(1,2)+4*(1*gm(2,3)**2+3*gm(2,2)&
&   *gm(3,3))*dgm01(1,3)+4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))&
&   *dgm01(2,3)+2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,1)+6*(gm(2,3)*(8*gm(3,3)*dgm01(1,1)+8*gm(1,3)*dgm01(1,3)&
&   -4*gm(1,1)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(1,3)+12*gm(1,3)&
&   *dgm01(3,3)))*dgm10(1,2)+12*(2*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))&
&   *dgm01(1,1)+4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,3)+2*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,3))*dgm10(1,3)+4*(6*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,1)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(1,3))*dgm10(2,3)+2*(6*(3*gm(1,3)*gm(2,2)+gm(1,2)&
&   *gm(2,3))*dgm01(1,1)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(1,2))*dgm10(3,3))/12.d0
   cm(5,8)=(6*(8*gm(1,3)*gm(3,3)*dgm01(1,2)+4*(1*gm(1,3)*gm(2,3)&
&   +3*gm(1,2)*gm(3,3))*dgm01(1,3)+4*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))&
&   *dgm01(2,3)+2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,1)+6*(8*gm(1,3)**2*dgm01(1,3)+24*gm(1,1)*gm(3,3)*dgm01(1,3)&
&   +8*gm(1,3)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))*dgm10(1,2)&
&   +12*(2*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,1)+4*(1*gm(1,3)&
&   **2+3*gm(1,1)*gm(3,3))*dgm01(1,2)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(1,3)+8*gm(1,1)*gm(1,3)*dgm01(2,3))*dgm10(1,3)&
&   +4*(6*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(1,1)+24*gm(1,1)&
&   *gm(1,3)*dgm01(1,3))*dgm10(2,3)+2*(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(1,1)+24*gm(1,1)*gm(1,3)*dgm01(1,2))*dgm10(3,3))&
&   /12.d0
   cm(6,8)=(6*(2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,3)+4*(1*gm(1,2)&
&   *gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,3)+2*(1*gm(1,2)**2+3*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,1)+6*(2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)&
&   *gm(3,3))*dgm01(1,1)+8*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))&
&   *dgm01(1,3)+8*gm(1,1)*gm(1,2)*dgm01(3,3))*dgm10(1,2)+12*(2*(3*gm(1,3)&
&   *gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,1)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(1,2)+4*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(1,3)&
&   +8*gm(1,1)*gm(1,2)*dgm01(2,3))*dgm10(1,3)+4*(6*(1*gm(1,2)*gm(1,3)&
&   +3*gm(1,1)*gm(2,3))*dgm01(1,1)+24*gm(1,1)*gm(1,2)*dgm01(1,3))&
&   *dgm10(2,3)+2*(6*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(1,1)&
&   +24*gm(1,1)*gm(1,2)*dgm01(1,2))*dgm10(3,3))/12.d0
   cm(1,9)=(6*(4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,2)&
&   +2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,3)+8*gm(1,1)*gm(1,3)&
&   *dgm01(2,2)+16*gm(1,1)*gm(1,2)*dgm01(2,3))*dgm10(1,1)+12*(gm(1,2)&
&   *(12*gm(1,3)*dgm01(1,1)+16*gm(1,1)*dgm01(1,3))+gm(1,1)*(-4*gm(2,3)&
&   *dgm01(1,1)+16*gm(1,3)*dgm01(1,2)+8*gm(1,1)*dgm01(2,3)))*dgm10(1,2)&
&   +6*(2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,1)+32*gm(1,1)&
&   *gm(1,2)*dgm01(1,2)+8*gm(1,1)**2*dgm01(2,2))*dgm10(1,3)+48*gm(1,1)&
&   *(gm(1,3)*dgm01(1,1)+gm(1,1)*dgm01(1,3))*dgm10(2,2)+96*gm(1,1)&
&   *(gm(1,2)*dgm01(1,1)+gm(1,1)*dgm01(1,2))*dgm10(2,3))/24.d0
   cm(2,9)=(6*(8*gm(2,2)**2*dgm01(1,3)+12*gm(1,2)*gm(2,3)*dgm01(2,2)&
&   +gm(2,2)*(16*gm(2,3)*dgm01(1,2)-4*gm(1,3)*dgm01(2,2)+16*gm(1,2)&
&   *dgm01(2,3)))*dgm10(1,1)+12*(gm(2,2)*(8*gm(2,3)*dgm01(1,1)-8*gm(1,3)&
&   *dgm01(1,2)+16*gm(1,2)*dgm01(1,3)-4*gm(1,1)*dgm01(2,3))+gm(1,2)&
&   *(24*gm(2,3)*dgm01(1,2)+12*gm(1,2)*dgm01(2,3)))*dgm10(1,2)+6*(8*gm(2,2)&
&   **2*dgm01(1,1)+32*gm(1,2)*gm(2,2)*dgm01(1,2)+2*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(2,2))*dgm10(1,3)+2*(6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*dgm01(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,3))&
&   *dgm10(2,2)+4*(24*gm(1,2)*gm(2,2)*dgm01(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,2))*dgm10(2,3))/24.d0
   cm(3,9)=(3*(16*gm(2,3)*gm(3,3)*dgm01(1,2)+2*(6*gm(2,3)**2-2*gm(2,2)&
&   *gm(3,3))*dgm01(1,3)+8*gm(1,3)*gm(3,3)*dgm01(2,2)+4*(6*gm(1,3)&
&   *gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3))*dgm10(1,1)+6*(8*gm(2,3)&
&   *gm(3,3)*dgm01(1,1)+16*gm(1,3)*gm(3,3)*dgm01(1,2)+4*(6*gm(1,3)&
&   *gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)+2*(6*gm(1,3)**2-2*gm(1,1)&
&   *gm(3,3))*dgm01(2,3))*dgm10(1,2)+6*((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *dgm01(1,1)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,2))*dgm10(1,3)+(24*gm(1,3)&
&   *gm(3,3)*dgm01(1,1)+6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3))&
&   *dgm10(2,2)+2*(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)&
&   +6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,2))*dgm10(2,3))/12.d0
   cm(4,9)=(6*(4*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(1,2)+8*gm(2,2)&
&   *gm(2,3)*dgm01(1,3)+2*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,2)&
&   +4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(1,1)&
&   +12*(2*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(3*gm(1,3)*gm(2,2)+gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+2*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(2,3))*dgm10(1,2)+6*(8*gm(2,2)*gm(2,3)*dgm01(1,1)+8*(3*gm(1,3)&
&   *gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,2)+2*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(2,2))*dgm10(1,3)+2*(6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)&
&   *gm(3,3))*dgm01(1,1)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(1,3))*dgm10(2,2)+4*(6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))&
&   *dgm01(1,1)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,2))&
&   *dgm10(2,3))/12.d0
   cm(5,9)=(6*(4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +2*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,3)+2*(1*gm(1,3)&
&   **2+3*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(2,3))*dgm10(1,1)+12*(2*(1*gm(1,3)*gm(2,3)+3*gm(1,2)&
&   *gm(3,3))*dgm01(1,1)+4*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(1,2)&
&   +4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,3)+8*gm(1,1)&
&   *gm(1,3)*dgm01(2,3))*dgm10(1,2)+6*(gm(2,3)*(12*gm(1,2)*dgm01(1,1)&
&   +24*gm(1,1)*dgm01(1,2))+gm(1,3)*(-4*gm(2,2)*dgm01(1,1)+8*(gm(1,2)&
&   *dgm01(1,2)+gm(1,1)*dgm01(2,2))))*dgm10(1,3)+2*(6*(1*gm(1,3)&
&   **2+3*gm(1,1)*gm(3,3))*dgm01(1,1)+24*gm(1,1)*gm(1,3)*dgm01(1,3))&
&   *dgm10(2,2)+4*(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,1)&
&   +24*gm(1,1)*gm(1,3)*dgm01(1,2))*dgm10(2,3))/12.d0
   cm(6,9)=(6*(4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,2)&
&   +8*gm(1,2)*gm(2,2)*dgm01(1,3)+2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(2,2)+4*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(2,3))&
&   *dgm10(1,1)+12*(2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,1)&
&   +4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,2)+4*(1*gm(1,2)&
&   **2+3*gm(1,1)*gm(2,2))*dgm01(1,3)+8*gm(1,1)*gm(1,2)*dgm01(2,3))&
&   *dgm10(1,2)+6*(8*gm(1,2)**2*dgm01(1,2)+24*gm(1,1)*gm(2,2)*dgm01(1,2)&
&   +8*gm(1,2)*(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2)))*dgm10(1,3)&
&   +2*(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,1)+24*gm(1,1)&
&   *gm(1,2)*dgm01(1,3))*dgm10(2,2)+4*(6*(1*gm(1,2)**2+3*gm(1,1)&
&   *gm(2,2))*dgm01(1,1)+24*gm(1,1)*gm(1,2)*dgm01(1,2))*dgm10(2,3))&
&   /12.d0
   cm(1,10)=((6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3)+24*gm(1,1)&
&   *gm(1,3)*dgm01(3,3))*dgm10(1,1)+3*(2*(6*gm(1,3)**2-2*gm(1,1)&
&   *gm(3,3))*dgm01(1,1)+32*gm(1,1)*gm(1,3)*dgm01(1,3)+8*gm(1,1)&
&   **2*dgm01(3,3))*dgm10(1,3)+24*gm(1,1)*(gm(1,3)*dgm01(1,1)+gm(1,1)&
&   *dgm01(1,3))*dgm10(3,3))/12.d0
   cm(2,10)=((6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,3)+6*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,1)+6*((6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(1,3)+(6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,1)&
&   +6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,3))*dgm10(3,3))/12.d0
   cm(3,10)=(24*gm(3,3)*(gm(3,3)*dgm01(1,3)+gm(1,3)*dgm01(3,3))&
&   *dgm10(1,1)+3*(8*gm(3,3)**2*dgm01(1,1)+32*gm(1,3)*gm(3,3)*dgm01(1,3)&
&   +2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(3,3))*dgm10(1,3)+(24*gm(1,3)&
&   *gm(3,3)*dgm01(1,1)+6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3))&
&   *dgm10(3,3))/12.d0
   cm(4,10)=((24*gm(2,3)*gm(3,3)*dgm01(1,3)+6*(1*gm(1,3)*gm(2,3)&
&   +3*gm(1,2)*gm(3,3))*dgm01(3,3))*dgm10(1,1)+3*(gm(2,3)*(8*gm(3,3)&
&   *dgm01(1,1)+8*gm(1,3)*dgm01(1,3)-4*gm(1,1)*dgm01(3,3))+gm(1,2)&
&   *(24*gm(3,3)*dgm01(1,3)+12*gm(1,3)*dgm01(3,3)))*dgm10(1,3)+(6*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,1)+6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(1,3))*dgm10(3,3))/6.d0
   cm(5,10)=((24*gm(1,3)*gm(3,3)*dgm01(1,3)+6*(1*gm(1,3)**2+3*gm(1,1)&
&   *gm(3,3))*dgm01(3,3))*dgm10(1,1)+3*(8*gm(1,3)**2*dgm01(1,3)+24*gm(1,1)&
&   *gm(3,3)*dgm01(1,3)+8*gm(1,3)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))&
&   *dgm10(1,3)+(6*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(1,1)+24*gm(1,1)&
&   *gm(1,3)*dgm01(1,3))*dgm10(3,3))/6.d0
   cm(6,10)=((6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   +6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(3,3))*dgm10(1,1)&
&   +3*(2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)+8*(1*gm(1,2)&
&   *gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,3)+8*gm(1,1)*gm(1,2)*dgm01(3,3))&
&   *dgm10(1,3)+(6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,1)&
&   +24*gm(1,1)*gm(1,2)*dgm01(1,3))*dgm10(3,3))/6.d0
   cm(1,11)=(6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(1,1)&
&   +4*(6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,2)+24*gm(1,1)&
&   *gm(1,2)*dgm01(2,2))*dgm10(1,2)+3*(2*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,1)+32*gm(1,1)*gm(1,2)*dgm01(1,2)+8*gm(1,1)&
&   **2*dgm01(2,2))*dgm10(2,2))/24.d0
   cm(2,11)=(96*gm(2,2)**2*dgm01(2,2)*dgm10(1,1)+384*gm(2,2)*(gm(2,2)&
&   *dgm01(1,2)+gm(1,2)*dgm01(2,2))*dgm10(1,2)+12*(8*gm(2,2)**2*dgm01(1,1)&
&   +32*gm(1,2)*gm(2,2)*dgm01(1,2)+2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))&
&   *dgm01(2,2))*dgm10(2,2))/96.d0
   cm(3,11)=(6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,2)*dgm10(1,1)&
&   +4*(6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,2)+6*(6*gm(1,3)&
&   *gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,2))*dgm10(1,2)+6*((6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)&
&   *gm(3,3))*dgm01(1,2)+(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,2))&
&   *dgm10(2,2))/24.d0
   cm(4,11)=(96*gm(2,2)*gm(2,3)*dgm01(2,2)*dgm10(1,1)+16*(24*gm(2,2)&
&   *gm(2,3)*dgm01(1,2)+6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,2))&
&   *dgm10(1,2)+12*(8*gm(2,2)*gm(2,3)*dgm01(1,1)+8*(3*gm(1,3)*gm(2,2)&
&   +gm(1,2)*gm(2,3))*dgm01(1,2)+2*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(2,2))*dgm10(2,2))/48.d0
   cm(5,11)=(6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,2)&
&   *dgm10(1,1)+4*(6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,2)&
&   +6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,2))*dgm10(1,2)&
&   +3*(gm(2,3)*(12*gm(1,2)*dgm01(1,1)+24*gm(1,1)*dgm01(1,2))+gm(1,3)&
&   *(-4*gm(2,2)*dgm01(1,1)+8*(gm(1,2)*dgm01(1,2)+gm(1,1)*dgm01(2,2))))&
&   *dgm10(2,2))/12.d0
   cm(6,11)=(96*gm(1,2)*gm(2,2)*dgm01(2,2)*dgm10(1,1)+16*(24*gm(1,2)&
&   *gm(2,2)*dgm01(1,2)+6*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(2,2))&
&   *dgm10(1,2)+12*(8*gm(1,2)**2*dgm01(1,2)+24*gm(1,1)*gm(2,2)*dgm01(1,2)&
&   +8*gm(1,2)*(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2)))*dgm10(2,2))&
&   /48.d0
   cm(1,12)=(2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,2)*dgm10(1,1)&
&   +8*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,3)*dgm10(1,1)&
&   +2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(3,3)*dgm10(1,1)+4*(2*(6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*dgm01(1,2)+4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(1,3)+16*gm(1,1)*gm(1,3)*dgm01(2,3)+8*gm(1,1)&
&   *gm(1,2)*dgm01(3,3))*dgm10(1,2)+4*(4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(1,2)+2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,3)&
&   +8*gm(1,1)*gm(1,3)*dgm01(2,2)+16*gm(1,1)*gm(1,2)*dgm01(2,3))&
&   *dgm10(1,3)+(2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,1)+32*gm(1,1)&
&   *gm(1,3)*dgm01(1,3)+8*gm(1,1)**2*dgm01(3,3))*dgm10(2,2)+4*(gm(1,2)&
&   *(12*gm(1,3)*dgm01(1,1)+16*gm(1,1)*dgm01(1,3))+gm(1,1)*(-4*gm(2,3)&
&   *dgm01(1,1)+16*gm(1,3)*dgm01(1,2)+8*gm(1,1)*dgm01(2,3)))*dgm10(2,3)&
&   +(2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,1)+32*gm(1,1)*gm(1,2)&
&   *dgm01(1,2)+8*gm(1,1)**2*dgm01(2,2))*dgm10(3,3))/8.d0
   cm(2,12)=(4*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,2)*dgm10(1,1)&
&   +64*gm(2,2)*gm(2,3)*dgm01(2,3)*dgm10(1,1)+16*gm(2,2)**2*dgm01(3,3)&
&   *dgm10(1,1)+8*(2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +16*gm(2,2)*gm(2,3)*dgm01(1,3)+4*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*dgm01(2,3)+8*gm(1,2)*gm(2,2)*dgm01(3,3))*dgm10(1,2)&
&   +8*(8*gm(2,2)**2*dgm01(1,3)+12*gm(1,2)*gm(2,3)*dgm01(2,2)+gm(2,2)&
&   *(16*gm(2,3)*dgm01(1,2)-4*gm(1,3)*dgm01(2,2)+16*gm(1,2)*dgm01(2,3)))&
&   *dgm10(1,3)+4*((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,3)+(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(2,2)+8*(gm(2,2)*(8*gm(2,3)*dgm01(1,1)&
&   -8*gm(1,3)*dgm01(1,2)+16*gm(1,2)*dgm01(1,3)-4*gm(1,1)*dgm01(2,3))&
&   +gm(1,2)*(24*gm(2,3)*dgm01(1,2)+12*gm(1,2)*dgm01(2,3)))*dgm10(2,3)&
&   +2*(8*gm(2,2)**2*dgm01(1,1)+32*gm(1,2)*gm(2,2)*dgm01(1,2)+2*(6*gm(1,2)&
&   **2-2*gm(1,1)*gm(2,2))*dgm01(2,2))*dgm10(3,3))/16.d0
   cm(3,12)=(16*gm(3,3)**2*dgm01(2,2)*dgm10(1,1)+64*gm(2,3)*gm(3,3)&
&   *dgm01(2,3)*dgm10(1,1)+4*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+8*(8*gm(3,3)**2*dgm01(1,2)+12*gm(1,3)*gm(2,3)*dgm01(3,3)&
&   +gm(3,3)*(16*gm(2,3)*dgm01(1,3)+16*gm(1,3)*dgm01(2,3)-4*gm(1,2)&
&   *dgm01(3,3)))*dgm10(1,2)+8*(16*gm(2,3)*gm(3,3)*dgm01(1,2)+2*(6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(1,3)+8*gm(1,3)*gm(3,3)*dgm01(2,2)&
&   +4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3))*dgm10(1,3)&
&   +2*(8*gm(3,3)**2*dgm01(1,1)+32*gm(1,3)*gm(3,3)*dgm01(1,3)+2*(6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*dgm01(3,3))*dgm10(2,2)+8*(8*gm(2,3)*gm(3,3)&
&   *dgm01(1,1)+16*gm(1,3)*gm(3,3)*dgm01(1,2)+4*(6*gm(1,3)*gm(2,3)&
&   -2*gm(1,2)*gm(3,3))*dgm01(1,3)+2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))&
&   *dgm01(2,3))*dgm10(2,3)+4*((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,1)&
&   +4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,2)+(6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*dgm01(2,2))*dgm10(3,3))/16.d0
   cm(4,12)=(16*gm(2,3)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+16*(1*gm(2,3)&
&   **2+3*gm(2,2)*gm(3,3))*dgm01(2,3)*dgm10(1,1)+16*gm(2,2)*gm(2,3)&
&   *dgm01(3,3)*dgm10(1,1)+8*(8*gm(2,3)*gm(3,3)*dgm01(1,2)+4*(1*gm(2,3)&
&   **2+3*gm(2,2)*gm(3,3))*dgm01(1,3)+4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)&
&   *gm(3,3))*dgm01(2,3)+2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,2)+8*(4*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +8*gm(2,2)*gm(2,3)*dgm01(1,3)+2*(1*gm(1,3)*gm(2,3)+3*gm(1,2)&
&   *gm(3,3))*dgm01(2,2)+4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,3))&
&   *dgm10(1,3)+2*(gm(2,3)*(8*gm(3,3)*dgm01(1,1)+8*gm(1,3)*dgm01(1,3)&
&   -4*gm(1,1)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(1,3)+12*gm(1,3)&
&   *dgm01(3,3)))*dgm10(2,2)+8*(2*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))&
&   *dgm01(1,1)+4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,3)+2*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,3))*dgm10(2,3)+2*(8*gm(2,2)&
&   *gm(2,3)*dgm01(1,1)+8*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,2)&
&   +2*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,2))*dgm10(3,3))&
&   /8.d0
   cm(5,12)=(16*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+16*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,3)*dgm10(1,1)+4*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(3,3)*dgm10(1,1)+8*(8*gm(1,3)&
&   *gm(3,3)*dgm01(1,2)+4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   +4*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(2,3)+2*(1*gm(1,2)*gm(1,3)&
&   +3*gm(1,1)*gm(2,3))*dgm01(3,3))*dgm10(1,2)+8*(4*(1*gm(1,3)*gm(2,3)&
&   +3*gm(1,2)*gm(3,3))*dgm01(1,2)+2*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+2*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(2,2)&
&   +4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,3))*dgm10(1,3)&
&   +2*(8*gm(1,3)**2*dgm01(1,3)+24*gm(1,1)*gm(3,3)*dgm01(1,3)+8*gm(1,3)&
&   *(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))*dgm10(2,2)+8*(2*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,1)+4*(1*gm(1,3)**2+3*gm(1,1)&
&   *gm(3,3))*dgm01(1,2)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))&
&   *dgm01(1,3)+8*gm(1,1)*gm(1,3)*dgm01(2,3))*dgm10(2,3)+2*(gm(2,3)&
&   *(12*gm(1,2)*dgm01(1,1)+24*gm(1,1)*dgm01(1,2))+gm(1,3)*(-4*gm(2,2)&
&   *dgm01(1,1)+8*(gm(1,2)*dgm01(1,2)+gm(1,1)*dgm01(2,2))))*dgm10(3,3))&
&   /8.d0
   cm(6,12)=(4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+16*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,3)&
&   *dgm10(1,1)+16*gm(1,2)*gm(2,2)*dgm01(3,3)*dgm10(1,1)+8*(2*(6*gm(1,3)&
&   *gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(3*gm(1,3)*gm(2,2)+gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))&
&   *dgm01(2,3)+2*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(1,2)&
&   +8*(4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,2)+8*gm(1,2)&
&   *gm(2,2)*dgm01(1,3)+2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,2)&
&   +4*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(1,3)+2*(2*(6*gm(1,3)&
&   *gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)+8*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(1,3)+8*gm(1,1)*gm(1,2)*dgm01(3,3))*dgm10(2,2)&
&   +8*(2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,1)+4*(1*gm(1,2)&
&   *gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,2)+4*(1*gm(1,2)**2+3*gm(1,1)&
&   *gm(2,2))*dgm01(1,3)+8*gm(1,1)*gm(1,2)*dgm01(2,3))*dgm10(2,3)&
&   +2*(8*gm(1,2)**2*dgm01(1,2)+24*gm(1,1)*gm(2,2)*dgm01(1,2)+8*gm(1,2)&
&   *(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2)))*dgm10(3,3))/8.d0
   cm(1,13)=(6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,2)&
&   *dgm10(1,1)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,3)*dgm10(1,1)&
&   +6*(4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,2)+2*(6*gm(1,2)&
&   **2-2*gm(1,1)*gm(2,2))*dgm01(1,3)+8*gm(1,1)*gm(1,3)*dgm01(2,2)&
&   +16*gm(1,1)*gm(1,2)*dgm01(2,3))*dgm10(1,2)+2*(6*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,2)+24*gm(1,1)*gm(1,2)*dgm01(2,2))*dgm10(1,3)&
&   +3*(gm(1,2)*(12*gm(1,3)*dgm01(1,1)+16*gm(1,1)*dgm01(1,3))+gm(1,1)&
&   *(-4*gm(2,3)*dgm01(1,1)+16*gm(1,3)*dgm01(1,2)+8*gm(1,1)*dgm01(2,3)))&
&   *dgm10(2,2)+3*(2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,1)&
&   +32*gm(1,1)*gm(1,2)*dgm01(1,2)+8*gm(1,1)**2*dgm01(2,2))*dgm10(2,3))&
&   /12.d0
   cm(2,13)=(48*gm(2,2)*gm(2,3)*dgm01(2,2)*dgm10(1,1)+48*gm(2,2)&
&   **2*dgm01(2,3)*dgm10(1,1)+12*(8*gm(2,2)**2*dgm01(1,3)+12*gm(1,2)&
&   *gm(2,3)*dgm01(2,2)+gm(2,2)*(16*gm(2,3)*dgm01(1,2)-4*gm(1,3)&
&   *dgm01(2,2)+16*gm(1,2)*dgm01(2,3)))*dgm10(1,2)+96*gm(2,2)*(gm(2,2)&
&   *dgm01(1,2)+gm(1,2)*dgm01(2,2))*dgm10(1,3)+6*(gm(2,2)*(8*gm(2,3)&
&   *dgm01(1,1)-8*gm(1,3)*dgm01(1,2)+16*gm(1,2)*dgm01(1,3)-4*gm(1,1)&
&   *dgm01(2,3))+gm(1,2)*(24*gm(2,3)*dgm01(1,2)+12*gm(1,2)*dgm01(2,3)))&
&   *dgm10(2,2)+6*(8*gm(2,2)**2*dgm01(1,1)+32*gm(1,2)*gm(2,2)*dgm01(1,2)&
&   +2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,2))*dgm10(2,3))/24.d0
   cm(3,13)=(48*gm(2,3)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+12*(6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(2,3)*dgm10(1,1)+12*(16*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,3)+8*gm(1,3)&
&   *gm(3,3)*dgm01(2,2)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3))&
&   *dgm10(1,2)+4*(6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,2))*dgm10(1,3)&
&   +6*(8*gm(2,3)*gm(3,3)*dgm01(1,1)+16*gm(1,3)*gm(3,3)*dgm01(1,2)&
&   +4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)+2*(6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*dgm01(2,3))*dgm10(2,2)+12*((6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)&
&   *gm(3,3))*dgm01(1,2)+(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,2))&
&   *dgm10(2,3))/24.d0
   cm(4,13)=(12*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(2,2)*dgm10(1,1)&
&   +48*gm(2,2)*gm(2,3)*dgm01(2,3)*dgm10(1,1)+12*(4*(1*gm(2,3)**2+3*gm(2,2)&
&   *gm(3,3))*dgm01(1,2)+8*gm(2,2)*gm(2,3)*dgm01(1,3)+2*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,2)+4*(3*gm(1,3)*gm(2,2)+gm(1,2)&
&   *gm(2,3))*dgm01(2,3))*dgm10(1,2)+4*(24*gm(2,2)*gm(2,3)*dgm01(1,2)&
&   +6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,2))*dgm10(1,3)&
&   +6*(2*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(3*gm(1,3)*gm(2,2)+gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+2*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(2,3))*dgm10(2,2)+6*(8*gm(2,2)*gm(2,3)*dgm01(1,1)+8*(3*gm(1,3)&
&   *gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,2)+2*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(2,2))*dgm10(2,3))/12.d0
   cm(5,13)=(6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,3)&
&   *dgm10(1,1)+6*(4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +2*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,3)+2*(1*gm(1,3)&
&   **2+3*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(2,3))*dgm10(1,2)+2*(6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*dgm01(1,2)+6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))&
&   *dgm01(2,2))*dgm10(1,3)+3*(2*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))&
&   *dgm01(1,1)+4*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(1,2)+4*(1*gm(1,2)&
&   *gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,3)+8*gm(1,1)*gm(1,3)*dgm01(2,3))&
&   *dgm10(2,2)+3*(gm(2,3)*(12*gm(1,2)*dgm01(1,1)+24*gm(1,1)*dgm01(1,2))&
&   +gm(1,3)*(-4*gm(2,2)*dgm01(1,1)+8*(gm(1,2)*dgm01(1,2)+gm(1,1)&
&   *dgm01(2,2))))*dgm10(2,3))/6.d0
   cm(6,13)=(12*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,2)&
&   *dgm10(1,1)+48*gm(1,2)*gm(2,2)*dgm01(2,3)*dgm10(1,1)+12*(4*(3*gm(1,3)&
&   *gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,2)+8*gm(1,2)*gm(2,2)*dgm01(1,3)&
&   +2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,2)+4*(1*gm(1,2)&
&   **2+3*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(1,2)+4*(24*gm(1,2)*gm(2,2)&
&   *dgm01(1,2)+6*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(2,2))*dgm10(1,3)&
&   +6*(2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,1)+4*(1*gm(1,2)&
&   *gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,2)+4*(1*gm(1,2)**2+3*gm(1,1)&
&   *gm(2,2))*dgm01(1,3)+8*gm(1,1)*gm(1,2)*dgm01(2,3))*dgm10(2,2)&
&   +6*(8*gm(1,2)**2*dgm01(1,2)+24*gm(1,1)*gm(2,2)*dgm01(1,2)+8*gm(1,2)&
&   *(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2)))*dgm10(2,3))/12.d0
   cm(1,14)=(6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,3)*dgm10(1,1)&
&   +6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(3,3)*dgm10(1,1)&
&   +2*(6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3)+24*gm(1,1)&
&   *gm(1,3)*dgm01(3,3))*dgm10(1,2)+6*(2*(6*gm(1,3)**2-2*gm(1,1)&
&   *gm(3,3))*dgm01(1,2)+4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(1,3)+16*gm(1,1)*gm(1,3)*dgm01(2,3)+8*gm(1,1)*gm(1,2)*dgm01(3,3))&
&   *dgm10(1,3)+3*(2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,1)&
&   +32*gm(1,1)*gm(1,3)*dgm01(1,3)+8*gm(1,1)**2*dgm01(3,3))*dgm10(2,3)&
&   +3*(gm(1,2)*(12*gm(1,3)*dgm01(1,1)+16*gm(1,1)*dgm01(1,3))+gm(1,1)&
&   *(-4*gm(2,3)*dgm01(1,1)+16*gm(1,3)*dgm01(1,2)+8*gm(1,1)*dgm01(2,3)))&
&   *dgm10(3,3))/12.d0
   cm(2,14)=(12*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,3)*dgm10(1,1)&
&   +48*gm(2,2)*gm(2,3)*dgm01(3,3)*dgm10(1,1)+4*(6*(6*gm(2,3)**2-2*gm(2,2)&
&   *gm(3,3))*dgm01(1,3)+6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(1,2)+12*(2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *dgm01(1,2)+16*gm(2,2)*gm(2,3)*dgm01(1,3)+4*(-2*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)*gm(2,3))*dgm01(2,3)+8*gm(1,2)*gm(2,2)*dgm01(3,3))&
&   *dgm10(1,3)+12*((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,3)+(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(2,3)+6*(gm(2,2)*(8*gm(2,3)*dgm01(1,1)&
&   -8*gm(1,3)*dgm01(1,2)+16*gm(1,2)*dgm01(1,3)-4*gm(1,1)*dgm01(2,3))&
&   +gm(1,2)*(24*gm(2,3)*dgm01(1,2)+12*gm(1,2)*dgm01(2,3)))*dgm10(3,3))&
&   /24.d0
   cm(3,14)=(48*gm(3,3)**2*dgm01(2,3)*dgm10(1,1)+48*gm(2,3)*gm(3,3)&
&   *dgm01(3,3)*dgm10(1,1)+96*gm(3,3)*(gm(3,3)*dgm01(1,3)+gm(1,3)&
&   *dgm01(3,3))*dgm10(1,2)+12*(8*gm(3,3)**2*dgm01(1,2)+12*gm(1,3)&
&   *gm(2,3)*dgm01(3,3)+gm(3,3)*(16*gm(2,3)*dgm01(1,3)+16*gm(1,3)&
&   *dgm01(2,3)-4*gm(1,2)*dgm01(3,3)))*dgm10(1,3)+6*(8*gm(3,3)**2*dgm01(1,1)&
&   +32*gm(1,3)*gm(3,3)*dgm01(1,3)+2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))&
&   *dgm01(3,3))*dgm10(2,3)+6*(8*gm(2,3)*gm(3,3)*dgm01(1,1)+16*gm(1,3)&
&   *gm(3,3)*dgm01(1,2)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   +2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,3))*dgm10(3,3))/24.d0
   cm(4,14)=(48*gm(2,3)*gm(3,3)*dgm01(2,3)*dgm10(1,1)+12*(1*gm(2,3)&
&   **2+3*gm(2,2)*gm(3,3))*dgm01(3,3)*dgm10(1,1)+4*(24*gm(2,3)*gm(3,3)&
&   *dgm01(1,3)+6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(3,3))&
&   *dgm10(1,2)+12*(8*gm(2,3)*gm(3,3)*dgm01(1,2)+4*(1*gm(2,3)**2+3*gm(2,2)&
&   *gm(3,3))*dgm01(1,3)+4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))&
&   *dgm01(2,3)+2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,3)+6*(gm(2,3)*(8*gm(3,3)*dgm01(1,1)+8*gm(1,3)*dgm01(1,3)&
&   -4*gm(1,1)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(1,3)+12*gm(1,3)&
&   *dgm01(3,3)))*dgm10(2,3)+6*(2*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))&
&   *dgm01(1,1)+4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,3)+2*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,3))*dgm10(3,3))/12.d0
   cm(5,14)=(48*gm(1,3)*gm(3,3)*dgm01(2,3)*dgm10(1,1)+12*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(3,3)*dgm10(1,1)+4*(24*gm(1,3)&
&   *gm(3,3)*dgm01(1,3)+6*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(3,3))&
&   *dgm10(1,2)+12*(8*gm(1,3)*gm(3,3)*dgm01(1,2)+4*(1*gm(1,3)*gm(2,3)&
&   +3*gm(1,2)*gm(3,3))*dgm01(1,3)+4*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))&
&   *dgm01(2,3)+2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,3)+6*(8*gm(1,3)**2*dgm01(1,3)+24*gm(1,1)*gm(3,3)*dgm01(1,3)&
&   +8*gm(1,3)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))*dgm10(2,3)&
&   +6*(2*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,1)+4*(1*gm(1,3)&
&   **2+3*gm(1,1)*gm(3,3))*dgm01(1,2)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(1,3)+8*gm(1,1)*gm(1,3)*dgm01(2,3))*dgm10(3,3))&
&   /12.d0
   cm(6,14)=(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3)&
&   *dgm10(1,1)+2*(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   +6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(3,3))*dgm10(1,2)&
&   +6*(2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(3*gm(1,3)&
&   *gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,3)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(2,3)+2*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(1,3)+3*(2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)&
&   +8*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,3)+8*gm(1,1)&
&   *gm(1,2)*dgm01(3,3))*dgm10(2,3)+3*(2*(3*gm(1,3)*gm(2,2)+gm(1,2)&
&   *gm(2,3))*dgm01(1,1)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))&
&   *dgm01(1,2)+4*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(1,3)+8*gm(1,1)&
&   *gm(1,2)*dgm01(2,3))*dgm10(3,3))/6.d0
   cm(1,15)=(6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(3,3)*dgm10(1,1)&
&   +4*(6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3)+24*gm(1,1)&
&   *gm(1,3)*dgm01(3,3))*dgm10(1,3)+3*(2*(6*gm(1,3)**2-2*gm(1,1)&
&   *gm(3,3))*dgm01(1,1)+32*gm(1,1)*gm(1,3)*dgm01(1,3)+8*gm(1,1)&
&   **2*dgm01(3,3))*dgm10(3,3))/24.d0
   cm(2,15)=(6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3)*dgm10(1,1)&
&   +4*(6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,3)+6*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,3)+6*((6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(1,1)+4*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(3,3))/24.d0
   cm(3,15)=(96*gm(3,3)**2*dgm01(3,3)*dgm10(1,1)+384*gm(3,3)*(gm(3,3)&
&   *dgm01(1,3)+gm(1,3)*dgm01(3,3))*dgm10(1,3)+12*(8*gm(3,3)**2*dgm01(1,1)&
&   +32*gm(1,3)*gm(3,3)*dgm01(1,3)+2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))&
&   *dgm01(3,3))*dgm10(3,3))/96.d0
   cm(4,15)=(96*gm(2,3)*gm(3,3)*dgm01(3,3)*dgm10(1,1)+16*(24*gm(2,3)&
&   *gm(3,3)*dgm01(1,3)+6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(3,3))&
&   *dgm10(1,3)+12*(gm(2,3)*(8*gm(3,3)*dgm01(1,1)+8*gm(1,3)*dgm01(1,3)&
&   -4*gm(1,1)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(1,3)+12*gm(1,3)&
&   *dgm01(3,3)))*dgm10(3,3))/48.d0
   cm(5,15)=(96*gm(1,3)*gm(3,3)*dgm01(3,3)*dgm10(1,1)+16*(24*gm(1,3)&
&   *gm(3,3)*dgm01(1,3)+6*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(3,3))&
&   *dgm10(1,3)+12*(8*gm(1,3)**2*dgm01(1,3)+24*gm(1,1)*gm(3,3)*dgm01(1,3)&
&   +8*gm(1,3)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))*dgm10(3,3))&
&   /48.d0
   cm(6,15)=(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+4*(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   +6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(3,3))*dgm10(1,3)&
&   +3*(2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,1)+8*(1*gm(1,2)&
&   *gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(1,3)+8*gm(1,1)*gm(1,2)*dgm01(3,3))&
&   *dgm10(3,3))/12.d0
   cm(1,16)=(120*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(1,2)&
&   +120*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,2)*dgm10(2,2)+480*gm(1,1)&
&   *gm(1,2)*dgm01(2,2)*dgm10(2,2))/240.d0
   cm(2,16)=2*gm(2,2)*(gm(1,2)*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
   cm(3,16)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,2)*dgm10(1,2)&
&   +(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,2)*dgm10(2,2)+(6*gm(1,3)&
&   *gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,2)*dgm10(2,2))/2.d0
   cm(4,16)=(120*gm(1,2)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)&
&   *(360*gm(1,3)*dgm01(2,2)*dgm10(2,2)+480*gm(2,3)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2))))/120.d0
   cm(5,16)=(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(1,2)&
&   +(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,2)*dgm10(2,2)&
&   +(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,2)*dgm10(2,2)
   cm(6,16)=(480*gm(1,2)*gm(2,2)*dgm01(2,2)*dgm10(1,2)+480*gm(1,2)&
&   *gm(2,2)*dgm01(1,2)*dgm10(2,2)+120*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))&
&   *dgm01(2,2)*dgm10(2,2))/120.d0
   cm(1,17)=(12*((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,2)+4*(6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(2,2)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,3))&
&   *dgm10(1,3)+6*(2*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,2)&
&   +4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,3))*dgm10(2,2)&
&   +6*gm(1,1)*(16*gm(1,3)*dgm01(2,3)+8*gm(1,2)*dgm01(3,3))*dgm10(2,2)&
&   +12*(4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,2)+2*(6*gm(1,2)&
&   **2-2*gm(1,1)*gm(2,2))*dgm01(1,3))*dgm10(2,3)+12*gm(1,1)*(8*gm(1,3)&
&   *dgm01(2,2)+16*gm(1,2)*dgm01(2,3))*dgm10(2,3)+12*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,2)*dgm10(3,3)+48*gm(1,1)*gm(1,2)*dgm01(2,2)&
&   *dgm10(3,3))/24.d0
   cm(2,17)=(6*(2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,2)+32*gm(2,2)&
&   *gm(2,3)*dgm01(2,3)+8*gm(2,2)**2*dgm01(3,3))*dgm10(1,2)+96*gm(2,2)&
&   *(gm(2,3)*dgm01(2,2)+gm(2,2)*dgm01(2,3))*dgm10(1,3)+6*(2*(6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(1,2)+16*gm(2,2)*gm(2,3)*dgm01(1,3))&
&   *dgm10(2,2)+6*(4*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,3)&
&   +8*gm(1,2)*gm(2,2)*dgm01(3,3))*dgm10(2,2)+12*gm(2,2)*(16*gm(2,3)&
&   *dgm01(1,2)+8*gm(2,2)*dgm01(1,3))*dgm10(2,3)+12*(2*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,2)+16*gm(1,2)*gm(2,2)*dgm01(2,3))&
&   *dgm10(2,3)+48*gm(2,2)**2*dgm01(1,2)*dgm10(3,3)+48*gm(1,2)*gm(2,2)&
&   *dgm01(2,2)*dgm10(3,3))/24.d0
   cm(3,17)=(3*(8*gm(3,3)**2*dgm01(2,2)+32*gm(2,3)*gm(3,3)*dgm01(2,3)&
&   +2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3))*dgm10(1,2)+2*(24*gm(2,3)&
&   *gm(3,3)*dgm01(2,2)+6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,3))&
&   *dgm10(1,3)+3*gm(3,3)*(8*gm(3,3)*dgm01(1,2)+16*gm(2,3)*dgm01(1,3))&
&   *dgm10(2,2)+3*(16*gm(1,3)*gm(3,3)*dgm01(2,3)+2*(6*gm(1,3)*gm(2,3)&
&   -2*gm(1,2)*gm(3,3))*dgm01(3,3))*dgm10(2,2)+6*(16*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,3))*dgm10(2,3)&
&   +6*(8*gm(1,3)*gm(3,3)*dgm01(2,2)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)&
&   *gm(3,3))*dgm01(2,3))*dgm10(2,3)+6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *dgm01(1,2)*dgm10(3,3)+6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))&
&   *dgm01(2,2)*dgm10(3,3))/12.d0
   cm(4,17)=(6*(8*gm(2,3)**2*dgm01(2,3)+24*gm(2,2)*gm(3,3)*dgm01(2,3)&
&   +8*gm(2,3)*(gm(3,3)*dgm01(2,2)+gm(2,2)*dgm01(3,3)))*dgm10(1,2)&
&   +4*(6*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(2,2)+24*gm(2,2)&
&   *gm(2,3)*dgm01(2,3))*dgm10(1,3)+6*(8*gm(2,3)*gm(3,3)*dgm01(1,2)&
&   +4*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(1,3))*dgm10(2,2)+6*(4*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,3)+2*(3*gm(1,3)*gm(2,2)+gm(1,2)&
&   *gm(2,3))*dgm01(3,3))*dgm10(2,2)+12*(4*(1*gm(2,3)**2+3*gm(2,2)&
&   *gm(3,3))*dgm01(1,2)+8*gm(2,2)*gm(2,3)*dgm01(1,3))*dgm10(2,3)&
&   +12*(2*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,2)+4*(3*gm(1,3)&
&   *gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(2,3)+48*gm(2,2)*gm(2,3)&
&   *dgm01(1,2)*dgm10(3,3)+12*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))&
&   *dgm01(2,2)*dgm10(3,3))/12.d0
   cm(5,17)=(3*(gm(1,3)*(8*gm(3,3)*dgm01(2,2)+8*gm(2,3)*dgm01(2,3)&
&   -4*gm(2,2)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(2,3)+12*gm(2,3)&
&   *dgm01(3,3)))*dgm10(1,2)+2*(6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))&
&   *dgm01(2,2)+6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,3))&
&   *dgm10(1,3)+3*(8*gm(1,3)*gm(3,3)*dgm01(1,2)+4*(1*gm(1,3)*gm(2,3)&
&   +3*gm(1,2)*gm(3,3))*dgm01(1,3))*dgm10(2,2)+3*(4*(1*gm(1,3)**2+3*gm(1,1)&
&   *gm(3,3))*dgm01(2,3)+2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))&
&   *dgm01(3,3))*dgm10(2,2)+6*(4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))&
&   *dgm01(1,2)+2*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,3))&
&   *dgm10(2,3)+6*(2*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))*dgm01(2,2)&
&   +4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,3))*dgm10(2,3)&
&   +6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,2)*dgm10(3,3)&
&   +6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,2)*dgm10(3,3))&
&   /6.d0
   cm(6,17)=(6*(gm(1,3)*(12*gm(2,3)*dgm01(2,2)+24*gm(2,2)*dgm01(2,3))&
&   +gm(1,2)*(-4*gm(3,3)*dgm01(2,2)+8*(gm(2,3)*dgm01(2,3)+gm(2,2)&
&   *dgm01(3,3))))*dgm10(1,2)+4*(6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))&
&   *dgm01(2,2)+24*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(1,3)+6*(2*(6*gm(1,3)&
&   *gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(3*gm(1,3)*gm(2,2)+gm(1,2)&
&   *gm(2,3))*dgm01(1,3))*dgm10(2,2)+6*(4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(2,3)+2*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(2,2)+12*(4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,2)&
&   +8*gm(1,2)*gm(2,2)*dgm01(1,3))*dgm10(2,3)+12*(2*(1*gm(1,2)*gm(1,3)&
&   +3*gm(1,1)*gm(2,3))*dgm01(2,2)+4*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))&
&   *dgm01(2,3))*dgm10(2,3)+48*gm(1,2)*gm(2,2)*dgm01(1,2)*dgm10(3,3)&
&   +12*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(3,3))/12.d0
   cm(1,18)=(8*(6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,2)&
&   +6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(1,2)+24*(6*gm(1,2)&
&   **2-2*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(1,3)+12*(4*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(1,2)+2*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,3))*dgm10(2,2)+12*gm(1,1)*(8*gm(1,3)*dgm01(2,2)&
&   +16*gm(1,2)*dgm01(2,3))*dgm10(2,2)+48*(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(1,2)*dgm10(2,3)+192*gm(1,1)*gm(1,2)*dgm01(2,2)&
&   *dgm10(2,3))/48.d0
   cm(2,18)=(144*gm(1,2)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)&
&   **2*(192*dgm01(2,3)*dgm10(1,2)+96*dgm01(2,2)*dgm10(1,3)+96*dgm01(1,3)&
&   *dgm10(2,2)+192*dgm01(1,2)*dgm10(2,3))+gm(2,2)*(-48*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,2)+192*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +192*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))))&
&   /48.d0
   cm(3,18)=(2*(24*gm(2,3)*gm(3,3)*dgm01(2,2)+6*(6*gm(2,3)**2-2*gm(2,2)&
&   *gm(3,3))*dgm01(2,3))*dgm10(1,2)+6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *dgm01(2,2)*dgm10(1,3)+3*(16*gm(2,3)*gm(3,3)*dgm01(1,2)+2*(6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*dgm01(1,3))*dgm10(2,2)+3*(8*gm(1,3)*gm(3,3)&
&   *dgm01(2,2)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3))&
&   *dgm10(2,2)+12*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,2)*dgm10(2,3)&
&   +12*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,2)*dgm10(2,3))&
&   /12.d0
   cm(4,18)=(8*(6*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(2,2)+24*gm(2,2)&
&   *gm(2,3)*dgm01(2,3))*dgm10(1,2)+96*gm(2,2)*gm(2,3)*dgm01(2,2)&
&   *dgm10(1,3)+12*(4*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +8*gm(2,2)*gm(2,3)*dgm01(1,3))*dgm10(2,2)+12*(2*(1*gm(1,3)*gm(2,3)&
&   +3*gm(1,2)*gm(3,3))*dgm01(2,2)+4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))&
&   *dgm01(2,3))*dgm10(2,2)+192*gm(2,2)*gm(2,3)*dgm01(1,2)*dgm10(2,3)&
&   +48*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,3))&
&   /24.d0
   cm(5,18)=(2*(6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,2)&
&   +6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(1,2)&
&   +6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(1,3)&
&   +3*(4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,2)+2*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(1,3))*dgm10(2,2)+3*(2*(1*gm(1,3)&
&   **2+3*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(2,3))*dgm10(2,2)+12*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*dgm01(1,2)*dgm10(2,3)+12*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(2,2)*dgm10(2,3))/6.d0
   cm(6,18)=(8*(6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,2)&
&   +24*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(1,2)+96*gm(1,2)*gm(2,2)&
&   *dgm01(2,2)*dgm10(1,3)+12*(4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))&
&   *dgm01(1,2)+8*gm(1,2)*gm(2,2)*dgm01(1,3))*dgm10(2,2)+12*(2*(1*gm(1,2)&
&   *gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,2)+4*(1*gm(1,2)**2+3*gm(1,1)&
&   *gm(2,2))*dgm01(2,3))*dgm10(2,2)+192*gm(1,2)*gm(2,2)*dgm01(1,2)&
&   *dgm10(2,3)+48*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(2,3))&
&   /24.d0
   cm(1,19)=(4*(6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,3)+6*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(3,3))*dgm10(1,2)+12*((6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(2,3)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(1,3)+12*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3)*dgm10(2,2)&
&   +48*gm(1,1)*gm(1,3)*dgm01(3,3)*dgm10(2,2)+12*(2*(6*gm(1,3)**2-2*gm(1,1)&
&   *gm(3,3))*dgm01(1,2)+4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(1,3))*dgm10(2,3)+12*gm(1,1)*(16*gm(1,3)*dgm01(2,3)+8*gm(1,2)&
&   *dgm01(3,3))*dgm10(2,3)+6*(4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))&
&   *dgm01(1,2)+2*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(1,3))*dgm10(3,3)&
&   +6*gm(1,1)*(8*gm(1,3)*dgm01(2,2)+16*gm(1,2)*dgm01(2,3))*dgm10(3,3))&
&   /24.d0
   cm(2,19)=(2*(6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,3)+24*gm(2,2)&
&   *gm(2,3)*dgm01(3,3))*dgm10(1,2)+3*(2*(6*gm(2,3)**2-2*gm(2,2)&
&   *gm(3,3))*dgm01(2,2)+32*gm(2,2)*gm(2,3)*dgm01(2,3)+8*gm(2,2)&
&   **2*dgm01(3,3))*dgm10(1,3)+6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *dgm01(1,3)*dgm10(2,2)+6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))&
&   *dgm01(3,3)*dgm10(2,2)+6*(2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *dgm01(1,2)+16*gm(2,2)*gm(2,3)*dgm01(1,3))*dgm10(2,3)+6*(4*(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,3)+8*gm(1,2)*gm(2,2)*dgm01(3,3))&
&   *dgm10(2,3)+3*gm(2,2)*(16*gm(2,3)*dgm01(1,2)+8*gm(2,2)*dgm01(1,3))&
&   *dgm10(3,3)+3*(2*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,2)&
&   +16*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(3,3))/12.d0
   cm(3,19)=(96*gm(3,3)*(gm(3,3)*dgm01(2,3)+gm(2,3)*dgm01(3,3))&
&   *dgm10(1,2)+6*(8*gm(3,3)**2*dgm01(2,2)+32*gm(2,3)*gm(3,3)*dgm01(2,3)&
&   +2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3))*dgm10(1,3)+48*gm(3,3)&
&   **2*dgm01(1,3)*dgm10(2,2)+48*gm(1,3)*gm(3,3)*dgm01(3,3)*dgm10(2,2)&
&   +12*gm(3,3)*(8*gm(3,3)*dgm01(1,2)+16*gm(2,3)*dgm01(1,3))*dgm10(2,3)&
&   +12*(16*gm(1,3)*gm(3,3)*dgm01(2,3)+2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)&
&   *gm(3,3))*dgm01(3,3))*dgm10(2,3)+6*(16*gm(2,3)*gm(3,3)*dgm01(1,2)&
&   +2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,3))*dgm10(3,3)+6*(8*gm(1,3)&
&   *gm(3,3)*dgm01(2,2)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3))&
&   *dgm10(3,3))/24.d0
   cm(4,19)=(4*(24*gm(2,3)*gm(3,3)*dgm01(2,3)+6*(1*gm(2,3)**2+3*gm(2,2)&
&   *gm(3,3))*dgm01(3,3))*dgm10(1,2)+6*(8*gm(2,3)**2*dgm01(2,3)+24*gm(2,2)&
&   *gm(3,3)*dgm01(2,3)+8*gm(2,3)*(gm(3,3)*dgm01(2,2)+gm(2,2)*dgm01(3,3)))&
&   *dgm10(1,3)+48*gm(2,3)*gm(3,3)*dgm01(1,3)*dgm10(2,2)+12*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(3,3)*dgm10(2,2)+12*(8*gm(2,3)&
&   *gm(3,3)*dgm01(1,2)+4*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(1,3))&
&   *dgm10(2,3)+12*(4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,3)&
&   +2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)&
&   +6*(4*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(1,2)+8*gm(2,2)*gm(2,3)&
&   *dgm01(1,3))*dgm10(3,3)+6*(2*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))&
&   *dgm01(2,2)+4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,3))&
&   *dgm10(3,3))/12.d0
   cm(5,19)=(4*(24*gm(1,3)*gm(3,3)*dgm01(2,3)+6*(1*gm(1,3)*gm(2,3)&
&   +3*gm(1,2)*gm(3,3))*dgm01(3,3))*dgm10(1,2)+6*(gm(1,3)*(8*gm(3,3)&
&   *dgm01(2,2)+8*gm(2,3)*dgm01(2,3)-4*gm(2,2)*dgm01(3,3))+gm(1,2)&
&   *(24*gm(3,3)*dgm01(2,3)+12*gm(2,3)*dgm01(3,3)))*dgm10(1,3)+48*gm(1,3)&
&   *gm(3,3)*dgm01(1,3)*dgm10(2,2)+12*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))&
&   *dgm01(3,3)*dgm10(2,2)+12*(8*gm(1,3)*gm(3,3)*dgm01(1,2)+4*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(1,3))*dgm10(2,3)+12*(4*(1*gm(1,3)&
&   **2+3*gm(1,1)*gm(3,3))*dgm01(2,3)+2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)&
&   *gm(2,3))*dgm01(3,3))*dgm10(2,3)+6*(4*(1*gm(1,3)*gm(2,3)+3*gm(1,2)&
&   *gm(3,3))*dgm01(1,2)+2*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))&
&   *dgm01(1,3))*dgm10(3,3)+6*(2*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))&
&   *dgm01(2,2)+4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,3))&
&   *dgm10(3,3))/12.d0
   cm(6,19)=(2*(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3)&
&   +6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,2)&
&   +3*(gm(1,3)*(12*gm(2,3)*dgm01(2,2)+24*gm(2,2)*dgm01(2,3))+gm(1,2)&
&   *(-4*gm(3,3)*dgm01(2,2)+8*(gm(2,3)*dgm01(2,3)+gm(2,2)*dgm01(3,3))))&
&   *dgm10(1,3)+6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   *dgm10(2,2)+6*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(3,3)&
&   *dgm10(2,2)+6*(2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +4*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,3))*dgm10(2,3)&
&   +6*(4*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,3)+2*(1*gm(1,2)&
&   **2+3*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(2,3)+3*(4*(3*gm(1,3)&
&   *gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,2)+8*gm(1,2)*gm(2,2)*dgm01(1,3))&
&   *dgm10(3,3)+3*(2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,2)&
&   +4*(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(3,3))/6.d0
   cm(1,20)=(gm(1,3)**2*(144*dgm01(3,3)*dgm10(1,2)+288*dgm01(2,3)&
&   *dgm10(1,3)+288*dgm01(1,3)*dgm10(2,3)+144*dgm01(1,2)*dgm10(3,3))&
&   +gm(1,1)*(96*gm(1,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)*(-48*dgm01(3,3)&
&   *dgm10(1,2)-96*dgm01(2,3)*dgm10(1,3)-96*dgm01(1,3)*dgm10(2,3)&
&   -48*dgm01(1,2)*dgm10(3,3))-96*gm(2,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3)))+gm(1,3)*(288*gm(1,2)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3))+192*gm(1,1)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))))/48.d0
   cm(2,20)=(gm(2,3)**2*(144*dgm01(3,3)*dgm10(1,2)+288*dgm01(2,3)&
&   *dgm10(1,3)+288*dgm01(1,3)*dgm10(2,3)+144*dgm01(1,2)*dgm10(3,3))&
&   +gm(2,3)*(192*gm(2,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +288*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))+gm(2,2)&
&   *(96*gm(1,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)*(-48*dgm01(3,3)*dgm10(1,2)&
&   -96*dgm01(2,3)*dgm10(1,3)-96*dgm01(1,3)*dgm10(2,3)-48*dgm01(1,2)&
&   *dgm10(3,3))-96*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))&
&   /48.d0
   cm(3,20)=(144*gm(1,3)*gm(2,3)*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&   **2*(96*dgm01(3,3)*dgm10(1,2)+192*dgm01(2,3)*dgm10(1,3)+192*dgm01(1,3)&
&   *dgm10(2,3)+96*dgm01(1,2)*dgm10(3,3))+gm(3,3)*(-48*gm(1,2)*dgm01(3,3)&
&   *dgm10(3,3)+192*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +192*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))&
&   /48.d0
   cm(4,20)=(48*gm(2,3)**2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +144*gm(1,2)*gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +gm(2,2)*(72*gm(1,3)*dgm01(3,3)*dgm10(3,3)+144*gm(3,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+gm(2,3)*(24*gm(1,2)*dgm01(3,3)&
&   *dgm10(3,3)+gm(3,3)*(96*dgm01(3,3)*dgm10(1,2)+192*dgm01(2,3)&
&   *dgm10(1,3)+192*dgm01(1,3)*dgm10(2,3)+96*dgm01(1,2)*dgm10(3,3))&
&   +48*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))/24.d0
   cm(5,20)=(144*gm(1,2)*gm(3,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))+48*gm(1,3)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+gm(1,3)*(24*gm(1,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&   *(96*dgm01(3,3)*dgm10(1,2)+192*dgm01(2,3)*dgm10(1,3)+192*dgm01(1,3)&
&   *dgm10(2,3)+96*dgm01(1,2)*dgm10(3,3))+48*gm(2,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+gm(1,1)*(72*gm(2,3)*dgm01(3,3)&
&   *dgm10(3,3)+144*gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))&
&   /24.d0
   cm(6,20)=(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(3,3)*dgm10(1,2)&
&   +2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3)*dgm10(1,3)&
&   +2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3)*dgm10(1,3)&
&   +2*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)*dgm10(2,3)&
&   +2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(3,3)*dgm10(2,3)&
&   +(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,2)*dgm10(3,3)&
&   +2*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(1,3)*dgm10(3,3)&
&   +2*(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(2,3)*dgm10(3,3)&
&   +(1*gm(1,2)**2+3*gm(1,1)*gm(2,2))*dgm01(3,3)*dgm10(3,3)
   cm(1,21)=(120*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(3,3)*dgm10(1,3)&
&   +120*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(1,3)*dgm10(3,3)+480*gm(1,1)&
&   *gm(1,3)*dgm01(3,3)*dgm10(3,3))/240.d0
   cm(2,21)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3)*dgm10(1,3)&
&   +(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(1,3)*dgm10(3,3)+(-2*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(3,3)*dgm10(3,3))/2.d0
   cm(3,21)=2*gm(3,3)*(gm(1,3)*dgm01(3,3)*dgm10(3,3)+gm(3,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
   cm(4,21)=(360*gm(1,2)*gm(3,3)*dgm01(3,3)*dgm10(3,3)+gm(2,3)&
&   *(120*gm(1,3)*dgm01(3,3)*dgm10(3,3)+480*gm(3,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3))))/120.d0
   cm(5,21)=(480*gm(1,3)*gm(3,3)*dgm01(3,3)*dgm10(1,3)+480*gm(1,3)&
&   *gm(3,3)*dgm01(1,3)*dgm10(3,3)+120*(1*gm(1,3)**2+3*gm(1,1)*gm(3,3))&
&   *dgm01(3,3)*dgm10(3,3))/120.d0
   cm(6,21)=(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(3,3)*dgm10(1,3)&
&   +(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(1,3)*dgm10(3,3)&
&   +(1*gm(1,2)*gm(1,3)+3*gm(1,1)*gm(2,3))*dgm01(3,3)*dgm10(3,3)
   cm(1,22)=((6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(2,2))&
&   /4.d0
   cm(2,22)=gm(2,2)**2*dgm01(2,2)*dgm10(2,2)
   cm(3,22)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,2)*dgm10(2,2))&
&   /4.d0
   cm(4,22)=2*gm(2,2)*gm(2,3)*dgm01(2,2)*dgm10(2,2)
   cm(5,22)=((-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,2)&
&   *dgm10(2,2))/2.d0
   cm(6,22)=2*gm(1,2)*gm(2,2)*dgm01(2,2)*dgm10(2,2)
   cm(1,23)=(6*((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)**2-2*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(2,2)+4*(6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(2,2)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,3))&
&   *dgm10(2,3)+6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(3,3))&
&   /24.d0
   cm(2,23)=(12*(2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   +32*gm(2,2)*gm(2,3)*dgm01(2,3)+8*gm(2,2)**2*dgm01(3,3))*dgm10(2,2)&
&   +384*gm(2,2)*(gm(2,3)*dgm01(2,2)+gm(2,2)*dgm01(2,3))*dgm10(2,3)&
&   +96*gm(2,2)**2*dgm01(2,2)*dgm10(3,3))/96.d0
   cm(3,23)=(3*(8*gm(3,3)**2*dgm01(2,2)+32*gm(2,3)*gm(3,3)*dgm01(2,3)&
&   +2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3))*dgm10(2,2)+4*(24*gm(2,3)&
&   *gm(3,3)*dgm01(2,2)+6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,3))&
&   *dgm10(2,3)+6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,2)*dgm10(3,3))&
&   /24.d0
   cm(4,23)=(12*(8*gm(2,3)**2*dgm01(2,3)+24*gm(2,2)*gm(3,3)*dgm01(2,3)&
&   +8*gm(2,3)*(gm(3,3)*dgm01(2,2)+gm(2,2)*dgm01(3,3)))*dgm10(2,2)&
&   +16*(6*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(2,2)+24*gm(2,2)&
&   *gm(2,3)*dgm01(2,3))*dgm10(2,3)+96*gm(2,2)*gm(2,3)*dgm01(2,2)&
&   *dgm10(3,3))/48.d0
   cm(5,23)=(3*(gm(1,3)*(8*gm(3,3)*dgm01(2,2)+8*gm(2,3)*dgm01(2,3)&
&   -4*gm(2,2)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(2,3)+12*gm(2,3)&
&   *dgm01(3,3)))*dgm10(2,2)+4*(6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))&
&   *dgm01(2,2)+6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,3))&
&   *dgm10(2,3)+6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,2)&
&   *dgm10(3,3))/12.d0
   cm(6,23)=(12*(gm(1,3)*(12*gm(2,3)*dgm01(2,2)+24*gm(2,2)*dgm01(2,3))&
&   +gm(1,2)*(-4*gm(3,3)*dgm01(2,2)+8*(gm(2,3)*dgm01(2,3)+gm(2,2)&
&   *dgm01(3,3))))*dgm10(2,2)+16*(6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))&
&   *dgm01(2,2)+24*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(2,3)+96*gm(1,2)&
&   *gm(2,2)*dgm01(2,2)*dgm10(3,3))/48.d0
   cm(1,24)=((6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,2)&
&   +6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(2,2)+6*(6*gm(1,2)&
&   **2-2*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(2,3))/12.d0
   cm(2,24)=2*gm(2,2)*(gm(2,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))
   cm(3,24)=((24*gm(2,3)*gm(3,3)*dgm01(2,2)+6*(6*gm(2,3)**2-2*gm(2,2)&
&   *gm(3,3))*dgm01(2,3))*dgm10(2,2)+6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *dgm01(2,2)*dgm10(2,3))/12.d0
   cm(4,24)=(20*(6*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   +24*gm(2,2)*gm(2,3)*dgm01(2,3))*dgm10(2,2)+480*gm(2,2)*gm(2,3)&
&   *dgm01(2,2)*dgm10(2,3))/120.d0
   cm(5,24)=((6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,2)&
&   +6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(2,2)&
&   +6*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,3))&
&   /6.d0
   cm(6,24)=(20*(6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,2)&
&   +24*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(2,2)+480*gm(1,2)*gm(2,2)&
&   *dgm01(2,2)*dgm10(2,3))/120.d0
   cm(1,25)=((6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,3)+6*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(3,3))*dgm10(2,2)+6*((6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(2,3)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(2,3)+(6*(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(2,2)&
&   +6*(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(3,3))/12.d0
   cm(2,25)=((6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,3)+24*gm(2,2)&
&   *gm(2,3)*dgm01(3,3))*dgm10(2,2)+3*(2*(6*gm(2,3)**2-2*gm(2,2)&
&   *gm(3,3))*dgm01(2,2)+32*gm(2,2)*gm(2,3)*dgm01(2,3)+8*gm(2,2)&
&   **2*dgm01(3,3))*dgm10(2,3)+24*gm(2,2)*(gm(2,3)*dgm01(2,2)+gm(2,2)&
&   *dgm01(2,3))*dgm10(3,3))/12.d0
   cm(3,25)=(24*gm(3,3)*(gm(3,3)*dgm01(2,3)+gm(2,3)*dgm01(3,3))&
&   *dgm10(2,2)+3*(8*gm(3,3)**2*dgm01(2,2)+32*gm(2,3)*gm(3,3)*dgm01(2,3)&
&   +2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3))*dgm10(2,3)+(24*gm(2,3)&
&   *gm(3,3)*dgm01(2,2)+6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,3))&
&   *dgm10(3,3))/12.d0
   cm(4,25)=((24*gm(2,3)*gm(3,3)*dgm01(2,3)+6*(1*gm(2,3)**2+3*gm(2,2)&
&   *gm(3,3))*dgm01(3,3))*dgm10(2,2)+3*(8*gm(2,3)**2*dgm01(2,3)+24*gm(2,2)&
&   *gm(3,3)*dgm01(2,3)+8*gm(2,3)*(gm(3,3)*dgm01(2,2)+gm(2,2)*dgm01(3,3)))&
&   *dgm10(2,3)+(6*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(2,2)+24*gm(2,2)&
&   *gm(2,3)*dgm01(2,3))*dgm10(3,3))/6.d0
   cm(5,25)=((24*gm(1,3)*gm(3,3)*dgm01(2,3)+6*(1*gm(1,3)*gm(2,3)&
&   +3*gm(1,2)*gm(3,3))*dgm01(3,3))*dgm10(2,2)+3*(gm(1,3)*(8*gm(3,3)&
&   *dgm01(2,2)+8*gm(2,3)*dgm01(2,3)-4*gm(2,2)*dgm01(3,3))+gm(1,2)&
&   *(24*gm(3,3)*dgm01(2,3)+12*gm(2,3)*dgm01(3,3)))*dgm10(2,3)+(6*(1*gm(1,3)&
&   *gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(2,2)+6*(-2*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(3,3))/6.d0
   cm(6,25)=((6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3)&
&   +6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,2)&
&   +3*(gm(1,3)*(12*gm(2,3)*dgm01(2,2)+24*gm(2,2)*dgm01(2,3))+gm(1,2)&
&   *(-4*gm(3,3)*dgm01(2,2)+8*(gm(2,3)*dgm01(2,3)+gm(2,2)*dgm01(3,3))))&
&   *dgm10(2,3)+(6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(2,2)&
&   +24*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(3,3))/6.d0
   cm(1,26)=(6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(3,3)*dgm10(2,2)&
&   +4*(6*(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,3)+6*(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(3,3))*dgm10(2,3)+6*((6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*dgm01(2,3)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(3,3))/24.d0
   cm(2,26)=(6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3)*dgm10(2,2)&
&   +4*(6*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,3)+24*gm(2,2)&
&   *gm(2,3)*dgm01(3,3))*dgm10(2,3)+3*(2*(6*gm(2,3)**2-2*gm(2,2)&
&   *gm(3,3))*dgm01(2,2)+32*gm(2,2)*gm(2,3)*dgm01(2,3)+8*gm(2,2)&
&   **2*dgm01(3,3))*dgm10(3,3))/24.d0
   cm(3,26)=(96*gm(3,3)**2*dgm01(3,3)*dgm10(2,2)+384*gm(3,3)*(gm(3,3)&
&   *dgm01(2,3)+gm(2,3)*dgm01(3,3))*dgm10(2,3)+12*(8*gm(3,3)**2*dgm01(2,2)&
&   +32*gm(2,3)*gm(3,3)*dgm01(2,3)+2*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *dgm01(3,3))*dgm10(3,3))/96.d0
   cm(4,26)=(96*gm(2,3)*gm(3,3)*dgm01(3,3)*dgm10(2,2)+16*(24*gm(2,3)&
&   *gm(3,3)*dgm01(2,3)+6*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))*dgm01(3,3))&
&   *dgm10(2,3)+12*(8*gm(2,3)**2*dgm01(2,3)+24*gm(2,2)*gm(3,3)*dgm01(2,3)&
&   +8*gm(2,3)*(gm(3,3)*dgm01(2,2)+gm(2,2)*dgm01(3,3)))*dgm10(3,3))&
&   /48.d0
   cm(5,26)=(96*gm(1,3)*gm(3,3)*dgm01(3,3)*dgm10(2,2)+16*(24*gm(1,3)&
&   *gm(3,3)*dgm01(2,3)+6*(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*dgm01(3,3))&
&   *dgm10(2,3)+12*(gm(1,3)*(8*gm(3,3)*dgm01(2,2)+8*gm(2,3)*dgm01(2,3)&
&   -4*gm(2,2)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(2,3)+12*gm(2,3)&
&   *dgm01(3,3)))*dgm10(3,3))/48.d0
   cm(6,26)=(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,2)+4*(6*(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3)&
&   +6*(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)&
&   +3*(gm(1,3)*(12*gm(2,3)*dgm01(2,2)+24*gm(2,2)*dgm01(2,3))+gm(1,2)&
&   *(-4*gm(3,3)*dgm01(2,2)+8*(gm(2,3)*dgm01(2,3)+gm(2,2)*dgm01(3,3))))&
&   *dgm10(3,3))/12.d0
   cm(1,27)=((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(3,3)*dgm10(2,3)&
&   +(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(2,3)*dgm10(3,3)+(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*dgm01(3,3)*dgm10(3,3))/2.d0
   cm(2,27)=(120*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3)*dgm10(2,3)&
&   +120*(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(2,3)*dgm10(3,3)+480*gm(2,2)&
&   *gm(2,3)*dgm01(3,3)*dgm10(3,3))/240.d0
   cm(3,27)=2*gm(3,3)*(gm(2,3)*dgm01(3,3)*dgm10(3,3)+gm(3,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
   cm(4,27)=(480*gm(2,3)*gm(3,3)*dgm01(3,3)*dgm10(2,3)+480*gm(2,3)&
&   *gm(3,3)*dgm01(2,3)*dgm10(3,3)+120*(1*gm(2,3)**2+3*gm(2,2)*gm(3,3))&
&   *dgm01(3,3)*dgm10(3,3))/120.d0
   cm(5,27)=(360*gm(1,2)*gm(3,3)*dgm01(3,3)*dgm10(3,3)+gm(1,3)&
&   *(120*gm(2,3)*dgm01(3,3)*dgm10(3,3)+480*gm(3,3)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3))))/120.d0
   cm(6,27)=(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(3,3)*dgm10(2,3)&
&   +(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(2,3)*dgm10(3,3)&
&   +(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*dgm01(3,3)*dgm10(3,3)
   cm(1,28)=((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm01(3,3)*dgm10(3,3))&
&   /4.d0
   cm(2,28)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm01(3,3)*dgm10(3,3))&
&   /4.d0
   cm(3,28)=gm(3,3)**2*dgm01(3,3)*dgm10(3,3)
   cm(4,28)=2*gm(2,3)*gm(3,3)*dgm01(3,3)*dgm10(3,3)
   cm(5,28)=2*gm(1,3)*gm(3,3)*dgm01(3,3)*dgm10(3,3)
   cm(6,28)=((6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm01(3,3)*dgm10(3,3))&
&   /2.d0
 elseif(rank==3)then
   cm(1,1)=gm(1,1)**3*dgm01(1,1)*dgm10(1,1)
   cm(2,1)=(gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,1)&
&   *dgm10(1,1))/12.d0
   cm(3,1)=(gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,1)&
&   *dgm10(1,1))/12.d0
   cm(4,1)=(gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(1,1)&
&   *dgm10(1,1))/6.d0
   cm(5,1)=3*gm(1,1)**2*gm(1,3)*dgm01(1,1)*dgm10(1,1)
   cm(6,1)=3*gm(1,1)**2*gm(1,2)*dgm01(1,1)*dgm10(1,1)
   cm(7,1)=((90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)&
&   *dgm10(1,1))/36.d0
   cm(8,1)=((-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(1,1)*dgm10(1,1))/12.d0
   cm(9,1)=((90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(1,1)*dgm10(1,1))/12.d0
   cm(10,1)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)&
&   *dgm10(1,1))/36.d0
   cm(1,2)=(gm(1,1)*(20*(6*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,1)&
&   +432*gm(1,1)*gm(1,2)*dgm01(1,2)+72*gm(1,1)**2*dgm01(2,2))*dgm10(1,1)&
&   +20*gm(1,1)*(432*gm(1,2)*dgm01(1,1)+288*gm(1,1)*dgm01(1,2))*dgm10(1,2)&
&   +1440*gm(1,1)**2*dgm01(1,1)*dgm10(2,2)))/1440.d0
   cm(2,2)=((72*gm(1,2)**3*dgm01(1,2)+360*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm01(1,2)+gm(1,1)*gm(2,2)*(144*gm(2,2)*dgm01(1,1)-36*gm(1,1)&
&   *dgm01(2,2))+gm(1,2)**2*(72*gm(2,2)*dgm01(1,1)+108*gm(1,1)*dgm01(2,2)))&
&   *dgm10(1,1)+(72*gm(1,2)**3*dgm01(1,1)+360*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm01(1,1)+432*gm(1,1)*gm(1,2)**2*dgm01(1,2)-144*gm(1,1)**2*gm(2,2)&
&   *dgm01(1,2))*dgm10(1,2)+2*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&   *dgm01(1,1)*dgm10(2,2))/24.d0
   cm(3,2)=((6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)+2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(2,2))*dgm10(1,1)+(gm(1,2)*(72*gm(1,3)**2-216*gm(1,1)&
&   *gm(3,3))*dgm01(1,1)+gm(1,1)*(576*gm(1,3)*gm(2,3)*dgm01(1,1)&
&   +432*gm(1,3)**2*dgm01(1,2)-144*gm(1,1)*gm(3,3)*dgm01(1,2)))*dgm10(1,2)&
&   +2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,1)*dgm10(2,2))&
&   /24.d0
   cm(4,2)=gm(1,2)**2*(3*gm(2,3)*dgm01(1,1)*dgm10(1,1)+6*gm(1,3)&
&   *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)))+gm(1,1)*(gm(2,2)&
&   *(12*gm(2,3)*dgm01(1,1)*dgm10(1,1)+24*gm(1,3)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)))+gm(1,1)*gm(2,3)*(-3*dgm01(2,2)*dgm10(1,1)&
&   -12*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)*dgm10(2,2)))+gm(1,2)*(6*gm(1,1)&
&   *gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,3)&
&   *(3*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(9*dgm01(2,2)*dgm10(1,1)&
&   +36*dgm01(1,2)*dgm10(1,2)+9*dgm01(1,1)*dgm10(2,2))))
   cm(5,2)=1.5d0*gm(1,2)**2*gm(1,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&   *gm(1,2)*(12*gm(2,3)*dgm01(1,1)*dgm10(1,1)+6*gm(1,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)))+gm(1,1)*(12*gm(1,1)*gm(2,3)&
&   *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,3)*(-4.5d0*gm(2,2)&
&   *dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)&
&   *dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))))
   cm(6,2)=1.5d0*gm(1,2)**3*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)*gm(1,2)&
&   **2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*gm(1,1)&
&   **2*gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)&
&   *gm(1,2)*(7.5d0*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(2,2)&
&   *dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)))
   cm(7,2)=((648*gm(1,2)**2*gm(2,2)*dgm01(1,2)-216*gm(1,1)*gm(2,2)&
&   **2*dgm01(1,2)+180*gm(1,2)**3*dgm01(2,2)+gm(1,2)*gm(2,2)*(216*gm(2,2)&
&   *dgm01(1,1)-108*gm(1,1)*dgm01(2,2)))*dgm10(1,1)+(648*gm(1,2)&
&   **2*gm(2,2)*dgm01(1,1)-216*gm(1,1)*gm(2,2)**2*dgm01(1,1)+720*gm(1,2)&
&   **3*dgm01(1,2)-432*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2))*dgm10(1,2)&
&   +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)*dgm10(2,2))&
&   /72.d0
   cm(8,2)=((6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2))*dgm10(1,1)+(12*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)+8*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(1,2))*dgm10(1,2)+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)*dgm10(2,2))/24.d0
   cm(9,2)=((gm(2,3)*(72*gm(1,2)**2*dgm01(1,2)-216*gm(1,1)*gm(2,2)&
&   *dgm01(1,2)+gm(1,2)*(72*gm(2,2)*dgm01(1,1)-72*gm(1,1)*dgm01(2,2)))&
&   +gm(1,3)*(144*gm(2,2)**2*dgm01(1,1)+180*gm(1,2)**2*dgm01(2,2)&
&   +gm(2,2)*(576*gm(1,2)*dgm01(1,2)-36*gm(1,1)*dgm01(2,2))))*dgm10(1,1)&
&   +(gm(1,1)*gm(2,2)*(-216*gm(2,3)*dgm01(1,1)-144*gm(1,3)*dgm01(1,2))&
&   +gm(1,2)**2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,3)*dgm01(1,2))+gm(1,2)&
&   *(576*gm(1,3)*gm(2,2)*dgm01(1,1)-288*gm(1,1)*gm(2,3)*dgm01(1,2)))&
&   *dgm10(1,2)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,1)*dgm10(2,2))/24.d0
   cm(10,2)=((6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(90*gm(1,3)**2*gm(2,3)&
&   -18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,2)+2*(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2))*dgm10(1,1)+(12*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,1)&
&   +8*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2))*dgm10(1,2)&
&   +2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)*dgm10(2,2))&
&   /72.d0
   cm(1,3)=(gm(1,1)*(20*(6*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,1)&
&   +432*gm(1,1)*gm(1,3)*dgm01(1,3)+72*gm(1,1)**2*dgm01(3,3))*dgm10(1,1)&
&   +20*gm(1,1)*(432*gm(1,3)*dgm01(1,1)+288*gm(1,1)*dgm01(1,3))*dgm10(1,3)&
&   +1440*gm(1,1)**2*dgm01(1,1)*dgm10(3,3)))/1440.d0
   cm(2,3)=((6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3)+2*gm(1,1)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(1,1)+(576*gm(1,1)*gm(1,2)&
&   *gm(2,3)*dgm01(1,1)+gm(1,1)*gm(2,2)*(-216*gm(1,3)*dgm01(1,1)&
&   -144*gm(1,1)*dgm01(1,3))+gm(1,2)**2*(72*gm(1,3)*dgm01(1,1)+432*gm(1,1)&
&   *dgm01(1,3)))*dgm10(1,3)+2*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,1)*dgm10(3,3))/24.d0
   cm(3,3)=((72*gm(1,3)**3*dgm01(1,3)+360*gm(1,1)*gm(1,3)*gm(3,3)&
&   *dgm01(1,3)+gm(1,1)*gm(3,3)*(144*gm(3,3)*dgm01(1,1)-36*gm(1,1)&
&   *dgm01(3,3))+gm(1,3)**2*(72*gm(3,3)*dgm01(1,1)+108*gm(1,1)*dgm01(3,3)))&
&   *dgm10(1,1)+(72*gm(1,3)**3*dgm01(1,1)+360*gm(1,1)*gm(1,3)*gm(3,3)&
&   *dgm01(1,1)+432*gm(1,1)*gm(1,3)**2*dgm01(1,3)-144*gm(1,1)**2*gm(3,3)&
&   *dgm01(1,3))*dgm10(1,3)+2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(1,1)*dgm10(3,3))/24.d0
   cm(4,3)=gm(1,3)**2*(3*gm(2,3)*dgm01(1,1)*dgm10(1,1)+6*gm(1,2)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))+gm(1,1)*(24*gm(1,2)&
&   *gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(2,3)&
&   *(12*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(-3*dgm01(3,3)*dgm10(1,1)&
&   -12*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)*dgm10(3,3))))+gm(1,3)&
&   *(6*gm(1,1)*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,2)*(3*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(9*dgm01(3,3)&
&   *dgm10(1,1)+36*dgm01(1,3)*dgm10(1,3)+9*dgm01(1,1)*dgm10(3,3))))
   cm(5,3)=1.5d0*gm(1,3)**3*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)*gm(1,3)&
&   **2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*gm(1,1)&
&   **2*gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)&
&   *gm(1,3)*(7.5d0*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(3,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)))
   cm(6,3)=12*gm(1,1)*gm(2,3)*(gm(1,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))+gm(1,2)*(1.5d0*gm(1,3)&
&   **2*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)*gm(1,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3))+gm(1,1)*(-4.5d0*gm(3,3)*dgm01(1,1)*dgm10(1,1)&
&   +gm(1,1)*(3*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)&
&   *dgm10(3,3))))
   cm(7,3)=((6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(90*gm(1,2)**2*gm(2,3)&
&   -18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,3)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(1,1)+(12*(90*gm(1,2)&
&   **2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,1)&
&   +8*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,3))*dgm10(1,3)&
&   +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)*dgm10(3,3))&
&   /72.d0
   cm(8,3)=((gm(1,3)**2*(72*gm(2,3)*dgm01(1,3)+180*gm(1,2)*dgm01(3,3))&
&   +gm(1,3)*(576*gm(1,2)*gm(3,3)*dgm01(1,3)+gm(2,3)*(72*gm(3,3)&
&   *dgm01(1,1)-72*gm(1,1)*dgm01(3,3)))+gm(3,3)*(-216*gm(1,1)*gm(2,3)&
&   *dgm01(1,3)+gm(1,2)*(144*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3))))&
&   *dgm10(1,1)+(gm(1,1)*gm(3,3)*(-216*gm(2,3)*dgm01(1,1)-144*gm(1,2)&
&   *dgm01(1,3))+gm(1,3)**2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,2)*dgm01(1,3))&
&   +gm(1,3)*(576*gm(1,2)*gm(3,3)*dgm01(1,1)-288*gm(1,1)*gm(2,3)&
&   *dgm01(1,3)))*dgm10(1,3)+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)*dgm10(3,3))/24.d0
   cm(9,3)=((6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,1)+(12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)+8*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(1,3))*dgm10(1,3)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,1)*dgm10(3,3))/24.d0
   cm(10,3)=((648*gm(1,3)**2*gm(3,3)*dgm01(1,3)-216*gm(1,1)*gm(3,3)&
&   **2*dgm01(1,3)+180*gm(1,3)**3*dgm01(3,3)+gm(1,3)*gm(3,3)*(216*gm(3,3)&
&   *dgm01(1,1)-108*gm(1,1)*dgm01(3,3)))*dgm10(1,1)+(648*gm(1,3)&
&   **2*gm(3,3)*dgm01(1,1)-216*gm(1,1)*gm(3,3)**2*dgm01(1,1)+720*gm(1,3)&
&   **3*dgm01(1,3)-432*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3))*dgm10(1,3)&
&   +2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)*dgm10(3,3))&
&   /72.d0
   cm(1,4)=(gm(1,1)*(1440*(gm(1,2)*(4.5d0*gm(1,3)*dgm01(1,1)+3*gm(1,1)&
&   *dgm01(1,3))+gm(1,1)*(-1.5d0*gm(2,3)*dgm01(1,1)+3*gm(1,3)*dgm01(1,2)&
&   +gm(1,1)*dgm01(2,3)))*dgm10(1,1)+10*gm(1,1)*(432*gm(1,3)*dgm01(1,1)&
&   +288*gm(1,1)*dgm01(1,3))*dgm10(1,2)+10*gm(1,1)*(432*gm(1,2)*dgm01(1,1)&
&   +288*gm(1,1)*dgm01(1,2))*dgm10(1,3)+1440*gm(1,1)**2*dgm01(1,1)&
&   *dgm10(2,3)))/720.d0
   cm(2,4)=(2*(36*gm(1,2)**3*dgm01(1,3)+gm(1,2)*(36*gm(1,3)*gm(2,2)&
&   *dgm01(1,1)+gm(1,1)*(288*gm(2,3)*dgm01(1,2)+180*gm(2,2)*dgm01(1,3)))&
&   +gm(1,1)*gm(2,2)*(144*gm(2,3)*dgm01(1,1)-108*gm(1,3)*dgm01(1,2)&
&   -36*gm(1,1)*dgm01(2,3))+gm(1,2)**2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,3)&
&   *dgm01(1,2)+108*gm(1,1)*dgm01(2,3)))*dgm10(1,1)+(576*gm(1,1)&
&   *gm(1,2)*gm(2,3)*dgm01(1,1)+gm(1,1)*gm(2,2)*(-216*gm(1,3)*dgm01(1,1)&
&   -144*gm(1,1)*dgm01(1,3))+gm(1,2)**2*(72*gm(1,3)*dgm01(1,1)+432*gm(1,1)&
&   *dgm01(1,3)))*dgm10(1,2)+(72*gm(1,2)**3*dgm01(1,1)+360*gm(1,1)&
&   *gm(1,2)*gm(2,2)*dgm01(1,1)+432*gm(1,1)*gm(1,2)**2*dgm01(1,2)&
&   -144*gm(1,1)**2*gm(2,2)*dgm01(1,2))*dgm10(1,3)+4*gm(1,1)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(1,1)*dgm10(2,3))/24.d0
   cm(3,4)=(2*(36*gm(1,3)**3*dgm01(1,2)+gm(1,3)*(36*gm(1,2)*gm(3,3)&
&   *dgm01(1,1)+gm(1,1)*(180*gm(3,3)*dgm01(1,2)+288*gm(2,3)*dgm01(1,3)))&
&   +gm(1,1)*gm(3,3)*(144*gm(2,3)*dgm01(1,1)-108*gm(1,2)*dgm01(1,3)&
&   -36*gm(1,1)*dgm01(2,3))+gm(1,3)**2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,2)&
&   *dgm01(1,3)+108*gm(1,1)*dgm01(2,3)))*dgm10(1,1)+(72*gm(1,3)**3*dgm01(1,1)&
&   +360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,1)+432*gm(1,1)*gm(1,3)**2*dgm01(1,3)&
&   -144*gm(1,1)**2*gm(3,3)*dgm01(1,3))*dgm10(1,2)+(gm(1,2)*(72*gm(1,3)&
&   **2-216*gm(1,1)*gm(3,3))*dgm01(1,1)+gm(1,1)*(576*gm(1,3)*gm(2,3)&
&   *dgm01(1,1)+432*gm(1,3)**2*dgm01(1,2)-144*gm(1,1)*gm(3,3)*dgm01(1,2)))&
&   *dgm10(1,3)+4*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,1)&
&   *dgm10(2,3))/24.d0
   cm(4,4)=9*gm(1,2)**2*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,3)**2*(9*gm(2,2)&
&   *dgm01(1,1)*dgm10(1,1)+6*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)))+gm(1,1)*(9*gm(2,3)**2*dgm01(1,1)*dgm10(1,1)+gm(3,3)&
&   *(15*gm(2,2)*dgm01(1,1)*dgm10(1,1)+24*gm(1,2)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)))+6*gm(1,2)*gm(2,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)))+gm(1,1)**2*gm(2,3)*(-6*dgm01(2,3)*dgm10(1,1)&
&   -12*dgm01(1,3)*dgm10(1,2)-12*dgm01(1,2)*dgm10(1,3)-6*dgm01(1,1)&
&   *dgm10(2,3))+gm(1,3)*(6*gm(1,2)**2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))+gm(1,1)*(6*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))+24*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))&
&   +gm(1,2)*(-6*gm(2,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(18*dgm01(2,3)&
&   *dgm10(1,1)+36*dgm01(1,3)*dgm10(1,2)+36*dgm01(1,2)*dgm10(1,3)&
&   +18*dgm01(1,1)*dgm10(2,3))))
   cm(5,4)=gm(1,2)*(3*gm(1,3)**2*dgm01(1,1)*dgm10(1,1)+12*gm(1,1)&
&   *gm(3,3)*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)*gm(1,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))+gm(1,1)*(6*gm(1,3)**2*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*gm(1,1)*(gm(3,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(2,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)))+gm(1,3)*(3*gm(2,3)*dgm01(1,1)*dgm10(1,1)&
&   +gm(1,1)*(6*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3))))
   cm(6,4)=gm(1,2)**2*(3*gm(1,3)*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))+12*gm(1,1)*(gm(1,3)&
&   *gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(gm(2,3)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2))+gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))))+gm(1,1)*gm(1,2)*(3*gm(2,3)*dgm01(1,1)*dgm10(1,1)&
&   +6*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)&
&   *(6*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)))
   cm(7,4)=(2*(6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(1,1)+6*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3)))*dgm01(1,2)+6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,3)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))&
&   *dgm01(2,3))*dgm10(1,1)+(12*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)&
&   *(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,1)+8*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,3))*dgm10(1,2)+(648*gm(1,2)&
&   **2*gm(2,2)*dgm01(1,1)-216*gm(1,1)*gm(2,2)**2*dgm01(1,1)+720*gm(1,2)&
&   **3*dgm01(1,2)-432*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2))*dgm10(1,3)&
&   +4*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)*dgm10(2,3))&
&   /72.d0
   cm(8,4)=(2*(6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)+6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)+6*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+2*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(2,3))*dgm10(1,1)+(gm(1,1)*gm(3,3)*(-216*gm(2,3)*dgm01(1,1)&
&   -144*gm(1,2)*dgm01(1,3))+gm(1,3)**2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,2)&
&   *dgm01(1,3))+gm(1,3)*(576*gm(1,2)*gm(3,3)*dgm01(1,1)-288*gm(1,1)&
&   *gm(2,3)*dgm01(1,3)))*dgm10(1,2)+(12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+8*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2))*dgm10(1,3)+4*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(1,1)*dgm10(2,3))/24.d0
   cm(9,4)=(2*(6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)+6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)&
&   +2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(1,1)+(12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)+8*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3))*dgm10(1,2)&
&   +(gm(1,1)*gm(2,2)*(-216*gm(2,3)*dgm01(1,1)-144*gm(1,3)*dgm01(1,2))&
&   +gm(1,2)**2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,3)*dgm01(1,2))+gm(1,2)&
&   *(576*gm(1,3)*gm(2,2)*dgm01(1,1)-288*gm(1,1)*gm(2,3)*dgm01(1,2)))&
&   *dgm10(1,3)+4*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,1)*dgm10(2,3))/24.d0
   cm(10,4)=(2*(6*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(1,1)+6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)&
&   +6*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))&
&   *gm(3,3))*dgm01(1,3)+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))&
&   *dgm01(2,3))*dgm10(1,1)+(648*gm(1,3)**2*gm(3,3)*dgm01(1,1)-216*gm(1,1)&
&   *gm(3,3)**2*dgm01(1,1)+720*gm(1,3)**3*dgm01(1,3)-432*gm(1,1)&
&   *gm(1,3)*gm(3,3)*dgm01(1,3))*dgm10(1,2)+(12*(90*gm(1,3)**2*gm(2,3)&
&   -18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,1)+8*(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2))*dgm10(1,3)+4*(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)*dgm10(2,3))/72.d0
   cm(1,5)=gm(1,1)**2*(3*gm(1,3)*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))
   cm(2,5)=(30*(576*gm(1,1)*gm(1,2)*gm(2,3)*dgm01(1,1)+gm(1,1)&
&   *gm(2,2)*(-216*gm(1,3)*dgm01(1,1)-144*gm(1,1)*dgm01(1,3))+gm(1,2)&
&   **2*(72*gm(1,3)*dgm01(1,1)+432*gm(1,1)*dgm01(1,3)))*dgm10(1,1)&
&   +240*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,1)*dgm10(1,3))&
&   /1440.d0
   cm(3,5)=1.5d0*gm(1,3)**3*dgm01(1,1)*dgm10(1,1)+7.5d0*gm(1,1)&
&   *gm(1,3)*gm(3,3)*dgm01(1,1)*dgm10(1,1)+9*gm(1,1)*gm(1,3)**2*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))-3*gm(1,1)**2*gm(3,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))
   cm(4,5)=gm(1,1)*gm(2,3)*(3*gm(1,3)*dgm01(1,1)*dgm10(1,1)-6*gm(1,1)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))+gm(1,2)*(3*gm(1,3)&
&   **2*dgm01(1,1)*dgm10(1,1)+12*gm(1,1)*gm(3,3)*dgm01(1,1)*dgm10(1,1)&
&   +18*gm(1,1)*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))
   cm(5,5)=gm(1,1)*(3*gm(1,3)**2*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)&
&   *gm(3,3)*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)*gm(1,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))
   cm(6,5)=gm(1,1)*(6*gm(1,1)*gm(2,3)*dgm01(1,1)*dgm10(1,1)+gm(1,2)&
&   *(3*gm(1,3)*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3))))
   cm(7,5)=(30*(12*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,1)+8*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(1,3))*dgm10(1,1)+240*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(1,1)*dgm10(1,3))/4320.d0
   cm(8,5)=(30*(gm(1,1)*gm(3,3)*(-216*gm(2,3)*dgm01(1,1)-144*gm(1,2)&
&   *dgm01(1,3))+gm(1,3)**2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,2)*dgm01(1,3))&
&   +gm(1,3)*(576*gm(1,2)*gm(3,3)*dgm01(1,1)-288*gm(1,1)*gm(2,3)&
&   *dgm01(1,3)))*dgm10(1,1)+240*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)*dgm10(1,3))/1440.d0
   cm(9,5)=(30*(12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+8*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3))*dgm10(1,1)+240*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(1,1)*dgm10(1,3))/1440.d0
   cm(10,5)=(30*(648*gm(1,3)**2*gm(3,3)*dgm01(1,1)-216*gm(1,1)&
&   *gm(3,3)**2*dgm01(1,1)+720*gm(1,3)**3*dgm01(1,3)-432*gm(1,1)&
&   *gm(1,3)*gm(3,3)*dgm01(1,3))*dgm10(1,1)+240*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(1,1)*dgm10(1,3))/4320.d0
   cm(1,6)=gm(1,1)**2*(3*gm(1,2)*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)&
&   *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)))
   cm(2,6)=1.5d0*gm(1,2)**3*dgm01(1,1)*dgm10(1,1)+7.5d0*gm(1,1)&
&   *gm(1,2)*gm(2,2)*dgm01(1,1)*dgm10(1,1)+9*gm(1,1)*gm(1,2)**2*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))-3*gm(1,1)**2*gm(2,2)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))
   cm(3,6)=(30*(gm(1,2)*(72*gm(1,3)**2-216*gm(1,1)*gm(3,3))*dgm01(1,1)&
&   +gm(1,1)*(576*gm(1,3)*gm(2,3)*dgm01(1,1)+432*gm(1,3)**2*dgm01(1,2)&
&   -144*gm(1,1)*gm(3,3)*dgm01(1,2)))*dgm10(1,1)+240*gm(1,1)*(54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(1,1)*dgm10(1,2))/1440.d0
   cm(4,6)=3*gm(1,2)**2*gm(1,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*gm(1,2)&
&   *(3*gm(2,3)*dgm01(1,1)*dgm10(1,1)+18*gm(1,3)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)))+gm(1,1)*(12*gm(1,3)*gm(2,2)*dgm01(1,1)&
&   *dgm10(1,1)-6*gm(1,1)*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)))
   cm(5,6)=gm(1,1)*(3*gm(1,2)*gm(1,3)*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)&
&   *(gm(2,3)*dgm01(1,1)*dgm10(1,1)+gm(1,3)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2))))
   cm(6,6)=gm(1,1)*(3*gm(1,2)**2*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)&
&   *gm(2,2)*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)*gm(1,2)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)))
   cm(7,6)=(30*(648*gm(1,2)**2*gm(2,2)*dgm01(1,1)-216*gm(1,1)*gm(2,2)&
&   **2*dgm01(1,1)+720*gm(1,2)**3*dgm01(1,2)-432*gm(1,1)*gm(1,2)&
&   *gm(2,2)*dgm01(1,2))*dgm10(1,1)+240*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(1,1)*dgm10(1,2))/4320.d0
   cm(8,6)=(30*(12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+8*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(1,2))*dgm10(1,1)+240*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(1,1)*dgm10(1,2))/1440.d0
   cm(9,6)=(30*(gm(1,1)*gm(2,2)*(-216*gm(2,3)*dgm01(1,1)-144*gm(1,3)&
&   *dgm01(1,2))+gm(1,2)**2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,3)*dgm01(1,2))&
&   +gm(1,2)*(576*gm(1,3)*gm(2,2)*dgm01(1,1)-288*gm(1,1)*gm(2,3)&
&   *dgm01(1,2)))*dgm10(1,1)+240*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,1)*dgm10(1,2))&
&   /1440.d0
   cm(10,6)=(30*(12*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,1)+8*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(1,2))*dgm10(1,1)+240*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(1,1)*dgm10(1,2))/4320.d0
   cm(1,7)=(2*(2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)&
&   +12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,2)+216*gm(1,1)&
&   **2*gm(1,2)*dgm01(2,2))*dgm10(1,1)+4*gm(1,1)*(6*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,1)+432*gm(1,1)*gm(1,2)*dgm01(1,2)+72*gm(1,1)&
&   **2*dgm01(2,2))*dgm10(1,2)+gm(1,1)**2*(432*gm(1,2)*dgm01(1,1)&
&   +288*gm(1,1)*dgm01(1,2))*dgm10(2,2))/144.d0
   cm(2,7)=(2*(144*gm(1,2)**2*gm(2,2)*dgm01(1,2)+288*gm(1,1)*gm(2,2)&
&   **2*dgm01(1,2)+36*gm(1,2)**3*dgm01(2,2)+gm(1,2)*gm(2,2)*(72*gm(2,2)&
&   *dgm01(1,1)+180*gm(1,1)*dgm01(2,2)))*dgm10(1,1)+4*(72*gm(1,2)&
&   **3*dgm01(1,2)+360*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2)+gm(1,1)&
&   *gm(2,2)*(144*gm(2,2)*dgm01(1,1)-36*gm(1,1)*dgm01(2,2))+gm(1,2)&
&   **2*(72*gm(2,2)*dgm01(1,1)+108*gm(1,1)*dgm01(2,2)))*dgm10(1,2)&
&   +(72*gm(1,2)**3*dgm01(1,1)+360*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,1)&
&   +432*gm(1,1)*gm(1,2)**2*dgm01(1,2)-144*gm(1,1)**2*gm(2,2)*dgm01(1,2))&
&   *dgm10(2,2))/48.d0
   cm(3,7)=(2*(2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2))*dgm10(1,1)&
&   +4*(6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(1,2)+2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(2,2))*dgm10(1,2)+(gm(1,2)*(72*gm(1,3)**2-216*gm(1,1)*gm(3,3))&
&   *dgm01(1,1)+gm(1,1)*(576*gm(1,3)*gm(2,3)*dgm01(1,1)+432*gm(1,3)&
&   **2*dgm01(1,2)-144*gm(1,1)*gm(3,3)*dgm01(1,2)))*dgm10(2,2))/48.d0
   cm(4,7)=gm(2,3)*(6*gm(1,2)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))+gm(1,2)*(9*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&   *(3*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)&
&   *dgm10(2,2)))+gm(1,1)*(24*gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))-6*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))))&
&   +gm(1,3)*(-3*gm(2,2)**2*dgm01(1,1)*dgm10(1,1)+gm(2,2)*(6*gm(1,2)&
&   *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)*(12*dgm01(2,2)&
&   *dgm10(1,1)+48*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)))&
&   +gm(1,2)*(gm(1,2)*(3*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&   +3*dgm01(1,1)*dgm10(2,2))+18*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))))
   cm(5,7)=(2*(gm(1,2)**2*(180*gm(2,3)*dgm01(1,1)+72*gm(1,3)*dgm01(1,2))&
&   +gm(1,1)*(gm(2,2)*(-36*gm(2,3)*dgm01(1,1)-216*gm(1,3)*dgm01(1,2))&
&   +144*gm(1,1)*gm(2,3)*dgm01(2,2))+gm(1,2)*(576*gm(1,1)*gm(2,3)&
&   *dgm01(1,2)+gm(1,3)*(-72*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))))&
&   *dgm10(1,1)+4*(36*gm(1,2)**2*gm(1,3)*dgm01(1,1)+gm(1,1)*gm(1,2)&
&   *(288*gm(2,3)*dgm01(1,1)+144*gm(1,3)*dgm01(1,2))+gm(1,1)*(288*gm(1,1)&
&   *gm(2,3)*dgm01(1,2)+gm(1,3)*(-108*gm(2,2)*dgm01(1,1)+72*gm(1,1)&
&   *dgm01(2,2))))*dgm10(1,2)+gm(1,1)*(144*gm(1,2)*gm(1,3)*dgm01(1,1)&
&   +288*gm(1,1)*(gm(2,3)*dgm01(1,1)+gm(1,3)*dgm01(1,2)))*dgm10(2,2))&
&   /48.d0
   cm(6,7)=3*gm(1,2)**3*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&   +gm(1,2)**2*(4.5d0*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(2,2)&
&   *dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)))&
&   +gm(1,1)*gm(2,2)*(-1.5d0*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&   *(6*dgm01(2,2)*dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)+6*dgm01(1,1)&
&   *dgm10(2,2)))+gm(1,1)*gm(1,2)*(15*gm(2,2)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2))+6*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)))
   cm(7,7)=(2*gm(2,2)*(72*gm(2,2)**2*dgm01(1,1)+324*gm(1,2)**2*dgm01(2,2)&
&   +gm(2,2)*(432*gm(1,2)*dgm01(1,2)-108*gm(1,1)*dgm01(2,2)))*dgm10(1,1)&
&   +4*(648*gm(1,2)**2*gm(2,2)*dgm01(1,2)-216*gm(1,1)*gm(2,2)**2*dgm01(1,2)&
&   +180*gm(1,2)**3*dgm01(2,2)+gm(1,2)*gm(2,2)*(216*gm(2,2)*dgm01(1,1)&
&   -108*gm(1,1)*dgm01(2,2)))*dgm10(1,2)+(648*gm(1,2)**2*gm(2,2)&
&   *dgm01(1,1)-216*gm(1,1)*gm(2,2)**2*dgm01(1,1)+720*gm(1,2)**3*dgm01(1,2)&
&   -432*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2))*dgm10(2,2))/144.d0
   cm(8,7)=(2*(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&   +12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2))*dgm10(1,1)+4*(6*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(2,2))*dgm10(1,2)+(12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+8*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2))*dgm10(2,2))&
&   /48.d0
   cm(9,7)=(2*(gm(2,2)**2*(72*gm(2,3)*dgm01(1,1)+288*gm(1,3)*dgm01(1,2))&
&   +36*gm(1,2)**2*gm(2,3)*dgm01(2,2)+gm(2,2)*(-108*gm(1,1)*gm(2,3)&
&   *dgm01(2,2)+gm(1,2)*(144*gm(2,3)*dgm01(1,2)+288*gm(1,3)*dgm01(2,2))))&
&   *dgm10(1,1)+4*(gm(2,3)*(72*gm(1,2)**2*dgm01(1,2)-216*gm(1,1)&
&   *gm(2,2)*dgm01(1,2)+gm(1,2)*(72*gm(2,2)*dgm01(1,1)-72*gm(1,1)&
&   *dgm01(2,2)))+gm(1,3)*(144*gm(2,2)**2*dgm01(1,1)+180*gm(1,2)&
&   **2*dgm01(2,2)+gm(2,2)*(576*gm(1,2)*dgm01(1,2)-36*gm(1,1)*dgm01(2,2))))&
&   *dgm10(1,2)+(gm(1,1)*gm(2,2)*(-216*gm(2,3)*dgm01(1,1)-144*gm(1,3)&
&   *dgm01(1,2))+gm(1,2)**2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,3)*dgm01(1,2))&
&   +gm(1,2)*(576*gm(1,3)*gm(2,2)*dgm01(1,1)-288*gm(1,1)*gm(2,3)&
&   *dgm01(1,2)))*dgm10(2,2))/48.d0
   cm(10,7)=(2*(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&   +12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+6*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(2,2))*dgm10(1,1)+4*(6*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+12*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&   *gm(2,3))*gm(3,3))*dgm01(1,2)+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm01(2,2))*dgm10(1,2)+(12*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,1)+8*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(1,2))*dgm10(2,2))/144.d0
   cm(1,8)=(6*(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)+4*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(1,2)+8*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)&
&   *gm(2,3))*dgm01(1,3)+144*gm(1,1)**2*gm(1,3)*dgm01(2,3)+72*gm(1,1)&
&   **2*gm(1,2)*dgm01(3,3))*dgm10(1,1)+4*gm(1,1)*(6*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(1,1)+432*gm(1,1)*gm(1,3)*dgm01(1,3)+72*gm(1,1)&
&   **2*dgm01(3,3))*dgm10(1,2)+576*gm(1,1)*(gm(1,2)*(4.5d0*gm(1,3)&
&   *dgm01(1,1)+3*gm(1,1)*dgm01(1,3))+gm(1,1)*(-1.5d0*gm(2,3)*dgm01(1,1)&
&   +3*gm(1,3)*dgm01(1,2)+gm(1,1)*dgm01(2,3)))*dgm10(1,3)+2*gm(1,1)&
&   **2*(432*gm(1,3)*dgm01(1,1)+288*gm(1,1)*dgm01(1,3))*dgm10(2,3)&
&   +gm(1,1)**2*(432*gm(1,2)*dgm01(1,1)+288*gm(1,1)*dgm01(1,2))*dgm10(3,3))&
&   /144.d0
   cm(2,8)=(12*((48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+2*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+4*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)&
&   +2*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,1)+4*(6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3)+2*gm(1,1)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(1,2)+8*(36*gm(1,2)&
&   **3*dgm01(1,3)+gm(1,2)*(36*gm(1,3)*gm(2,2)*dgm01(1,1)+gm(1,1)&
&   *(288*gm(2,3)*dgm01(1,2)+180*gm(2,2)*dgm01(1,3)))+gm(1,1)*gm(2,2)&
&   *(144*gm(2,3)*dgm01(1,1)-108*gm(1,3)*dgm01(1,2)-36*gm(1,1)*dgm01(2,3))&
&   +gm(1,2)**2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,3)*dgm01(1,2)+108*gm(1,1)&
&   *dgm01(2,3)))*dgm10(1,3)+2*(576*gm(1,1)*gm(1,2)*gm(2,3)*dgm01(1,1)&
&   +gm(1,1)*gm(2,2)*(-216*gm(1,3)*dgm01(1,1)-144*gm(1,1)*dgm01(1,3))&
&   +gm(1,2)**2*(72*gm(1,3)*dgm01(1,1)+432*gm(1,1)*dgm01(1,3)))*dgm10(2,3)&
&   +(72*gm(1,2)**3*dgm01(1,1)+360*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,1)&
&   +432*gm(1,1)*gm(1,2)**2*dgm01(1,2)-144*gm(1,1)**2*gm(2,2)*dgm01(1,2))&
&   *dgm10(3,3))/48.d0
   cm(3,8)=(6*(24*gm(1,3)**3*dgm01(2,3)+gm(1,3)**2*(48*gm(3,3)&
&   *dgm01(1,2)+48*gm(2,3)*dgm01(1,3)+12*gm(1,2)*dgm01(3,3))+gm(3,3)&
&   *(gm(1,1)*(96*gm(3,3)*dgm01(1,2)+192*gm(2,3)*dgm01(1,3))+gm(1,2)&
&   *(48*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3)))+gm(1,3)*(gm(3,3)&
&   *(48*gm(1,2)*dgm01(1,3)+120*gm(1,1)*dgm01(2,3))+gm(2,3)*(24*gm(3,3)&
&   *dgm01(1,1)+96*gm(1,1)*dgm01(3,3))))*dgm10(1,1)+4*(72*gm(1,3)&
&   **3*dgm01(1,3)+360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3)+gm(1,1)&
&   *gm(3,3)*(144*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3))+gm(1,3)&
&   **2*(72*gm(3,3)*dgm01(1,1)+108*gm(1,1)*dgm01(3,3)))*dgm10(1,2)&
&   +8*(36*gm(1,3)**3*dgm01(1,2)+gm(1,3)*(36*gm(1,2)*gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*(180*gm(3,3)*dgm01(1,2)+288*gm(2,3)*dgm01(1,3)))+gm(1,1)&
&   *gm(3,3)*(144*gm(2,3)*dgm01(1,1)-108*gm(1,2)*dgm01(1,3)-36*gm(1,1)&
&   *dgm01(2,3))+gm(1,3)**2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,2)*dgm01(1,3)&
&   +108*gm(1,1)*dgm01(2,3)))*dgm10(1,3)+2*(72*gm(1,3)**3*dgm01(1,1)&
&   +360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,1)+432*gm(1,1)*gm(1,3)**2*dgm01(1,3)&
&   -144*gm(1,1)**2*gm(3,3)*dgm01(1,3))*dgm10(2,3)+(gm(1,2)*(72*gm(1,3)&
&   **2-216*gm(1,1)*gm(3,3))*dgm01(1,1)+gm(1,1)*(576*gm(1,3)*gm(2,3)&
&   *dgm01(1,1)+432*gm(1,3)**2*dgm01(1,2)-144*gm(1,1)*gm(3,3)*dgm01(1,2)))&
&   *dgm10(3,3))/48.d0
   cm(4,8)=18*gm(1,2)**2*gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))+gm(1,3)**2*(6*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))+18*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,2)*(6*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)))+gm(1,2)*(gm(1,1)*gm(3,3)&
&   *(24*dgm01(2,3)*dgm10(1,1)+48*dgm01(1,3)*dgm10(1,2)+48*dgm01(1,2)&
&   *dgm10(1,3)+24*dgm01(1,1)*dgm10(2,3))+gm(2,3)*(3*gm(3,3)*dgm01(1,1)&
&   *dgm10(1,1)+gm(1,1)*(3*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&   +3*dgm01(1,1)*dgm10(3,3))))+gm(1,1)*(18*gm(2,3)**2*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+30*gm(2,2)*gm(3,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(2,3)*(24*gm(3,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)*(-6*dgm01(3,3)*dgm10(1,2)&
&   -12*dgm01(2,3)*dgm10(1,3)-12*dgm01(1,3)*dgm10(2,3)-6*dgm01(1,2)&
&   *dgm10(3,3))))+gm(1,3)*(3*gm(2,3)**2*dgm01(1,1)*dgm10(1,1)+gm(2,3)&
&   *(-12*gm(1,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)&
&   *(6*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)))+gm(2,2)*(12*gm(3,3)*dgm01(1,1)&
&   *dgm10(1,1)+gm(1,1)*(12*dgm01(3,3)*dgm10(1,1)+48*dgm01(1,3)*dgm10(1,3)&
&   +12*dgm01(1,1)*dgm10(3,3)))+gm(1,2)*(6*gm(3,3)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2))+gm(1,2)*(3*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)&
&   *dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3))+gm(1,1)*(18*dgm01(3,3)*dgm10(1,2)&
&   +36*dgm01(2,3)*dgm10(1,3)+36*dgm01(1,3)*dgm10(2,3)+18*dgm01(1,2)&
&   *dgm10(3,3))))
   cm(5,8)=(6*(24*gm(1,3)**3*dgm01(1,2)+gm(1,3)**2*(12*gm(2,3)&
&   *dgm01(1,1)+48*(gm(1,2)*dgm01(1,3)+gm(1,1)*dgm01(2,3)))+gm(1,3)&
&   *(gm(1,1)*(120*gm(3,3)*dgm01(1,2)+48*gm(2,3)*dgm01(1,3))+gm(1,2)&
&   *(96*gm(3,3)*dgm01(1,1)+24*gm(1,1)*dgm01(3,3)))+gm(1,1)*(gm(3,3)&
&   *(192*gm(1,2)*dgm01(1,3)+96*gm(1,1)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)&
&   *dgm01(1,1)+48*gm(1,1)*dgm01(3,3))))*dgm10(1,1)+4*(36*gm(1,3)&
&   **3*dgm01(1,1)+144*gm(1,1)*gm(1,3)**2*dgm01(1,3)+288*gm(1,1)&
&   **2*gm(3,3)*dgm01(1,3)+gm(1,1)*gm(1,3)*(180*gm(3,3)*dgm01(1,1)&
&   +72*gm(1,1)*dgm01(3,3)))*dgm10(1,2)+8*(gm(1,2)*(36*gm(1,3)**2*dgm01(1,1)&
&   +144*gm(1,1)*gm(3,3)*dgm01(1,1)+72*gm(1,1)*gm(1,3)*dgm01(1,3))&
&   +gm(1,1)*(72*gm(1,3)**2*dgm01(1,2)+144*gm(1,1)*(gm(3,3)*dgm01(1,2)&
&   +gm(2,3)*dgm01(1,3))+gm(1,3)*(36*gm(2,3)*dgm01(1,1)+72*gm(1,1)&
&   *dgm01(2,3))))*dgm10(1,3)+2*gm(1,1)*(144*gm(1,3)**2*dgm01(1,1)&
&   +288*gm(1,1)*gm(3,3)*dgm01(1,1)+288*gm(1,1)*gm(1,3)*dgm01(1,3))&
&   *dgm10(2,3)+gm(1,1)*(144*gm(1,2)*gm(1,3)*dgm01(1,1)+288*gm(1,1)&
&   *(gm(2,3)*dgm01(1,1)+gm(1,3)*dgm01(1,2)))*dgm10(3,3))/48.d0
   cm(6,8)=(6*(-24*gm(1,2)**2*gm(3,3)*dgm01(1,1)+gm(1,3)**2*(60*gm(2,2)&
&   *dgm01(1,1)+24*gm(1,2)*dgm01(1,2))+gm(1,3)*(48*gm(1,2)**2*dgm01(1,3)&
&   +192*gm(1,1)*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3))+gm(1,2)&
&   *(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)*dgm01(2,3)))+gm(1,1)*(-24*gm(2,3)&
&   **2*dgm01(1,1)-12*gm(2,2)*gm(3,3)*dgm01(1,1)-72*gm(1,2)*gm(3,3)&
&   *dgm01(1,2)+48*gm(1,2)*gm(2,3)*dgm01(1,3)+24*gm(1,2)**2*dgm01(3,3))&
&   +gm(1,1)**2*(96*gm(2,3)*dgm01(2,3)+48*gm(2,2)*dgm01(3,3)))*dgm10(1,1)&
&   +4*(288*gm(1,1)*gm(2,3)*(gm(1,3)*dgm01(1,1)+gm(1,1)*dgm01(1,3))&
&   +gm(1,2)*(36*gm(1,3)**2*dgm01(1,1)+144*gm(1,1)*gm(1,3)*dgm01(1,3)&
&   +gm(1,1)*(-108*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3))))*dgm10(1,2)&
&   +8*(gm(1,2)**2*(36*gm(1,3)*dgm01(1,1)+72*gm(1,1)*dgm01(1,3))&
&   +144*gm(1,1)*(gm(1,3)*gm(2,2)*dgm01(1,1)+gm(1,1)*(gm(2,3)*dgm01(1,2)&
&   +gm(2,2)*dgm01(1,3)))+gm(1,1)*gm(1,2)*(36*gm(2,3)*dgm01(1,1)&
&   +72*(gm(1,3)*dgm01(1,2)+gm(1,1)*dgm01(2,3))))*dgm10(1,3)+2*gm(1,1)&
&   *(288*gm(1,1)*gm(2,3)*dgm01(1,1)+gm(1,2)*(144*gm(1,3)*dgm01(1,1)&
&   +288*gm(1,1)*dgm01(1,3)))*dgm10(2,3)+gm(1,1)*(144*gm(1,2)**2*dgm01(1,1)&
&   +288*gm(1,1)*gm(2,2)*dgm01(1,1)+288*gm(1,1)*gm(1,2)*dgm01(1,2))&
&   *dgm10(3,3))/48.d0
   cm(7,8)=(12*(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&   +2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+2*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(2,3)+gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,1)+4*(6*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(90*gm(1,2)&
&   **2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,3)&
&   +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(1,2)&
&   +8*(6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,1)&
&   +6*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&   *gm(2,3)))*dgm01(1,2)+6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&   *dgm01(1,3)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))&
&   *dgm10(1,3)+2*(12*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,1)+8*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(1,3))*dgm10(2,3)+(648*gm(1,2)**2*gm(2,2)&
&   *dgm01(1,1)-216*gm(1,1)*gm(2,2)**2*dgm01(1,1)+720*gm(1,2)**3*dgm01(1,2)&
&   -432*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2))*dgm10(3,3))/144.d0
   cm(8,8)=(12*(gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,1)&
&   +2*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +4*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)&
&   *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,1)+4*(gm(1,3)**2*(72*gm(2,3)&
&   *dgm01(1,3)+180*gm(1,2)*dgm01(3,3))+gm(1,3)*(576*gm(1,2)*gm(3,3)&
&   *dgm01(1,3)+gm(2,3)*(72*gm(3,3)*dgm01(1,1)-72*gm(1,1)*dgm01(3,3)))&
&   +gm(3,3)*(-216*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)*(144*gm(3,3)&
&   *dgm01(1,1)-36*gm(1,1)*dgm01(3,3))))*dgm10(1,2)+8*(6*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3))*dgm10(1,3)+2*(gm(1,1)&
&   *gm(3,3)*(-216*gm(2,3)*dgm01(1,1)-144*gm(1,2)*dgm01(1,3))+gm(1,3)&
&   **2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,2)*dgm01(1,3))+gm(1,3)*(576*gm(1,2)&
&   *gm(3,3)*dgm01(1,1)-288*gm(1,1)*gm(2,3)*dgm01(1,3)))*dgm10(2,3)&
&   +(12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +8*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(1,2))*dgm10(3,3))/48.d0
   cm(9,8)=(12*((6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&   +2*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)+2*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+(48*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,1)+4*(6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,2)&
&   +8*(6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)+2*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(2,3))*dgm10(1,3)+2*(12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+8*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3))*dgm10(2,3)+(gm(1,1)&
&   *gm(2,2)*(-216*gm(2,3)*dgm01(1,1)-144*gm(1,3)*dgm01(1,2))+gm(1,2)&
&   **2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,3)*dgm01(1,2))+gm(1,2)*(576*gm(1,3)&
&   *gm(2,2)*dgm01(1,1)-288*gm(1,1)*gm(2,3)*dgm01(1,2)))*dgm10(3,3))&
&   /48.d0
   cm(10,8)=(6*(72*gm(2,3)*gm(3,3)**2*dgm01(1,1)+144*gm(1,3)*gm(3,3)&
&   **2*dgm01(1,2)+8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(1,3)+4*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)&
&   +2*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))&
&   *gm(3,3))*dgm01(3,3))*dgm10(1,1)+4*(648*gm(1,3)**2*gm(3,3)*dgm01(1,3)&
&   -216*gm(1,1)*gm(3,3)**2*dgm01(1,3)+180*gm(1,3)**3*dgm01(3,3)&
&   +gm(1,3)*gm(3,3)*(216*gm(3,3)*dgm01(1,1)-108*gm(1,1)*dgm01(3,3)))&
&   *dgm10(1,2)+8*(6*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(1,1)+6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)&
&   +6*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))&
&   *gm(3,3))*dgm01(1,3)+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))&
&   *dgm01(2,3))*dgm10(1,3)+2*(648*gm(1,3)**2*gm(3,3)*dgm01(1,1)&
&   -216*gm(1,1)*gm(3,3)**2*dgm01(1,1)+720*gm(1,3)**3*dgm01(1,3)&
&   -432*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3))*dgm10(2,3)+(12*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,1)&
&   +8*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2))*dgm10(3,3))&
&   /144.d0
   cm(1,9)=(6*(2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&   +2*gm(1,2)*gm(2,3)))*dgm01(1,1)+8*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&   -18*gm(1,1)*gm(2,3))*dgm01(1,2)+4*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,3)+72*gm(1,1)**2*gm(1,3)*dgm01(2,2)+144*gm(1,1)&
&   **2*gm(1,2)*dgm01(2,3))*dgm10(1,1)+576*gm(1,1)*(gm(1,2)*(4.5d0*gm(1,3)&
&   *dgm01(1,1)+3*gm(1,1)*dgm01(1,3))+gm(1,1)*(-1.5d0*gm(2,3)*dgm01(1,1)&
&   +3*gm(1,3)*dgm01(1,2)+gm(1,1)*dgm01(2,3)))*dgm10(1,2)+4*gm(1,1)&
&   *(6*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,1)+432*gm(1,1)&
&   *gm(1,2)*dgm01(1,2)+72*gm(1,1)**2*dgm01(2,2))*dgm10(1,3)+gm(1,1)&
&   **2*(432*gm(1,3)*dgm01(1,1)+288*gm(1,1)*dgm01(1,3))*dgm10(2,2)&
&   +2*gm(1,1)**2*(432*gm(1,2)*dgm01(1,1)+288*gm(1,1)*dgm01(1,2))&
&   *dgm10(2,3))/144.d0
   cm(2,9)=(6*(48*gm(1,2)**2*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3))&
&   +gm(1,1)*gm(2,2)*(192*gm(2,3)*dgm01(1,2)+96*gm(2,2)*dgm01(1,3))&
&   +gm(1,3)*(48*gm(2,2)**2*dgm01(1,1)+12*gm(1,2)**2*dgm01(2,2)+gm(2,2)&
&   *(48*gm(1,2)*dgm01(1,2)-36*gm(1,1)*dgm01(2,2)))+24*gm(1,2)**3*dgm01(2,3)&
&   +gm(1,2)*(96*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(2,2)*(24*gm(2,3)*dgm01(1,1)&
&   +120*gm(1,1)*dgm01(2,3))))*dgm10(1,1)+8*(36*gm(1,2)**3*dgm01(1,3)&
&   +gm(1,2)*(36*gm(1,3)*gm(2,2)*dgm01(1,1)+gm(1,1)*(288*gm(2,3)&
&   *dgm01(1,2)+180*gm(2,2)*dgm01(1,3)))+gm(1,1)*gm(2,2)*(144*gm(2,3)&
&   *dgm01(1,1)-108*gm(1,3)*dgm01(1,2)-36*gm(1,1)*dgm01(2,3))+gm(1,2)&
&   **2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,3)*dgm01(1,2)+108*gm(1,1)&
&   *dgm01(2,3)))*dgm10(1,2)+4*(72*gm(1,2)**3*dgm01(1,2)+360*gm(1,1)&
&   *gm(1,2)*gm(2,2)*dgm01(1,2)+gm(1,1)*gm(2,2)*(144*gm(2,2)*dgm01(1,1)&
&   -36*gm(1,1)*dgm01(2,2))+gm(1,2)**2*(72*gm(2,2)*dgm01(1,1)+108*gm(1,1)&
&   *dgm01(2,2)))*dgm10(1,3)+(576*gm(1,1)*gm(1,2)*gm(2,3)*dgm01(1,1)&
&   +gm(1,1)*gm(2,2)*(-216*gm(1,3)*dgm01(1,1)-144*gm(1,1)*dgm01(1,3))&
&   +gm(1,2)**2*(72*gm(1,3)*dgm01(1,1)+432*gm(1,1)*dgm01(1,3)))*dgm10(2,2)&
&   +2*(72*gm(1,2)**3*dgm01(1,1)+360*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,1)&
&   +432*gm(1,1)*gm(1,2)**2*dgm01(1,2)-144*gm(1,1)**2*gm(2,2)*dgm01(1,2))&
&   *dgm10(2,3))/48.d0
   cm(3,9)=(6*(-48*gm(1,2)**2*gm(3,3)*dgm01(1,3)+gm(1,1)*(192*gm(2,3)&
&   *gm(3,3)*dgm01(1,2)+120*gm(2,3)**2*dgm01(1,3)-24*gm(2,2)*gm(3,3)&
&   *dgm01(1,3))+12*gm(1,3)**3*dgm01(2,2)+gm(1,2)*gm(3,3)*(96*gm(2,3)&
&   *dgm01(1,1)-72*gm(1,1)*dgm01(2,3))+gm(1,3)**2*(48*gm(2,3)*dgm01(1,2)&
&   -48*gm(2,2)*dgm01(1,3)+24*gm(1,2)*dgm01(2,3))+gm(1,3)*(12*gm(2,3)&
&   **2*dgm01(1,1)+gm(3,3)*(-36*gm(2,2)*dgm01(1,1)+48*gm(1,2)*dgm01(1,2)&
&   +60*gm(1,1)*dgm01(2,2))+gm(2,3)*(144*gm(1,2)*dgm01(1,3)+192*gm(1,1)&
&   *dgm01(2,3))))*dgm10(1,1)+8*(36*gm(1,3)**3*dgm01(1,2)+gm(1,3)&
&   *(36*gm(1,2)*gm(3,3)*dgm01(1,1)+gm(1,1)*(180*gm(3,3)*dgm01(1,2)&
&   +288*gm(2,3)*dgm01(1,3)))+gm(1,1)*gm(3,3)*(144*gm(2,3)*dgm01(1,1)&
&   -108*gm(1,2)*dgm01(1,3)-36*gm(1,1)*dgm01(2,3))+gm(1,3)**2*(36*gm(2,3)&
&   *dgm01(1,1)+36*gm(1,2)*dgm01(1,3)+108*gm(1,1)*dgm01(2,3)))*dgm10(1,2)&
&   +4*(6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(1,2)+2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(2,2))*dgm10(1,3)+(72*gm(1,3)**3*dgm01(1,1)+360*gm(1,1)&
&   *gm(1,3)*gm(3,3)*dgm01(1,1)+432*gm(1,1)*gm(1,3)**2*dgm01(1,3)&
&   -144*gm(1,1)**2*gm(3,3)*dgm01(1,3))*dgm10(2,2)+2*(gm(1,2)*(72*gm(1,3)&
&   **2-216*gm(1,1)*gm(3,3))*dgm01(1,1)+gm(1,1)*(576*gm(1,3)*gm(2,3)&
&   *dgm01(1,1)+432*gm(1,3)**2*dgm01(1,2)-144*gm(1,1)*gm(3,3)*dgm01(1,2)))&
&   *dgm10(2,3))/48.d0
   cm(4,9)=gm(1,2)**2*(18*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))+6*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))&
&   +gm(1,3)**2*(18*gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&   +gm(1,2)*(3*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)&
&   *dgm10(2,2)))+gm(1,2)*(3*gm(2,3)**2*dgm01(1,1)*dgm10(1,1)+gm(3,3)&
&   *(12*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(12*dgm01(2,2)*dgm10(1,1)&
&   +48*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)))+gm(1,1)&
&   *gm(2,3)*(6*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)))+gm(1,1)*(18*gm(2,3)**2*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+30*gm(2,2)*gm(3,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(2,3)*(24*gm(2,2)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)*(-12*dgm01(2,3)*dgm10(1,2)&
&   -6*dgm01(2,2)*dgm10(1,3)-6*dgm01(1,3)*dgm10(2,2)-12*dgm01(1,2)&
&   *dgm10(2,3))))+gm(1,3)*(gm(1,1)*gm(2,3)*(3*dgm01(2,2)*dgm10(1,1)&
&   +12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))+gm(1,2)**2*(6*dgm01(2,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)&
&   +6*dgm01(1,1)*dgm10(2,3))+gm(2,2)*(3*gm(2,3)*dgm01(1,1)*dgm10(1,1)&
&   +6*gm(1,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)&
&   *(24*dgm01(2,3)*dgm10(1,1)+48*dgm01(1,3)*dgm10(1,2)+48*dgm01(1,2)&
&   *dgm10(1,3)+24*dgm01(1,1)*dgm10(2,3)))+gm(1,2)*(-12*gm(2,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)*(36*dgm01(2,3)*dgm10(1,2)&
&   +18*dgm01(2,2)*dgm10(1,3)+18*dgm01(1,3)*dgm10(2,2)+36*dgm01(1,2)&
&   *dgm10(2,3))))
   cm(5,9)=(6*(60*gm(1,2)**2*gm(3,3)*dgm01(1,1)+gm(1,1)*(-24*gm(2,3)&
&   **2*dgm01(1,1)+gm(3,3)*(-12*gm(2,2)*dgm01(1,1)+192*gm(1,2)*dgm01(1,2))&
&   +192*gm(1,2)*gm(2,3)*dgm01(1,3))+gm(1,3)**2*(-24*gm(2,2)*dgm01(1,1)&
&   +48*gm(1,2)*dgm01(1,2)+24*gm(1,1)*dgm01(2,2))+gm(1,1)**2*(48*gm(3,3)&
&   *dgm01(2,2)+96*gm(2,3)*dgm01(2,3))+gm(1,3)*(24*gm(1,2)**2*dgm01(1,3)&
&   +gm(1,1)*(48*gm(2,3)*dgm01(1,2)-72*gm(2,2)*dgm01(1,3))+gm(1,2)&
&   *(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)*dgm01(2,3))))*dgm10(1,1)+8*(gm(1,2)&
&   *(36*gm(1,3)**2*dgm01(1,1)+144*gm(1,1)*gm(3,3)*dgm01(1,1)+72*gm(1,1)&
&   *gm(1,3)*dgm01(1,3))+gm(1,1)*(72*gm(1,3)**2*dgm01(1,2)+144*gm(1,1)&
&   *(gm(3,3)*dgm01(1,2)+gm(2,3)*dgm01(1,3))+gm(1,3)*(36*gm(2,3)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(2,3))))*dgm10(1,2)+4*(36*gm(1,2)&
&   **2*gm(1,3)*dgm01(1,1)+gm(1,1)*gm(1,2)*(288*gm(2,3)*dgm01(1,1)&
&   +144*gm(1,3)*dgm01(1,2))+gm(1,1)*(288*gm(1,1)*gm(2,3)*dgm01(1,2)&
&   +gm(1,3)*(-108*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))))*dgm10(1,3)&
&   +gm(1,1)*(144*gm(1,3)**2*dgm01(1,1)+288*gm(1,1)*gm(3,3)*dgm01(1,1)&
&   +288*gm(1,1)*gm(1,3)*dgm01(1,3))*dgm10(2,2)+2*gm(1,1)*(144*gm(1,2)&
&   *gm(1,3)*dgm01(1,1)+288*gm(1,1)*(gm(2,3)*dgm01(1,1)+gm(1,3)*dgm01(1,2)))&
&   *dgm10(2,3))/48.d0
   cm(6,9)=(6*(24*gm(1,2)**3*dgm01(1,3)+gm(1,2)*(gm(1,1)*(48*gm(2,3)&
&   *dgm01(1,2)+120*gm(2,2)*dgm01(1,3))+gm(1,3)*(96*gm(2,2)*dgm01(1,1)&
&   +24*gm(1,1)*dgm01(2,2)))+gm(1,2)**2*(12*gm(2,3)*dgm01(1,1)+48*(gm(1,3)&
&   *dgm01(1,2)+gm(1,1)*dgm01(2,3)))+gm(1,1)*(48*gm(1,1)*gm(2,3)&
&   *dgm01(2,2)+gm(2,2)*(-36*gm(2,3)*dgm01(1,1)+192*gm(1,3)*dgm01(1,2)&
&   +96*gm(1,1)*dgm01(2,3))))*dgm10(1,1)+8*(gm(1,2)**2*(36*gm(1,3)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(1,3))+144*gm(1,1)*(gm(1,3)*gm(2,2)&
&   *dgm01(1,1)+gm(1,1)*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3)))&
&   +gm(1,1)*gm(1,2)*(36*gm(2,3)*dgm01(1,1)+72*(gm(1,3)*dgm01(1,2)&
&   +gm(1,1)*dgm01(2,3))))*dgm10(1,2)+4*(36*gm(1,2)**3*dgm01(1,1)&
&   +144*gm(1,1)*gm(1,2)**2*dgm01(1,2)+288*gm(1,1)**2*gm(2,2)*dgm01(1,2)&
&   +gm(1,1)*gm(1,2)*(180*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2)))&
&   *dgm10(1,3)+gm(1,1)*(288*gm(1,1)*gm(2,3)*dgm01(1,1)+gm(1,2)*(144*gm(1,3)&
&   *dgm01(1,1)+288*gm(1,1)*dgm01(1,3)))*dgm10(2,2)+2*gm(1,1)*(144*gm(1,2)&
&   **2*dgm01(1,1)+288*gm(1,1)*gm(2,2)*dgm01(1,1)+288*gm(1,1)*gm(1,2)&
&   *dgm01(1,2))*dgm10(2,3))/48.d0
   cm(7,9)=(6*(180*gm(1,2)**2*gm(2,3)*dgm01(2,2)+gm(2,2)**2*(72*gm(2,3)&
&   *dgm01(1,1)-144*gm(1,3)*dgm01(1,2)+144*gm(1,2)*dgm01(1,3)-72*gm(1,1)&
&   *dgm01(2,3))+gm(2,2)*(-36*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(1,2)&
&   *(432*gm(2,3)*dgm01(1,2)-72*gm(1,3)*dgm01(2,2))+216*gm(1,2)**2*dgm01(2,3)))&
&   *dgm10(1,1)+8*(6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(1,1)+6*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3)))*dgm01(1,2)+6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,3)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))&
&   *dgm01(2,3))*dgm10(1,2)+4*(648*gm(1,2)**2*gm(2,2)*dgm01(1,2)&
&   -216*gm(1,1)*gm(2,2)**2*dgm01(1,2)+180*gm(1,2)**3*dgm01(2,2)&
&   +gm(1,2)*gm(2,2)*(216*gm(2,2)*dgm01(1,1)-108*gm(1,1)*dgm01(2,2)))&
&   *dgm10(1,3)+(12*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,1)+8*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(1,3))*dgm10(2,2)+2*(648*gm(1,2)**2*gm(2,2)&
&   *dgm01(1,1)-216*gm(1,1)*gm(2,2)**2*dgm01(1,1)+720*gm(1,2)**3*dgm01(1,2)&
&   -432*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2))*dgm10(2,3))/144.d0
   cm(8,9)=(6*(12*gm(2,3)**3*dgm01(1,1)+gm(1,3)*gm(3,3)*(192*gm(2,2)&
&   *dgm01(1,2)+96*gm(1,2)*dgm01(2,2))+120*gm(1,3)**2*gm(2,2)*dgm01(2,3)&
&   +gm(2,3)**2*(48*gm(1,3)*dgm01(1,2)+24*gm(1,2)*dgm01(1,3)-48*gm(1,1)&
&   *dgm01(2,3))+gm(3,3)*(-72*gm(1,2)*gm(2,2)*dgm01(1,3)-48*gm(1,2)&
&   **2*dgm01(2,3)-24*gm(1,1)*gm(2,2)*dgm01(2,3))+gm(2,3)*(gm(2,2)&
&   *(60*gm(3,3)*dgm01(1,1)+192*gm(1,3)*dgm01(1,3))+(12*gm(1,3)**2-36*gm(1,1)&
&   *gm(3,3))*dgm01(2,2)+gm(1,2)*(48*gm(3,3)*dgm01(1,2)+144*gm(1,3)&
&   *dgm01(2,3))))*dgm10(1,1)+8*(6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)+6*(6*gm(1,3)**2*gm(2,3)&
&   +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(2,3))*dgm10(1,2)+4*(6*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(2,2))*dgm10(1,3)+(gm(1,1)*gm(3,3)*(-216*gm(2,3)*dgm01(1,1)&
&   -144*gm(1,2)*dgm01(1,3))+gm(1,3)**2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,2)&
&   *dgm01(1,3))+gm(1,3)*(576*gm(1,2)*gm(3,3)*dgm01(1,1)-288*gm(1,1)&
&   *gm(2,3)*dgm01(1,3)))*dgm10(2,2)+2*(12*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)+8*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2))*dgm10(2,3))&
&   /48.d0
   cm(9,9)=(6*(gm(2,2)**2*(48*gm(3,3)*dgm01(1,1)+96*gm(1,3)*dgm01(1,3))&
&   -24*gm(1,1)*gm(2,3)**2*dgm01(2,2)+gm(1,2)*gm(2,3)*(48*gm(2,3)&
&   *dgm01(1,2)+72*gm(1,3)*dgm01(2,2))+gm(1,2)**2*(60*gm(3,3)*dgm01(2,2)&
&   +24*gm(2,3)*dgm01(2,3))+gm(2,2)*(24*gm(2,3)**2*dgm01(1,1)+(-24*gm(1,3)&
&   **2-12*gm(1,1)*gm(3,3))*dgm01(2,2)+gm(2,3)*(48*gm(1,3)*dgm01(1,2)&
&   +48*gm(1,2)*dgm01(1,3)-72*gm(1,1)*dgm01(2,3))+192*gm(1,2)*(gm(3,3)&
&   *dgm01(1,2)+gm(1,3)*dgm01(2,3))))*dgm10(1,1)+8*(6*(6*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(1,3)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(1,2)&
&   +4*(gm(2,3)*(72*gm(1,2)**2*dgm01(1,2)-216*gm(1,1)*gm(2,2)*dgm01(1,2)&
&   +gm(1,2)*(72*gm(2,2)*dgm01(1,1)-72*gm(1,1)*dgm01(2,2)))+gm(1,3)&
&   *(144*gm(2,2)**2*dgm01(1,1)+180*gm(1,2)**2*dgm01(2,2)+gm(2,2)&
&   *(576*gm(1,2)*dgm01(1,2)-36*gm(1,1)*dgm01(2,2))))*dgm10(1,3)&
&   +(12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +8*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(1,3))*dgm10(2,2)+2*(gm(1,1)*gm(2,2)*(-216*gm(2,3)&
&   *dgm01(1,1)-144*gm(1,3)*dgm01(1,2))+gm(1,2)**2*(72*gm(2,3)*dgm01(1,1)&
&   +720*gm(1,3)*dgm01(1,2))+gm(1,2)*(576*gm(1,3)*gm(2,2)*dgm01(1,1)&
&   -288*gm(1,1)*gm(2,3)*dgm01(1,2)))*dgm10(2,3))/48.d0
   cm(10,9)=(12*(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&   +4*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(2,2)+2*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&   *gm(2,3))*gm(3,3))*dgm01(2,3))*dgm10(1,1)+8*(6*gm(3,3)*(54*gm(1,3)&
&   *gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,1)+6*gm(3,3)*(54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(1,2)+6*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,3)+2*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(2,3))*dgm10(1,2)+4*(6*(-36*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +12*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))&
&   *gm(3,3))*dgm01(1,2)+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))&
&   *dgm01(2,2))*dgm10(1,3)+(648*gm(1,3)**2*gm(3,3)*dgm01(1,1)-216*gm(1,1)&
&   *gm(3,3)**2*dgm01(1,1)+720*gm(1,3)**3*dgm01(1,3)-432*gm(1,1)&
&   *gm(1,3)*gm(3,3)*dgm01(1,3))*dgm10(2,2)+2*(12*(90*gm(1,3)**2*gm(2,3)&
&   -18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,1)+8*(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2))*dgm10(2,3))/144.d0
   cm(1,10)=(2*(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)&
&   +12*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)+216*gm(1,1)&
&   **2*gm(1,3)*dgm01(3,3))*dgm10(1,1)+4*gm(1,1)*(6*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(1,1)+432*gm(1,1)*gm(1,3)*dgm01(1,3)+72*gm(1,1)&
&   **2*dgm01(3,3))*dgm10(1,3)+gm(1,1)**2*(432*gm(1,3)*dgm01(1,1)&
&   +288*gm(1,1)*dgm01(1,3))*dgm10(3,3))/144.d0
   cm(2,10)=(2*(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,1)&
&   +4*(6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(1,3)+2*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,3)+(576*gm(1,1)*gm(1,2)*gm(2,3)&
&   *dgm01(1,1)+gm(1,1)*gm(2,2)*(-216*gm(1,3)*dgm01(1,1)-144*gm(1,1)&
&   *dgm01(1,3))+gm(1,2)**2*(72*gm(1,3)*dgm01(1,1)+432*gm(1,1)*dgm01(1,3)))&
&   *dgm10(3,3))/48.d0
   cm(3,10)=(2*(144*gm(1,3)**2*gm(3,3)*dgm01(1,3)+288*gm(1,1)*gm(3,3)&
&   **2*dgm01(1,3)+36*gm(1,3)**3*dgm01(3,3)+gm(1,3)*gm(3,3)*(72*gm(3,3)&
&   *dgm01(1,1)+180*gm(1,1)*dgm01(3,3)))*dgm10(1,1)+4*(72*gm(1,3)&
&   **3*dgm01(1,3)+360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3)+gm(1,1)&
&   *gm(3,3)*(144*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3))+gm(1,3)&
&   **2*(72*gm(3,3)*dgm01(1,1)+108*gm(1,1)*dgm01(3,3)))*dgm10(1,3)&
&   +(72*gm(1,3)**3*dgm01(1,1)+360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,1)&
&   +432*gm(1,1)*gm(1,3)**2*dgm01(1,3)-144*gm(1,1)**2*gm(3,3)*dgm01(1,3))&
&   *dgm10(3,3))/48.d0
   cm(4,10)=gm(1,3)**2*(6*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))+gm(1,2)*(3*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&   +3*dgm01(1,1)*dgm10(3,3)))+gm(1,2)*gm(3,3)*(-3*gm(3,3)*dgm01(1,1)&
&   *dgm10(1,1)+gm(1,1)*(12*dgm01(3,3)*dgm10(1,1)+48*dgm01(1,3)*dgm10(1,3)&
&   +12*dgm01(1,1)*dgm10(3,3)))+gm(1,1)*gm(2,3)*(24*gm(3,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))-6*gm(1,1)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3)))+gm(1,3)*(gm(2,3)*(9*gm(3,3)*dgm01(1,1)&
&   *dgm10(1,1)+gm(1,1)*(3*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&   +3*dgm01(1,1)*dgm10(3,3)))+gm(1,2)*(6*gm(3,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3))+18*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))))
   cm(5,10)=3*gm(1,3)**3*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,3)**2*(4.5d0*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(3,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)))&
&   +gm(1,1)*gm(3,3)*(-1.5d0*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&   *(6*dgm01(3,3)*dgm10(1,1)+24*dgm01(1,3)*dgm10(1,3)+6*dgm01(1,1)&
&   *dgm10(3,3)))+gm(1,1)*gm(1,3)*(15*gm(3,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3))+6*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3)))
   cm(6,10)=(2*(gm(1,3)**2*(180*gm(2,3)*dgm01(1,1)+72*gm(1,2)*dgm01(1,3))&
&   +gm(1,3)*(576*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)*(-72*gm(3,3)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(3,3)))+gm(1,1)*(-216*gm(1,2)*gm(3,3)&
&   *dgm01(1,3)+gm(2,3)*(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3))))&
&   *dgm10(1,1)+4*(288*gm(1,1)*gm(2,3)*(gm(1,3)*dgm01(1,1)+gm(1,1)&
&   *dgm01(1,3))+gm(1,2)*(36*gm(1,3)**2*dgm01(1,1)+144*gm(1,1)*gm(1,3)&
&   *dgm01(1,3)+gm(1,1)*(-108*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3))))&
&   *dgm10(1,3)+gm(1,1)*(288*gm(1,1)*gm(2,3)*dgm01(1,1)+gm(1,2)*(144*gm(1,3)&
&   *dgm01(1,1)+288*gm(1,1)*dgm01(1,3)))*dgm10(3,3))/48.d0
   cm(7,10)=(2*(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&   +12*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+6*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(3,3))*dgm10(1,1)+4*(6*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+12*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3)))*dgm01(1,3)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,3)+(12*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)&
&   *(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,1)+8*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,3))*dgm10(3,3))/144.d0
   cm(8,10)=(2*(288*gm(1,2)*gm(3,3)*(gm(3,3)*dgm01(1,3)+gm(1,3)&
&   *dgm01(3,3))+gm(2,3)*(72*gm(3,3)**2*dgm01(1,1)+36*gm(1,3)**2*dgm01(3,3)&
&   +gm(3,3)*(144*gm(1,3)*dgm01(1,3)-108*gm(1,1)*dgm01(3,3))))*dgm10(1,1)&
&   +4*(gm(1,3)**2*(72*gm(2,3)*dgm01(1,3)+180*gm(1,2)*dgm01(3,3))&
&   +gm(1,3)*(576*gm(1,2)*gm(3,3)*dgm01(1,3)+gm(2,3)*(72*gm(3,3)&
&   *dgm01(1,1)-72*gm(1,1)*dgm01(3,3)))+gm(3,3)*(-216*gm(1,1)*gm(2,3)&
&   *dgm01(1,3)+gm(1,2)*(144*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3))))&
&   *dgm10(1,3)+(gm(1,1)*gm(3,3)*(-216*gm(2,3)*dgm01(1,1)-144*gm(1,2)&
&   *dgm01(1,3))+gm(1,3)**2*(72*gm(2,3)*dgm01(1,1)+720*gm(1,2)*dgm01(1,3))&
&   +gm(1,3)*(576*gm(1,2)*gm(3,3)*dgm01(1,1)-288*gm(1,1)*gm(2,3)&
&   *dgm01(1,3)))*dgm10(3,3))/48.d0
   cm(9,10)=(2*(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&   +12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,1)+4*(6*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+2*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(1,3)+(12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+8*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3))*dgm10(3,3))&
&   /48.d0
   cm(10,10)=(2*gm(3,3)*(72*gm(3,3)**2*dgm01(1,1)+324*gm(1,3)**2*dgm01(3,3)&
&   +gm(3,3)*(432*gm(1,3)*dgm01(1,3)-108*gm(1,1)*dgm01(3,3)))*dgm10(1,1)&
&   +4*(648*gm(1,3)**2*gm(3,3)*dgm01(1,3)-216*gm(1,1)*gm(3,3)**2*dgm01(1,3)&
&   +180*gm(1,3)**3*dgm01(3,3)+gm(1,3)*gm(3,3)*(216*gm(3,3)*dgm01(1,1)&
&   -108*gm(1,1)*dgm01(3,3)))*dgm10(1,3)+(648*gm(1,3)**2*gm(3,3)&
&   *dgm01(1,1)-216*gm(1,1)*gm(3,3)**2*dgm01(1,1)+720*gm(1,3)**3*dgm01(1,3)&
&   -432*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3))*dgm10(3,3))/144.d0
   cm(1,11)=((8*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,2)&
&   +12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2))*dgm10(1,1)&
&   +4*(2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)+12*gm(1,1)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,2)+216*gm(1,1)**2*gm(1,2)&
&   *dgm01(2,2))*dgm10(1,2)+2*gm(1,1)*(6*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,1)+432*gm(1,1)*gm(1,2)*dgm01(1,2)+72*gm(1,1)&
&   **2*dgm01(2,2))*dgm10(2,2))/144.d0
   cm(2,11)=(gm(2,2)*(288*gm(1,2)*gm(2,2)*dgm01(1,2)+12*(12*gm(1,2)&
&   **2+24*gm(1,1)*gm(2,2))*dgm01(2,2))*dgm10(1,1)+4*(144*gm(1,2)&
&   **2*gm(2,2)*dgm01(1,2)+288*gm(1,1)*gm(2,2)**2*dgm01(1,2)+36*gm(1,2)&
&   **3*dgm01(2,2)+gm(1,2)*gm(2,2)*(72*gm(2,2)*dgm01(1,1)+180*gm(1,1)&
&   *dgm01(2,2)))*dgm10(1,2)+2*(72*gm(1,2)**3*dgm01(1,2)+360*gm(1,1)&
&   *gm(1,2)*gm(2,2)*dgm01(1,2)+gm(1,1)*gm(2,2)*(144*gm(2,2)*dgm01(1,1)&
&   -36*gm(1,1)*dgm01(2,2))+gm(1,2)**2*(72*gm(2,2)*dgm01(1,1)+108*gm(1,1)&
&   *dgm01(2,2)))*dgm10(2,2))/48.d0
   cm(3,11)=((8*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,2))*dgm10(1,1)+4*(2*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,2))*dgm10(1,2)+2*(6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(48*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)&
&   +2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2))*dgm10(2,2))&
&   /48.d0
   cm(4,11)=((8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(1,2)+12*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2))*dgm10(1,1)+4*(gm(2,3)&
&   *(72*gm(1,2)**2*dgm01(1,2)+288*gm(1,1)*gm(2,2)*dgm01(1,2)+gm(1,2)&
&   *(108*gm(2,2)*dgm01(1,1)+36*gm(1,1)*dgm01(2,2)))+gm(1,3)*(-36*gm(2,2)&
&   **2*dgm01(1,1)+36*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(72*gm(1,2)*dgm01(1,2)&
&   +144*gm(1,1)*dgm01(2,2))))*dgm10(1,2)+2*(gm(1,2)**2*(36*gm(2,3)&
&   *dgm01(1,1)+72*gm(1,3)*dgm01(1,2))+gm(1,1)*(gm(2,2)*(144*gm(2,3)&
&   *dgm01(1,1)+288*gm(1,3)*dgm01(1,2))-36*gm(1,1)*gm(2,3)*dgm01(2,2))&
&   +gm(1,2)*(72*gm(1,1)*gm(2,3)*dgm01(1,2)+gm(1,3)*(36*gm(2,2)*dgm01(1,1)&
&   +108*gm(1,1)*dgm01(2,2))))*dgm10(2,2))/24.d0
   cm(5,11)=((8*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,2)+12*(6*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2))&
&   *dgm10(1,1)+4*(gm(1,2)**2*(180*gm(2,3)*dgm01(1,1)+72*gm(1,3)&
&   *dgm01(1,2))+gm(1,1)*(gm(2,2)*(-36*gm(2,3)*dgm01(1,1)-216*gm(1,3)&
&   *dgm01(1,2))+144*gm(1,1)*gm(2,3)*dgm01(2,2))+gm(1,2)*(576*gm(1,1)&
&   *gm(2,3)*dgm01(1,2)+gm(1,3)*(-72*gm(2,2)*dgm01(1,1)+72*gm(1,1)&
&   *dgm01(2,2))))*dgm10(1,2)+2*(36*gm(1,2)**2*gm(1,3)*dgm01(1,1)&
&   +gm(1,1)*gm(1,2)*(288*gm(2,3)*dgm01(1,1)+144*gm(1,3)*dgm01(1,2))&
&   +gm(1,1)*(288*gm(1,1)*gm(2,3)*dgm01(1,2)+gm(1,3)*(-108*gm(2,2)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(2,2))))*dgm10(2,2))/48.d0
   cm(6,11)=((8*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,2)&
&   +12*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2))*dgm10(1,1)&
&   +4*(72*gm(1,2)**3*dgm01(1,2)+360*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2)&
&   +gm(1,2)**2*(108*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))+gm(1,1)&
&   *gm(2,2)*(-36*gm(2,2)*dgm01(1,1)+144*gm(1,1)*dgm01(2,2)))*dgm10(1,2)&
&   +2*(36*gm(1,2)**3*dgm01(1,1)+144*gm(1,1)*gm(1,2)**2*dgm01(1,2)&
&   +288*gm(1,1)**2*gm(2,2)*dgm01(1,2)+gm(1,1)*gm(1,2)*(180*gm(2,2)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(2,2)))*dgm10(2,2))/48.d0
   cm(7,11)=(gm(2,2)**2*(288*gm(2,2)*dgm01(1,2)+432*gm(1,2)*dgm01(2,2))&
&   *dgm10(1,1)+4*gm(2,2)*(72*gm(2,2)**2*dgm01(1,1)+324*gm(1,2)**2*dgm01(2,2)&
&   +gm(2,2)*(432*gm(1,2)*dgm01(1,2)-108*gm(1,1)*dgm01(2,2)))*dgm10(1,2)&
&   +2*(648*gm(1,2)**2*gm(2,2)*dgm01(1,2)-216*gm(1,1)*gm(2,2)**2*dgm01(1,2)&
&   +180*gm(1,2)**3*dgm01(2,2)+gm(1,2)*gm(2,2)*(216*gm(2,2)*dgm01(1,1)&
&   -108*gm(1,1)*dgm01(2,2)))*dgm10(2,2))/144.d0
   cm(8,11)=((8*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2))*dgm10(1,1)+4*(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,1)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,2))*dgm10(1,2)+2*(6*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(2,2))*dgm10(2,2))/48.d0
   cm(9,11)=(gm(2,2)*(144*gm(1,2)*gm(2,3)*dgm01(2,2)+288*gm(2,2)&
&   *(gm(2,3)*dgm01(1,2)+gm(1,3)*dgm01(2,2)))*dgm10(1,1)+4*(gm(2,2)&
&   **2*(72*gm(2,3)*dgm01(1,1)+288*gm(1,3)*dgm01(1,2))+36*gm(1,2)&
&   **2*gm(2,3)*dgm01(2,2)+gm(2,2)*(-108*gm(1,1)*gm(2,3)*dgm01(2,2)&
&   +gm(1,2)*(144*gm(2,3)*dgm01(1,2)+288*gm(1,3)*dgm01(2,2))))*dgm10(1,2)&
&   +2*(gm(2,3)*(72*gm(1,2)**2*dgm01(1,2)-216*gm(1,1)*gm(2,2)*dgm01(1,2)&
&   +gm(1,2)*(72*gm(2,2)*dgm01(1,1)-72*gm(1,1)*dgm01(2,2)))+gm(1,3)&
&   *(144*gm(2,2)**2*dgm01(1,1)+180*gm(1,2)**2*dgm01(2,2)+gm(2,2)&
&   *(576*gm(1,2)*dgm01(1,2)-36*gm(1,1)*dgm01(2,2))))*dgm10(2,2))&
&   /48.d0
   cm(10,11)=((8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2))*dgm10(1,1)+4*(2*(90*gm(2,3)**3-54*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,1)+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(2,2))&
&   *dgm10(1,2)+2*(6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(90*gm(1,3)**2*gm(2,3)&
&   -18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,2)+2*(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2))*dgm10(2,2))/144.d0
   cm(1,12)=(6*(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)+2*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,3)+gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+4*gm(1,1)*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,3)+gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,1)+6*(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)+4*gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)+8*gm(1,1)*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(1,3)+144*gm(1,1)**2*gm(1,3)&
&   *dgm01(2,3)+72*gm(1,1)**2*gm(1,2)*dgm01(3,3))*dgm10(1,2)+6*(2*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,1)&
&   +8*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(1,2)&
&   +4*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,3)+72*gm(1,1)&
&   **2*gm(1,3)*dgm01(2,2)+144*gm(1,1)**2*gm(1,2)*dgm01(2,3))*dgm10(1,3)&
&   +gm(1,1)*(6*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,1)+432*gm(1,1)&
&   *gm(1,3)*dgm01(1,3)+72*gm(1,1)**2*dgm01(3,3))*dgm10(2,2)+288*gm(1,1)&
&   *(gm(1,2)*(4.5d0*gm(1,3)*dgm01(1,1)+3*gm(1,1)*dgm01(1,3))+gm(1,1)&
&   *(-1.5d0*gm(2,3)*dgm01(1,1)+3*gm(1,3)*dgm01(1,2)+gm(1,1)*dgm01(2,3)))&
&   *dgm10(2,3)+gm(1,1)*(6*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,1)&
&   +432*gm(1,1)*gm(1,2)*dgm01(1,2)+72*gm(1,1)**2*dgm01(2,2))*dgm10(3,3))&
&   /72.d0
   cm(2,12)=(6*(2*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&   +12*gm(1,2)*gm(2,3))*dgm01(1,3)+(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)+gm(2,2)*(12*gm(1,2)&
&   **2+24*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(1,1)+12*((48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(1,3)+2*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)&
&   **3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(1,2)+6*(48*gm(1,2)&
&   **2*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3))+gm(1,1)*gm(2,2)*(192*gm(2,3)&
&   *dgm01(1,2)+96*gm(2,2)*dgm01(1,3))+gm(1,3)*(48*gm(2,2)**2*dgm01(1,1)&
&   +12*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(48*gm(1,2)*dgm01(1,2)-36*gm(1,1)&
&   *dgm01(2,2)))+24*gm(1,2)**3*dgm01(2,3)+gm(1,2)*(96*gm(1,1)*gm(2,3)&
&   *dgm01(2,2)+gm(2,2)*(24*gm(2,3)*dgm01(1,1)+120*gm(1,1)*dgm01(2,3))))&
&   *dgm10(1,3)+(6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3)+2*gm(1,1)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(2,2)+4*(36*gm(1,2)&
&   **3*dgm01(1,3)+gm(1,2)*(36*gm(1,3)*gm(2,2)*dgm01(1,1)+gm(1,1)&
&   *(288*gm(2,3)*dgm01(1,2)+180*gm(2,2)*dgm01(1,3)))+gm(1,1)*gm(2,2)&
&   *(144*gm(2,3)*dgm01(1,1)-108*gm(1,3)*dgm01(1,2)-36*gm(1,1)*dgm01(2,3))&
&   +gm(1,2)**2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,3)*dgm01(1,2)+108*gm(1,1)&
&   *dgm01(2,3)))*dgm10(2,3)+(72*gm(1,2)**3*dgm01(1,2)+360*gm(1,1)&
&   *gm(1,2)*gm(2,2)*dgm01(1,2)+gm(1,1)*gm(2,2)*(144*gm(2,2)*dgm01(1,1)&
&   -36*gm(1,1)*dgm01(2,2))+gm(1,2)**2*(72*gm(2,2)*dgm01(1,1)+108*gm(1,1)&
&   *dgm01(2,2)))*dgm10(3,3))/24.d0
   cm(3,12)=(6*(2*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&   *dgm01(1,2)+2*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3))*dgm01(2,2)+4*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,1)&
&   +6*(24*gm(1,3)**3*dgm01(2,3)+gm(1,3)**2*(48*gm(3,3)*dgm01(1,2)&
&   +48*gm(2,3)*dgm01(1,3)+12*gm(1,2)*dgm01(3,3))+gm(3,3)*(gm(1,1)&
&   *(96*gm(3,3)*dgm01(1,2)+192*gm(2,3)*dgm01(1,3))+gm(1,2)*(48*gm(3,3)&
&   *dgm01(1,1)-36*gm(1,1)*dgm01(3,3)))+gm(1,3)*(gm(3,3)*(48*gm(1,2)&
&   *dgm01(1,3)+120*gm(1,1)*dgm01(2,3))+gm(2,3)*(24*gm(3,3)*dgm01(1,1)&
&   +96*gm(1,1)*dgm01(3,3))))*dgm10(1,2)+6*(-48*gm(1,2)**2*gm(3,3)&
&   *dgm01(1,3)+gm(1,1)*(192*gm(2,3)*gm(3,3)*dgm01(1,2)+120*gm(2,3)&
&   **2*dgm01(1,3)-24*gm(2,2)*gm(3,3)*dgm01(1,3))+12*gm(1,3)**3*dgm01(2,2)&
&   +gm(1,2)*gm(3,3)*(96*gm(2,3)*dgm01(1,1)-72*gm(1,1)*dgm01(2,3))&
&   +gm(1,3)**2*(48*gm(2,3)*dgm01(1,2)-48*gm(2,2)*dgm01(1,3)+24*gm(1,2)&
&   *dgm01(2,3))+gm(1,3)*(12*gm(2,3)**2*dgm01(1,1)+gm(3,3)*(-36*gm(2,2)&
&   *dgm01(1,1)+48*gm(1,2)*dgm01(1,2)+60*gm(1,1)*dgm01(2,2))+gm(2,3)&
&   *(144*gm(1,2)*dgm01(1,3)+192*gm(1,1)*dgm01(2,3))))*dgm10(1,3)&
&   +(72*gm(1,3)**3*dgm01(1,3)+360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3)&
&   +gm(1,1)*gm(3,3)*(144*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3))&
&   +gm(1,3)**2*(72*gm(3,3)*dgm01(1,1)+108*gm(1,1)*dgm01(3,3)))*dgm10(2,2)&
&   +4*(36*gm(1,3)**3*dgm01(1,2)+gm(1,3)*(36*gm(1,2)*gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*(180*gm(3,3)*dgm01(1,2)+288*gm(2,3)*dgm01(1,3)))+gm(1,1)&
&   *gm(3,3)*(144*gm(2,3)*dgm01(1,1)-108*gm(1,2)*dgm01(1,3)-36*gm(1,1)&
&   *dgm01(2,3))+gm(1,3)**2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,2)*dgm01(1,3)&
&   +108*gm(1,1)*dgm01(2,3)))*dgm10(2,3)+(6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)+2*gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2))*dgm10(3,3))/24.d0
   cm(4,12)=(6*(2*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)+(6*gm(1,3)&
&   **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,2)+4*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)&
&   +9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(1,1)+6*(72*gm(1,2)**2*gm(3,3)&
&   *dgm01(1,3)+gm(1,1)*(96*gm(2,3)*gm(3,3)*dgm01(1,2)+72*gm(2,3)&
&   **2*dgm01(1,3)+120*gm(2,2)*gm(3,3)*dgm01(1,3))+gm(1,3)**2*(24*gm(2,3)&
&   *dgm01(1,2)+72*gm(2,2)*dgm01(1,3)+24*gm(1,2)*dgm01(2,3))+gm(1,2)&
&   *(96*gm(1,1)*gm(3,3)*dgm01(2,3)+12*gm(2,3)*(gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*dgm01(3,3)))+gm(1,3)*(12*gm(2,3)**2*dgm01(1,1)+gm(2,3)&
&   *(-48*gm(1,2)*dgm01(1,3)+24*gm(1,1)*dgm01(2,3))+48*gm(2,2)*(gm(3,3)&
&   *dgm01(1,1)+gm(1,1)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(1,2)&
&   +12*gm(1,2)*dgm01(3,3))))*dgm10(1,2)+6*(gm(1,2)**2*(72*gm(3,3)&
&   *dgm01(1,2)+24*gm(2,3)*dgm01(1,3))+gm(1,1)*(72*gm(2,3)**2*dgm01(1,2)&
&   +120*gm(2,2)*gm(3,3)*dgm01(1,2)+96*gm(2,2)*gm(2,3)*dgm01(1,3))&
&   +gm(1,3)**2*(72*gm(2,2)*dgm01(1,2)+12*gm(1,2)*dgm01(2,2))+gm(1,2)&
&   *(12*gm(2,3)**2*dgm01(1,1)+48*gm(3,3)*(gm(2,2)*dgm01(1,1)+gm(1,1)&
&   *dgm01(2,2))+24*gm(1,1)*gm(2,3)*dgm01(2,3))+gm(1,3)*(-48*gm(1,2)&
&   *gm(2,3)*dgm01(1,2)+12*gm(1,1)*gm(2,3)*dgm01(2,2)+24*gm(1,2)&
&   **2*dgm01(2,3)+gm(2,2)*(12*gm(2,3)*dgm01(1,1)+24*gm(1,2)*dgm01(1,3)&
&   +96*gm(1,1)*dgm01(2,3))))*dgm10(1,3)+(gm(1,3)**2*(36*gm(2,3)&
&   *dgm01(1,1)+72*gm(1,2)*dgm01(1,3))+gm(1,1)*(288*gm(1,2)*gm(3,3)&
&   *dgm01(1,3)+gm(2,3)*(144*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3)))&
&   +gm(1,3)*(72*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)*(36*gm(3,3)*dgm01(1,1)&
&   +108*gm(1,1)*dgm01(3,3))))*dgm10(2,2)+4*(54*gm(1,2)**2*gm(3,3)&
&   *dgm01(1,1)+gm(1,3)**2*(54*gm(2,2)*dgm01(1,1)+36*gm(1,2)*dgm01(1,2))&
&   +gm(1,1)*(54*gm(2,3)**2*dgm01(1,1)+gm(3,3)*(90*gm(2,2)*dgm01(1,1)&
&   +144*gm(1,2)*dgm01(1,2))+36*gm(1,2)*gm(2,3)*dgm01(1,3))-36*gm(1,1)&
&   **2*gm(2,3)*dgm01(2,3)+gm(1,3)*(36*gm(1,2)**2*dgm01(1,3)+gm(1,1)&
&   *(36*gm(2,3)*dgm01(1,2)+144*gm(2,2)*dgm01(1,3))+gm(1,2)*(-36*gm(2,3)&
&   *dgm01(1,1)+108*gm(1,1)*dgm01(2,3))))*dgm10(2,3)+(gm(1,2)**2*(36*gm(2,3)&
&   *dgm01(1,1)+72*gm(1,3)*dgm01(1,2))+gm(1,1)*(gm(2,2)*(144*gm(2,3)&
&   *dgm01(1,1)+288*gm(1,3)*dgm01(1,2))-36*gm(1,1)*gm(2,3)*dgm01(2,2))&
&   +gm(1,2)*(72*gm(1,1)*gm(2,3)*dgm01(1,2)+gm(1,3)*(36*gm(2,2)*dgm01(1,1)&
&   +108*gm(1,1)*dgm01(2,2))))*dgm10(3,3))/12.d0
   cm(5,12)=(6*(2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)+2*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm01(2,2)+4*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm01(2,3)+(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,1)&
&   +6*(24*gm(1,3)**3*dgm01(1,2)+gm(1,3)**2*(12*gm(2,3)*dgm01(1,1)&
&   +48*(gm(1,2)*dgm01(1,3)+gm(1,1)*dgm01(2,3)))+gm(1,3)*(gm(1,1)&
&   *(120*gm(3,3)*dgm01(1,2)+48*gm(2,3)*dgm01(1,3))+gm(1,2)*(96*gm(3,3)&
&   *dgm01(1,1)+24*gm(1,1)*dgm01(3,3)))+gm(1,1)*(gm(3,3)*(192*gm(1,2)&
&   *dgm01(1,3)+96*gm(1,1)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)*dgm01(1,1)&
&   +48*gm(1,1)*dgm01(3,3))))*dgm10(1,2)+6*(60*gm(1,2)**2*gm(3,3)&
&   *dgm01(1,1)+gm(1,1)*(-24*gm(2,3)**2*dgm01(1,1)+gm(3,3)*(-12*gm(2,2)&
&   *dgm01(1,1)+192*gm(1,2)*dgm01(1,2))+192*gm(1,2)*gm(2,3)*dgm01(1,3))&
&   +gm(1,3)**2*(-24*gm(2,2)*dgm01(1,1)+48*gm(1,2)*dgm01(1,2)+24*gm(1,1)&
&   *dgm01(2,2))+gm(1,1)**2*(48*gm(3,3)*dgm01(2,2)+96*gm(2,3)*dgm01(2,3))&
&   +gm(1,3)*(24*gm(1,2)**2*dgm01(1,3)+gm(1,1)*(48*gm(2,3)*dgm01(1,2)&
&   -72*gm(2,2)*dgm01(1,3))+gm(1,2)*(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)&
&   *dgm01(2,3))))*dgm10(1,3)+(36*gm(1,3)**3*dgm01(1,1)+144*gm(1,1)&
&   *gm(1,3)**2*dgm01(1,3)+288*gm(1,1)**2*gm(3,3)*dgm01(1,3)+gm(1,1)&
&   *gm(1,3)*(180*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3)))*dgm10(2,2)&
&   +4*(gm(1,2)*(36*gm(1,3)**2*dgm01(1,1)+144*gm(1,1)*gm(3,3)*dgm01(1,1)&
&   +72*gm(1,1)*gm(1,3)*dgm01(1,3))+gm(1,1)*(72*gm(1,3)**2*dgm01(1,2)&
&   +144*gm(1,1)*(gm(3,3)*dgm01(1,2)+gm(2,3)*dgm01(1,3))+gm(1,3)&
&   *(36*gm(2,3)*dgm01(1,1)+72*gm(1,1)*dgm01(2,3))))*dgm10(2,3)+(36*gm(1,2)&
&   **2*gm(1,3)*dgm01(1,1)+gm(1,1)*gm(1,2)*(288*gm(2,3)*dgm01(1,1)&
&   +144*gm(1,3)*dgm01(1,2))+gm(1,1)*(288*gm(1,1)*gm(2,3)*dgm01(1,2)&
&   +gm(1,3)*(-108*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))))*dgm10(3,3))&
&   /24.d0
   cm(6,12)=(6*(2*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+2*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)+(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)+4*(6*gm(1,2)&
&   **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))&
&   *dgm10(1,1)+6*(-24*gm(1,2)**2*gm(3,3)*dgm01(1,1)+gm(1,3)**2*(60*gm(2,2)&
&   *dgm01(1,1)+24*gm(1,2)*dgm01(1,2))+gm(1,3)*(48*gm(1,2)**2*dgm01(1,3)&
&   +192*gm(1,1)*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3))+gm(1,2)&
&   *(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)*dgm01(2,3)))+gm(1,1)*(-24*gm(2,3)&
&   **2*dgm01(1,1)-12*gm(2,2)*gm(3,3)*dgm01(1,1)-72*gm(1,2)*gm(3,3)&
&   *dgm01(1,2)+48*gm(1,2)*gm(2,3)*dgm01(1,3)+24*gm(1,2)**2*dgm01(3,3))&
&   +gm(1,1)**2*(96*gm(2,3)*dgm01(2,3)+48*gm(2,2)*dgm01(3,3)))*dgm10(1,2)&
&   +6*(24*gm(1,2)**3*dgm01(1,3)+gm(1,2)*(gm(1,1)*(48*gm(2,3)*dgm01(1,2)&
&   +120*gm(2,2)*dgm01(1,3))+gm(1,3)*(96*gm(2,2)*dgm01(1,1)+24*gm(1,1)&
&   *dgm01(2,2)))+gm(1,2)**2*(12*gm(2,3)*dgm01(1,1)+48*(gm(1,3)*dgm01(1,2)&
&   +gm(1,1)*dgm01(2,3)))+gm(1,1)*(48*gm(1,1)*gm(2,3)*dgm01(2,2)&
&   +gm(2,2)*(-36*gm(2,3)*dgm01(1,1)+192*gm(1,3)*dgm01(1,2)+96*gm(1,1)&
&   *dgm01(2,3))))*dgm10(1,3)+(288*gm(1,1)*gm(2,3)*(gm(1,3)*dgm01(1,1)&
&   +gm(1,1)*dgm01(1,3))+gm(1,2)*(36*gm(1,3)**2*dgm01(1,1)+144*gm(1,1)&
&   *gm(1,3)*dgm01(1,3)+gm(1,1)*(-108*gm(3,3)*dgm01(1,1)+72*gm(1,1)&
&   *dgm01(3,3))))*dgm10(2,2)+4*(gm(1,2)**2*(36*gm(1,3)*dgm01(1,1)&
&   +72*gm(1,1)*dgm01(1,3))+144*gm(1,1)*(gm(1,3)*gm(2,2)*dgm01(1,1)&
&   +gm(1,1)*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3)))+gm(1,1)*gm(1,2)&
&   *(36*gm(2,3)*dgm01(1,1)+72*(gm(1,3)*dgm01(1,2)+gm(1,1)*dgm01(2,3))))&
&   *dgm10(2,3)+(36*gm(1,2)**3*dgm01(1,1)+144*gm(1,1)*gm(1,2)**2*dgm01(1,2)&
&   +288*gm(1,1)**2*gm(2,2)*dgm01(1,2)+gm(1,1)*gm(1,2)*(180*gm(2,2)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(2,2)))*dgm10(3,3))/24.d0
   cm(7,12)=(3*(180*gm(1,2)*gm(2,3)**2*dgm01(2,2)+gm(2,2)*(216*gm(2,3)&
&   **2*dgm01(1,2)-36*gm(1,2)*gm(3,3)*dgm01(2,2)+gm(2,3)*(-72*gm(1,3)&
&   *dgm01(2,2)+432*gm(1,2)*dgm01(2,3)))+gm(2,2)**2*(-72*gm(3,3)&
&   *dgm01(1,2)+144*gm(2,3)*dgm01(1,3)-144*gm(1,3)*dgm01(2,3)+72*gm(1,2)&
&   *dgm01(3,3)))*dgm10(1,1)+12*(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,1)+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&   +54*gm(1,2)*gm(2,3))*dgm01(1,3)+2*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)&
&   *(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(2,3)+gm(2,2)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(1,2)+6*(180*gm(1,2)&
&   **2*gm(2,3)*dgm01(2,2)+gm(2,2)**2*(72*gm(2,3)*dgm01(1,1)-144*gm(1,3)&
&   *dgm01(1,2)+144*gm(1,2)*dgm01(1,3)-72*gm(1,1)*dgm01(2,3))+gm(2,2)&
&   *(-36*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(1,2)*(432*gm(2,3)*dgm01(1,2)&
&   -72*gm(1,3)*dgm01(2,2))+216*gm(1,2)**2*dgm01(2,3)))*dgm10(1,3)&
&   +(6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+12*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,3)+2*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(2,2)+4*(6*gm(2,2)*(-18*gm(1,3)&
&   *gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,1)+6*(90*gm(1,2)**2*gm(2,3)&
&   -18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,2)+6*gm(2,2)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,3)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(2,3)+(648*gm(1,2)&
&   **2*gm(2,2)*dgm01(1,2)-216*gm(1,1)*gm(2,2)**2*dgm01(1,2)+180*gm(1,2)&
&   **3*dgm01(2,2)+gm(1,2)*gm(2,2)*(216*gm(2,2)*dgm01(1,1)-108*gm(1,1)&
&   *dgm01(2,2)))*dgm10(3,3))/72.d0
   cm(8,12)=(6*(2*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)+gm(3,3)&
&   *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,1)+12*(gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3))*dgm01(1,1)+2*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&   *gm(3,3))*dgm01(1,2)+4*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)+2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,2)&
&   +6*(12*gm(2,3)**3*dgm01(1,1)+gm(1,3)*gm(3,3)*(192*gm(2,2)*dgm01(1,2)&
&   +96*gm(1,2)*dgm01(2,2))+120*gm(1,3)**2*gm(2,2)*dgm01(2,3)+gm(2,3)&
&   **2*(48*gm(1,3)*dgm01(1,2)+24*gm(1,2)*dgm01(1,3)-48*gm(1,1)*dgm01(2,3))&
&   +gm(3,3)*(-72*gm(1,2)*gm(2,2)*dgm01(1,3)-48*gm(1,2)**2*dgm01(2,3)&
&   -24*gm(1,1)*gm(2,2)*dgm01(2,3))+gm(2,3)*(gm(2,2)*(60*gm(3,3)&
&   *dgm01(1,1)+192*gm(1,3)*dgm01(1,3))+(12*gm(1,3)**2-36*gm(1,1)&
&   *gm(3,3))*dgm01(2,2)+gm(1,2)*(48*gm(3,3)*dgm01(1,2)+144*gm(1,3)&
&   *dgm01(2,3))))*dgm10(1,3)+(gm(1,3)**2*(72*gm(2,3)*dgm01(1,3)&
&   +180*gm(1,2)*dgm01(3,3))+gm(1,3)*(576*gm(1,2)*gm(3,3)*dgm01(1,3)&
&   +gm(2,3)*(72*gm(3,3)*dgm01(1,1)-72*gm(1,1)*dgm01(3,3)))+gm(3,3)&
&   *(-216*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)*(144*gm(3,3)*dgm01(1,1)&
&   -36*gm(1,1)*dgm01(3,3))))*dgm10(2,2)+4*(6*(6*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3))*dgm10(2,3)+(6*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(2,2))*dgm10(3,3))/24.d0
   cm(9,12)=(6*(2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +2*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,3)+(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3)+gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&   *gm(2,3))*dgm01(3,3))*dgm10(1,1)+12*((6*gm(2,3)**3+30*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,1)+2*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+4*(6*gm(1,3)*gm(2,2)&
&   *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(1,2)+6*(gm(2,2)**2*(48*gm(3,3)&
&   *dgm01(1,1)+96*gm(1,3)*dgm01(1,3))-24*gm(1,1)*gm(2,3)**2*dgm01(2,2)&
&   +gm(1,2)*gm(2,3)*(48*gm(2,3)*dgm01(1,2)+72*gm(1,3)*dgm01(2,2))&
&   +gm(1,2)**2*(60*gm(3,3)*dgm01(2,2)+24*gm(2,3)*dgm01(2,3))+gm(2,2)&
&   *(24*gm(2,3)**2*dgm01(1,1)+(-24*gm(1,3)**2-12*gm(1,1)*gm(3,3))&
&   *dgm01(2,2)+gm(2,3)*(48*gm(1,3)*dgm01(1,2)+48*gm(1,2)*dgm01(1,3)&
&   -72*gm(1,1)*dgm01(2,3))+192*gm(1,2)*(gm(3,3)*dgm01(1,2)+gm(1,3)&
&   *dgm01(2,3))))*dgm10(1,3)+(6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+2*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(2,2)+4*(6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)+6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(48*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,3)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(2,3)+(gm(2,3)&
&   *(72*gm(1,2)**2*dgm01(1,2)-216*gm(1,1)*gm(2,2)*dgm01(1,2)+gm(1,2)&
&   *(72*gm(2,2)*dgm01(1,1)-72*gm(1,1)*dgm01(2,2)))+gm(1,3)*(144*gm(2,2)&
&   **2*dgm01(1,1)+180*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(576*gm(1,2)&
&   *dgm01(1,2)-36*gm(1,1)*dgm01(2,2))))*dgm10(3,3))/24.d0
   cm(10,12)=(3*(144*gm(2,3)*gm(3,3)**2*dgm01(1,2)+4*gm(3,3)*(54*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3))*dgm01(1,3)+72*gm(1,3)*gm(3,3)**2*dgm01(2,2)&
&   +8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(2,3)&
&   +2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,1)+6*(72*gm(2,3)*gm(3,3)**2*dgm01(1,1)&
&   +144*gm(1,3)*gm(3,3)**2*dgm01(1,2)+8*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&   -18*gm(1,2)*gm(3,3))*dgm01(1,3)+4*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(2,3)+2*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3))*gm(3,3))*dgm01(3,3))*dgm10(1,2)+12*(gm(3,3)&
&   *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+4*gm(3,3)*(54*gm(1,3)&
&   *gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,2)+2*(-36*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+2*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(2,3))&
&   *dgm10(1,3)+(648*gm(1,3)**2*gm(3,3)*dgm01(1,3)-216*gm(1,1)*gm(3,3)&
&   **2*dgm01(1,3)+180*gm(1,3)**3*dgm01(3,3)+gm(1,3)*gm(3,3)*(216*gm(3,3)&
&   *dgm01(1,1)-108*gm(1,1)*dgm01(3,3)))*dgm10(2,2)+4*(6*gm(3,3)&
&   *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,1)+6*gm(3,3)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)+6*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,3)&
&   +2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3))*dgm10(2,3)&
&   +(6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+12*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,2)+2*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(2,2))*dgm10(3,3))/72.d0
   cm(1,13)=((6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&   +2*gm(1,2)*gm(2,3)))*dgm01(1,2)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(1,3)+6*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)&
&   *gm(2,3))*dgm01(2,2)+6*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&   *dgm01(2,3))*dgm10(1,1)+3*(2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,1)+8*gm(1,1)*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(1,2)+4*gm(1,1)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(1,3)+72*gm(1,1)**2*gm(1,3)*dgm01(2,2)&
&   +144*gm(1,1)**2*gm(1,2)*dgm01(2,3))*dgm10(1,2)+(2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)+12*gm(1,1)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(1,2)+216*gm(1,1)**2*gm(1,2)*dgm01(2,2))&
&   *dgm10(1,3)+72*gm(1,1)*(gm(1,2)*(4.5d0*gm(1,3)*dgm01(1,1)+3*gm(1,1)&
&   *dgm01(1,3))+gm(1,1)*(-1.5d0*gm(2,3)*dgm01(1,1)+3*gm(1,3)*dgm01(1,2)&
&   +gm(1,1)*dgm01(2,3)))*dgm10(2,2)+gm(1,1)*(6*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,1)+432*gm(1,1)*gm(1,2)*dgm01(1,2)+72*gm(1,1)&
&   **2*dgm01(2,2))*dgm10(2,3))/36.d0
   cm(2,13)=((6*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&   *dgm01(1,2)+72*gm(1,2)*gm(2,2)**2*dgm01(1,3)+6*(6*gm(1,2)*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)&
&   +6*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(1,1)&
&   +3*(48*gm(1,2)**2*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3))+gm(1,1)&
&   *gm(2,2)*(192*gm(2,3)*dgm01(1,2)+96*gm(2,2)*dgm01(1,3))+gm(1,3)&
&   *(48*gm(2,2)**2*dgm01(1,1)+12*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(48*gm(1,2)&
&   *dgm01(1,2)-36*gm(1,1)*dgm01(2,2)))+24*gm(1,2)**3*dgm01(2,3)&
&   +gm(1,2)*(96*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(2,2)*(24*gm(2,3)*dgm01(1,1)&
&   +120*gm(1,1)*dgm01(2,3))))*dgm10(1,2)+(144*gm(1,2)**2*gm(2,2)&
&   *dgm01(1,2)+288*gm(1,1)*gm(2,2)**2*dgm01(1,2)+36*gm(1,2)**3*dgm01(2,2)&
&   +gm(1,2)*gm(2,2)*(72*gm(2,2)*dgm01(1,1)+180*gm(1,1)*dgm01(2,2)))&
&   *dgm10(1,3)+(36*gm(1,2)**3*dgm01(1,3)+gm(1,2)*(36*gm(1,3)*gm(2,2)&
&   *dgm01(1,1)+gm(1,1)*(288*gm(2,3)*dgm01(1,2)+180*gm(2,2)*dgm01(1,3)))&
&   +gm(1,1)*gm(2,2)*(144*gm(2,3)*dgm01(1,1)-108*gm(1,3)*dgm01(1,2)&
&   -36*gm(1,1)*dgm01(2,3))+gm(1,2)**2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,3)&
&   *dgm01(1,2)+108*gm(1,1)*dgm01(2,3)))*dgm10(2,2)+(72*gm(1,2)**3*dgm01(1,2)&
&   +360*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2)+gm(1,1)*gm(2,2)*(144*gm(2,2)&
&   *dgm01(1,1)-36*gm(1,1)*dgm01(2,2))+gm(1,2)**2*(72*gm(2,2)*dgm01(1,1)&
&   +108*gm(1,1)*dgm01(2,2)))*dgm10(2,3))/12.d0
   cm(3,13)=((6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(6*gm(1,3)&
&   **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,2)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3))*dgm10(1,1)+3*(-48*gm(1,2)**2*gm(3,3)*dgm01(1,3)&
&   +gm(1,1)*(192*gm(2,3)*gm(3,3)*dgm01(1,2)+120*gm(2,3)**2*dgm01(1,3)&
&   -24*gm(2,2)*gm(3,3)*dgm01(1,3))+12*gm(1,3)**3*dgm01(2,2)+gm(1,2)&
&   *gm(3,3)*(96*gm(2,3)*dgm01(1,1)-72*gm(1,1)*dgm01(2,3))+gm(1,3)&
&   **2*(48*gm(2,3)*dgm01(1,2)-48*gm(2,2)*dgm01(1,3)+24*gm(1,2)*dgm01(2,3))&
&   +gm(1,3)*(12*gm(2,3)**2*dgm01(1,1)+gm(3,3)*(-36*gm(2,2)*dgm01(1,1)&
&   +48*gm(1,2)*dgm01(1,2)+60*gm(1,1)*dgm01(2,2))+gm(2,3)*(144*gm(1,2)&
&   *dgm01(1,3)+192*gm(1,1)*dgm01(2,3))))*dgm10(1,2)+(2*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,2))*dgm10(1,3)+(36*gm(1,3)**3*dgm01(1,2)&
&   +gm(1,3)*(36*gm(1,2)*gm(3,3)*dgm01(1,1)+gm(1,1)*(180*gm(3,3)&
&   *dgm01(1,2)+288*gm(2,3)*dgm01(1,3)))+gm(1,1)*gm(3,3)*(144*gm(2,3)&
&   *dgm01(1,1)-108*gm(1,2)*dgm01(1,3)-36*gm(1,1)*dgm01(2,3))+gm(1,3)&
&   **2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,2)*dgm01(1,3)+108*gm(1,1)&
&   *dgm01(2,3)))*dgm10(2,2)+(6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)+2*gm(1,1)*(54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(2,2))*dgm10(2,3))/12.d0
   cm(4,13)=((6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+2*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+6*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)&
&   *gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)+6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3))*dgm10(1,1)&
&   +3*(gm(1,2)**2*(72*gm(3,3)*dgm01(1,2)+24*gm(2,3)*dgm01(1,3))&
&   +gm(1,1)*(72*gm(2,3)**2*dgm01(1,2)+120*gm(2,2)*gm(3,3)*dgm01(1,2)&
&   +96*gm(2,2)*gm(2,3)*dgm01(1,3))+gm(1,3)**2*(72*gm(2,2)*dgm01(1,2)&
&   +12*gm(1,2)*dgm01(2,2))+gm(1,2)*(12*gm(2,3)**2*dgm01(1,1)+48*gm(3,3)&
&   *(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2))+24*gm(1,1)*gm(2,3)*dgm01(2,3))&
&   +gm(1,3)*(-48*gm(1,2)*gm(2,3)*dgm01(1,2)+12*gm(1,1)*gm(2,3)*dgm01(2,2)&
&   +24*gm(1,2)**2*dgm01(2,3)+gm(2,2)*(12*gm(2,3)*dgm01(1,1)+24*gm(1,2)&
&   *dgm01(1,3)+96*gm(1,1)*dgm01(2,3))))*dgm10(1,2)+(gm(2,3)*(72*gm(1,2)&
&   **2*dgm01(1,2)+288*gm(1,1)*gm(2,2)*dgm01(1,2)+gm(1,2)*(108*gm(2,2)&
&   *dgm01(1,1)+36*gm(1,1)*dgm01(2,2)))+gm(1,3)*(-36*gm(2,2)**2*dgm01(1,1)&
&   +36*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(72*gm(1,2)*dgm01(1,2)+144*gm(1,1)&
&   *dgm01(2,2))))*dgm10(1,3)+(54*gm(1,2)**2*gm(3,3)*dgm01(1,1)+gm(1,3)&
&   **2*(54*gm(2,2)*dgm01(1,1)+36*gm(1,2)*dgm01(1,2))+gm(1,1)*(54*gm(2,3)&
&   **2*dgm01(1,1)+gm(3,3)*(90*gm(2,2)*dgm01(1,1)+144*gm(1,2)*dgm01(1,2))&
&   +36*gm(1,2)*gm(2,3)*dgm01(1,3))-36*gm(1,1)**2*gm(2,3)*dgm01(2,3)&
&   +gm(1,3)*(36*gm(1,2)**2*dgm01(1,3)+gm(1,1)*(36*gm(2,3)*dgm01(1,2)&
&   +144*gm(2,2)*dgm01(1,3))+gm(1,2)*(-36*gm(2,3)*dgm01(1,1)+108*gm(1,1)&
&   *dgm01(2,3))))*dgm10(2,2)+(gm(1,2)**2*(36*gm(2,3)*dgm01(1,1)&
&   +72*gm(1,3)*dgm01(1,2))+gm(1,1)*(gm(2,2)*(144*gm(2,3)*dgm01(1,1)&
&   +288*gm(1,3)*dgm01(1,2))-36*gm(1,1)*gm(2,3)*dgm01(2,2))+gm(1,2)&
&   *(72*gm(1,1)*gm(2,3)*dgm01(1,2)+gm(1,3)*(36*gm(2,2)*dgm01(1,1)&
&   +108*gm(1,1)*dgm01(2,2))))*dgm10(2,3))/6.d0
   cm(5,13)=((6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)+6*(6*gm(1,1)*gm(1,3)&
&   *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(2,2)&
&   +6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(1,1)+3*(60*gm(1,2)**2*gm(3,3)&
&   *dgm01(1,1)+gm(1,1)*(-24*gm(2,3)**2*dgm01(1,1)+gm(3,3)*(-12*gm(2,2)&
&   *dgm01(1,1)+192*gm(1,2)*dgm01(1,2))+192*gm(1,2)*gm(2,3)*dgm01(1,3))&
&   +gm(1,3)**2*(-24*gm(2,2)*dgm01(1,1)+48*gm(1,2)*dgm01(1,2)+24*gm(1,1)&
&   *dgm01(2,2))+gm(1,1)**2*(48*gm(3,3)*dgm01(2,2)+96*gm(2,3)*dgm01(2,3))&
&   +gm(1,3)*(24*gm(1,2)**2*dgm01(1,3)+gm(1,1)*(48*gm(2,3)*dgm01(1,2)&
&   -72*gm(2,2)*dgm01(1,3))+gm(1,2)*(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)&
&   *dgm01(2,3))))*dgm10(1,2)+(gm(1,2)**2*(180*gm(2,3)*dgm01(1,1)&
&   +72*gm(1,3)*dgm01(1,2))+gm(1,1)*(gm(2,2)*(-36*gm(2,3)*dgm01(1,1)&
&   -216*gm(1,3)*dgm01(1,2))+144*gm(1,1)*gm(2,3)*dgm01(2,2))+gm(1,2)&
&   *(576*gm(1,1)*gm(2,3)*dgm01(1,2)+gm(1,3)*(-72*gm(2,2)*dgm01(1,1)&
&   +72*gm(1,1)*dgm01(2,2))))*dgm10(1,3)+(gm(1,2)*(36*gm(1,3)**2*dgm01(1,1)&
&   +144*gm(1,1)*gm(3,3)*dgm01(1,1)+72*gm(1,1)*gm(1,3)*dgm01(1,3))&
&   +gm(1,1)*(72*gm(1,3)**2*dgm01(1,2)+144*gm(1,1)*(gm(3,3)*dgm01(1,2)&
&   +gm(2,3)*dgm01(1,3))+gm(1,3)*(36*gm(2,3)*dgm01(1,1)+72*gm(1,1)&
&   *dgm01(2,3))))*dgm10(2,2)+(36*gm(1,2)**2*gm(1,3)*dgm01(1,1)+gm(1,1)&
&   *gm(1,2)*(288*gm(2,3)*dgm01(1,1)+144*gm(1,3)*dgm01(1,2))+gm(1,1)&
&   *(288*gm(1,1)*gm(2,3)*dgm01(1,2)+gm(1,3)*(-108*gm(2,2)*dgm01(1,1)&
&   +72*gm(1,1)*dgm01(2,2))))*dgm10(2,3))/12.d0
   cm(6,13)=((6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,2)+2*gm(2,2)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(1,3)+6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&   *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)+6*(6*gm(1,2)&
&   **3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(1,1)+3*(24*gm(1,2)&
&   **3*dgm01(1,3)+gm(1,2)*(gm(1,1)*(48*gm(2,3)*dgm01(1,2)+120*gm(2,2)&
&   *dgm01(1,3))+gm(1,3)*(96*gm(2,2)*dgm01(1,1)+24*gm(1,1)*dgm01(2,2)))&
&   +gm(1,2)**2*(12*gm(2,3)*dgm01(1,1)+48*(gm(1,3)*dgm01(1,2)+gm(1,1)&
&   *dgm01(2,3)))+gm(1,1)*(48*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(2,2)&
&   *(-36*gm(2,3)*dgm01(1,1)+192*gm(1,3)*dgm01(1,2)+96*gm(1,1)*dgm01(2,3))))&
&   *dgm10(1,2)+(72*gm(1,2)**3*dgm01(1,2)+360*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm01(1,2)+gm(1,2)**2*(108*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))&
&   +gm(1,1)*gm(2,2)*(-36*gm(2,2)*dgm01(1,1)+144*gm(1,1)*dgm01(2,2)))&
&   *dgm10(1,3)+(gm(1,2)**2*(36*gm(1,3)*dgm01(1,1)+72*gm(1,1)*dgm01(1,3))&
&   +144*gm(1,1)*(gm(1,3)*gm(2,2)*dgm01(1,1)+gm(1,1)*(gm(2,3)*dgm01(1,2)&
&   +gm(2,2)*dgm01(1,3)))+gm(1,1)*gm(1,2)*(36*gm(2,3)*dgm01(1,1)&
&   +72*(gm(1,3)*dgm01(1,2)+gm(1,1)*dgm01(2,3))))*dgm10(2,2)+(36*gm(1,2)&
&   **3*dgm01(1,1)+144*gm(1,1)*gm(1,2)**2*dgm01(1,2)+288*gm(1,1)&
&   **2*gm(2,2)*dgm01(1,2)+gm(1,1)*gm(1,2)*(180*gm(2,2)*dgm01(1,1)&
&   +72*gm(1,1)*dgm01(2,2)))*dgm10(2,3))/12.d0
   cm(7,13)=(72*gm(2,2)*(1*gm(2,2)**2*dgm01(1,3)+4.5d0*gm(1,2)&
&   *gm(2,3)*dgm01(2,2)+gm(2,2)*(3*gm(2,3)*dgm01(1,2)-1.5d0*gm(1,3)&
&   *dgm01(2,2)+3*gm(1,2)*dgm01(2,3)))*dgm10(1,1)+3*(180*gm(1,2)&
&   **2*gm(2,3)*dgm01(2,2)+gm(2,2)**2*(72*gm(2,3)*dgm01(1,1)-144*gm(1,3)&
&   *dgm01(1,2)+144*gm(1,2)*dgm01(1,3)-72*gm(1,1)*dgm01(2,3))+gm(2,2)&
&   *(-36*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(1,2)*(432*gm(2,3)*dgm01(1,2)&
&   -72*gm(1,3)*dgm01(2,2))+216*gm(1,2)**2*dgm01(2,3)))*dgm10(1,2)&
&   +gm(2,2)*(72*gm(2,2)**2*dgm01(1,1)+324*gm(1,2)**2*dgm01(2,2)&
&   +gm(2,2)*(432*gm(1,2)*dgm01(1,2)-108*gm(1,1)*dgm01(2,2)))*dgm10(1,3)&
&   +(6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,1)&
&   +6*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&   *gm(2,3)))*dgm01(1,2)+6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&   *dgm01(1,3)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))&
&   *dgm10(2,2)+(648*gm(1,2)**2*gm(2,2)*dgm01(1,2)-216*gm(1,1)*gm(2,2)&
&   **2*dgm01(1,2)+180*gm(1,2)**3*dgm01(2,2)+gm(1,2)*gm(2,2)*(216*gm(2,2)&
&   *dgm01(1,1)-108*gm(1,1)*dgm01(2,2)))*dgm10(2,3))/36.d0
   cm(8,13)=((6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)+6*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3))*dgm10(1,1)+3*(12*gm(2,3)**3*dgm01(1,1)&
&   +gm(1,3)*gm(3,3)*(192*gm(2,2)*dgm01(1,2)+96*gm(1,2)*dgm01(2,2))&
&   +120*gm(1,3)**2*gm(2,2)*dgm01(2,3)+gm(2,3)**2*(48*gm(1,3)*dgm01(1,2)&
&   +24*gm(1,2)*dgm01(1,3)-48*gm(1,1)*dgm01(2,3))+gm(3,3)*(-72*gm(1,2)&
&   *gm(2,2)*dgm01(1,3)-48*gm(1,2)**2*dgm01(2,3)-24*gm(1,1)*gm(2,2)&
&   *dgm01(2,3))+gm(2,3)*(gm(2,2)*(60*gm(3,3)*dgm01(1,1)+192*gm(1,3)&
&   *dgm01(1,3))+(12*gm(1,3)**2-36*gm(1,1)*gm(3,3))*dgm01(2,2)+gm(1,2)&
&   *(48*gm(3,3)*dgm01(1,2)+144*gm(1,3)*dgm01(2,3))))*dgm10(1,2)&
&   +(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+12*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,2))&
&   *dgm10(1,3)+(6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)+6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)+6*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+2*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(2,3))*dgm10(2,2)+(6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(2,2))*dgm10(2,3))/12.d0
   cm(9,13)=((36*gm(1,2)*gm(2,3)**2*dgm01(2,2)+gm(2,2)**2*(144*gm(3,3)&
&   *dgm01(1,2)+72*gm(2,3)*dgm01(1,3)+144*gm(1,3)*dgm01(2,3))+gm(2,2)&
&   *(72*gm(2,3)**2*dgm01(1,2)+144*gm(1,2)*gm(3,3)*dgm01(2,2)+gm(2,3)&
&   *(36*gm(1,3)*dgm01(2,2)+72*gm(1,2)*dgm01(2,3))))*dgm10(1,1)+3*(gm(2,2)&
&   **2*(48*gm(3,3)*dgm01(1,1)+96*gm(1,3)*dgm01(1,3))-24*gm(1,1)&
&   *gm(2,3)**2*dgm01(2,2)+gm(1,2)*gm(2,3)*(48*gm(2,3)*dgm01(1,2)&
&   +72*gm(1,3)*dgm01(2,2))+gm(1,2)**2*(60*gm(3,3)*dgm01(2,2)+24*gm(2,3)&
&   *dgm01(2,3))+gm(2,2)*(24*gm(2,3)**2*dgm01(1,1)+(-24*gm(1,3)**2-12*gm(1,1)&
&   *gm(3,3))*dgm01(2,2)+gm(2,3)*(48*gm(1,3)*dgm01(1,2)+48*gm(1,2)&
&   *dgm01(1,3)-72*gm(1,1)*dgm01(2,3))+192*gm(1,2)*(gm(3,3)*dgm01(1,2)&
&   +gm(1,3)*dgm01(2,3))))*dgm10(1,2)+(gm(2,2)**2*(72*gm(2,3)*dgm01(1,1)&
&   +288*gm(1,3)*dgm01(1,2))+36*gm(1,2)**2*gm(2,3)*dgm01(2,2)+gm(2,2)&
&   *(-108*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(1,2)*(144*gm(2,3)*dgm01(1,2)&
&   +288*gm(1,3)*dgm01(2,2))))*dgm10(1,3)+(6*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)+6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(48*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,3)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(2,2)+(gm(2,3)&
&   *(72*gm(1,2)**2*dgm01(1,2)-216*gm(1,1)*gm(2,2)*dgm01(1,2)+gm(1,2)&
&   *(72*gm(2,2)*dgm01(1,1)-72*gm(1,1)*dgm01(2,2)))+gm(1,3)*(144*gm(2,2)&
&   **2*dgm01(1,1)+180*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(576*gm(1,2)&
&   *dgm01(1,2)-36*gm(1,1)*dgm01(2,2))))*dgm10(2,3))/12.d0
   cm(10,13)=((6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)+6*gm(3,3)&
&   *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(2,2)+6*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3))*dgm10(1,1)+6*(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,1)+4*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(1,2)+2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(2,2)+2*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3))*gm(3,3))*dgm01(2,3))*dgm10(1,2)+(2*(90*gm(2,3)&
&   **3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)+12*(-36*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +6*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))&
&   *gm(3,3))*dgm01(2,2))*dgm10(1,3)+(6*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&   -18*gm(1,2)*gm(3,3))*dgm01(1,1)+6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(1,2)+6*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3))*gm(3,3))*dgm01(1,3)+2*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(2,3))*dgm10(2,2)+(6*(-36*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +12*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))&
&   *gm(3,3))*dgm01(1,2)+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))&
&   *dgm01(2,2))*dgm10(2,3))/36.d0
   cm(1,14)=((2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2)&
&   +6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(1,3)+6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(2,3)+6*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&   *dgm01(3,3))*dgm10(1,1)+(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm01(1,1)+12*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(1,3)+216*gm(1,1)**2*gm(1,3)*dgm01(3,3))*dgm10(1,2)+3*(2*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(1,1)+4*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)&
&   +8*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(1,3)&
&   +144*gm(1,1)**2*gm(1,3)*dgm01(2,3)+72*gm(1,1)**2*gm(1,2)*dgm01(3,3))&
&   *dgm10(1,3)+gm(1,1)*(6*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,1)&
&   +432*gm(1,1)*gm(1,3)*dgm01(1,3)+72*gm(1,1)**2*dgm01(3,3))*dgm10(2,3)&
&   +72*gm(1,1)*(gm(1,2)*(4.5d0*gm(1,3)*dgm01(1,1)+3*gm(1,1)*dgm01(1,3))&
&   +gm(1,1)*(-1.5d0*gm(2,3)*dgm01(1,1)+3*gm(1,3)*dgm01(1,2)+gm(1,1)&
&   *dgm01(2,3)))*dgm10(3,3))/36.d0
   cm(2,14)=((2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*(6*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(1,1)+(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(1,2)+6*((48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+2*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+4*(6*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,3)+2*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(1,3)+(6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3)+2*gm(1,1)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(2,3)+(36*gm(1,2)&
&   **3*dgm01(1,3)+gm(1,2)*(36*gm(1,3)*gm(2,2)*dgm01(1,1)+gm(1,1)&
&   *(288*gm(2,3)*dgm01(1,2)+180*gm(2,2)*dgm01(1,3)))+gm(1,1)*gm(2,2)&
&   *(144*gm(2,3)*dgm01(1,1)-108*gm(1,3)*dgm01(1,2)-36*gm(1,1)*dgm01(2,3))&
&   +gm(1,2)**2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,3)*dgm01(1,2)+108*gm(1,1)&
&   *dgm01(2,3)))*dgm10(3,3))/12.d0
   cm(3,14)=((gm(1,3)*gm(3,3)*(72*gm(3,3)*dgm01(1,2)+72*gm(2,3)&
&   *dgm01(1,3)+36*gm(1,2)*dgm01(3,3))+gm(1,3)**2*(72*gm(3,3)*dgm01(2,3)&
&   +36*gm(2,3)*dgm01(3,3))+144*gm(3,3)*(gm(1,2)*gm(3,3)*dgm01(1,3)&
&   +gm(1,1)*(gm(3,3)*dgm01(2,3)+gm(2,3)*dgm01(3,3))))*dgm10(1,1)&
&   +(144*gm(1,3)**2*gm(3,3)*dgm01(1,3)+288*gm(1,1)*gm(3,3)**2*dgm01(1,3)&
&   +36*gm(1,3)**3*dgm01(3,3)+gm(1,3)*gm(3,3)*(72*gm(3,3)*dgm01(1,1)&
&   +180*gm(1,1)*dgm01(3,3)))*dgm10(1,2)+3*(24*gm(1,3)**3*dgm01(2,3)&
&   +gm(1,3)**2*(48*gm(3,3)*dgm01(1,2)+48*gm(2,3)*dgm01(1,3)+12*gm(1,2)&
&   *dgm01(3,3))+gm(3,3)*(gm(1,1)*(96*gm(3,3)*dgm01(1,2)+192*gm(2,3)&
&   *dgm01(1,3))+gm(1,2)*(48*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3)))&
&   +gm(1,3)*(gm(3,3)*(48*gm(1,2)*dgm01(1,3)+120*gm(1,1)*dgm01(2,3))&
&   +gm(2,3)*(24*gm(3,3)*dgm01(1,1)+96*gm(1,1)*dgm01(3,3))))*dgm10(1,3)&
&   +(72*gm(1,3)**3*dgm01(1,3)+360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3)&
&   +gm(1,1)*gm(3,3)*(144*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3))&
&   +gm(1,3)**2*(72*gm(3,3)*dgm01(1,1)+108*gm(1,1)*dgm01(3,3)))*dgm10(2,3)&
&   +(36*gm(1,3)**3*dgm01(1,2)+gm(1,3)*(36*gm(1,2)*gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*(180*gm(3,3)*dgm01(1,2)+288*gm(2,3)*dgm01(1,3)))+gm(1,1)&
&   *gm(3,3)*(144*gm(2,3)*dgm01(1,1)-108*gm(1,2)*dgm01(1,3)-36*gm(1,1)&
&   *dgm01(2,3))+gm(1,3)**2*(36*gm(2,3)*dgm01(1,1)+36*gm(1,2)*dgm01(1,3)&
&   +108*gm(1,1)*dgm01(2,3)))*dgm10(3,3))/12.d0
   cm(4,14)=((2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(1,2)+6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+6*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,1)+(gm(1,3)&
&   **2*(72*gm(2,3)*dgm01(1,3)+36*gm(1,2)*dgm01(3,3))+gm(1,3)*(72*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)+gm(2,3)*(108*gm(3,3)*dgm01(1,1)+36*gm(1,1)&
&   *dgm01(3,3)))+gm(3,3)*(288*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)&
&   *(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3))))*dgm10(1,2)&
&   +3*(72*gm(1,2)**2*gm(3,3)*dgm01(1,3)+gm(1,1)*(96*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+72*gm(2,3)**2*dgm01(1,3)+120*gm(2,2)*gm(3,3)*dgm01(1,3))&
&   +gm(1,3)**2*(24*gm(2,3)*dgm01(1,2)+72*gm(2,2)*dgm01(1,3)+24*gm(1,2)&
&   *dgm01(2,3))+gm(1,2)*(96*gm(1,1)*gm(3,3)*dgm01(2,3)+12*gm(2,3)&
&   *(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))+gm(1,3)*(12*gm(2,3)&
&   **2*dgm01(1,1)+gm(2,3)*(-48*gm(1,2)*dgm01(1,3)+24*gm(1,1)*dgm01(2,3))&
&   +48*gm(2,2)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)&
&   *dgm01(1,2)+12*gm(1,2)*dgm01(3,3))))*dgm10(1,3)+(gm(1,3)**2*(36*gm(2,3)&
&   *dgm01(1,1)+72*gm(1,2)*dgm01(1,3))+gm(1,1)*(288*gm(1,2)*gm(3,3)&
&   *dgm01(1,3)+gm(2,3)*(144*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3)))&
&   +gm(1,3)*(72*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)*(36*gm(3,3)*dgm01(1,1)&
&   +108*gm(1,1)*dgm01(3,3))))*dgm10(2,3)+(54*gm(1,2)**2*gm(3,3)&
&   *dgm01(1,1)+gm(1,3)**2*(54*gm(2,2)*dgm01(1,1)+36*gm(1,2)*dgm01(1,2))&
&   +gm(1,1)*(54*gm(2,3)**2*dgm01(1,1)+gm(3,3)*(90*gm(2,2)*dgm01(1,1)&
&   +144*gm(1,2)*dgm01(1,2))+36*gm(1,2)*gm(2,3)*dgm01(1,3))-36*gm(1,1)&
&   **2*gm(2,3)*dgm01(2,3)+gm(1,3)*(36*gm(1,2)**2*dgm01(1,3)+gm(1,1)&
&   *(36*gm(2,3)*dgm01(1,2)+144*gm(2,2)*dgm01(1,3))+gm(1,2)*(-36*gm(2,3)&
&   *dgm01(1,1)+108*gm(1,1)*dgm01(2,3))))*dgm10(3,3))/6.d0
   cm(5,14)=((2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)&
&   +6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(1,3)+6*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm01(2,3)+6*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm01(3,3))*dgm10(1,1)+(72*gm(1,3)**3*dgm01(1,3)&
&   +360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3)+gm(1,3)**2*(108*gm(3,3)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(3,3))+gm(1,1)*gm(3,3)*(-36*gm(3,3)&
&   *dgm01(1,1)+144*gm(1,1)*dgm01(3,3)))*dgm10(1,2)+3*(24*gm(1,3)&
&   **3*dgm01(1,2)+gm(1,3)**2*(12*gm(2,3)*dgm01(1,1)+48*(gm(1,2)&
&   *dgm01(1,3)+gm(1,1)*dgm01(2,3)))+gm(1,3)*(gm(1,1)*(120*gm(3,3)&
&   *dgm01(1,2)+48*gm(2,3)*dgm01(1,3))+gm(1,2)*(96*gm(3,3)*dgm01(1,1)&
&   +24*gm(1,1)*dgm01(3,3)))+gm(1,1)*(gm(3,3)*(192*gm(1,2)*dgm01(1,3)&
&   +96*gm(1,1)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)*dgm01(1,1)+48*gm(1,1)&
&   *dgm01(3,3))))*dgm10(1,3)+(36*gm(1,3)**3*dgm01(1,1)+144*gm(1,1)&
&   *gm(1,3)**2*dgm01(1,3)+288*gm(1,1)**2*gm(3,3)*dgm01(1,3)+gm(1,1)&
&   *gm(1,3)*(180*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3)))*dgm10(2,3)&
&   +(gm(1,2)*(36*gm(1,3)**2*dgm01(1,1)+144*gm(1,1)*gm(3,3)*dgm01(1,1)&
&   +72*gm(1,1)*gm(1,3)*dgm01(1,3))+gm(1,1)*(72*gm(1,3)**2*dgm01(1,2)&
&   +144*gm(1,1)*(gm(3,3)*dgm01(1,2)+gm(2,3)*dgm01(1,3))+gm(1,3)&
&   *(36*gm(2,3)*dgm01(1,1)+72*gm(1,1)*dgm01(2,3))))*dgm10(3,3))&
&   /12.d0
   cm(6,14)=((2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)+6*(6*gm(1,2)&
&   **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(1,1)+(gm(1,3)**2*(180*gm(2,3)*dgm01(1,1)+72*gm(1,2)&
&   *dgm01(1,3))+gm(1,3)*(576*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)&
&   *(-72*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3)))+gm(1,1)*(-216*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)+gm(2,3)*(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)&
&   *dgm01(3,3))))*dgm10(1,2)+3*(-24*gm(1,2)**2*gm(3,3)*dgm01(1,1)&
&   +gm(1,3)**2*(60*gm(2,2)*dgm01(1,1)+24*gm(1,2)*dgm01(1,2))+gm(1,3)&
&   *(48*gm(1,2)**2*dgm01(1,3)+192*gm(1,1)*(gm(2,3)*dgm01(1,2)+gm(2,2)&
&   *dgm01(1,3))+gm(1,2)*(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)*dgm01(2,3)))&
&   +gm(1,1)*(-24*gm(2,3)**2*dgm01(1,1)-12*gm(2,2)*gm(3,3)*dgm01(1,1)&
&   -72*gm(1,2)*gm(3,3)*dgm01(1,2)+48*gm(1,2)*gm(2,3)*dgm01(1,3)&
&   +24*gm(1,2)**2*dgm01(3,3))+gm(1,1)**2*(96*gm(2,3)*dgm01(2,3)&
&   +48*gm(2,2)*dgm01(3,3)))*dgm10(1,3)+(288*gm(1,1)*gm(2,3)*(gm(1,3)&
&   *dgm01(1,1)+gm(1,1)*dgm01(1,3))+gm(1,2)*(36*gm(1,3)**2*dgm01(1,1)&
&   +144*gm(1,1)*gm(1,3)*dgm01(1,3)+gm(1,1)*(-108*gm(3,3)*dgm01(1,1)&
&   +72*gm(1,1)*dgm01(3,3))))*dgm10(2,3)+(gm(1,2)**2*(36*gm(1,3)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(1,3))+144*gm(1,1)*(gm(1,3)*gm(2,2)&
&   *dgm01(1,1)+gm(1,1)*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3)))&
&   +gm(1,1)*gm(1,2)*(36*gm(2,3)*dgm01(1,1)+72*(gm(1,3)*dgm01(1,2)&
&   +gm(1,1)*dgm01(2,3))))*dgm10(3,3))/12.d0
   cm(7,14)=((2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)+6*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(1,1)+(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)&
&   *gm(3,3))*dgm01(1,1)+12*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(90*gm(1,2)&
&   **2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(3,3))&
&   *dgm10(1,2)+6*(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&   +2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&   *gm(2,3))*dgm01(1,3)+2*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(2,3)+gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,3)+(6*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(90*gm(1,2)&
&   **2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,3)&
&   +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(2,3)&
&   +(6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,1)&
&   +6*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&   *gm(2,3)))*dgm01(1,2)+6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&   *dgm01(1,3)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))&
&   *dgm10(3,3))/36.d0
   cm(8,14)=((72*gm(2,3)*gm(3,3)**2*dgm01(1,2)+6*gm(3,3)*(12*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3))*dgm01(1,3)+6*gm(3,3)*(12*gm(1,3)*gm(2,3)&
&   +24*gm(1,2)*gm(3,3))*dgm01(2,3)+6*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,1)&
&   +(288*gm(1,2)*gm(3,3)*(gm(3,3)*dgm01(1,3)+gm(1,3)*dgm01(3,3))&
&   +gm(2,3)*(72*gm(3,3)**2*dgm01(1,1)+36*gm(1,3)**2*dgm01(3,3)+gm(3,3)&
&   *(144*gm(1,3)*dgm01(1,3)-108*gm(1,1)*dgm01(3,3))))*dgm10(1,2)&
&   +6*(gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,1)+2*gm(3,3)&
&   *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(2,3)+(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,3)+(gm(1,3)**2*(72*gm(2,3)*dgm01(1,3)&
&   +180*gm(1,2)*dgm01(3,3))+gm(1,3)*(576*gm(1,2)*gm(3,3)*dgm01(1,3)&
&   +gm(2,3)*(72*gm(3,3)*dgm01(1,1)-72*gm(1,1)*dgm01(3,3)))+gm(3,3)&
&   *(-216*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)*(144*gm(3,3)*dgm01(1,1)&
&   -36*gm(1,1)*dgm01(3,3))))*dgm10(2,3)+(6*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)+6*(6*gm(1,3)&
&   **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3)+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,3))*dgm10(3,3))/12.d0
   cm(9,14)=((2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)+6*(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,1)+(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,1)+12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,2)+6*((6*gm(2,3)&
&   **3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)+2*(48*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3)+(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(1,3)+(6*(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)+(6*(6*gm(1,3)*gm(2,2)&
&   *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(1,3)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(3,3))&
&   /12.d0
   cm(10,14)=(72*gm(3,3)*(1*gm(3,3)**2*dgm01(1,2)+4.5d0*gm(1,3)&
&   *gm(2,3)*dgm01(3,3)+gm(3,3)*(3*gm(2,3)*dgm01(1,3)+3*gm(1,3)*dgm01(2,3)&
&   -1.5d0*gm(1,2)*dgm01(3,3)))*dgm10(1,1)+gm(3,3)*(72*gm(3,3)**2*dgm01(1,1)&
&   +324*gm(1,3)**2*dgm01(3,3)+gm(3,3)*(432*gm(1,3)*dgm01(1,3)-108*gm(1,1)&
&   *dgm01(3,3)))*dgm10(1,2)+3*(72*gm(2,3)*gm(3,3)**2*dgm01(1,1)&
&   +144*gm(1,3)*gm(3,3)**2*dgm01(1,2)+8*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&   -18*gm(1,2)*gm(3,3))*dgm01(1,3)+4*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(2,3)+2*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3))*gm(3,3))*dgm01(3,3))*dgm10(1,3)+(648*gm(1,3)&
&   **2*gm(3,3)*dgm01(1,3)-216*gm(1,1)*gm(3,3)**2*dgm01(1,3)+180*gm(1,3)&
&   **3*dgm01(3,3)+gm(1,3)*gm(3,3)*(216*gm(3,3)*dgm01(1,1)-108*gm(1,1)&
&   *dgm01(3,3)))*dgm10(2,3)+(6*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(1,1)+6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(1,2)+6*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&   *gm(2,3))*gm(3,3))*dgm01(1,3)+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm01(2,3))*dgm10(3,3))/36.d0
   cm(1,15)=((8*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,3)&
&   +12*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3))*dgm10(1,1)&
&   +4*(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)+12*gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)+216*gm(1,1)**2*gm(1,3)&
&   *dgm01(3,3))*dgm10(1,3)+2*gm(1,1)*(6*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(1,1)+432*gm(1,1)*gm(1,3)*dgm01(1,3)+72*gm(1,1)&
&   **2*dgm01(3,3))*dgm10(3,3))/144.d0
   cm(2,15)=((8*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,1)+4*(2*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3)+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,3)+2*(6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(1,3)+2*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(3,3))/48.d0
   cm(3,15)=(gm(3,3)*(288*gm(1,3)*gm(3,3)*dgm01(1,3)+12*(12*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3))*dgm01(3,3))*dgm10(1,1)+4*(144*gm(1,3)&
&   **2*gm(3,3)*dgm01(1,3)+288*gm(1,1)*gm(3,3)**2*dgm01(1,3)+36*gm(1,3)&
&   **3*dgm01(3,3)+gm(1,3)*gm(3,3)*(72*gm(3,3)*dgm01(1,1)+180*gm(1,1)&
&   *dgm01(3,3)))*dgm10(1,3)+2*(72*gm(1,3)**3*dgm01(1,3)+360*gm(1,1)&
&   *gm(1,3)*gm(3,3)*dgm01(1,3)+gm(1,1)*gm(3,3)*(144*gm(3,3)*dgm01(1,1)&
&   -36*gm(1,1)*dgm01(3,3))+gm(1,3)**2*(72*gm(3,3)*dgm01(1,1)+108*gm(1,1)&
&   *dgm01(3,3)))*dgm10(3,3))/48.d0
   cm(4,15)=((8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(1,3)+12*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&   +24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(1,1)+4*(gm(1,3)&
&   **2*(72*gm(2,3)*dgm01(1,3)+36*gm(1,2)*dgm01(3,3))+gm(1,3)*(72*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)+gm(2,3)*(108*gm(3,3)*dgm01(1,1)+36*gm(1,1)&
&   *dgm01(3,3)))+gm(3,3)*(288*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)&
&   *(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3))))*dgm10(1,3)&
&   +2*(gm(1,3)**2*(36*gm(2,3)*dgm01(1,1)+72*gm(1,2)*dgm01(1,3))&
&   +gm(1,1)*(288*gm(1,2)*gm(3,3)*dgm01(1,3)+gm(2,3)*(144*gm(3,3)&
&   *dgm01(1,1)-36*gm(1,1)*dgm01(3,3)))+gm(1,3)*(72*gm(1,1)*gm(2,3)&
&   *dgm01(1,3)+gm(1,2)*(36*gm(3,3)*dgm01(1,1)+108*gm(1,1)*dgm01(3,3))))&
&   *dgm10(3,3))/24.d0
   cm(5,15)=((8*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)&
&   +12*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3))*dgm10(1,1)&
&   +4*(72*gm(1,3)**3*dgm01(1,3)+360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3)&
&   +gm(1,3)**2*(108*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3))+gm(1,1)&
&   *gm(3,3)*(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3)))*dgm10(1,3)&
&   +2*(36*gm(1,3)**3*dgm01(1,1)+144*gm(1,1)*gm(1,3)**2*dgm01(1,3)&
&   +288*gm(1,1)**2*gm(3,3)*dgm01(1,3)+gm(1,1)*gm(1,3)*(180*gm(3,3)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(3,3)))*dgm10(3,3))/48.d0
   cm(6,15)=((8*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,3)+12*(48*gm(1,1)*gm(1,3)&
&   *gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(1,1)+4*(gm(1,3)**2*(180*gm(2,3)*dgm01(1,1)+72*gm(1,2)&
&   *dgm01(1,3))+gm(1,3)*(576*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)&
&   *(-72*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3)))+gm(1,1)*(-216*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)+gm(2,3)*(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)&
&   *dgm01(3,3))))*dgm10(1,3)+2*(288*gm(1,1)*gm(2,3)*(gm(1,3)*dgm01(1,1)&
&   +gm(1,1)*dgm01(1,3))+gm(1,2)*(36*gm(1,3)**2*dgm01(1,1)+144*gm(1,1)&
&   *gm(1,3)*dgm01(1,3)+gm(1,1)*(-108*gm(3,3)*dgm01(1,1)+72*gm(1,1)&
&   *dgm01(3,3))))*dgm10(3,3))/48.d0
   cm(7,15)=((8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)&
&   +12*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,1)+4*(2*(90*gm(2,3)**3-54*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,1)+12*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(90*gm(1,2)&
&   **2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(3,3))&
&   *dgm10(1,3)+2*(6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(90*gm(1,2)**2*gm(2,3)&
&   -18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,3)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(3,3))/144.d0
   cm(8,15)=(gm(3,3)*(288*gm(2,3)*gm(3,3)*dgm01(1,3)+12*(12*gm(1,3)&
&   *gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(3,3))*dgm10(1,1)+4*(288*gm(1,2)&
&   *gm(3,3)*(gm(3,3)*dgm01(1,3)+gm(1,3)*dgm01(3,3))+gm(2,3)*(72*gm(3,3)&
&   **2*dgm01(1,1)+36*gm(1,3)**2*dgm01(3,3)+gm(3,3)*(144*gm(1,3)&
&   *dgm01(1,3)-108*gm(1,1)*dgm01(3,3))))*dgm10(1,3)+2*(gm(1,3)**2*(72*gm(2,3)&
&   *dgm01(1,3)+180*gm(1,2)*dgm01(3,3))+gm(1,3)*(576*gm(1,2)*gm(3,3)&
&   *dgm01(1,3)+gm(2,3)*(72*gm(3,3)*dgm01(1,1)-72*gm(1,1)*dgm01(3,3)))&
&   +gm(3,3)*(-216*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)*(144*gm(3,3)&
&   *dgm01(1,1)-36*gm(1,1)*dgm01(3,3))))*dgm10(3,3))/48.d0
   cm(9,15)=((8*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)&
&   +12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,1)+4*(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,1)+12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,3)+2*(6*(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)&
&   +12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(3,3))/48.d0
   cm(10,15)=(gm(3,3)**2*(288*gm(3,3)*dgm01(1,3)+432*gm(1,3)*dgm01(3,3))&
&   *dgm10(1,1)+4*gm(3,3)*(72*gm(3,3)**2*dgm01(1,1)+324*gm(1,3)**2*dgm01(3,3)&
&   +gm(3,3)*(432*gm(1,3)*dgm01(1,3)-108*gm(1,1)*dgm01(3,3)))*dgm10(1,3)&
&   +2*(648*gm(1,3)**2*gm(3,3)*dgm01(1,3)-216*gm(1,1)*gm(3,3)**2*dgm01(1,3)&
&   +180*gm(1,3)**3*dgm01(3,3)+gm(1,3)*gm(3,3)*(216*gm(3,3)*dgm01(1,1)&
&   -108*gm(1,1)*dgm01(3,3)))*dgm10(3,3))/144.d0
   cm(1,16)=(2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2)&
&   *dgm10(1,1)+(8*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,2)&
&   +12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2))*dgm10(1,2)&
&   +(2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)+12*gm(1,1)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,2)+216*gm(1,1)**2*gm(1,2)&
&   *dgm01(2,2))*dgm10(2,2))/72.d0
   cm(2,16)=(1440*gm(1,2)*gm(2,2)**2*dgm01(2,2)*dgm10(1,1)+20*gm(2,2)&
&   *(288*gm(1,2)*gm(2,2)*dgm01(1,2)+12*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm01(2,2))*dgm10(1,2)+20*(144*gm(1,2)**2*gm(2,2)*dgm01(1,2)&
&   +288*gm(1,1)*gm(2,2)**2*dgm01(1,2)+36*gm(1,2)**3*dgm01(2,2)+gm(1,2)&
&   *gm(2,2)*(72*gm(2,2)*dgm01(1,1)+180*gm(1,1)*dgm01(2,2)))*dgm10(2,2))&
&   /480.d0
   cm(3,16)=(2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(1,1)+(8*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2))*dgm10(1,2)+(2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(48*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2))&
&   *dgm10(2,2))/24.d0
   cm(4,16)=(2*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(2,2)*dgm10(1,1)+(8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&   *gm(2,3))*dgm01(1,2)+12*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2))*dgm10(1,2)&
&   +(gm(2,3)*(72*gm(1,2)**2*dgm01(1,2)+288*gm(1,1)*gm(2,2)*dgm01(1,2)&
&   +gm(1,2)*(108*gm(2,2)*dgm01(1,1)+36*gm(1,1)*dgm01(2,2)))+gm(1,3)&
&   *(-36*gm(2,2)**2*dgm01(1,1)+36*gm(1,2)**2*dgm01(2,2)+gm(2,2)&
&   *(72*gm(1,2)*dgm01(1,2)+144*gm(1,1)*dgm01(2,2))))*dgm10(2,2))&
&   /12.d0
   cm(5,16)=(2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(1,1)+(8*(-36*gm(1,2)&
&   *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,2)+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2))*dgm10(1,2)+(gm(1,2)&
&   **2*(180*gm(2,3)*dgm01(1,1)+72*gm(1,3)*dgm01(1,2))+gm(1,1)*(gm(2,2)&
&   *(-36*gm(2,3)*dgm01(1,1)-216*gm(1,3)*dgm01(1,2))+144*gm(1,1)&
&   *gm(2,3)*dgm01(2,2))+gm(1,2)*(576*gm(1,1)*gm(2,3)*dgm01(1,2)&
&   +gm(1,3)*(-72*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))))*dgm10(2,2))&
&   /24.d0
   cm(6,16)=(2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2)&
&   *dgm10(1,1)+(8*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,2)&
&   +12*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2))*dgm10(1,2)&
&   +(72*gm(1,2)**3*dgm01(1,2)+360*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2)&
&   +gm(1,2)**2*(108*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))+gm(1,1)&
&   *gm(2,2)*(-36*gm(2,2)*dgm01(1,1)+144*gm(1,1)*dgm01(2,2)))*dgm10(2,2))&
&   /24.d0
   cm(7,16)=(gm(2,2)*(1440*gm(2,2)**2*dgm01(2,2)*dgm10(1,1)+20*gm(2,2)&
&   *(288*gm(2,2)*dgm01(1,2)+432*gm(1,2)*dgm01(2,2))*dgm10(1,2)+20*(72*gm(2,2)&
&   **2*dgm01(1,1)+324*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(432*gm(1,2)&
&   *dgm01(1,2)-108*gm(1,1)*dgm01(2,2)))*dgm10(2,2)))/1440.d0
   cm(8,16)=(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+(8*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2))*dgm10(1,2)+(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,1)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,2))*dgm10(2,2))/24.d0
   cm(9,16)=(1440*gm(2,2)**2*gm(2,3)*dgm01(2,2)*dgm10(1,1)+20*gm(2,2)&
&   *(144*gm(1,2)*gm(2,3)*dgm01(2,2)+288*gm(2,2)*(gm(2,3)*dgm01(1,2)&
&   +gm(1,3)*dgm01(2,2)))*dgm10(1,2)+20*(gm(2,2)**2*(72*gm(2,3)*dgm01(1,1)&
&   +288*gm(1,3)*dgm01(1,2))+36*gm(1,2)**2*gm(2,3)*dgm01(2,2)+gm(2,2)&
&   *(-108*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(1,2)*(144*gm(2,3)*dgm01(1,2)&
&   +288*gm(1,3)*dgm01(2,2))))*dgm10(2,2))/480.d0
   cm(10,16)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+(8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2))*dgm10(1,2)+(2*(90*gm(2,3)**3-54*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,1)+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(2,2))&
&   *dgm10(2,2))/72.d0
   cm(1,17)=(6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)*dgm10(1,1)+12*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(2,3)&
&   *dgm10(1,1)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3)&
&   *dgm10(1,1)+12*(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)+2*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,3)+gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+4*gm(1,1)*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,3)+gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,2)+4*(6*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,2)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,3)+6*gm(1,1)*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,2)+6*gm(1,1)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(1,3)+3*(2*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(1,1)+4*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)&
&   +8*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(1,3)&
&   +144*gm(1,1)**2*gm(1,3)*dgm01(2,3)+72*gm(1,1)**2*gm(1,2)*dgm01(3,3))&
&   *dgm10(2,2)+6*(2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&   +2*gm(1,2)*gm(2,3)))*dgm01(1,1)+8*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&   -18*gm(1,1)*gm(2,3))*dgm01(1,2)+4*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,3)+72*gm(1,1)**2*gm(1,3)*dgm01(2,2)+144*gm(1,1)&
&   **2*gm(1,2)*dgm01(2,3))*dgm10(2,3)+(2*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(1,1)+12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,2)+216*gm(1,1)**2*gm(1,2)*dgm01(2,2))*dgm10(3,3))&
&   /72.d0
   cm(2,17)=(12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(1,1)+24*gm(2,2)*(24*gm(1,3)&
&   *gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(1,1)+144*gm(1,2)&
&   *gm(2,2)**2*dgm01(3,3)*dgm10(1,1)+24*(2*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*gm(2,2)&
&   *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(1,3)+(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,2)+4*(6*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(2,3)+gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(1,2)+8*(6*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&   *dgm01(1,2)+72*gm(1,2)*gm(2,2)**2*dgm01(1,3)+6*(6*gm(1,2)*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)&
&   +6*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(1,3)&
&   +12*((48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)+2*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))&
&   *dgm10(2,2)+12*(48*gm(1,2)**2*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3))&
&   +gm(1,1)*gm(2,2)*(192*gm(2,3)*dgm01(1,2)+96*gm(2,2)*dgm01(1,3))&
&   +gm(1,3)*(48*gm(2,2)**2*dgm01(1,1)+12*gm(1,2)**2*dgm01(2,2)+gm(2,2)&
&   *(48*gm(1,2)*dgm01(1,2)-36*gm(1,1)*dgm01(2,2)))+24*gm(1,2)**3*dgm01(2,3)&
&   +gm(1,2)*(96*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(2,2)*(24*gm(2,3)*dgm01(1,1)&
&   +120*gm(1,1)*dgm01(2,3))))*dgm10(2,3)+2*(144*gm(1,2)**2*gm(2,2)&
&   *dgm01(1,2)+288*gm(1,1)*gm(2,2)**2*dgm01(1,2)+36*gm(1,2)**3*dgm01(2,2)&
&   +gm(1,2)*gm(2,2)*(72*gm(2,2)*dgm01(1,1)+180*gm(1,1)*dgm01(2,2)))&
&   *dgm10(3,3))/48.d0
   cm(3,17)=(6*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&   *dgm01(2,2)*dgm10(1,1)+12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(1,1)+2*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(3,3)*dgm10(1,1)+12*(2*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&   *gm(3,3))*dgm01(1,2)+2*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3))*dgm01(2,2)+4*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,2)&
&   +4*(6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(6*gm(1,3)&
&   **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,2)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3))*dgm10(1,3)+3*(24*gm(1,3)**3*dgm01(2,3)+gm(1,3)**2*(48*gm(3,3)&
&   *dgm01(1,2)+48*gm(2,3)*dgm01(1,3)+12*gm(1,2)*dgm01(3,3))+gm(3,3)&
&   *(gm(1,1)*(96*gm(3,3)*dgm01(1,2)+192*gm(2,3)*dgm01(1,3))+gm(1,2)&
&   *(48*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3)))+gm(1,3)*(gm(3,3)&
&   *(48*gm(1,2)*dgm01(1,3)+120*gm(1,1)*dgm01(2,3))+gm(2,3)*(24*gm(3,3)&
&   *dgm01(1,1)+96*gm(1,1)*dgm01(3,3))))*dgm10(2,2)+6*(-48*gm(1,2)&
&   **2*gm(3,3)*dgm01(1,3)+gm(1,1)*(192*gm(2,3)*gm(3,3)*dgm01(1,2)&
&   +120*gm(2,3)**2*dgm01(1,3)-24*gm(2,2)*gm(3,3)*dgm01(1,3))+12*gm(1,3)&
&   **3*dgm01(2,2)+gm(1,2)*gm(3,3)*(96*gm(2,3)*dgm01(1,1)-72*gm(1,1)&
&   *dgm01(2,3))+gm(1,3)**2*(48*gm(2,3)*dgm01(1,2)-48*gm(2,2)*dgm01(1,3)&
&   +24*gm(1,2)*dgm01(2,3))+gm(1,3)*(12*gm(2,3)**2*dgm01(1,1)+gm(3,3)&
&   *(-36*gm(2,2)*dgm01(1,1)+48*gm(1,2)*dgm01(1,2)+60*gm(1,1)*dgm01(2,2))&
&   +gm(2,3)*(144*gm(1,2)*dgm01(1,3)+192*gm(1,1)*dgm01(2,3))))*dgm10(2,3)&
&   +(2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,2))*dgm10(3,3))/24.d0
   cm(4,17)=(6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)*dgm10(1,1)+12*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(1,1)&
&   +2*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(3,3)&
&   *dgm10(1,1)+12*(2*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)+(6*gm(1,3)&
&   **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,2)+4*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)&
&   +9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(1,2)+4*(6*(6*gm(1,3)*gm(2,2)&
&   *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +2*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,3)&
&   +6*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(2,3))*dgm10(1,3)+3*(72*gm(1,2)**2*gm(3,3)&
&   *dgm01(1,3)+gm(1,1)*(96*gm(2,3)*gm(3,3)*dgm01(1,2)+72*gm(2,3)&
&   **2*dgm01(1,3)+120*gm(2,2)*gm(3,3)*dgm01(1,3))+gm(1,3)**2*(24*gm(2,3)&
&   *dgm01(1,2)+72*gm(2,2)*dgm01(1,3)+24*gm(1,2)*dgm01(2,3))+gm(1,2)&
&   *(96*gm(1,1)*gm(3,3)*dgm01(2,3)+12*gm(2,3)*(gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*dgm01(3,3)))+gm(1,3)*(12*gm(2,3)**2*dgm01(1,1)+gm(2,3)&
&   *(-48*gm(1,2)*dgm01(1,3)+24*gm(1,1)*dgm01(2,3))+48*gm(2,2)*(gm(3,3)&
&   *dgm01(1,1)+gm(1,1)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)*dgm01(1,2)&
&   +12*gm(1,2)*dgm01(3,3))))*dgm10(2,2)+6*(gm(1,2)**2*(72*gm(3,3)&
&   *dgm01(1,2)+24*gm(2,3)*dgm01(1,3))+gm(1,1)*(72*gm(2,3)**2*dgm01(1,2)&
&   +120*gm(2,2)*gm(3,3)*dgm01(1,2)+96*gm(2,2)*gm(2,3)*dgm01(1,3))&
&   +gm(1,3)**2*(72*gm(2,2)*dgm01(1,2)+12*gm(1,2)*dgm01(2,2))+gm(1,2)&
&   *(12*gm(2,3)**2*dgm01(1,1)+48*gm(3,3)*(gm(2,2)*dgm01(1,1)+gm(1,1)&
&   *dgm01(2,2))+24*gm(1,1)*gm(2,3)*dgm01(2,3))+gm(1,3)*(-48*gm(1,2)&
&   *gm(2,3)*dgm01(1,2)+12*gm(1,1)*gm(2,3)*dgm01(2,2)+24*gm(1,2)&
&   **2*dgm01(2,3)+gm(2,2)*(12*gm(2,3)*dgm01(1,1)+24*gm(1,2)*dgm01(1,3)&
&   +96*gm(1,1)*dgm01(2,3))))*dgm10(2,3)+(gm(2,3)*(72*gm(1,2)**2*dgm01(1,2)&
&   +288*gm(1,1)*gm(2,2)*dgm01(1,2)+gm(1,2)*(108*gm(2,2)*dgm01(1,1)&
&   +36*gm(1,1)*dgm01(2,2)))+gm(1,3)*(-36*gm(2,2)**2*dgm01(1,1)+36*gm(1,2)&
&   **2*dgm01(2,2)+gm(2,2)*(72*gm(1,2)*dgm01(1,2)+144*gm(1,1)*dgm01(2,2))))&
&   *dgm10(3,3))/12.d0
   cm(5,17)=(6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)*dgm10(1,1)+12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(1,1)&
&   +2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(3,3)*dgm10(1,1)+12*(2*(6*gm(1,3)**2*gm(2,3)&
&   +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(2,3)&
&   +(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,2)+4*(6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)&
&   +6*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3)))*dgm01(2,2)+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(1,3)+3*(24*gm(1,3)&
&   **3*dgm01(1,2)+gm(1,3)**2*(12*gm(2,3)*dgm01(1,1)+48*(gm(1,2)&
&   *dgm01(1,3)+gm(1,1)*dgm01(2,3)))+gm(1,3)*(gm(1,1)*(120*gm(3,3)&
&   *dgm01(1,2)+48*gm(2,3)*dgm01(1,3))+gm(1,2)*(96*gm(3,3)*dgm01(1,1)&
&   +24*gm(1,1)*dgm01(3,3)))+gm(1,1)*(gm(3,3)*(192*gm(1,2)*dgm01(1,3)&
&   +96*gm(1,1)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)*dgm01(1,1)+48*gm(1,1)&
&   *dgm01(3,3))))*dgm10(2,2)+6*(60*gm(1,2)**2*gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*(-24*gm(2,3)**2*dgm01(1,1)+gm(3,3)*(-12*gm(2,2)*dgm01(1,1)&
&   +192*gm(1,2)*dgm01(1,2))+192*gm(1,2)*gm(2,3)*dgm01(1,3))+gm(1,3)&
&   **2*(-24*gm(2,2)*dgm01(1,1)+48*gm(1,2)*dgm01(1,2)+24*gm(1,1)&
&   *dgm01(2,2))+gm(1,1)**2*(48*gm(3,3)*dgm01(2,2)+96*gm(2,3)*dgm01(2,3))&
&   +gm(1,3)*(24*gm(1,2)**2*dgm01(1,3)+gm(1,1)*(48*gm(2,3)*dgm01(1,2)&
&   -72*gm(2,2)*dgm01(1,3))+gm(1,2)*(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)&
&   *dgm01(2,3))))*dgm10(2,3)+(gm(1,2)**2*(180*gm(2,3)*dgm01(1,1)&
&   +72*gm(1,3)*dgm01(1,2))+gm(1,1)*(gm(2,2)*(-36*gm(2,3)*dgm01(1,1)&
&   -216*gm(1,3)*dgm01(1,2))+144*gm(1,1)*gm(2,3)*dgm01(2,2))+gm(1,2)&
&   *(576*gm(1,1)*gm(2,3)*dgm01(1,2)+gm(1,3)*(-72*gm(2,2)*dgm01(1,1)&
&   +72*gm(1,1)*dgm01(2,2))))*dgm10(3,3))/24.d0
   cm(6,17)=(6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)*dgm10(1,1)+12*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)*dgm10(1,1)&
&   +2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3)*dgm10(1,1)&
&   +12*(2*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +2*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(1,3)+(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)+4*(6*gm(1,2)**2*gm(1,3)&
&   +24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)&
&   +(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(1,2)&
&   +4*(6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(1,2)+2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,3)+6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&   *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)+6*(6*gm(1,2)**3+30*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(1,3)+3*(-24*gm(1,2)**2*gm(3,3)&
&   *dgm01(1,1)+gm(1,3)**2*(60*gm(2,2)*dgm01(1,1)+24*gm(1,2)*dgm01(1,2))&
&   +gm(1,3)*(48*gm(1,2)**2*dgm01(1,3)+192*gm(1,1)*(gm(2,3)*dgm01(1,2)&
&   +gm(2,2)*dgm01(1,3))+gm(1,2)*(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)&
&   *dgm01(2,3)))+gm(1,1)*(-24*gm(2,3)**2*dgm01(1,1)-12*gm(2,2)*gm(3,3)&
&   *dgm01(1,1)-72*gm(1,2)*gm(3,3)*dgm01(1,2)+48*gm(1,2)*gm(2,3)&
&   *dgm01(1,3)+24*gm(1,2)**2*dgm01(3,3))+gm(1,1)**2*(96*gm(2,3)&
&   *dgm01(2,3)+48*gm(2,2)*dgm01(3,3)))*dgm10(2,2)+6*(24*gm(1,2)&
&   **3*dgm01(1,3)+gm(1,2)*(gm(1,1)*(48*gm(2,3)*dgm01(1,2)+120*gm(2,2)&
&   *dgm01(1,3))+gm(1,3)*(96*gm(2,2)*dgm01(1,1)+24*gm(1,1)*dgm01(2,2)))&
&   +gm(1,2)**2*(12*gm(2,3)*dgm01(1,1)+48*(gm(1,3)*dgm01(1,2)+gm(1,1)&
&   *dgm01(2,3)))+gm(1,1)*(48*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(2,2)&
&   *(-36*gm(2,3)*dgm01(1,1)+192*gm(1,3)*dgm01(1,2)+96*gm(1,1)*dgm01(2,3))))&
&   *dgm10(2,3)+(72*gm(1,2)**3*dgm01(1,2)+360*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm01(1,2)+gm(1,2)**2*(108*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))&
&   +gm(1,1)*gm(2,2)*(-36*gm(2,2)*dgm01(1,1)+144*gm(1,1)*dgm01(2,2)))&
&   *dgm10(3,3))/24.d0
   cm(7,17)=(12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+864*gm(2,2)**2*gm(2,3)*dgm01(2,3)*dgm10(1,1)+144*gm(2,2)&
&   **3*dgm01(3,3)*dgm10(1,1)+12*(180*gm(1,2)*gm(2,3)**2*dgm01(2,2)&
&   +gm(2,2)*(216*gm(2,3)**2*dgm01(1,2)-36*gm(1,2)*gm(3,3)*dgm01(2,2)&
&   +gm(2,3)*(-72*gm(1,3)*dgm01(2,2)+432*gm(1,2)*dgm01(2,3)))+gm(2,2)&
&   **2*(-72*gm(3,3)*dgm01(1,2)+144*gm(2,3)*dgm01(1,3)-144*gm(1,3)&
&   *dgm01(2,3)+72*gm(1,2)*dgm01(3,3)))*dgm10(1,2)+576*gm(2,2)*(1*gm(2,2)&
&   **2*dgm01(1,3)+4.5d0*gm(1,2)*gm(2,3)*dgm01(2,2)+gm(2,2)*(3*gm(2,3)&
&   *dgm01(1,2)-1.5d0*gm(1,3)*dgm01(2,2)+3*gm(1,2)*dgm01(2,3)))*dgm10(1,3)&
&   +12*(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+2*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(1,3)+2*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3)))*dgm01(2,3)+gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(2,2)+12*(180*gm(1,2)**2*gm(2,3)*dgm01(2,2)&
&   +gm(2,2)**2*(72*gm(2,3)*dgm01(1,1)-144*gm(1,3)*dgm01(1,2)+144*gm(1,2)&
&   *dgm01(1,3)-72*gm(1,1)*dgm01(2,3))+gm(2,2)*(-36*gm(1,1)*gm(2,3)&
&   *dgm01(2,2)+gm(1,2)*(432*gm(2,3)*dgm01(1,2)-72*gm(1,3)*dgm01(2,2))&
&   +216*gm(1,2)**2*dgm01(2,3)))*dgm10(2,3)+2*gm(2,2)*(72*gm(2,2)&
&   **2*dgm01(1,1)+324*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(432*gm(1,2)&
&   *dgm01(1,2)-108*gm(1,1)*dgm01(2,2)))*dgm10(3,3))/144.d0
   cm(8,17)=(6*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+12*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+12*(2*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&   *dgm01(1,2)+2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)&
&   +gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,2)+4*(6*(6*gm(2,3)**3+30*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,2)+2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,3)+6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3))*dgm10(1,3)&
&   +6*(gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,1)+2*gm(3,3)&
&   *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(2,3)+(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(2,2)+6*(12*gm(2,3)**3*dgm01(1,1)&
&   +gm(1,3)*gm(3,3)*(192*gm(2,2)*dgm01(1,2)+96*gm(1,2)*dgm01(2,2))&
&   +120*gm(1,3)**2*gm(2,2)*dgm01(2,3)+gm(2,3)**2*(48*gm(1,3)*dgm01(1,2)&
&   +24*gm(1,2)*dgm01(1,3)-48*gm(1,1)*dgm01(2,3))+gm(3,3)*(-72*gm(1,2)&
&   *gm(2,2)*dgm01(1,3)-48*gm(1,2)**2*dgm01(2,3)-24*gm(1,1)*gm(2,2)&
&   *dgm01(2,3))+gm(2,3)*(gm(2,2)*(60*gm(3,3)*dgm01(1,1)+192*gm(1,3)&
&   *dgm01(1,3))+(12*gm(1,3)**2-36*gm(1,1)*gm(3,3))*dgm01(2,2)+gm(1,2)&
&   *(48*gm(3,3)*dgm01(1,2)+144*gm(1,3)*dgm01(2,3))))*dgm10(2,3)&
&   +(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+12*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,2))&
&   *dgm10(3,3))/24.d0
   cm(9,17)=(12*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+24*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+144*gm(2,2)**2*gm(2,3)*dgm01(3,3)*dgm10(1,1)+24*(2*(6*gm(2,3)&
&   **3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)+2*gm(2,2)*(12*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3))*dgm01(1,3)+(48*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)+4*(6*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,2)+8*(36*gm(1,2)*gm(2,3)**2*dgm01(2,2)+gm(2,2)**2*(144*gm(3,3)&
&   *dgm01(1,2)+72*gm(2,3)*dgm01(1,3)+144*gm(1,3)*dgm01(2,3))+gm(2,2)&
&   *(72*gm(2,3)**2*dgm01(1,2)+144*gm(1,2)*gm(3,3)*dgm01(2,2)+gm(2,3)&
&   *(36*gm(1,3)*dgm01(2,2)+72*gm(1,2)*dgm01(2,3))))*dgm10(1,3)+12*((6*gm(2,3)&
&   **3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)+2*(48*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3)+(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(2,2)+12*(gm(2,2)&
&   **2*(48*gm(3,3)*dgm01(1,1)+96*gm(1,3)*dgm01(1,3))-24*gm(1,1)&
&   *gm(2,3)**2*dgm01(2,2)+gm(1,2)*gm(2,3)*(48*gm(2,3)*dgm01(1,2)&
&   +72*gm(1,3)*dgm01(2,2))+gm(1,2)**2*(60*gm(3,3)*dgm01(2,2)+24*gm(2,3)&
&   *dgm01(2,3))+gm(2,2)*(24*gm(2,3)**2*dgm01(1,1)+(-24*gm(1,3)**2-12*gm(1,1)&
&   *gm(3,3))*dgm01(2,2)+gm(2,3)*(48*gm(1,3)*dgm01(1,2)+48*gm(1,2)&
&   *dgm01(1,3)-72*gm(1,1)*dgm01(2,3))+192*gm(1,2)*(gm(3,3)*dgm01(1,2)&
&   +gm(1,3)*dgm01(2,3))))*dgm10(2,3)+2*(gm(2,2)**2*(72*gm(2,3)*dgm01(1,1)&
&   +288*gm(1,3)*dgm01(1,2))+36*gm(1,2)**2*gm(2,3)*dgm01(2,2)+gm(2,2)&
&   *(-108*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(1,2)*(144*gm(2,3)*dgm01(1,2)&
&   +288*gm(1,3)*dgm01(2,2))))*dgm10(3,3))/48.d0
   cm(10,17)=(432*gm(2,3)*gm(3,3)**2*dgm01(2,2)*dgm10(1,1)+24*gm(3,3)&
&   *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)*dgm10(1,1)+4*(90*gm(2,3)&
&   **3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(1,1)+12*(144*gm(2,3)&
&   *gm(3,3)**2*dgm01(1,2)+4*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&   *dgm01(1,3)+72*gm(1,3)*gm(3,3)**2*dgm01(2,2)+8*gm(3,3)*(54*gm(1,3)&
&   *gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(2,3)+2*(-36*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(1,2)+8*(6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)+6*gm(3,3)&
&   *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(2,2)+6*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3))*dgm10(1,3)+6*(72*gm(2,3)*gm(3,3)**2*dgm01(1,1)+144*gm(1,3)&
&   *gm(3,3)**2*dgm01(1,2)+8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(1,3)+4*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(2,3)+2*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&   *gm(2,3))*gm(3,3))*dgm01(3,3))*dgm10(2,2)+24*(gm(3,3)*(54*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+4*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&   -18*gm(1,2)*gm(3,3))*dgm01(1,2)+2*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+gm(3,3)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+2*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(2,3))&
&   *dgm10(2,3)+2*(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&   +12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+6*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(2,2))*dgm10(3,3))/144.d0
   cm(1,18)=(6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&   +2*gm(1,2)*gm(2,3)))*dgm01(2,2)*dgm10(1,1)+4*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(2,3)*dgm10(1,1)+4*(6*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,2)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,3)+6*gm(1,1)*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,2)+6*gm(1,1)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(1,2)+(8*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,2)+12*gm(1,1)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(2,2))*dgm10(1,3)+3*(2*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,1)&
&   +8*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(1,2)&
&   +4*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,3)+72*gm(1,1)&
&   **2*gm(1,3)*dgm01(2,2)+144*gm(1,1)**2*gm(1,2)*dgm01(2,3))*dgm10(2,2)&
&   +2*(2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)+12*gm(1,1)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,2)+216*gm(1,1)**2*gm(1,2)&
&   *dgm01(2,2))*dgm10(2,3))/72.d0
   cm(2,18)=(24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&   *dgm01(2,2)*dgm10(1,1)+576*gm(1,2)*gm(2,2)**2*dgm01(2,3)*dgm10(1,1)&
&   +16*(6*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(1,2)&
&   +72*gm(1,2)*gm(2,2)**2*dgm01(1,3)+6*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)&
&   +6*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(1,2)&
&   +4*gm(2,2)*(288*gm(1,2)*gm(2,2)*dgm01(1,2)+12*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm01(2,2))*dgm10(1,3)+12*(48*gm(1,2)**2*(gm(2,3)*dgm01(1,2)&
&   +gm(2,2)*dgm01(1,3))+gm(1,1)*gm(2,2)*(192*gm(2,3)*dgm01(1,2)&
&   +96*gm(2,2)*dgm01(1,3))+gm(1,3)*(48*gm(2,2)**2*dgm01(1,1)+12*gm(1,2)&
&   **2*dgm01(2,2)+gm(2,2)*(48*gm(1,2)*dgm01(1,2)-36*gm(1,1)*dgm01(2,2)))&
&   +24*gm(1,2)**3*dgm01(2,3)+gm(1,2)*(96*gm(1,1)*gm(2,3)*dgm01(2,2)&
&   +gm(2,2)*(24*gm(2,3)*dgm01(1,1)+120*gm(1,1)*dgm01(2,3))))*dgm10(2,2)&
&   +8*(144*gm(1,2)**2*gm(2,2)*dgm01(1,2)+288*gm(1,1)*gm(2,2)**2*dgm01(1,2)&
&   +36*gm(1,2)**3*dgm01(2,2)+gm(1,2)*gm(2,2)*(72*gm(2,2)*dgm01(1,1)&
&   +180*gm(1,1)*dgm01(2,2)))*dgm10(2,3))/96.d0
   cm(3,18)=(6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)*dgm10(1,1)+4*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(1,1)&
&   +4*(6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(6*gm(1,3)&
&   **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,2)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3))*dgm10(1,2)+(8*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,2))*dgm10(1,3)&
&   +3*(-48*gm(1,2)**2*gm(3,3)*dgm01(1,3)+gm(1,1)*(192*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+120*gm(2,3)**2*dgm01(1,3)-24*gm(2,2)*gm(3,3)*dgm01(1,3))&
&   +12*gm(1,3)**3*dgm01(2,2)+gm(1,2)*gm(3,3)*(96*gm(2,3)*dgm01(1,1)&
&   -72*gm(1,1)*dgm01(2,3))+gm(1,3)**2*(48*gm(2,3)*dgm01(1,2)-48*gm(2,2)&
&   *dgm01(1,3)+24*gm(1,2)*dgm01(2,3))+gm(1,3)*(12*gm(2,3)**2*dgm01(1,1)&
&   +gm(3,3)*(-36*gm(2,2)*dgm01(1,1)+48*gm(1,2)*dgm01(1,2)+60*gm(1,1)&
&   *dgm01(2,2))+gm(2,3)*(144*gm(1,2)*dgm01(1,3)+192*gm(1,1)*dgm01(2,3))))&
&   *dgm10(2,2)+2*(2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2))*dgm10(2,3))&
&   /24.d0
   cm(4,18)=(6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)*dgm10(1,1)+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&   +54*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(1,1)+4*(6*(6*gm(1,3)*gm(2,2)&
&   *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +2*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,3)&
&   +6*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(2,3))*dgm10(1,2)+(8*gm(2,2)*(-18*gm(1,3)&
&   *gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,2)+12*(6*gm(1,2)*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2))&
&   *dgm10(1,3)+3*(gm(1,2)**2*(72*gm(3,3)*dgm01(1,2)+24*gm(2,3)*dgm01(1,3))&
&   +gm(1,1)*(72*gm(2,3)**2*dgm01(1,2)+120*gm(2,2)*gm(3,3)*dgm01(1,2)&
&   +96*gm(2,2)*gm(2,3)*dgm01(1,3))+gm(1,3)**2*(72*gm(2,2)*dgm01(1,2)&
&   +12*gm(1,2)*dgm01(2,2))+gm(1,2)*(12*gm(2,3)**2*dgm01(1,1)+48*gm(3,3)&
&   *(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2))+24*gm(1,1)*gm(2,3)*dgm01(2,3))&
&   +gm(1,3)*(-48*gm(1,2)*gm(2,3)*dgm01(1,2)+12*gm(1,1)*gm(2,3)*dgm01(2,2)&
&   +24*gm(1,2)**2*dgm01(2,3)+gm(2,2)*(12*gm(2,3)*dgm01(1,1)+24*gm(1,2)&
&   *dgm01(1,3)+96*gm(1,1)*dgm01(2,3))))*dgm10(2,2)+2*(gm(2,3)*(72*gm(1,2)&
&   **2*dgm01(1,2)+288*gm(1,1)*gm(2,2)*dgm01(1,2)+gm(1,2)*(108*gm(2,2)&
&   *dgm01(1,1)+36*gm(1,1)*dgm01(2,2)))+gm(1,3)*(-36*gm(2,2)**2*dgm01(1,1)&
&   +36*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(72*gm(1,2)*dgm01(1,2)+144*gm(1,1)&
&   *dgm01(2,2))))*dgm10(2,3))/12.d0
   cm(5,18)=(6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)*dgm10(1,1)+4*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&   **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)*dgm10(1,1)&
&   +4*(6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(1,3)+6*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(2,2)+6*(6*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3))&
&   *dgm10(1,2)+(8*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,2)+12*(6*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2))&
&   *dgm10(1,3)+3*(60*gm(1,2)**2*gm(3,3)*dgm01(1,1)+gm(1,1)*(-24*gm(2,3)&
&   **2*dgm01(1,1)+gm(3,3)*(-12*gm(2,2)*dgm01(1,1)+192*gm(1,2)*dgm01(1,2))&
&   +192*gm(1,2)*gm(2,3)*dgm01(1,3))+gm(1,3)**2*(-24*gm(2,2)*dgm01(1,1)&
&   +48*gm(1,2)*dgm01(1,2)+24*gm(1,1)*dgm01(2,2))+gm(1,1)**2*(48*gm(3,3)&
&   *dgm01(2,2)+96*gm(2,3)*dgm01(2,3))+gm(1,3)*(24*gm(1,2)**2*dgm01(1,3)&
&   +gm(1,1)*(48*gm(2,3)*dgm01(1,2)-72*gm(2,2)*dgm01(1,3))+gm(1,2)&
&   *(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)*dgm01(2,3))))*dgm10(2,2)+2*(gm(1,2)&
&   **2*(180*gm(2,3)*dgm01(1,1)+72*gm(1,3)*dgm01(1,2))+gm(1,1)*(gm(2,2)&
&   *(-36*gm(2,3)*dgm01(1,1)-216*gm(1,3)*dgm01(1,2))+144*gm(1,1)&
&   *gm(2,3)*dgm01(2,2))+gm(1,2)*(576*gm(1,1)*gm(2,3)*dgm01(1,2)&
&   +gm(1,3)*(-72*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))))*dgm10(2,3))&
&   /24.d0
   cm(6,18)=(6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(1,1)+4*gm(2,2)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,3)*dgm10(1,1)+4*(6*(48*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,2)+2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,3)&
&   +6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(2,2)+6*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(2,3))*dgm10(1,2)+(8*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(1,2)+12*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))&
&   *dgm01(2,2))*dgm10(1,3)+3*(24*gm(1,2)**3*dgm01(1,3)+gm(1,2)*(gm(1,1)&
&   *(48*gm(2,3)*dgm01(1,2)+120*gm(2,2)*dgm01(1,3))+gm(1,3)*(96*gm(2,2)&
&   *dgm01(1,1)+24*gm(1,1)*dgm01(2,2)))+gm(1,2)**2*(12*gm(2,3)*dgm01(1,1)&
&   +48*(gm(1,3)*dgm01(1,2)+gm(1,1)*dgm01(2,3)))+gm(1,1)*(48*gm(1,1)&
&   *gm(2,3)*dgm01(2,2)+gm(2,2)*(-36*gm(2,3)*dgm01(1,1)+192*gm(1,3)&
&   *dgm01(1,2)+96*gm(1,1)*dgm01(2,3))))*dgm10(2,2)+2*(72*gm(1,2)&
&   **3*dgm01(1,2)+360*gm(1,1)*gm(1,2)*gm(2,2)*dgm01(1,2)+gm(1,2)&
&   **2*(108*gm(2,2)*dgm01(1,1)+72*gm(1,1)*dgm01(2,2))+gm(1,1)*gm(2,2)&
&   *(-36*gm(2,2)*dgm01(1,1)+144*gm(1,1)*dgm01(2,2)))*dgm10(2,3))&
&   /24.d0
   cm(7,18)=(864*gm(2,2)**2*gm(2,3)*dgm01(2,2)*dgm10(1,1)+576*gm(2,2)&
&   **3*dgm01(2,3)*dgm10(1,1)+1152*gm(2,2)*(1*gm(2,2)**2*dgm01(1,3)&
&   +4.5d0*gm(1,2)*gm(2,3)*dgm01(2,2)+gm(2,2)*(3*gm(2,3)*dgm01(1,2)&
&   -1.5d0*gm(1,3)*dgm01(2,2)+3*gm(1,2)*dgm01(2,3)))*dgm10(1,2)+4*gm(2,2)&
&   **2*(288*gm(2,2)*dgm01(1,2)+432*gm(1,2)*dgm01(2,2))*dgm10(1,3)&
&   +12*(180*gm(1,2)**2*gm(2,3)*dgm01(2,2)+gm(2,2)**2*(72*gm(2,3)&
&   *dgm01(1,1)-144*gm(1,3)*dgm01(1,2)+144*gm(1,2)*dgm01(1,3)-72*gm(1,1)&
&   *dgm01(2,3))+gm(2,2)*(-36*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(1,2)&
&   *(432*gm(2,3)*dgm01(1,2)-72*gm(1,3)*dgm01(2,2))+216*gm(1,2)**2*dgm01(2,3)))&
&   *dgm10(2,2)+8*gm(2,2)*(72*gm(2,2)**2*dgm01(1,1)+324*gm(1,2)**2*dgm01(2,2)&
&   +gm(2,2)*(432*gm(1,2)*dgm01(1,2)-108*gm(1,1)*dgm01(2,2)))*dgm10(2,3))&
&   /288.d0
   cm(8,18)=(6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+4*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+4*(6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)+6*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3))*dgm10(1,2)+(8*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,2)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2))*dgm10(1,3)+3*(12*gm(2,3)&
&   **3*dgm01(1,1)+gm(1,3)*gm(3,3)*(192*gm(2,2)*dgm01(1,2)+96*gm(1,2)&
&   *dgm01(2,2))+120*gm(1,3)**2*gm(2,2)*dgm01(2,3)+gm(2,3)**2*(48*gm(1,3)&
&   *dgm01(1,2)+24*gm(1,2)*dgm01(1,3)-48*gm(1,1)*dgm01(2,3))+gm(3,3)&
&   *(-72*gm(1,2)*gm(2,2)*dgm01(1,3)-48*gm(1,2)**2*dgm01(2,3)-24*gm(1,1)&
&   *gm(2,2)*dgm01(2,3))+gm(2,3)*(gm(2,2)*(60*gm(3,3)*dgm01(1,1)&
&   +192*gm(1,3)*dgm01(1,3))+(12*gm(1,3)**2-36*gm(1,1)*gm(3,3))*dgm01(2,2)&
&   +gm(1,2)*(48*gm(3,3)*dgm01(1,2)+144*gm(1,3)*dgm01(2,3))))*dgm10(2,2)&
&   +2*(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+12*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,2))&
&   *dgm10(2,3))/24.d0
   cm(9,18)=(24*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+576*gm(2,2)**2*gm(2,3)*dgm01(2,3)*dgm10(1,1)+16*(36*gm(1,2)&
&   *gm(2,3)**2*dgm01(2,2)+gm(2,2)**2*(144*gm(3,3)*dgm01(1,2)+72*gm(2,3)&
&   *dgm01(1,3)+144*gm(1,3)*dgm01(2,3))+gm(2,2)*(72*gm(2,3)**2*dgm01(1,2)&
&   +144*gm(1,2)*gm(3,3)*dgm01(2,2)+gm(2,3)*(36*gm(1,3)*dgm01(2,2)&
&   +72*gm(1,2)*dgm01(2,3))))*dgm10(1,2)+4*gm(2,2)*(144*gm(1,2)*gm(2,3)&
&   *dgm01(2,2)+288*gm(2,2)*(gm(2,3)*dgm01(1,2)+gm(1,3)*dgm01(2,2)))&
&   *dgm10(1,3)+12*(gm(2,2)**2*(48*gm(3,3)*dgm01(1,1)+96*gm(1,3)&
&   *dgm01(1,3))-24*gm(1,1)*gm(2,3)**2*dgm01(2,2)+gm(1,2)*gm(2,3)&
&   *(48*gm(2,3)*dgm01(1,2)+72*gm(1,3)*dgm01(2,2))+gm(1,2)**2*(60*gm(3,3)&
&   *dgm01(2,2)+24*gm(2,3)*dgm01(2,3))+gm(2,2)*(24*gm(2,3)**2*dgm01(1,1)&
&   +(-24*gm(1,3)**2-12*gm(1,1)*gm(3,3))*dgm01(2,2)+gm(2,3)*(48*gm(1,3)&
&   *dgm01(1,2)+48*gm(1,2)*dgm01(1,3)-72*gm(1,1)*dgm01(2,3))+192*gm(1,2)&
&   *(gm(3,3)*dgm01(1,2)+gm(1,3)*dgm01(2,3))))*dgm10(2,2)+8*(gm(2,2)&
&   **2*(72*gm(2,3)*dgm01(1,1)+288*gm(1,3)*dgm01(1,2))+36*gm(1,2)&
&   **2*gm(2,3)*dgm01(2,2)+gm(2,2)*(-108*gm(1,1)*gm(2,3)*dgm01(2,2)&
&   +gm(1,2)*(144*gm(2,3)*dgm01(1,2)+288*gm(1,3)*dgm01(2,2))))*dgm10(2,3))&
&   /96.d0
   cm(10,18)=(6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+4*(6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)+6*gm(3,3)&
&   *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(2,2)+6*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3))*dgm10(1,2)+(8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)&
&   *gm(3,3))*dgm01(1,2)+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2))*dgm10(1,3)+6*(gm(3,3)&
&   *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+4*gm(3,3)*(54*gm(1,3)&
&   *gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,2)+2*(-36*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+2*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(2,3))&
&   *dgm10(2,2)+2*(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&   +12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+6*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(2,2))*dgm10(2,3))/72.d0
   cm(1,19)=(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)*dgm10(1,1)+6*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(3,3)&
&   *dgm10(1,1)+4*(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2)&
&   +6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(1,3)+6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(2,3)+6*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&   *dgm01(3,3))*dgm10(1,2)+12*(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)+2*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,3)&
&   +gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+4*gm(1,1)&
&   *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,3)+gm(1,1)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(1,3)+(2*(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)+12*gm(1,1)*(54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(1,3)+216*gm(1,1)**2*gm(1,3)*dgm01(3,3))&
&   *dgm10(2,2)+6*(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)+4*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(1,2)+8*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)&
&   *gm(2,3))*dgm01(1,3)+144*gm(1,1)**2*gm(1,3)*dgm01(2,3)+72*gm(1,1)&
&   **2*gm(1,2)*dgm01(3,3))*dgm10(2,3)+3*(2*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,1)+8*gm(1,1)&
&   *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(1,2)+4*gm(1,1)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,3)+72*gm(1,1)**2*gm(1,3)&
&   *dgm01(2,2)+144*gm(1,1)**2*gm(1,2)*dgm01(2,3))*dgm10(3,3))/72.d0
   cm(2,19)=(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(1,1)+12*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   *dgm10(1,1)+6*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&   *dgm01(3,3)*dgm10(1,1)+4*(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(1,2)+12*(2*(48*gm(1,3)*gm(2,2)&
&   *gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +2*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(1,3)&
&   +(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(2,3)+gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,3)+(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(2,2)+12*((48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+2*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+4*(6*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,3)+2*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(2,3)+3*(48*gm(1,2)**2*(gm(2,3)&
&   *dgm01(1,2)+gm(2,2)*dgm01(1,3))+gm(1,1)*gm(2,2)*(192*gm(2,3)&
&   *dgm01(1,2)+96*gm(2,2)*dgm01(1,3))+gm(1,3)*(48*gm(2,2)**2*dgm01(1,1)&
&   +12*gm(1,2)**2*dgm01(2,2)+gm(2,2)*(48*gm(1,2)*dgm01(1,2)-36*gm(1,1)&
&   *dgm01(2,2)))+24*gm(1,2)**3*dgm01(2,3)+gm(1,2)*(96*gm(1,1)*gm(2,3)&
&   *dgm01(2,2)+gm(2,2)*(24*gm(2,3)*dgm01(1,1)+120*gm(1,1)*dgm01(2,3))))&
&   *dgm10(3,3))/24.d0
   cm(3,19)=(144*gm(1,3)*gm(3,3)**2*dgm01(2,2)*dgm10(1,1)+24*gm(3,3)&
&   *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(2,3)*dgm10(1,1)&
&   +12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(1,1)+8*(gm(1,3)*gm(3,3)*(72*gm(3,3)&
&   *dgm01(1,2)+72*gm(2,3)*dgm01(1,3)+36*gm(1,2)*dgm01(3,3))+gm(1,3)&
&   **2*(72*gm(3,3)*dgm01(2,3)+36*gm(2,3)*dgm01(3,3))+144*gm(3,3)&
&   *(gm(1,2)*gm(3,3)*dgm01(1,3)+gm(1,1)*(gm(3,3)*dgm01(2,3)+gm(2,3)&
&   *dgm01(3,3))))*dgm10(1,2)+24*(2*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&   *gm(3,3))*dgm01(1,2)+2*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3))*dgm01(2,2)+4*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,3)&
&   +2*(144*gm(1,3)**2*gm(3,3)*dgm01(1,3)+288*gm(1,1)*gm(3,3)**2*dgm01(1,3)&
&   +36*gm(1,3)**3*dgm01(3,3)+gm(1,3)*gm(3,3)*(72*gm(3,3)*dgm01(1,1)&
&   +180*gm(1,1)*dgm01(3,3)))*dgm10(2,2)+12*(24*gm(1,3)**3*dgm01(2,3)&
&   +gm(1,3)**2*(48*gm(3,3)*dgm01(1,2)+48*gm(2,3)*dgm01(1,3)+12*gm(1,2)&
&   *dgm01(3,3))+gm(3,3)*(gm(1,1)*(96*gm(3,3)*dgm01(1,2)+192*gm(2,3)&
&   *dgm01(1,3))+gm(1,2)*(48*gm(3,3)*dgm01(1,1)-36*gm(1,1)*dgm01(3,3)))&
&   +gm(1,3)*(gm(3,3)*(48*gm(1,2)*dgm01(1,3)+120*gm(1,1)*dgm01(2,3))&
&   +gm(2,3)*(24*gm(3,3)*dgm01(1,1)+96*gm(1,1)*dgm01(3,3))))*dgm10(2,3)&
&   +6*(-48*gm(1,2)**2*gm(3,3)*dgm01(1,3)+gm(1,1)*(192*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+120*gm(2,3)**2*dgm01(1,3)-24*gm(2,2)*gm(3,3)*dgm01(1,3))&
&   +12*gm(1,3)**3*dgm01(2,2)+gm(1,2)*gm(3,3)*(96*gm(2,3)*dgm01(1,1)&
&   -72*gm(1,1)*dgm01(2,3))+gm(1,3)**2*(48*gm(2,3)*dgm01(1,2)-48*gm(2,2)&
&   *dgm01(1,3)+24*gm(1,2)*dgm01(2,3))+gm(1,3)*(12*gm(2,3)**2*dgm01(1,1)&
&   +gm(3,3)*(-36*gm(2,2)*dgm01(1,1)+48*gm(1,2)*dgm01(1,2)+60*gm(1,1)&
&   *dgm01(2,2))+gm(2,3)*(144*gm(1,2)*dgm01(1,3)+192*gm(1,1)*dgm01(2,3))))&
&   *dgm10(3,3))/48.d0
   cm(4,19)=(2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(2,2)*dgm10(1,1)+12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(1,1)+6*(6*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(3,3)&
&   *dgm10(1,1)+4*(2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(1,2)+6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+6*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,2)+12*(2*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +2*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+4*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(2,3)+(6*gm(1,2)*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,3)+(gm(1,3)**2*(72*gm(2,3)*dgm01(1,3)+36*gm(1,2)*dgm01(3,3))&
&   +gm(1,3)*(72*gm(1,2)*gm(3,3)*dgm01(1,3)+gm(2,3)*(108*gm(3,3)&
&   *dgm01(1,1)+36*gm(1,1)*dgm01(3,3)))+gm(3,3)*(288*gm(1,1)*gm(2,3)&
&   *dgm01(1,3)+gm(1,2)*(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3))))&
&   *dgm10(2,2)+6*(72*gm(1,2)**2*gm(3,3)*dgm01(1,3)+gm(1,1)*(96*gm(2,3)&
&   *gm(3,3)*dgm01(1,2)+72*gm(2,3)**2*dgm01(1,3)+120*gm(2,2)*gm(3,3)&
&   *dgm01(1,3))+gm(1,3)**2*(24*gm(2,3)*dgm01(1,2)+72*gm(2,2)*dgm01(1,3)&
&   +24*gm(1,2)*dgm01(2,3))+gm(1,2)*(96*gm(1,1)*gm(3,3)*dgm01(2,3)&
&   +12*gm(2,3)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))+gm(1,3)&
&   *(12*gm(2,3)**2*dgm01(1,1)+gm(2,3)*(-48*gm(1,2)*dgm01(1,3)+24*gm(1,1)&
&   *dgm01(2,3))+48*gm(2,2)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3))&
&   +gm(1,2)*(24*gm(3,3)*dgm01(1,2)+12*gm(1,2)*dgm01(3,3))))*dgm10(2,3)&
&   +3*(gm(1,2)**2*(72*gm(3,3)*dgm01(1,2)+24*gm(2,3)*dgm01(1,3))&
&   +gm(1,1)*(72*gm(2,3)**2*dgm01(1,2)+120*gm(2,2)*gm(3,3)*dgm01(1,2)&
&   +96*gm(2,2)*gm(2,3)*dgm01(1,3))+gm(1,3)**2*(72*gm(2,2)*dgm01(1,2)&
&   +12*gm(1,2)*dgm01(2,2))+gm(1,2)*(12*gm(2,3)**2*dgm01(1,1)+48*gm(3,3)&
&   *(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2))+24*gm(1,1)*gm(2,3)*dgm01(2,3))&
&   +gm(1,3)*(-48*gm(1,2)*gm(2,3)*dgm01(1,2)+12*gm(1,1)*gm(2,3)*dgm01(2,2)&
&   +24*gm(1,2)**2*dgm01(2,3)+gm(2,2)*(12*gm(2,3)*dgm01(1,1)+24*gm(1,2)&
&   *dgm01(1,3)+96*gm(1,1)*dgm01(2,3))))*dgm10(3,3))/12.d0
   cm(5,19)=(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)*dgm10(1,1)+6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(1,1)&
&   +4*(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)+6*(6*gm(1,3)&
&   **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3)+6*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)&
&   +6*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,2)+12*(2*(6*gm(1,3)**2*gm(2,3)&
&   +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(2,3)&
&   +(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,3)+(72*gm(1,3)**3*dgm01(1,3)&
&   +360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3)+gm(1,3)**2*(108*gm(3,3)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(3,3))+gm(1,1)*gm(3,3)*(-36*gm(3,3)&
&   *dgm01(1,1)+144*gm(1,1)*dgm01(3,3)))*dgm10(2,2)+6*(24*gm(1,3)&
&   **3*dgm01(1,2)+gm(1,3)**2*(12*gm(2,3)*dgm01(1,1)+48*(gm(1,2)&
&   *dgm01(1,3)+gm(1,1)*dgm01(2,3)))+gm(1,3)*(gm(1,1)*(120*gm(3,3)&
&   *dgm01(1,2)+48*gm(2,3)*dgm01(1,3))+gm(1,2)*(96*gm(3,3)*dgm01(1,1)&
&   +24*gm(1,1)*dgm01(3,3)))+gm(1,1)*(gm(3,3)*(192*gm(1,2)*dgm01(1,3)&
&   +96*gm(1,1)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)*dgm01(1,1)+48*gm(1,1)&
&   *dgm01(3,3))))*dgm10(2,3)+3*(60*gm(1,2)**2*gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*(-24*gm(2,3)**2*dgm01(1,1)+gm(3,3)*(-12*gm(2,2)*dgm01(1,1)&
&   +192*gm(1,2)*dgm01(1,2))+192*gm(1,2)*gm(2,3)*dgm01(1,3))+gm(1,3)&
&   **2*(-24*gm(2,2)*dgm01(1,1)+48*gm(1,2)*dgm01(1,2)+24*gm(1,1)&
&   *dgm01(2,2))+gm(1,1)**2*(48*gm(3,3)*dgm01(2,2)+96*gm(2,3)*dgm01(2,3))&
&   +gm(1,3)*(24*gm(1,2)**2*dgm01(1,3)+gm(1,1)*(48*gm(2,3)*dgm01(1,2)&
&   -72*gm(2,2)*dgm01(1,3))+gm(1,2)*(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)&
&   *dgm01(2,3))))*dgm10(3,3))/24.d0
   cm(6,19)=(2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)*dgm10(1,1)+12*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(1,1)&
&   +6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(3,3)*dgm10(1,1)+4*(2*(90*gm(1,3)**2*gm(2,3)&
&   -36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   +6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(2,3)+6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&   *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,2)+12*(2*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(48*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,3)+(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(2,2)+4*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&   *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(1,3)+(gm(1,3)**2*(180*gm(2,3)&
&   *dgm01(1,1)+72*gm(1,2)*dgm01(1,3))+gm(1,3)*(576*gm(1,1)*gm(2,3)&
&   *dgm01(1,3)+gm(1,2)*(-72*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3)))&
&   +gm(1,1)*(-216*gm(1,2)*gm(3,3)*dgm01(1,3)+gm(2,3)*(-36*gm(3,3)&
&   *dgm01(1,1)+144*gm(1,1)*dgm01(3,3))))*dgm10(2,2)+6*(-24*gm(1,2)&
&   **2*gm(3,3)*dgm01(1,1)+gm(1,3)**2*(60*gm(2,2)*dgm01(1,1)+24*gm(1,2)&
&   *dgm01(1,2))+gm(1,3)*(48*gm(1,2)**2*dgm01(1,3)+192*gm(1,1)*(gm(2,3)&
&   *dgm01(1,2)+gm(2,2)*dgm01(1,3))+gm(1,2)*(72*gm(2,3)*dgm01(1,1)&
&   +48*gm(1,1)*dgm01(2,3)))+gm(1,1)*(-24*gm(2,3)**2*dgm01(1,1)-12*gm(2,2)&
&   *gm(3,3)*dgm01(1,1)-72*gm(1,2)*gm(3,3)*dgm01(1,2)+48*gm(1,2)&
&   *gm(2,3)*dgm01(1,3)+24*gm(1,2)**2*dgm01(3,3))+gm(1,1)**2*(96*gm(2,3)&
&   *dgm01(2,3)+48*gm(2,2)*dgm01(3,3)))*dgm10(2,3)+3*(24*gm(1,2)&
&   **3*dgm01(1,3)+gm(1,2)*(gm(1,1)*(48*gm(2,3)*dgm01(1,2)+120*gm(2,2)&
&   *dgm01(1,3))+gm(1,3)*(96*gm(2,2)*dgm01(1,1)+24*gm(1,1)*dgm01(2,2)))&
&   +gm(1,2)**2*(12*gm(2,3)*dgm01(1,1)+48*(gm(1,3)*dgm01(1,2)+gm(1,1)&
&   *dgm01(2,3)))+gm(1,1)*(48*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(2,2)&
&   *(-36*gm(2,3)*dgm01(1,1)+192*gm(1,3)*dgm01(1,2)+96*gm(1,1)*dgm01(2,3))))&
&   *dgm10(3,3))/24.d0
   cm(7,19)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+24*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+432*gm(2,2)**2*gm(2,3)*dgm01(3,3)*dgm10(1,1)+8*(2*(90*gm(2,3)&
&   **3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)+6*gm(2,2)*(54*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3))*dgm01(1,3)+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*gm(2,2)&
&   *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,2)&
&   +12*(180*gm(1,2)*gm(2,3)**2*dgm01(2,2)+gm(2,2)*(216*gm(2,3)**2*dgm01(1,2)&
&   -36*gm(1,2)*gm(3,3)*dgm01(2,2)+gm(2,3)*(-72*gm(1,3)*dgm01(2,2)&
&   +432*gm(1,2)*dgm01(2,3)))+gm(2,2)**2*(-72*gm(3,3)*dgm01(1,2)&
&   +144*gm(2,3)*dgm01(1,3)-144*gm(1,3)*dgm01(2,3)+72*gm(1,2)*dgm01(3,3)))&
&   *dgm10(1,3)+2*(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&   +12*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+6*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)&
&   *gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(3,3))*dgm10(2,2)+24*(gm(2,2)&
&   *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+2*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(1,3)+2*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)&
&   +gm(1,1)*gm(2,3)))*dgm01(2,3)+gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(2,3)+6*(180*gm(1,2)**2*gm(2,3)*dgm01(2,2)&
&   +gm(2,2)**2*(72*gm(2,3)*dgm01(1,1)-144*gm(1,3)*dgm01(1,2)+144*gm(1,2)&
&   *dgm01(1,3)-72*gm(1,1)*dgm01(2,3))+gm(2,2)*(-36*gm(1,1)*gm(2,3)&
&   *dgm01(2,2)+gm(1,2)*(432*gm(2,3)*dgm01(1,2)-72*gm(1,3)*dgm01(2,2))&
&   +216*gm(1,2)**2*dgm01(2,3)))*dgm10(3,3))/144.d0
   cm(8,19)=(144*gm(2,3)*gm(3,3)**2*dgm01(2,2)*dgm10(1,1)+24*gm(3,3)&
&   *(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(2,3)*dgm10(1,1)+12*(6*gm(2,3)&
&   **3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(1,1)+8*(72*gm(2,3)&
&   *gm(3,3)**2*dgm01(1,2)+6*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&   *dgm01(1,3)+6*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&   *dgm01(2,3)+6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,2)+24*(2*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3))*dgm01(1,2)+2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3)+gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(2,2)&
&   +4*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3)+(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,3)+2*(288*gm(1,2)&
&   *gm(3,3)*(gm(3,3)*dgm01(1,3)+gm(1,3)*dgm01(3,3))+gm(2,3)*(72*gm(3,3)&
&   **2*dgm01(1,1)+36*gm(1,3)**2*dgm01(3,3)+gm(3,3)*(144*gm(1,3)&
&   *dgm01(1,3)-108*gm(1,1)*dgm01(3,3))))*dgm10(2,2)+24*(gm(3,3)&
&   *(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,1)+2*gm(3,3)*(12*gm(1,3)&
&   *gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(6*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(2,3)+(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(2,3)+6*(12*gm(2,3)**3*dgm01(1,1)&
&   +gm(1,3)*gm(3,3)*(192*gm(2,2)*dgm01(1,2)+96*gm(1,2)*dgm01(2,2))&
&   +120*gm(1,3)**2*gm(2,2)*dgm01(2,3)+gm(2,3)**2*(48*gm(1,3)*dgm01(1,2)&
&   +24*gm(1,2)*dgm01(1,3)-48*gm(1,1)*dgm01(2,3))+gm(3,3)*(-72*gm(1,2)&
&   *gm(2,2)*dgm01(1,3)-48*gm(1,2)**2*dgm01(2,3)-24*gm(1,1)*gm(2,2)&
&   *dgm01(2,3))+gm(2,3)*(gm(2,2)*(60*gm(3,3)*dgm01(1,1)+192*gm(1,3)&
&   *dgm01(1,3))+(12*gm(1,3)**2-36*gm(1,1)*gm(3,3))*dgm01(2,2)+gm(1,2)&
&   *(48*gm(3,3)*dgm01(1,2)+144*gm(1,3)*dgm01(2,3))))*dgm10(3,3))&
&   /48.d0
   cm(9,19)=(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,1)+12*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+6*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+4*(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   +6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)+6*(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,2)+12*(2*(6*gm(2,3)**3+30*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,2)+2*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3))*dgm01(1,3)+(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)+4*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)+gm(2,2)&
&   *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,3)&
&   +(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+12*(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(2,2)+12*((6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&   +2*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)+2*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+(48*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(2,3)+3*(gm(2,2)**2*(48*gm(3,3)*dgm01(1,1)+96*gm(1,3)*dgm01(1,3))&
&   -24*gm(1,1)*gm(2,3)**2*dgm01(2,2)+gm(1,2)*gm(2,3)*(48*gm(2,3)&
&   *dgm01(1,2)+72*gm(1,3)*dgm01(2,2))+gm(1,2)**2*(60*gm(3,3)*dgm01(2,2)&
&   +24*gm(2,3)*dgm01(2,3))+gm(2,2)*(24*gm(2,3)**2*dgm01(1,1)+(-24*gm(1,3)&
&   **2-12*gm(1,1)*gm(3,3))*dgm01(2,2)+gm(2,3)*(48*gm(1,3)*dgm01(1,2)&
&   +48*gm(1,2)*dgm01(1,3)-72*gm(1,1)*dgm01(2,3))+192*gm(1,2)*(gm(3,3)&
&   *dgm01(1,2)+gm(1,3)*dgm01(2,3))))*dgm10(3,3))/24.d0
   cm(10,19)=(144*gm(3,3)**3*dgm01(2,2)*dgm10(1,1)+864*gm(2,3)&
&   *gm(3,3)**2*dgm01(2,3)*dgm10(1,1)+12*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(3,3)*dgm10(1,1)+576*gm(3,3)*(1*gm(3,3)**2*dgm01(1,2)&
&   +4.5d0*gm(1,3)*gm(2,3)*dgm01(3,3)+gm(3,3)*(3*gm(2,3)*dgm01(1,3)&
&   +3*gm(1,3)*dgm01(2,3)-1.5d0*gm(1,2)*dgm01(3,3)))*dgm10(1,2)+12*(144*gm(2,3)&
&   *gm(3,3)**2*dgm01(1,2)+4*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&   *dgm01(1,3)+72*gm(1,3)*gm(3,3)**2*dgm01(2,2)+8*gm(3,3)*(54*gm(1,3)&
&   *gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(2,3)+2*(-36*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(1,3)+2*gm(3,3)*(72*gm(3,3)**2*dgm01(1,1)+324*gm(1,3)**2*dgm01(3,3)&
&   +gm(3,3)*(432*gm(1,3)*dgm01(1,3)-108*gm(1,1)*dgm01(3,3)))*dgm10(2,2)&
&   +12*(72*gm(2,3)*gm(3,3)**2*dgm01(1,1)+144*gm(1,3)*gm(3,3)**2*dgm01(1,2)&
&   +8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   +4*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)+2*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(3,3))&
&   *dgm10(2,3)+12*(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&   +4*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(2,2)+2*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&   *gm(2,3))*gm(3,3))*dgm01(2,3))*dgm10(3,3))/144.d0
   cm(1,20)=(4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(3,3)*dgm10(1,1)+(8*(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,3)+12*gm(1,1)*(54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(3,3))*dgm10(1,2)+4*(2*(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2)+6*(-36*gm(1,1)*gm(1,3)&
&   *gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,3)&
&   +6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)+6*gm(1,1)&
&   *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(3,3))*dgm10(1,3)&
&   +2*(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)+12*gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)+216*gm(1,1)**2*gm(1,3)&
&   *dgm01(3,3))*dgm10(2,3)+3*(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)+4*gm(1,1)*(54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(1,2)+8*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&   -18*gm(1,1)*gm(2,3))*dgm01(1,3)+144*gm(1,1)**2*gm(1,3)*dgm01(2,3)&
&   +72*gm(1,1)**2*gm(1,2)*dgm01(3,3))*dgm10(3,3))/72.d0
   cm(2,20)=(4*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(1,1)+6*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)&
&   *dgm10(1,1)+(8*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,2)+4*(2*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,3)+2*(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)&
&   +6*((48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,1)+2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)+2*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))&
&   *dgm10(3,3))/24.d0
   cm(3,20)=(576*gm(1,3)*gm(3,3)**2*dgm01(2,3)*dgm10(1,1)+24*gm(3,3)&
&   *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(3,3)*dgm10(1,1)&
&   +4*gm(3,3)*(288*gm(1,3)*gm(3,3)*dgm01(1,3)+12*(12*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3))*dgm01(3,3))*dgm10(1,2)+16*(gm(1,3)*gm(3,3)*(72*gm(3,3)&
&   *dgm01(1,2)+72*gm(2,3)*dgm01(1,3)+36*gm(1,2)*dgm01(3,3))+gm(1,3)&
&   **2*(72*gm(3,3)*dgm01(2,3)+36*gm(2,3)*dgm01(3,3))+144*gm(3,3)&
&   *(gm(1,2)*gm(3,3)*dgm01(1,3)+gm(1,1)*(gm(3,3)*dgm01(2,3)+gm(2,3)&
&   *dgm01(3,3))))*dgm10(1,3)+8*(144*gm(1,3)**2*gm(3,3)*dgm01(1,3)&
&   +288*gm(1,1)*gm(3,3)**2*dgm01(1,3)+36*gm(1,3)**3*dgm01(3,3)+gm(1,3)&
&   *gm(3,3)*(72*gm(3,3)*dgm01(1,1)+180*gm(1,1)*dgm01(3,3)))*dgm10(2,3)&
&   +12*(24*gm(1,3)**3*dgm01(2,3)+gm(1,3)**2*(48*gm(3,3)*dgm01(1,2)&
&   +48*gm(2,3)*dgm01(1,3)+12*gm(1,2)*dgm01(3,3))+gm(3,3)*(gm(1,1)&
&   *(96*gm(3,3)*dgm01(1,2)+192*gm(2,3)*dgm01(1,3))+gm(1,2)*(48*gm(3,3)&
&   *dgm01(1,1)-36*gm(1,1)*dgm01(3,3)))+gm(1,3)*(gm(3,3)*(48*gm(1,2)&
&   *dgm01(1,3)+120*gm(1,1)*dgm01(2,3))+gm(2,3)*(24*gm(3,3)*dgm01(1,1)&
&   +96*gm(1,1)*dgm01(3,3))))*dgm10(3,3))/96.d0
   cm(4,20)=(4*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(2,3)*dgm10(1,1)+6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(1,1)+(8*gm(3,3)*(54*gm(1,3)&
&   *gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,3)+12*(6*gm(1,3)**2*gm(2,3)&
&   +6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3))&
&   *dgm10(1,2)+4*(2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(1,2)+6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3)+6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+6*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,3)+2*(gm(1,3)&
&   **2*(72*gm(2,3)*dgm01(1,3)+36*gm(1,2)*dgm01(3,3))+gm(1,3)*(72*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)+gm(2,3)*(108*gm(3,3)*dgm01(1,1)+36*gm(1,1)&
&   *dgm01(3,3)))+gm(3,3)*(288*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)&
&   *(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3))))*dgm10(2,3)&
&   +3*(72*gm(1,2)**2*gm(3,3)*dgm01(1,3)+gm(1,1)*(96*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+72*gm(2,3)**2*dgm01(1,3)+120*gm(2,2)*gm(3,3)*dgm01(1,3))&
&   +gm(1,3)**2*(24*gm(2,3)*dgm01(1,2)+72*gm(2,2)*dgm01(1,3)+24*gm(1,2)&
&   *dgm01(2,3))+gm(1,2)*(96*gm(1,1)*gm(3,3)*dgm01(2,3)+12*gm(2,3)&
&   *(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))+gm(1,3)*(12*gm(2,3)&
&   **2*dgm01(1,1)+gm(2,3)*(-48*gm(1,2)*dgm01(1,3)+24*gm(1,1)*dgm01(2,3))&
&   +48*gm(2,2)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3))+gm(1,2)*(24*gm(3,3)&
&   *dgm01(1,2)+12*gm(1,2)*dgm01(3,3))))*dgm10(3,3))/12.d0
   cm(5,20)=(4*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(1,1)+(8*gm(3,3)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)+12*(6*gm(1,3)&
&   **3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3))*dgm10(1,2)+4*(2*gm(3,3)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)+6*(6*gm(1,3)**2*gm(2,3)&
&   +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,3)&
&   +6*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)+6*(6*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(1,3)+2*(72*gm(1,3)**3*dgm01(1,3)+360*gm(1,1)*gm(1,3)*gm(3,3)&
&   *dgm01(1,3)+gm(1,3)**2*(108*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3))&
&   +gm(1,1)*gm(3,3)*(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3)))&
&   *dgm10(2,3)+3*(24*gm(1,3)**3*dgm01(1,2)+gm(1,3)**2*(12*gm(2,3)&
&   *dgm01(1,1)+48*(gm(1,2)*dgm01(1,3)+gm(1,1)*dgm01(2,3)))+gm(1,3)&
&   *(gm(1,1)*(120*gm(3,3)*dgm01(1,2)+48*gm(2,3)*dgm01(1,3))+gm(1,2)&
&   *(96*gm(3,3)*dgm01(1,1)+24*gm(1,1)*dgm01(3,3)))+gm(1,1)*(gm(3,3)&
&   *(192*gm(1,2)*dgm01(1,3)+96*gm(1,1)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)&
&   *dgm01(1,1)+48*gm(1,1)*dgm01(3,3))))*dgm10(3,3))/24.d0
   cm(6,20)=(4*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)*dgm10(1,1)+6*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(1,1)&
&   +(8*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(1,3)+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(3,3))*dgm10(1,2)+4*(2*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3)+6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)+6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&   *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,3)&
&   +2*(gm(1,3)**2*(180*gm(2,3)*dgm01(1,1)+72*gm(1,2)*dgm01(1,3))&
&   +gm(1,3)*(576*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)*(-72*gm(3,3)&
&   *dgm01(1,1)+72*gm(1,1)*dgm01(3,3)))+gm(1,1)*(-216*gm(1,2)*gm(3,3)&
&   *dgm01(1,3)+gm(2,3)*(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3))))&
&   *dgm10(2,3)+3*(-24*gm(1,2)**2*gm(3,3)*dgm01(1,1)+gm(1,3)**2*(60*gm(2,2)&
&   *dgm01(1,1)+24*gm(1,2)*dgm01(1,2))+gm(1,3)*(48*gm(1,2)**2*dgm01(1,3)&
&   +192*gm(1,1)*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3))+gm(1,2)&
&   *(72*gm(2,3)*dgm01(1,1)+48*gm(1,1)*dgm01(2,3)))+gm(1,1)*(-24*gm(2,3)&
&   **2*dgm01(1,1)-12*gm(2,2)*gm(3,3)*dgm01(1,1)-72*gm(1,2)*gm(3,3)&
&   *dgm01(1,2)+48*gm(1,2)*gm(2,3)*dgm01(1,3)+24*gm(1,2)**2*dgm01(3,3))&
&   +gm(1,1)**2*(96*gm(2,3)*dgm01(2,3)+48*gm(2,2)*dgm01(3,3)))*dgm10(3,3))&
&   /24.d0
   cm(7,20)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+(8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)&
&   +12*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,2)+4*(2*(90*gm(2,3)**3-54*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,2)+6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,3)+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&   +54*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,3)+2*(2*(90*gm(2,3)&
&   **3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)+12*(-36*gm(1,3)*gm(2,2)&
&   *gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +6*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&   *gm(2,3)))*dgm01(3,3))*dgm10(2,3)+6*(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,1)+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&   +54*gm(1,2)*gm(2,3))*dgm01(1,3)+2*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)&
&   *(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(2,3)+gm(2,2)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(3,3))/72.d0
   cm(8,20)=(576*gm(2,3)*gm(3,3)**2*dgm01(2,3)*dgm10(1,1)+24*gm(3,3)&
&   *(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(3,3)*dgm10(1,1)+4*gm(3,3)&
&   *(288*gm(2,3)*gm(3,3)*dgm01(1,3)+12*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&   *gm(3,3))*dgm01(3,3))*dgm10(1,2)+16*(72*gm(2,3)*gm(3,3)**2*dgm01(1,2)&
&   +6*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,3)+6*gm(3,3)&
&   *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(2,3)+6*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(1,3)+8*(288*gm(1,2)*gm(3,3)*(gm(3,3)*dgm01(1,3)+gm(1,3)&
&   *dgm01(3,3))+gm(2,3)*(72*gm(3,3)**2*dgm01(1,1)+36*gm(1,3)**2*dgm01(3,3)&
&   +gm(3,3)*(144*gm(1,3)*dgm01(1,3)-108*gm(1,1)*dgm01(3,3))))*dgm10(2,3)&
&   +24*(gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,1)+2*gm(3,3)&
&   *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(1,2)+4*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(2,3)+(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(3,3))/96.d0
   cm(9,20)=(4*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,1)+6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+(8*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)&
&   +12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,2)+4*(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,2)+6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3)+6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,3)&
&   +2*(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)+12*(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   +6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(2,3)+6*((6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&   +2*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)+2*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+(48*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(3,3))/24.d0
   cm(10,20)=(576*gm(3,3)**3*dgm01(2,3)*dgm10(1,1)+864*gm(2,3)&
&   *gm(3,3)**2*dgm01(3,3)*dgm10(1,1)+4*gm(3,3)**2*(288*gm(3,3)*dgm01(1,3)&
&   +432*gm(1,3)*dgm01(3,3))*dgm10(1,2)+1152*gm(3,3)*(1*gm(3,3)**2*dgm01(1,2)&
&   +4.5d0*gm(1,3)*gm(2,3)*dgm01(3,3)+gm(3,3)*(3*gm(2,3)*dgm01(1,3)&
&   +3*gm(1,3)*dgm01(2,3)-1.5d0*gm(1,2)*dgm01(3,3)))*dgm10(1,3)+8*gm(3,3)&
&   *(72*gm(3,3)**2*dgm01(1,1)+324*gm(1,3)**2*dgm01(3,3)+gm(3,3)&
&   *(432*gm(1,3)*dgm01(1,3)-108*gm(1,1)*dgm01(3,3)))*dgm10(2,3)&
&   +12*(72*gm(2,3)*gm(3,3)**2*dgm01(1,1)+144*gm(1,3)*gm(3,3)**2*dgm01(1,2)&
&   +8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,3)&
&   +4*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)+2*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*dgm01(3,3))&
&   *dgm10(3,3))/288.d0
   cm(1,21)=(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+(8*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,3)&
&   +12*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3))*dgm10(1,3)&
&   +(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)+12*gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)+216*gm(1,1)**2*gm(1,3)&
&   *dgm01(3,3))*dgm10(3,3))/72.d0
   cm(2,21)=(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(1,1)+(8*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3)+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(3,3))*dgm10(1,3)+(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)+12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(3,3))/24.d0
   cm(3,21)=(1440*gm(1,3)*gm(3,3)**2*dgm01(3,3)*dgm10(1,1)+20*gm(3,3)&
&   *(288*gm(1,3)*gm(3,3)*dgm01(1,3)+12*(12*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3))*dgm01(3,3))*dgm10(1,3)+20*(144*gm(1,3)**2*gm(3,3)*dgm01(1,3)&
&   +288*gm(1,1)*gm(3,3)**2*dgm01(1,3)+36*gm(1,3)**3*dgm01(3,3)+gm(1,3)&
&   *gm(3,3)*(72*gm(3,3)*dgm01(1,1)+180*gm(1,1)*dgm01(3,3)))*dgm10(3,3))&
&   /480.d0
   cm(4,21)=(2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(3,3)*dgm10(1,1)+(8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(1,3)+12*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(1,3)+(gm(1,3)&
&   **2*(72*gm(2,3)*dgm01(1,3)+36*gm(1,2)*dgm01(3,3))+gm(1,3)*(72*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)+gm(2,3)*(108*gm(3,3)*dgm01(1,1)+36*gm(1,1)&
&   *dgm01(3,3)))+gm(3,3)*(288*gm(1,1)*gm(2,3)*dgm01(1,3)+gm(1,2)&
&   *(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3))))*dgm10(3,3))&
&   /12.d0
   cm(5,21)=(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+(8*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)&
&   +12*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3))*dgm10(1,3)&
&   +(72*gm(1,3)**3*dgm01(1,3)+360*gm(1,1)*gm(1,3)*gm(3,3)*dgm01(1,3)&
&   +gm(1,3)**2*(108*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3))+gm(1,1)&
&   *gm(3,3)*(-36*gm(3,3)*dgm01(1,1)+144*gm(1,1)*dgm01(3,3)))*dgm10(3,3))&
&   /24.d0
   cm(6,21)=(2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(1,1)+(8*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3)+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(3,3))*dgm10(1,3)+(gm(1,3)**2*(180*gm(2,3)&
&   *dgm01(1,1)+72*gm(1,2)*dgm01(1,3))+gm(1,3)*(576*gm(1,1)*gm(2,3)&
&   *dgm01(1,3)+gm(1,2)*(-72*gm(3,3)*dgm01(1,1)+72*gm(1,1)*dgm01(3,3)))&
&   +gm(1,1)*(-216*gm(1,2)*gm(3,3)*dgm01(1,3)+gm(2,3)*(-36*gm(3,3)&
&   *dgm01(1,1)+144*gm(1,1)*dgm01(3,3))))*dgm10(3,3))/24.d0
   cm(7,21)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+(8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)&
&   +12*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,3)+(2*(90*gm(2,3)**3-54*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,1)+12*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(90*gm(1,2)&
&   **2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(3,3))&
&   *dgm10(3,3))/72.d0
   cm(8,21)=(1440*gm(2,3)*gm(3,3)**2*dgm01(3,3)*dgm10(1,1)+20*gm(3,3)&
&   *(288*gm(2,3)*gm(3,3)*dgm01(1,3)+12*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&   *gm(3,3))*dgm01(3,3))*dgm10(1,3)+20*(288*gm(1,2)*gm(3,3)*(gm(3,3)&
&   *dgm01(1,3)+gm(1,3)*dgm01(3,3))+gm(2,3)*(72*gm(3,3)**2*dgm01(1,1)&
&   +36*gm(1,3)**2*dgm01(3,3)+gm(3,3)*(144*gm(1,3)*dgm01(1,3)-108*gm(1,1)&
&   *dgm01(3,3))))*dgm10(3,3))/480.d0
   cm(9,21)=(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,1)+(8*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)&
&   +12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,3)+(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,1)+12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)+6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(3,3))/24.d0
   cm(10,21)=(gm(3,3)*(1440*gm(3,3)**2*dgm01(3,3)*dgm10(1,1)+20*gm(3,3)&
&   *(288*gm(3,3)*dgm01(1,3)+432*gm(1,3)*dgm01(3,3))*dgm10(1,3)+20*(72*gm(3,3)&
&   **2*dgm01(1,1)+324*gm(1,3)**2*dgm01(3,3)+gm(3,3)*(432*gm(1,3)&
&   *dgm01(1,3)-108*gm(1,1)*dgm01(3,3)))*dgm10(3,3)))/1440.d0
   cm(1,22)=(2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2)&
&   *dgm10(1,2)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,2)&
&   *dgm10(2,2)+3*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2)&
&   *dgm10(2,2))/36.d0
   cm(2,22)=(gm(2,2)*(8640*gm(1,2)*gm(2,2)*dgm01(2,2)*dgm10(1,2)&
&   +8640*gm(1,2)*gm(2,2)*dgm01(1,2)*dgm10(2,2)+360*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm01(2,2)*dgm10(2,2)))/1440.d0
   cm(3,22)=(2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(1,2)+2*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)*dgm10(2,2)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)*dgm10(2,2))/12.d0
   cm(4,22)=(2*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(2,2)*dgm10(1,2)+2*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&   *gm(2,3))*dgm01(1,2)*dgm10(2,2)+3*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)&
&   *dgm10(2,2))/6.d0
   cm(5,22)=(2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(1,2)+2*(-36*gm(1,2)&
&   *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,2)*dgm10(2,2)+3*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2))/12.d0
   cm(6,22)=(2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2)&
&   *dgm10(1,2)+2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,2)&
&   *dgm10(2,2)+3*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2)&
&   *dgm10(2,2))/12.d0
   cm(7,22)=gm(2,2)**2*(3*gm(1,2)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
   cm(8,22)=(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,2)+2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   *dgm10(2,2)+3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2))/12.d0
   cm(9,22)=(gm(2,2)*(4320*gm(1,2)*gm(2,3)*dgm01(2,2)*dgm10(2,2)&
&   +8640*gm(2,2)*(gm(1,3)*dgm01(2,2)*dgm10(2,2)+gm(2,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))))/1440.d0
   cm(10,22)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&   *dgm10(1,2)+2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&   *dgm10(2,2)+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2))/36.d0
   cm(1,23)=(2*(6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)+12*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(2,3)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(1,2)+(12*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(2,2)&
&   +8*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(1,3)&
&   +3*(4*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(1,2)+4*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)&
&   *gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,3))*dgm10(2,2)+6*gm(1,1)&
&   *((54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,3)+(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(2,2)+4*(6*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,2)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,3))*dgm10(2,3)+24*gm(1,1)&
&   *((54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,2)+(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(2,3)+4*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,2)*dgm10(3,3)+6*gm(1,1)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(3,3))/72.d0
   cm(2,23)=(8*(6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)+12*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&   +12*gm(1,2)*gm(2,3))*dgm01(2,3)+72*gm(1,2)*gm(2,2)**2*dgm01(3,3))&
&   *dgm10(1,2)+4*gm(2,2)*(12*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&   *dgm01(2,2)+288*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(1,3)+12*(4*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +4*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(1,3))&
&   *dgm10(2,2)+24*((-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)+gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(2,2)+16*gm(2,2)*(144*gm(1,3)*gm(2,2)&
&   *dgm01(1,2)+72*gm(1,2)*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3)))&
&   *dgm10(2,3)+16*(6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)+6*gm(2,2)*(12*gm(1,2)&
&   **2+24*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(2,3)+576*gm(1,2)*gm(2,2)&
&   **2*dgm01(1,2)*dgm10(3,3)+24*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm01(2,2)*dgm10(3,3))/96.d0
   cm(3,23)=(2*(gm(1,3)*(72*gm(2,3)**2*dgm01(2,3)-216*gm(2,2)*gm(3,3)&
&   *dgm01(2,3)+gm(2,3)*(72*gm(3,3)*dgm01(2,2)-72*gm(2,2)*dgm01(3,3)))&
&   +gm(1,2)*(144*gm(3,3)**2*dgm01(2,2)+180*gm(2,3)**2*dgm01(3,3)&
&   +gm(3,3)*(576*gm(2,3)*dgm01(2,3)-36*gm(2,2)*dgm01(3,3))))*dgm10(1,2)&
&   +(gm(1,3)*(72*gm(2,3)**2*dgm01(2,2)-216*gm(2,2)*gm(3,3)*dgm01(2,2)&
&   -288*gm(2,2)*gm(2,3)*dgm01(2,3))+gm(1,2)*(576*gm(2,3)*gm(3,3)&
&   *dgm01(2,2)+720*gm(2,3)**2*dgm01(2,3)-144*gm(2,2)*gm(3,3)*dgm01(2,3)))&
&   *dgm10(1,3)+3*(gm(1,2)*gm(3,3)*(96*gm(3,3)*dgm01(1,2)+192*gm(2,3)&
&   *dgm01(1,3))+gm(1,3)*(48*gm(2,3)*gm(3,3)*dgm01(1,2)+24*gm(2,3)&
&   **2*dgm01(1,3)-72*gm(2,2)*gm(3,3)*dgm01(1,3)))*dgm10(2,2)+6*(gm(3,3)&
&   *(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,3)**2*gm(2,3)&
&   +6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   +(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(2,2)+4*(gm(1,3)*(36*gm(2,3)**2*dgm01(1,2)-108*gm(2,2)&
&   *gm(3,3)*dgm01(1,2)-72*gm(2,2)*gm(2,3)*dgm01(1,3))+gm(1,2)*(288*gm(2,3)&
&   *gm(3,3)*dgm01(1,2)+180*gm(2,3)**2*dgm01(1,3)-36*gm(2,2)*gm(3,3)&
&   *dgm01(1,3)))*dgm10(2,3)+4*(6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3))*dgm10(2,3)&
&   +4*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)*dgm10(3,3)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)*dgm10(3,3))/24.d0
   cm(4,23)=(2*(gm(1,3)*(36*gm(2,3)**2*dgm01(2,2)+72*gm(2,2)*gm(2,3)&
&   *dgm01(2,3)+gm(2,2)*(144*gm(3,3)*dgm01(2,2)-36*gm(2,2)*dgm01(3,3)))&
&   +gm(1,2)*(72*gm(2,3)**2*dgm01(2,3)+288*gm(2,2)*gm(3,3)*dgm01(2,3)&
&   +gm(2,3)*(36*gm(3,3)*dgm01(2,2)+108*gm(2,2)*dgm01(3,3))))*dgm10(1,2)&
&   +(gm(1,3)*gm(2,2)*(72*gm(2,3)*dgm01(2,2)-144*gm(2,2)*dgm01(2,3))&
&   +gm(1,2)*(72*gm(2,3)**2*dgm01(2,2)+288*gm(2,2)*gm(3,3)*dgm01(2,2)&
&   +432*gm(2,2)*gm(2,3)*dgm01(2,3)))*dgm10(1,3)+3*(gm(1,3)*(24*gm(2,3)&
&   **2*dgm01(1,2)+96*gm(2,2)*gm(3,3)*dgm01(1,2)+24*gm(2,2)*gm(2,3)&
&   *dgm01(1,3))+gm(1,2)*(24*gm(2,3)*gm(3,3)*dgm01(1,2)+24*gm(2,3)&
&   **2*dgm01(1,3)+96*gm(2,2)*gm(3,3)*dgm01(1,3)))*dgm10(2,2)+6*((6*gm(1,3)&
&   **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,2)+4*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)&
&   +9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(2,2)+4*(gm(1,3)*gm(2,2)*(36*gm(2,3)&
&   *dgm01(1,2)-36*gm(2,2)*dgm01(1,3))+gm(1,2)*(36*gm(2,3)**2*dgm01(1,2)&
&   +144*gm(2,2)*gm(3,3)*dgm01(1,2)+108*gm(2,2)*gm(2,3)*dgm01(1,3)))&
&   *dgm10(2,3)+4*(6*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)&
&   +9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3))*dgm10(2,3)+4*gm(2,2)&
&   *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,2)*dgm10(3,3)&
&   +6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(3,3))/12.d0
   cm(5,23)=(2*(6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,2)+(12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+8*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3))*dgm10(1,3)+3*(4*(6*gm(1,3)&
&   **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,2)+4*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3))*dgm10(2,2)+6*((6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&   *dgm01(2,2)+4*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3)))*dgm01(2,3)+(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,2)+4*(6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,2)&
&   *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,3))*dgm10(2,3)+4*(6*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(2,2)+6*(6*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3))&
&   *dgm10(2,3)+4*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,2)*dgm10(3,3)+6*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm01(2,2)*dgm10(3,3))/24.d0
   cm(6,23)=(2*(6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+12*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)+2*gm(2,2)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(1,2)+(576*gm(1,2)*gm(1,3)&
&   *gm(2,2)*dgm01(2,2)+gm(1,1)*gm(2,2)*(-216*gm(2,3)*dgm01(2,2)&
&   -144*gm(2,2)*dgm01(2,3))+gm(1,2)**2*(72*gm(2,3)*dgm01(2,2)+432*gm(2,2)&
&   *dgm01(2,3)))*dgm10(1,3)+3*(4*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+4*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3))*dgm10(2,2)&
&   +6*((48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(2,2)+4*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&   *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(2,2)+4*(288*gm(1,2)*gm(1,3)&
&   *gm(2,2)*dgm01(1,2)+gm(1,1)*gm(2,2)*(-108*gm(2,3)*dgm01(1,2)&
&   -36*gm(2,2)*dgm01(1,3))+gm(1,2)**2*(36*gm(2,3)*dgm01(1,2)+108*gm(2,2)&
&   *dgm01(1,3)))*dgm10(2,3)+4*(6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&   *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)+6*(6*gm(1,2)&
&   **3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(2,3)+4*gm(2,2)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,2)*dgm10(3,3)+6*(6*gm(1,2)&
&   **3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2)*dgm10(3,3))/24.d0
   cm(7,23)=(8*gm(2,2)*(6*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   +432*gm(2,2)*gm(2,3)*dgm01(2,3)+72*gm(2,2)**2*dgm01(3,3))*dgm10(1,2)&
&   +4*gm(2,2)**2*(432*gm(2,3)*dgm01(2,2)+288*gm(2,2)*dgm01(2,3))&
&   *dgm10(1,3)+12*gm(2,2)*(4*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&   *dgm01(1,2)+144*gm(2,2)*gm(2,3)*dgm01(1,3))*dgm10(2,2)+12*(2*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(2,3)+72*gm(1,2)*gm(2,2)**2*dgm01(3,3))*dgm10(2,2)+16*gm(2,2)&
&   **2*(216*gm(2,3)*dgm01(1,2)+72*gm(2,2)*dgm01(1,3))*dgm10(2,3)&
&   +16*gm(2,2)*(6*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(2,2)&
&   +216*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(2,3)+576*gm(2,2)**3*dgm01(1,2)&
&   *dgm10(3,3)+864*gm(1,2)*gm(2,2)**2*dgm01(2,2)*dgm10(3,3))/288.d0
   cm(8,23)=(2*(72*gm(2,3)**3*dgm01(2,3)+360*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm01(2,3)+gm(2,2)*gm(3,3)*(144*gm(3,3)*dgm01(2,2)-36*gm(2,2)&
&   *dgm01(3,3))+gm(2,3)**2*(72*gm(3,3)*dgm01(2,2)+108*gm(2,2)*dgm01(3,3)))&
&   *dgm10(1,2)+(72*gm(2,3)**3*dgm01(2,2)+360*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm01(2,2)+432*gm(2,2)*gm(2,3)**2*dgm01(2,3)-144*gm(2,2)**2*gm(3,3)&
&   *dgm01(2,3))*dgm10(1,3)+3*(48*gm(2,3)**2*gm(3,3)*dgm01(1,2)+96*gm(2,2)&
&   *gm(3,3)**2*dgm01(1,2)+24*gm(2,3)**3*dgm01(1,3)+120*gm(2,2)*gm(2,3)&
&   *gm(3,3)*dgm01(1,3))*dgm10(2,2)+6*(gm(3,3)*(12*gm(1,3)*gm(2,3)&
&   +24*gm(1,2)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)+(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(2,2)+4*(36*gm(2,3)**3*dgm01(1,2)+180*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+108*gm(2,2)*gm(2,3)**2*dgm01(1,3)-36*gm(2,2)**2*gm(3,3)&
&   *dgm01(1,3))*dgm10(2,3)+4*(6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)+6*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3))&
&   *dgm10(2,3)+4*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&   *dgm10(3,3)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(3,3))/24.d0
   cm(9,23)=(8*(36*gm(2,3)**3*dgm01(2,2)+144*gm(2,2)*gm(2,3)**2*dgm01(2,3)&
&   +288*gm(2,2)**2*gm(3,3)*dgm01(2,3)+gm(2,2)*gm(2,3)*(180*gm(3,3)&
&   *dgm01(2,2)+72*gm(2,2)*dgm01(3,3)))*dgm10(1,2)+4*gm(2,2)*(144*gm(2,3)&
&   **2*dgm01(2,2)+288*gm(2,2)*gm(3,3)*dgm01(2,2)+288*gm(2,2)*gm(2,3)&
&   *dgm01(2,3))*dgm10(1,3)+12*(24*gm(2,3)**3*dgm01(1,2)+120*gm(2,2)&
&   *gm(2,3)*gm(3,3)*dgm01(1,2)+48*gm(2,2)*gm(2,3)**2*dgm01(1,3)&
&   +96*gm(2,2)**2*gm(3,3)*dgm01(1,3))*dgm10(2,2)+24*((48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3)+gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&   *gm(2,3))*dgm01(3,3))*dgm10(2,2)+16*gm(2,2)*(72*gm(2,3)**2*dgm01(1,2)&
&   +144*gm(2,2)*gm(3,3)*dgm01(1,2)+72*gm(2,2)*gm(2,3)*dgm01(1,3))&
&   *dgm10(2,3)+16*(6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)+6*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&   +12*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(2,3)+576*gm(2,2)**2*gm(2,3)&
&   *dgm01(1,2)*dgm10(3,3)+24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&   *gm(2,3))*dgm01(2,2)*dgm10(3,3))/96.d0
   cm(10,23)=(2*(648*gm(2,3)**2*gm(3,3)*dgm01(2,3)-216*gm(2,2)&
&   *gm(3,3)**2*dgm01(2,3)+180*gm(2,3)**3*dgm01(3,3)+gm(2,3)*gm(3,3)&
&   *(216*gm(3,3)*dgm01(2,2)-108*gm(2,2)*dgm01(3,3)))*dgm10(1,2)&
&   +(648*gm(2,3)**2*gm(3,3)*dgm01(2,2)-216*gm(2,2)*gm(3,3)**2*dgm01(2,2)&
&   +720*gm(2,3)**3*dgm01(2,3)-432*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(2,3))&
&   *dgm10(1,3)+3*gm(3,3)*(144*gm(2,3)*gm(3,3)*dgm01(1,2)+216*gm(2,3)&
&   **2*dgm01(1,3)-72*gm(2,2)*gm(3,3)*dgm01(1,3))*dgm10(2,2)+3*(72*gm(1,3)&
&   *gm(3,3)**2*dgm01(2,2)+8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(2,3)+2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,2)+4*(324*gm(2,3)&
&   **2*gm(3,3)*dgm01(1,2)-108*gm(2,2)*gm(3,3)**2*dgm01(1,2)+180*gm(2,3)&
&   **3*dgm01(1,3)-108*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(1,3))*dgm10(2,3)&
&   +4*(6*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(2,2)&
&   +6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3))*dgm10(2,3)+4*(90*gm(2,3)**3-54*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,2)*dgm10(3,3)+6*(-36*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   *dgm10(3,3))/72.d0
   cm(1,24)=((12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&   +2*gm(1,2)*gm(2,3)))*dgm01(2,2)+8*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(2,3))*dgm10(1,2)+4*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(2,2)*dgm10(1,3)+2*(6*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,2)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,3))*dgm10(2,2)+12*gm(1,1)&
&   *((54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,2)+(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(2,2)+8*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,2)*dgm10(2,3)+12*gm(1,1)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(2,3))/72.d0
   cm(2,24)=(10*gm(2,2)*(12*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&   *dgm01(2,2)+288*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(1,2)+1440*gm(1,2)&
&   *gm(2,2)**2*dgm01(2,2)*dgm10(1,3)+20*gm(2,2)*(144*gm(1,3)*gm(2,2)&
&   *dgm01(1,2)+72*gm(1,2)*(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3)))&
&   *dgm10(2,2)+20*(6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)+6*gm(2,2)*(12*gm(1,2)&
&   **2+24*gm(1,1)*gm(2,2))*dgm01(2,3))*dgm10(2,2)+2880*gm(1,2)*gm(2,2)&
&   **2*dgm01(1,2)*dgm10(2,3)+120*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm01(2,2)*dgm10(2,3))/240.d0
   cm(3,24)=((gm(1,3)*(72*gm(2,3)**2*dgm01(2,2)-216*gm(2,2)*gm(3,3)&
&   *dgm01(2,2)-288*gm(2,2)*gm(2,3)*dgm01(2,3))+gm(1,2)*(576*gm(2,3)&
&   *gm(3,3)*dgm01(2,2)+720*gm(2,3)**2*dgm01(2,3)-144*gm(2,2)*gm(3,3)&
&   *dgm01(2,3)))*dgm10(1,2)+4*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(1,3)+2*(gm(1,3)&
&   *(36*gm(2,3)**2*dgm01(1,2)-108*gm(2,2)*gm(3,3)*dgm01(1,2)-72*gm(2,2)&
&   *gm(2,3)*dgm01(1,3))+gm(1,2)*(288*gm(2,3)*gm(3,3)*dgm01(1,2)&
&   +180*gm(2,3)**2*dgm01(1,3)-36*gm(2,2)*gm(3,3)*dgm01(1,3)))*dgm10(2,2)&
&   +2*(6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(2,2)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3))*dgm10(2,2)+8*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(2,3)&
&   +12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   *dgm10(2,3))/24.d0
   cm(4,24)=((gm(1,3)*gm(2,2)*(72*gm(2,3)*dgm01(2,2)-144*gm(2,2)&
&   *dgm01(2,3))+gm(1,2)*(72*gm(2,3)**2*dgm01(2,2)+288*gm(2,2)*gm(3,3)&
&   *dgm01(2,2)+432*gm(2,2)*gm(2,3)*dgm01(2,3)))*dgm10(1,2)+4*gm(2,2)&
&   *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(1,3)&
&   +2*(gm(1,3)*gm(2,2)*(36*gm(2,3)*dgm01(1,2)-36*gm(2,2)*dgm01(1,3))&
&   +gm(1,2)*(36*gm(2,3)**2*dgm01(1,2)+144*gm(2,2)*gm(3,3)*dgm01(1,2)&
&   +108*gm(2,2)*gm(2,3)*dgm01(1,3)))*dgm10(2,2)+2*(6*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(2,2)+6*(6*gm(1,2)*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3))&
&   *dgm10(2,2)+8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(1,2)*dgm10(2,3)+12*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(2,3))&
&   /12.d0
   cm(5,24)=((12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+8*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3))*dgm10(1,2)+4*(-36*gm(1,2)&
&   *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(2,2)*dgm10(1,3)+2*(6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&   **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3))*dgm10(2,2)&
&   +2*(6*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3)))*dgm01(2,2)+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(2,2)+8*(-36*gm(1,2)&
&   *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(1,2)*dgm10(2,3)+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,3))/24.d0
   cm(6,24)=((576*gm(1,2)*gm(1,3)*gm(2,2)*dgm01(2,2)+gm(1,1)*gm(2,2)&
&   *(-216*gm(2,3)*dgm01(2,2)-144*gm(2,2)*dgm01(2,3))+gm(1,2)**2*(72*gm(2,3)&
&   *dgm01(2,2)+432*gm(2,2)*dgm01(2,3)))*dgm10(1,2)+4*gm(2,2)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(1,3)+2*(288*gm(1,2)&
&   *gm(1,3)*gm(2,2)*dgm01(1,2)+gm(1,1)*gm(2,2)*(-108*gm(2,3)*dgm01(1,2)&
&   -36*gm(2,2)*dgm01(1,3))+gm(1,2)**2*(36*gm(2,3)*dgm01(1,2)+108*gm(2,2)&
&   *dgm01(1,3)))*dgm10(2,2)+2*(6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&   *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)+6*(6*gm(1,2)&
&   **3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(2,2)+8*gm(2,2)&
&   *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,2)*dgm10(2,3)+12*(6*gm(1,2)&
&   **3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2)*dgm10(2,3))/24.d0
   cm(7,24)=(gm(2,2)*(6480*gm(1,2)*gm(2,3)*dgm01(2,2)*dgm10(2,2)&
&   +gm(2,2)**2*(2880*dgm01(2,3)*dgm10(1,2)+1440*dgm01(2,2)*dgm10(1,3)&
&   +1440*dgm01(1,3)*dgm10(2,2)+2880*dgm01(1,2)*dgm10(2,3))+gm(2,2)&
&   *(-2160*gm(1,3)*dgm01(2,2)*dgm10(2,2)+4320*gm(2,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4320*gm(1,2)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)))))/720.d0
   cm(8,24)=((72*gm(2,3)**3*dgm01(2,2)+360*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm01(2,2)+432*gm(2,2)*gm(2,3)**2*dgm01(2,3)-144*gm(2,2)**2*gm(3,3)&
&   *dgm01(2,3))*dgm10(1,2)+4*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&   *dgm01(2,2)*dgm10(1,3)+2*(36*gm(2,3)**3*dgm01(1,2)+180*gm(2,2)&
&   *gm(2,3)*gm(3,3)*dgm01(1,2)+108*gm(2,2)*gm(2,3)**2*dgm01(1,3)&
&   -36*gm(2,2)**2*gm(3,3)*dgm01(1,3))*dgm10(2,2)+2*(6*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3))*dgm10(2,2)+8*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,2)*dgm10(2,3)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,3))&
&   /24.d0
   cm(9,24)=(10*gm(2,2)*(144*gm(2,3)**2*dgm01(2,2)+288*gm(2,2)&
&   *gm(3,3)*dgm01(2,2)+288*gm(2,2)*gm(2,3)*dgm01(2,3))*dgm10(1,2)&
&   +1440*gm(2,2)**2*gm(2,3)*dgm01(2,2)*dgm10(1,3)+20*gm(2,2)*(72*gm(2,3)&
&   **2*dgm01(1,2)+144*gm(2,2)*gm(3,3)*dgm01(1,2)+72*gm(2,2)*gm(2,3)&
&   *dgm01(1,3))*dgm10(2,2)+20*(6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)+6*gm(2,2)*(24*gm(1,3)&
&   *gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(2,2)+2880*gm(2,2)&
&   **2*gm(2,3)*dgm01(1,2)*dgm10(2,3)+120*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&   +12*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,3))/240.d0
   cm(10,24)=((648*gm(2,3)**2*gm(3,3)*dgm01(2,2)-216*gm(2,2)*gm(3,3)&
&   **2*dgm01(2,2)+720*gm(2,3)**3*dgm01(2,3)-432*gm(2,2)*gm(2,3)&
&   *gm(3,3)*dgm01(2,3))*dgm10(1,2)+4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)&
&   *gm(3,3))*dgm01(2,2)*dgm10(1,3)+2*(324*gm(2,3)**2*gm(3,3)*dgm01(1,2)&
&   -108*gm(2,2)*gm(3,3)**2*dgm01(1,2)+180*gm(2,3)**3*dgm01(1,3)&
&   -108*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(1,3))*dgm10(2,2)+2*(6*gm(3,3)&
&   *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(2,2)+6*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3))*dgm10(2,2)+8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))&
&   *dgm01(1,2)*dgm10(2,3)+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,3))/72.d0
   cm(1,25)=((2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)&
&   +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(2,3)+6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)&
&   *gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(3,3))*dgm10(1,2)+(6*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(2,2)+12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&   +2*gm(1,2)*gm(2,3)))*dgm01(2,3)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(3,3))*dgm10(1,3)+(2*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(1,2)+6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,3))*dgm10(2,2)+6*gm(1,1)&
&   *((54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)+(54*gm(1,2)*gm(1,3)&
&   -18*gm(1,1)*gm(2,3))*dgm01(3,3))*dgm10(2,2)+3*(4*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(1,2)+4*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&   +2*gm(1,2)*gm(2,3)))*dgm01(1,3))*dgm10(2,3)+6*gm(1,1)*((54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(54*gm(1,2)*gm(1,3)-18*gm(1,1)&
&   *gm(2,3))*dgm01(2,3)+(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(2,3)+(6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&   +2*gm(1,2)*gm(2,3)))*dgm01(1,2)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(1,3))*dgm10(3,3)+6*gm(1,1)*((54*gm(1,2)*gm(1,3)&
&   -18*gm(1,1)*gm(2,3))*dgm01(2,2)+(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&   *dgm01(2,3))*dgm10(3,3))/36.d0
   cm(2,25)=((2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*gm(2,2)&
&   *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,2)&
&   +(6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)+12*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&   *gm(2,3))*dgm01(2,3)+72*gm(1,2)*gm(2,2)**2*dgm01(3,3))*dgm10(1,3)&
&   +(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(1,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3))*dgm10(2,2)+(6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*(6*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(2,2)+3*(4*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+4*gm(2,2)*(24*gm(1,3)&
&   *gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(1,3))*dgm10(2,3)+6*((-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,2)+4*(6*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(2,3)+gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(2,3)+gm(2,2)*(144*gm(1,3)*gm(2,2)*dgm01(1,2)+72*gm(1,2)&
&   *(gm(2,3)*dgm01(1,2)+gm(2,2)*dgm01(1,3)))*dgm10(3,3)+(6*(6*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(2,2)+6*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*dgm01(2,3))&
&   *dgm10(3,3))/12.d0
   cm(3,25)=((288*gm(1,2)*gm(3,3)*(gm(3,3)*dgm01(2,3)+gm(2,3)*dgm01(3,3))&
&   +gm(1,3)*(72*gm(3,3)**2*dgm01(2,2)+36*gm(2,3)**2*dgm01(3,3)+gm(3,3)&
&   *(144*gm(2,3)*dgm01(2,3)-108*gm(2,2)*dgm01(3,3))))*dgm10(1,2)&
&   +(gm(1,3)*(72*gm(2,3)**2*dgm01(2,3)-216*gm(2,2)*gm(3,3)*dgm01(2,3)&
&   +gm(2,3)*(72*gm(3,3)*dgm01(2,2)-72*gm(2,2)*dgm01(3,3)))+gm(1,2)&
&   *(144*gm(3,3)**2*dgm01(2,2)+180*gm(2,3)**2*dgm01(3,3)+gm(3,3)&
&   *(576*gm(2,3)*dgm01(2,3)-36*gm(2,2)*dgm01(3,3))))*dgm10(1,3)&
&   +gm(3,3)*(144*gm(1,2)*gm(3,3)*dgm01(1,3)+72*gm(1,3)*(gm(3,3)&
&   *dgm01(1,2)+gm(2,3)*dgm01(1,3)))*dgm10(2,2)+(6*gm(3,3)*(12*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3))*dgm01(2,3)+6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(2,2)&
&   +3*(gm(1,2)*gm(3,3)*(96*gm(3,3)*dgm01(1,2)+192*gm(2,3)*dgm01(1,3))&
&   +gm(1,3)*(48*gm(2,3)*gm(3,3)*dgm01(1,2)+24*gm(2,3)**2*dgm01(1,3)&
&   -72*gm(2,2)*gm(3,3)*dgm01(1,3)))*dgm10(2,3)+6*(gm(3,3)*(12*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,3)&
&   +(gm(1,3)*(36*gm(2,3)**2*dgm01(1,2)-108*gm(2,2)*gm(3,3)*dgm01(1,2)&
&   -72*gm(2,2)*gm(2,3)*dgm01(1,3))+gm(1,2)*(288*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+180*gm(2,3)**2*dgm01(1,3)-36*gm(2,2)*gm(3,3)*dgm01(1,3)))&
&   *dgm10(3,3)+(6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&   +24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,3))*dgm10(3,3))/12.d0
   cm(4,25)=((gm(1,3)*(72*gm(2,3)**2*dgm01(2,3)+288*gm(2,2)*gm(3,3)&
&   *dgm01(2,3)+gm(2,3)*(108*gm(3,3)*dgm01(2,2)+36*gm(2,2)*dgm01(3,3)))&
&   +gm(1,2)*(-36*gm(3,3)**2*dgm01(2,2)+36*gm(2,3)**2*dgm01(3,3)&
&   +gm(3,3)*(72*gm(2,3)*dgm01(2,3)+144*gm(2,2)*dgm01(3,3))))*dgm10(1,2)&
&   +(gm(1,3)*(36*gm(2,3)**2*dgm01(2,2)+72*gm(2,2)*gm(2,3)*dgm01(2,3)&
&   +gm(2,2)*(144*gm(3,3)*dgm01(2,2)-36*gm(2,2)*dgm01(3,3)))+gm(1,2)&
&   *(72*gm(2,3)**2*dgm01(2,3)+288*gm(2,2)*gm(3,3)*dgm01(2,3)+gm(2,3)&
&   *(36*gm(3,3)*dgm01(2,2)+108*gm(2,2)*dgm01(3,3))))*dgm10(1,3)&
&   +(2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3))*dgm10(2,2)+(6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+6*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,2)+3*(gm(1,3)&
&   *(24*gm(2,3)**2*dgm01(1,2)+96*gm(2,2)*gm(3,3)*dgm01(1,2)+24*gm(2,2)&
&   *gm(2,3)*dgm01(1,3))+gm(1,2)*(24*gm(2,3)*gm(3,3)*dgm01(1,2)+24*gm(2,3)&
&   **2*dgm01(1,3)+96*gm(2,2)*gm(3,3)*dgm01(1,3)))*dgm10(2,3)+6*((6*gm(1,3)&
&   **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,2)+4*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)&
&   +9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)+(gm(1,3)*gm(2,2)*(36*gm(2,3)&
&   *dgm01(1,2)-36*gm(2,2)*dgm01(1,3))+gm(1,2)*(36*gm(2,3)**2*dgm01(1,2)&
&   +144*gm(2,2)*gm(3,3)*dgm01(1,2)+108*gm(2,2)*gm(2,3)*dgm01(1,3)))&
&   *dgm10(3,3)+(6*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)&
&   +9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3))*dgm10(3,3))/6.d0
   cm(5,25)=((2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)&
&   +12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(2,3)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,2)+(6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+12*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+2*(-36*gm(1,2)&
&   *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(1,3)+(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(1,2)+6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)&
&   *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,3))*dgm10(2,2)+(6*(6*gm(1,3)&
&   **3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)+6*(6*gm(1,1)*gm(1,3)&
&   *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(2,2)+3*(4*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)+4*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,3))*dgm10(2,3)+6*((6*gm(1,3)&
&   **3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,1)*gm(1,3)&
&   *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(2,3)&
&   +(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)+(6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(1,2)+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3))&
&   *dgm10(3,3)+(6*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm01(2,2)+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(3,3))&
&   /12.d0
   cm(6,25)=((2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+12*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(1,2)+(6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+12*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)+2*gm(2,2)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(1,3)+(2*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3))*dgm10(2,2)+(6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)+6*(6*gm(1,2)**2*gm(1,3)&
&   +24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(2,2)+3*(4*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+4*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3))*dgm10(2,3)+6*((48*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)&
&   +4*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(3,3))*dgm10(2,3)+(288*gm(1,2)*gm(1,3)*gm(2,2)&
&   *dgm01(1,2)+gm(1,1)*gm(2,2)*(-108*gm(2,3)*dgm01(1,2)-36*gm(2,2)&
&   *dgm01(1,3))+gm(1,2)**2*(36*gm(2,3)*dgm01(1,2)+108*gm(2,2)*dgm01(1,3)))&
&   *dgm10(3,3)+(6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&   +6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)+6*(6*gm(1,2)**3+30*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(3,3))/12.d0
   cm(7,25)=((2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&   +12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)+216*gm(2,2)&
&   **2*gm(2,3)*dgm01(3,3))*dgm10(1,2)+gm(2,2)*(6*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(2,2)+432*gm(2,2)*gm(2,3)*dgm01(2,3)+72*gm(2,2)&
&   **2*dgm01(3,3))*dgm10(1,3)+(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)&
&   *gm(3,3))*dgm01(1,2)+6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&   *dgm01(1,3))*dgm10(2,2)+(6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*gm(2,2)*(-18*gm(1,3)&
&   *gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,2)+3*gm(2,2)&
&   *(4*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)+144*gm(2,2)&
&   *gm(2,3)*dgm01(1,3))*dgm10(2,3)+3*(2*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)+8*gm(2,2)&
&   *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(2,3)+72*gm(1,2)&
&   *gm(2,2)**2*dgm01(3,3))*dgm10(2,3)+gm(2,2)**2*(216*gm(2,3)*dgm01(1,2)&
&   +72*gm(2,2)*dgm01(1,3))*dgm10(3,3)+gm(2,2)*(6*(-18*gm(1,3)*gm(2,2)&
&   +54*gm(1,2)*gm(2,3))*dgm01(2,2)+216*gm(1,2)*gm(2,2)*dgm01(2,3))&
&   *dgm10(3,3))/36.d0
   cm(8,25)=((144*gm(2,3)**2*gm(3,3)*dgm01(2,3)+288*gm(2,2)*gm(3,3)&
&   **2*dgm01(2,3)+36*gm(2,3)**3*dgm01(3,3)+gm(2,3)*gm(3,3)*(72*gm(3,3)&
&   *dgm01(2,2)+180*gm(2,2)*dgm01(3,3)))*dgm10(1,2)+(72*gm(2,3)**3*dgm01(2,3)&
&   +360*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(2,3)+gm(2,2)*gm(3,3)*(144*gm(3,3)&
&   *dgm01(2,2)-36*gm(2,2)*dgm01(3,3))+gm(2,3)**2*(72*gm(3,3)*dgm01(2,2)&
&   +108*gm(2,2)*dgm01(3,3)))*dgm10(1,3)+gm(3,3)*(72*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+6*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,3))&
&   *dgm10(2,2)+(6*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&   *dgm01(2,3)+6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(2,2)+3*(48*gm(2,3)**2*gm(3,3)*dgm01(1,2)&
&   +96*gm(2,2)*gm(3,3)**2*dgm01(1,2)+24*gm(2,3)**3*dgm01(1,3)+120*gm(2,2)&
&   *gm(2,3)*gm(3,3)*dgm01(1,3))*dgm10(2,3)+6*(gm(3,3)*(12*gm(1,3)&
&   *gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(2,3)+(36*gm(2,3)**3*dgm01(1,2)+180*gm(2,2)&
&   *gm(2,3)*gm(3,3)*dgm01(1,2)+108*gm(2,2)*gm(2,3)**2*dgm01(1,3)&
&   -36*gm(2,2)**2*gm(3,3)*dgm01(1,3))*dgm10(3,3)+(6*(6*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3))*dgm10(3,3))/12.d0
   cm(9,25)=((72*gm(2,3)**3*dgm01(2,3)+360*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm01(2,3)+gm(2,3)**2*(108*gm(3,3)*dgm01(2,2)+72*gm(2,2)*dgm01(3,3))&
&   +gm(2,2)*gm(3,3)*(-36*gm(3,3)*dgm01(2,2)+144*gm(2,2)*dgm01(3,3)))&
&   *dgm10(1,2)+(36*gm(2,3)**3*dgm01(2,2)+144*gm(2,2)*gm(2,3)**2*dgm01(2,3)&
&   +288*gm(2,2)**2*gm(3,3)*dgm01(2,3)+gm(2,2)*gm(2,3)*(180*gm(3,3)&
&   *dgm01(2,2)+72*gm(2,2)*dgm01(3,3)))*dgm10(1,3)+(2*gm(3,3)*(54*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3))*dgm01(1,2)+6*(6*gm(2,3)**3+30*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,3))*dgm10(2,2)+(6*(48*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(2,2)+3*(24*gm(2,3)**3*dgm01(1,2)&
&   +120*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(1,2)+48*gm(2,2)*gm(2,3)**2*dgm01(1,3)&
&   +96*gm(2,2)**2*gm(3,3)*dgm01(1,3))*dgm10(2,3)+6*((48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3)+gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&   *gm(2,3))*dgm01(3,3))*dgm10(2,3)+gm(2,2)*(72*gm(2,3)**2*dgm01(1,2)&
&   +144*gm(2,2)*gm(3,3)*dgm01(1,2)+72*gm(2,2)*gm(2,3)*dgm01(1,3))&
&   *dgm10(3,3)+(6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)+6*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&   +12*gm(1,2)*gm(2,3))*dgm01(2,3))*dgm10(3,3))/12.d0
   cm(10,25)=(gm(3,3)*(72*gm(3,3)**2*dgm01(2,2)+324*gm(2,3)**2*dgm01(3,3)&
&   +gm(3,3)*(432*gm(2,3)*dgm01(2,3)-108*gm(2,2)*dgm01(3,3)))*dgm10(1,2)&
&   +(648*gm(2,3)**2*gm(3,3)*dgm01(2,3)-216*gm(2,2)*gm(3,3)**2*dgm01(2,3)&
&   +180*gm(2,3)**3*dgm01(3,3)+gm(2,3)*gm(3,3)*(216*gm(3,3)*dgm01(2,2)&
&   -108*gm(2,2)*dgm01(3,3)))*dgm10(1,3)+gm(3,3)**2*(72*gm(3,3)*dgm01(1,2)&
&   +216*gm(2,3)*dgm01(1,3))*dgm10(2,2)+gm(3,3)*(216*gm(1,3)*gm(3,3)&
&   *dgm01(2,3)+6*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(3,3))&
&   *dgm10(2,2)+3*gm(3,3)*(144*gm(2,3)*gm(3,3)*dgm01(1,2)+216*gm(2,3)&
&   **2*dgm01(1,3)-72*gm(2,2)*gm(3,3)*dgm01(1,3))*dgm10(2,3)+3*(72*gm(1,3)&
&   *gm(3,3)**2*dgm01(2,2)+8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(2,3)+2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,3)+(324*gm(2,3)&
&   **2*gm(3,3)*dgm01(1,2)-108*gm(2,2)*gm(3,3)**2*dgm01(1,2)+180*gm(2,3)&
&   **3*dgm01(1,3)-108*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(1,3))*dgm10(3,3)&
&   +(6*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(2,2)&
&   +6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3))*dgm10(3,3))/36.d0
   cm(1,26)=((8*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)&
&   +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(1,2)+2*(2*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(2,2)+12*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)+6*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(3,3))&
&   *dgm10(1,3)+4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,3)&
&   *dgm10(2,2)+6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,2)+4*(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2)&
&   +6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(1,3))*dgm10(2,3)+24*gm(1,1)*((54*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3))*dgm01(2,3)+(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&   *dgm01(3,3))*dgm10(2,3)+3*(4*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)+4*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(1,3))&
&   *dgm10(3,3)+6*gm(1,1)*((54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)&
&   +4*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,3)+(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(3,3))/72.d0
   cm(2,26)=((8*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,2)&
&   +2*(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*gm(2,2)*(24*gm(1,3)&
&   *gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(1,3)+4*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3)*dgm10(2,2)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(2,2)+4*(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)+6*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3))&
&   *dgm10(2,3)+4*(6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+6*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)+3*(4*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)&
&   +4*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(1,3))&
&   *dgm10(3,3)+6*((-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)+gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm01(3,3))*dgm10(3,3))/24.d0
   cm(3,26)=(4*gm(3,3)*(288*gm(1,3)*gm(3,3)*dgm01(2,3)+12*(12*gm(1,3)&
&   *gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(3,3))*dgm10(1,2)+8*(288*gm(1,2)&
&   *gm(3,3)*(gm(3,3)*dgm01(2,3)+gm(2,3)*dgm01(3,3))+gm(1,3)*(72*gm(3,3)&
&   **2*dgm01(2,2)+36*gm(2,3)**2*dgm01(3,3)+gm(3,3)*(144*gm(2,3)&
&   *dgm01(2,3)-108*gm(2,2)*dgm01(3,3))))*dgm10(1,3)+576*gm(1,3)&
&   *gm(3,3)**2*dgm01(1,3)*dgm10(2,2)+24*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3))*dgm01(3,3)*dgm10(2,2)+16*gm(3,3)*(144*gm(1,2)*gm(3,3)&
&   *dgm01(1,3)+72*gm(1,3)*(gm(3,3)*dgm01(1,2)+gm(2,3)*dgm01(1,3)))&
&   *dgm10(2,3)+16*(6*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))&
&   *dgm01(2,3)+6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&   +24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(2,3)+12*(gm(1,2)&
&   *gm(3,3)*(96*gm(3,3)*dgm01(1,2)+192*gm(2,3)*dgm01(1,3))+gm(1,3)&
&   *(48*gm(2,3)*gm(3,3)*dgm01(1,2)+24*gm(2,3)**2*dgm01(1,3)-72*gm(2,2)&
&   *gm(3,3)*dgm01(1,3)))*dgm10(3,3)+24*(gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3))*dgm01(2,2)+4*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(3,3))&
&   /96.d0
   cm(4,26)=((8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(2,3)+12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,2)+2*(gm(1,3)*(72*gm(2,3)&
&   **2*dgm01(2,3)+288*gm(2,2)*gm(3,3)*dgm01(2,3)+gm(2,3)*(108*gm(3,3)&
&   *dgm01(2,2)+36*gm(2,2)*dgm01(3,3)))+gm(1,2)*(-36*gm(3,3)**2*dgm01(2,2)&
&   +36*gm(2,3)**2*dgm01(3,3)+gm(3,3)*(72*gm(2,3)*dgm01(2,3)+144*gm(2,2)&
&   *dgm01(3,3))))*dgm10(1,3)+4*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(1,3)*dgm10(2,2)+6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(2,2)&
&   +4*(2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,2)&
&   +6*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(1,3))*dgm10(2,3)+4*(6*(6*gm(1,3)**2*gm(2,3)&
&   +6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   +6*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(2,3)+3*(gm(1,3)*(24*gm(2,3)**2*dgm01(1,2)+96*gm(2,2)*gm(3,3)&
&   *dgm01(1,2)+24*gm(2,2)*gm(2,3)*dgm01(1,3))+gm(1,2)*(24*gm(2,3)&
&   *gm(3,3)*dgm01(1,2)+24*gm(2,3)**2*dgm01(1,3)+96*gm(2,2)*gm(3,3)&
&   *dgm01(1,3)))*dgm10(3,3)+6*((6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+4*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(2,3)+(6*gm(1,2)*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(3,3))/12.d0
   cm(5,26)=((8*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)&
&   +12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(1,2)+2*(2*gm(3,3)*(54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,3)&
&   +4*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)*dgm10(2,2)&
&   +6*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)*dgm10(2,2)&
&   +4*(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)+6*(6*gm(1,3)&
&   **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3))*dgm10(2,3)+4*(6*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm01(2,3)+6*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm01(3,3))*dgm10(2,3)+3*(4*(6*gm(1,3)&
&   **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,2)+4*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3))*dgm10(3,3)+6*((6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&   *dgm01(2,2)+4*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3)))*dgm01(2,3)+(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(3,3))&
&   /24.d0
   cm(6,26)=((8*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+12*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(1,2)+2*(2*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,2)+12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(1,3)+4*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3)*dgm10(2,2)+6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(3,3)*dgm10(2,2)+4*(2*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,2)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3))*dgm10(2,3)+4*(6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)+6*(6*gm(1,2)**2*gm(1,3)&
&   +24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(2,3)+3*(4*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(1,2)+4*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3))*dgm10(3,3)+6*((48*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)&
&   +4*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm01(2,3)+(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(3,3))*dgm10(3,3))/24.d0
   cm(7,26)=((8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   +12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3))*dgm10(1,2)&
&   +2*(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)+12*gm(2,2)&
&   *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)+216*gm(2,2)**2*gm(2,3)&
&   *dgm01(3,3))*dgm10(1,3)+4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3)*dgm10(2,2)+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(2,2)+4*(2*(90*gm(2,3)&
&   **3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)+6*gm(2,2)*(54*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3))*dgm01(1,3))*dgm10(2,3)+4*(6*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(2,3)+3*gm(2,2)*(4*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,2)+144*gm(2,2)*gm(2,3)*dgm01(1,3))*dgm10(3,3)&
&   +3*(2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)+8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&   *gm(2,3))*dgm01(2,3)+72*gm(1,2)*gm(2,2)**2*dgm01(3,3))*dgm10(3,3))&
&   /72.d0
   cm(8,26)=(4*gm(3,3)*(288*gm(2,3)*gm(3,3)*dgm01(2,3)+12*(12*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3))*dgm01(3,3))*dgm10(1,2)+8*(144*gm(2,3)&
&   **2*gm(3,3)*dgm01(2,3)+288*gm(2,2)*gm(3,3)**2*dgm01(2,3)+36*gm(2,3)&
&   **3*dgm01(3,3)+gm(2,3)*gm(3,3)*(72*gm(3,3)*dgm01(2,2)+180*gm(2,2)&
&   *dgm01(3,3)))*dgm10(1,3)+576*gm(2,3)*gm(3,3)**2*dgm01(1,3)*dgm10(2,2)&
&   +24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,2)+16*gm(3,3)*(72*gm(2,3)*gm(3,3)*dgm01(1,2)+6*(12*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3))*dgm01(1,3))*dgm10(2,3)+16*(6*gm(3,3)&
&   *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(2,3)+6*(6*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(2,3)+12*(48*gm(2,3)**2*gm(3,3)*dgm01(1,2)+96*gm(2,2)*gm(3,3)&
&   **2*dgm01(1,2)+24*gm(2,3)**3*dgm01(1,3)+120*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm01(1,3))*dgm10(3,3)+24*(gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&   *gm(3,3))*dgm01(2,2)+4*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)+(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(3,3))&
&   /96.d0
   cm(9,26)=((8*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)&
&   +12*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(1,2)&
&   +2*(72*gm(2,3)**3*dgm01(2,3)+360*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(2,3)&
&   +gm(2,3)**2*(108*gm(3,3)*dgm01(2,2)+72*gm(2,2)*dgm01(3,3))+gm(2,2)&
&   *gm(3,3)*(-36*gm(3,3)*dgm01(2,2)+144*gm(2,2)*dgm01(3,3)))*dgm10(1,3)&
&   +4*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)*dgm10(2,2)&
&   +6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(2,2)+4*(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(1,2)+6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3))*dgm10(2,3)+4*(6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*(6*gm(1,3)*gm(2,2)&
&   *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(3,3))&
&   *dgm10(2,3)+3*(24*gm(2,3)**3*dgm01(1,2)+120*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm01(1,2)+48*gm(2,2)*gm(2,3)**2*dgm01(1,3)+96*gm(2,2)**2*gm(3,3)&
&   *dgm01(1,3))*dgm10(3,3)+6*((48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)+4*(6*gm(1,3)*gm(2,2)&
&   *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(3,3))/24.d0
   cm(10,26)=(4*gm(3,3)**2*(288*gm(3,3)*dgm01(2,3)+432*gm(2,3)&
&   *dgm01(3,3))*dgm10(1,2)+8*gm(3,3)*(72*gm(3,3)**2*dgm01(2,2)+324*gm(2,3)&
&   **2*dgm01(3,3)+gm(3,3)*(432*gm(2,3)*dgm01(2,3)-108*gm(2,2)*dgm01(3,3)))&
&   *dgm10(1,3)+576*gm(3,3)**3*dgm01(1,3)*dgm10(2,2)+864*gm(1,3)&
&   *gm(3,3)**2*dgm01(3,3)*dgm10(2,2)+16*gm(3,3)**2*(72*gm(3,3)*dgm01(1,2)&
&   +216*gm(2,3)*dgm01(1,3))*dgm10(2,3)+16*gm(3,3)*(216*gm(1,3)*gm(3,3)&
&   *dgm01(2,3)+6*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(3,3))&
&   *dgm10(2,3)+12*gm(3,3)*(144*gm(2,3)*gm(3,3)*dgm01(1,2)+216*gm(2,3)&
&   **2*dgm01(1,3)-72*gm(2,2)*gm(3,3)*dgm01(1,3))*dgm10(3,3)+12*(72*gm(1,3)&
&   *gm(3,3)**2*dgm01(2,2)+8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(2,3)+2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(3,3))/288.d0
   cm(1,27)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,2)+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,3)+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(3,3)*dgm10(1,3)+2*(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,3)*dgm10(2,3)+3*gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)*dgm10(2,3)+(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2)*dgm10(3,3)+3*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(1,3)*dgm10(3,3)+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&   *dgm01(2,3)*dgm10(3,3)+3*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)&
&   *gm(2,3))*dgm01(3,3)*dgm10(3,3))/18.d0
   cm(2,27)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(1,2)+2*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(1,3)&
&   +3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(1,3)+2*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(2,3)&
&   +3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3)&
&   *dgm10(2,3)+(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(3,3)+3*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   *dgm10(3,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)*dgm10(3,3)+3*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3)*dgm10(3,3))&
&   /6.d0
   cm(3,27)=(gm(1,3)*gm(3,3)*(720*gm(1,2)*dgm01(3,3)*dgm10(3,3)&
&   +gm(3,3)*(1440*dgm01(3,3)*dgm10(1,2)+2880*dgm01(2,3)*dgm10(1,3)&
&   +2880*dgm01(1,3)*dgm10(2,3)+1440*dgm01(1,2)*dgm10(3,3))+1440*gm(2,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+gm(1,3)**2*(720*gm(2,3)&
&   *dgm01(3,3)*dgm10(3,3)+1440*gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3)))+2880*gm(3,3)*(gm(1,2)*gm(3,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3))+gm(1,1)*(gm(2,3)*dgm01(3,3)*dgm10(3,3)&
&   +gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))))/240.d0
   cm(4,27)=(gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,2)+2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(2,3)*dgm10(1,3)+3*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(1,3)+2*gm(3,3)*(54*gm(1,3)&
&   *gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(1,3)*dgm10(2,3)+3*(6*gm(1,3)&
&   **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(3,3)*dgm10(2,3)+gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(1,2)*dgm10(3,3)+3*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(3,3)&
&   +3*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(2,3)*dgm10(3,3)+3*(9*gm(1,3)**2*gm(2,2)&
&   -6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)&
&   **2+15*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3))/3.d0
   cm(5,27)=(gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,2)+2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,3)+3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(1,3)+2*gm(3,3)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)*dgm10(2,3)+3*(6*gm(1,3)&
&   **3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)*dgm10(2,3)+gm(3,3)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)*dgm10(3,3)+3*(6*gm(1,3)&
&   **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3)*dgm10(3,3)+3*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&   *dgm01(2,3)*dgm10(3,3)+3*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm01(3,3)*dgm10(3,3))/6.d0
   cm(6,27)=((90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(1,2)+2*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,3)*dgm10(1,3)+3*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(1,3)+2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,3)*dgm10(2,3)&
&   +3*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(2,3)+(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)*dgm10(3,3)&
&   +3*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&   *dgm10(3,3)+3*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)*dgm10(3,3)+3*(6*gm(1,2)**2*gm(1,3)&
&   +24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3)&
&   *dgm10(3,3))/6.d0
   cm(7,27)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,2)+2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,3)+3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,3)+2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)&
&   *dgm10(2,3)+3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(2,3)+(90*gm(2,3)**3-54*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,2)*dgm10(3,3)+3*gm(2,2)*(54*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3))*dgm01(1,3)*dgm10(3,3)+3*(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)*dgm10(3,3)+3*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&   *gm(2,3))*dgm01(3,3)*dgm10(3,3))/18.d0
   cm(8,27)=(gm(2,3)**2*(720*gm(1,3)*dgm01(3,3)*dgm10(3,3)+1440*gm(3,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+gm(2,3)*gm(3,3)&
&   *(720*gm(1,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)*(1440*dgm01(3,3)&
&   *dgm10(1,2)+2880*dgm01(2,3)*dgm10(1,3)+2880*dgm01(1,3)*dgm10(2,3)&
&   +1440*dgm01(1,2)*dgm10(3,3))+1440*gm(1,3)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3)))+2880*gm(3,3)*(gm(1,2)*gm(3,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(2,2)*(gm(1,3)*dgm01(3,3)&
&   *dgm10(3,3)+gm(3,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))))&
&   /240.d0
   cm(9,27)=(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,2)+2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)&
&   *dgm10(1,3)+3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,3)+2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)&
&   *dgm10(2,3)+3*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(2,3)+gm(3,3)*(54*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3))*dgm01(1,2)*dgm10(3,3)+3*(6*gm(2,3)**3+30*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(1,3)*dgm10(3,3)+3*(48*gm(1,2)*gm(2,3)&
&   *gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   *dgm10(3,3)+3*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(3,3))/6.d0
   cm(10,27)=2*gm(3,3)*(4.5d0*gm(1,3)*gm(2,3)*dgm01(3,3)*dgm10(3,3)&
&   +gm(3,3)**2*(1*dgm01(3,3)*dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)&
&   +2*dgm01(1,3)*dgm10(2,3)+dgm01(1,2)*dgm10(3,3))+gm(3,3)*(-1.5d0*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+3*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))+3*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))
   cm(1,28)=(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,3)+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,3)&
&   *dgm10(3,3)+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)&
&   *dgm10(3,3))/36.d0
   cm(2,28)=(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(1,3)+2*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(1,3)*dgm10(3,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(3,3))/12.d0
   cm(3,28)=(gm(3,3)*(8640*gm(1,3)*gm(3,3)*dgm01(3,3)*dgm10(1,3)&
&   +8640*gm(1,3)*gm(3,3)*dgm01(1,3)*dgm10(3,3)+360*(12*gm(1,3)**2+24*gm(1,1)&
&   *gm(3,3))*dgm01(3,3)*dgm10(3,3)))/1440.d0
   cm(4,28)=(2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(3,3)*dgm10(1,3)+2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(1,3)*dgm10(3,3)+3*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(3,3))&
&   /6.d0
   cm(5,28)=(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,3)+2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,3)&
&   *dgm10(3,3)+3*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(3,3))/12.d0
   cm(6,28)=(2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(1,3)+2*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(1,3)*dgm10(3,3)+3*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&   *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(3,3)*dgm10(3,3))/12.d0
   cm(7,28)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,3)+2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)&
&   *dgm10(3,3)+3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3))/36.d0
   cm(8,28)=(gm(3,3)*(8640*gm(1,2)*gm(3,3)*dgm01(3,3)*dgm10(3,3)&
&   +gm(2,3)*(4320*gm(1,3)*dgm01(3,3)*dgm10(3,3)+8640*gm(3,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))))/1440.d0
   cm(9,28)=(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(1,3)+2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)&
&   *dgm10(3,3)+3*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3))/12.d0
   cm(10,28)=gm(3,3)**2*(3*gm(1,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
   cm(1,29)=((90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2)&
&   *dgm10(2,2))/36.d0
   cm(2,29)=3*gm(1,2)*gm(2,2)**2*dgm01(2,2)*dgm10(2,2)
   cm(3,29)=((-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)*dgm10(2,2))/12.d0
   cm(4,29)=(gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(2,2)&
&   *dgm10(2,2))/6.d0
   cm(5,29)=((-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2))/12.d0
   cm(6,29)=(gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2)&
&   *dgm10(2,2))/12.d0
   cm(7,29)=gm(2,2)**3*dgm01(2,2)*dgm10(2,2)
   cm(8,29)=(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,2)&
&   *dgm10(2,2))/12.d0
   cm(9,29)=3*gm(2,2)**2*gm(2,3)*dgm01(2,2)*dgm10(2,2)
   cm(10,29)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&   *dgm10(2,2))/36.d0
   cm(1,30)=((6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)+12*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(2,3)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(2,2)+(12*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(2,2)&
&   +8*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(2,3)&
&   +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2)*dgm10(3,3))&
&   /72.d0
   cm(2,30)=(20*(6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)+12*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&   +12*gm(1,2)*gm(2,3))*dgm01(2,3)+72*gm(1,2)*gm(2,2)**2*dgm01(3,3))&
&   *dgm10(2,2)+20*gm(2,2)*(12*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&   *dgm01(2,2)+288*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(2,3)+1440*gm(1,2)&
&   *gm(2,2)**2*dgm01(2,2)*dgm10(3,3))/480.d0
   cm(3,30)=((gm(1,3)*(72*gm(2,3)**2*dgm01(2,3)-216*gm(2,2)*gm(3,3)&
&   *dgm01(2,3)+gm(2,3)*(72*gm(3,3)*dgm01(2,2)-72*gm(2,2)*dgm01(3,3)))&
&   +gm(1,2)*(144*gm(3,3)**2*dgm01(2,2)+180*gm(2,3)**2*dgm01(3,3)&
&   +gm(3,3)*(576*gm(2,3)*dgm01(2,3)-36*gm(2,2)*dgm01(3,3))))*dgm10(2,2)&
&   +(gm(1,3)*(72*gm(2,3)**2*dgm01(2,2)-216*gm(2,2)*gm(3,3)*dgm01(2,2)&
&   -288*gm(2,2)*gm(2,3)*dgm01(2,3))+gm(1,2)*(576*gm(2,3)*gm(3,3)&
&   *dgm01(2,2)+720*gm(2,3)**2*dgm01(2,3)-144*gm(2,2)*gm(3,3)*dgm01(2,3)))&
&   *dgm10(2,3)+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(3,3))/24.d0
   cm(4,30)=((gm(1,3)*(36*gm(2,3)**2*dgm01(2,2)+72*gm(2,2)*gm(2,3)&
&   *dgm01(2,3)+gm(2,2)*(144*gm(3,3)*dgm01(2,2)-36*gm(2,2)*dgm01(3,3)))&
&   +gm(1,2)*(72*gm(2,3)**2*dgm01(2,3)+288*gm(2,2)*gm(3,3)*dgm01(2,3)&
&   +gm(2,3)*(36*gm(3,3)*dgm01(2,2)+108*gm(2,2)*dgm01(3,3))))*dgm10(2,2)&
&   +(gm(1,3)*gm(2,2)*(72*gm(2,3)*dgm01(2,2)-144*gm(2,2)*dgm01(2,3))&
&   +gm(1,2)*(72*gm(2,3)**2*dgm01(2,2)+288*gm(2,2)*gm(3,3)*dgm01(2,2)&
&   +432*gm(2,2)*gm(2,3)*dgm01(2,3)))*dgm10(2,3)+2*gm(2,2)*(-18*gm(1,3)&
&   *gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(3,3))/12.d0
   cm(5,30)=((6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(2,2)+(12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+8*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3))*dgm10(2,3)+2*(-36*gm(1,2)&
&   *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(2,2)*dgm10(3,3))/24.d0
   cm(6,30)=((6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+12*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)+2*gm(2,2)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(2,2)+(576*gm(1,2)*gm(1,3)&
&   *gm(2,2)*dgm01(2,2)+gm(1,1)*gm(2,2)*(-216*gm(2,3)*dgm01(2,2)&
&   -144*gm(2,2)*dgm01(2,3))+gm(1,2)**2*(72*gm(2,3)*dgm01(2,2)+432*gm(2,2)&
&   *dgm01(2,3)))*dgm10(2,3)+2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm01(2,2)*dgm10(3,3))/24.d0
   cm(7,30)=(gm(2,2)*(20*(6*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&   *dgm01(2,2)+432*gm(2,2)*gm(2,3)*dgm01(2,3)+72*gm(2,2)**2*dgm01(3,3))&
&   *dgm10(2,2)+20*gm(2,2)*(432*gm(2,3)*dgm01(2,2)+288*gm(2,2)*dgm01(2,3))&
&   *dgm10(2,3)+1440*gm(2,2)**2*dgm01(2,2)*dgm10(3,3)))/1440.d0
   cm(8,30)=((72*gm(2,3)**3*dgm01(2,3)+360*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm01(2,3)+gm(2,2)*gm(3,3)*(144*gm(3,3)*dgm01(2,2)-36*gm(2,2)&
&   *dgm01(3,3))+gm(2,3)**2*(72*gm(3,3)*dgm01(2,2)+108*gm(2,2)*dgm01(3,3)))&
&   *dgm10(2,2)+(72*gm(2,3)**3*dgm01(2,2)+360*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm01(2,2)+432*gm(2,2)*gm(2,3)**2*dgm01(2,3)-144*gm(2,2)**2*gm(3,3)&
&   *dgm01(2,3))*dgm10(2,3)+2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&   *dgm01(2,2)*dgm10(3,3))/24.d0
   cm(9,30)=1.5d0*gm(2,3)**3*dgm01(2,2)*dgm10(2,2)+6*gm(2,2)*gm(2,3)&
&   **2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+12*gm(2,2)&
&   **2*gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)&
&   *gm(2,3)*(7.5d0*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(3*dgm01(3,3)&
&   *dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)+3*dgm01(2,2)*dgm10(3,3)))
   cm(10,30)=((648*gm(2,3)**2*gm(3,3)*dgm01(2,3)-216*gm(2,2)*gm(3,3)&
&   **2*dgm01(2,3)+180*gm(2,3)**3*dgm01(3,3)+gm(2,3)*gm(3,3)*(216*gm(3,3)&
&   *dgm01(2,2)-108*gm(2,2)*dgm01(3,3)))*dgm10(2,2)+(648*gm(2,3)&
&   **2*gm(3,3)*dgm01(2,2)-216*gm(2,2)*gm(3,3)**2*dgm01(2,2)+720*gm(2,3)&
&   **3*dgm01(2,3)-432*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(2,3))*dgm10(2,3)&
&   +2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)*dgm10(3,3))&
&   /72.d0
   cm(1,31)=(30*(12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)&
&   *gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(2,2)+8*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(2,2)+240*(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*dgm01(2,2)*dgm10(2,3))/4320.d0
   cm(2,31)=(gm(2,2)*(30*(12*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&   *dgm01(2,2)+288*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(2,2)+8640*gm(1,2)&
&   *gm(2,2)*dgm01(2,2)*dgm10(2,3)))/1440.d0
   cm(3,31)=(30*(gm(1,3)*(72*gm(2,3)**2*dgm01(2,2)-216*gm(2,2)&
&   *gm(3,3)*dgm01(2,2)-288*gm(2,2)*gm(2,3)*dgm01(2,3))+gm(1,2)*(576*gm(2,3)&
&   *gm(3,3)*dgm01(2,2)+720*gm(2,3)**2*dgm01(2,3)-144*gm(2,2)*gm(3,3)&
&   *dgm01(2,3)))*dgm10(2,2)+240*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,3))/1440.d0
   cm(4,31)=(30*(gm(1,3)*gm(2,2)*(72*gm(2,3)*dgm01(2,2)-144*gm(2,2)&
&   *dgm01(2,3))+gm(1,2)*(72*gm(2,3)**2*dgm01(2,2)+288*gm(2,2)*gm(3,3)&
&   *dgm01(2,2)+432*gm(2,2)*gm(2,3)*dgm01(2,3)))*dgm10(2,2)+240*gm(2,2)&
&   *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,3))&
&   /720.d0
   cm(5,31)=(30*(12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)+8*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&   **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3))*dgm10(2,2)&
&   +240*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(2,3))/1440.d0
   cm(6,31)=(30*(576*gm(1,2)*gm(1,3)*gm(2,2)*dgm01(2,2)+gm(1,1)&
&   *gm(2,2)*(-216*gm(2,3)*dgm01(2,2)-144*gm(2,2)*dgm01(2,3))+gm(1,2)&
&   **2*(72*gm(2,3)*dgm01(2,2)+432*gm(2,2)*dgm01(2,3)))*dgm10(2,2)&
&   +240*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(2,3))&
&   /1440.d0
   cm(7,31)=gm(2,2)**2*(3*gm(2,3)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))
   cm(8,31)=1.5d0*gm(2,3)**3*dgm01(2,2)*dgm10(2,2)+7.5d0*gm(2,2)&
&   *gm(2,3)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+9*gm(2,2)*gm(2,3)**2*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))-3*gm(2,2)**2*gm(3,3)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))
   cm(9,31)=gm(2,2)*(3*gm(2,3)**2*dgm01(2,2)*dgm10(2,2)+6*gm(2,2)&
&   *gm(3,3)*dgm01(2,2)*dgm10(2,2)+6*gm(2,2)*gm(2,3)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))
   cm(10,31)=(30*(648*gm(2,3)**2*gm(3,3)*dgm01(2,2)-216*gm(2,2)&
&   *gm(3,3)**2*dgm01(2,2)+720*gm(2,3)**3*dgm01(2,3)-432*gm(2,2)&
&   *gm(2,3)*gm(3,3)*dgm01(2,3))*dgm10(2,2)+240*(90*gm(2,3)**3-54*gm(2,2)&
&   *gm(2,3)*gm(3,3))*dgm01(2,2)*dgm10(2,3))/4320.d0
   cm(1,32)=(2*(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)&
&   +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(2,3)+6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)&
&   *gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(3,3))*dgm10(2,2)+4*(6*(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *dgm01(2,2)+12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&   +2*gm(1,2)*gm(2,3)))*dgm01(2,3)+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)&
&   *gm(2,2))*dgm01(3,3))*dgm10(2,3)+(12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(2,2)+8*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3))*dgm10(3,3))/144.d0
   cm(2,32)=(2*(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*gm(2,2)&
&   *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,2)&
&   +4*(6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)+12*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&   *gm(2,3))*dgm01(2,3)+72*gm(1,2)*gm(2,2)**2*dgm01(3,3))*dgm10(2,3)&
&   +gm(2,2)*(12*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(2,2)&
&   +288*gm(1,2)*gm(2,2)*dgm01(2,3))*dgm10(3,3))/48.d0
   cm(3,32)=(2*(288*gm(1,2)*gm(3,3)*(gm(3,3)*dgm01(2,3)+gm(2,3)&
&   *dgm01(3,3))+gm(1,3)*(72*gm(3,3)**2*dgm01(2,2)+36*gm(2,3)**2*dgm01(3,3)&
&   +gm(3,3)*(144*gm(2,3)*dgm01(2,3)-108*gm(2,2)*dgm01(3,3))))*dgm10(2,2)&
&   +4*(gm(1,3)*(72*gm(2,3)**2*dgm01(2,3)-216*gm(2,2)*gm(3,3)*dgm01(2,3)&
&   +gm(2,3)*(72*gm(3,3)*dgm01(2,2)-72*gm(2,2)*dgm01(3,3)))+gm(1,2)&
&   *(144*gm(3,3)**2*dgm01(2,2)+180*gm(2,3)**2*dgm01(3,3)+gm(3,3)&
&   *(576*gm(2,3)*dgm01(2,3)-36*gm(2,2)*dgm01(3,3))))*dgm10(2,3)&
&   +(gm(1,3)*(72*gm(2,3)**2*dgm01(2,2)-216*gm(2,2)*gm(3,3)*dgm01(2,2)&
&   -288*gm(2,2)*gm(2,3)*dgm01(2,3))+gm(1,2)*(576*gm(2,3)*gm(3,3)&
&   *dgm01(2,2)+720*gm(2,3)**2*dgm01(2,3)-144*gm(2,2)*gm(3,3)*dgm01(2,3)))&
&   *dgm10(3,3))/48.d0
   cm(4,32)=(2*(gm(1,3)*(72*gm(2,3)**2*dgm01(2,3)+288*gm(2,2)*gm(3,3)&
&   *dgm01(2,3)+gm(2,3)*(108*gm(3,3)*dgm01(2,2)+36*gm(2,2)*dgm01(3,3)))&
&   +gm(1,2)*(-36*gm(3,3)**2*dgm01(2,2)+36*gm(2,3)**2*dgm01(3,3)&
&   +gm(3,3)*(72*gm(2,3)*dgm01(2,3)+144*gm(2,2)*dgm01(3,3))))*dgm10(2,2)&
&   +4*(gm(1,3)*(36*gm(2,3)**2*dgm01(2,2)+72*gm(2,2)*gm(2,3)*dgm01(2,3)&
&   +gm(2,2)*(144*gm(3,3)*dgm01(2,2)-36*gm(2,2)*dgm01(3,3)))+gm(1,2)&
&   *(72*gm(2,3)**2*dgm01(2,3)+288*gm(2,2)*gm(3,3)*dgm01(2,3)+gm(2,3)&
&   *(36*gm(3,3)*dgm01(2,2)+108*gm(2,2)*dgm01(3,3))))*dgm10(2,3)&
&   +(gm(1,3)*gm(2,2)*(72*gm(2,3)*dgm01(2,2)-144*gm(2,2)*dgm01(2,3))&
&   +gm(1,2)*(72*gm(2,3)**2*dgm01(2,2)+288*gm(2,2)*gm(3,3)*dgm01(2,2)&
&   +432*gm(2,2)*gm(2,3)*dgm01(2,3)))*dgm10(3,3))/24.d0
   cm(5,32)=(2*(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)&
&   +12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(2,3)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(2,2)+4*(6*(6*gm(1,3)**2*gm(2,3)&
&   +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&   +12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&   **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)&
&   +2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&   *gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)+(12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,2)+8*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3))&
&   *dgm10(3,3))/48.d0
   cm(6,32)=(2*(2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+12*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&   +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))&
&   *dgm10(2,2)+4*(6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+12*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)+2*gm(2,2)*(54*gm(1,2)&
&   **2-18*gm(1,1)*gm(2,2))*dgm01(3,3))*dgm10(2,3)+(576*gm(1,2)*gm(1,3)&
&   *gm(2,2)*dgm01(2,2)+gm(1,1)*gm(2,2)*(-216*gm(2,3)*dgm01(2,2)&
&   -144*gm(2,2)*dgm01(2,3))+gm(1,2)**2*(72*gm(2,3)*dgm01(2,2)+432*gm(2,2)&
&   *dgm01(2,3)))*dgm10(3,3))/48.d0
   cm(7,32)=(2*(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&   +12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)+216*gm(2,2)&
&   **2*gm(2,3)*dgm01(3,3))*dgm10(2,2)+4*gm(2,2)*(6*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(2,2)+432*gm(2,2)*gm(2,3)*dgm01(2,3)+72*gm(2,2)&
&   **2*dgm01(3,3))*dgm10(2,3)+gm(2,2)**2*(432*gm(2,3)*dgm01(2,2)&
&   +288*gm(2,2)*dgm01(2,3))*dgm10(3,3))/144.d0
   cm(8,32)=(2*(144*gm(2,3)**2*gm(3,3)*dgm01(2,3)+288*gm(2,2)*gm(3,3)&
&   **2*dgm01(2,3)+36*gm(2,3)**3*dgm01(3,3)+gm(2,3)*gm(3,3)*(72*gm(3,3)&
&   *dgm01(2,2)+180*gm(2,2)*dgm01(3,3)))*dgm10(2,2)+4*(72*gm(2,3)&
&   **3*dgm01(2,3)+360*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(2,3)+gm(2,2)&
&   *gm(3,3)*(144*gm(3,3)*dgm01(2,2)-36*gm(2,2)*dgm01(3,3))+gm(2,3)&
&   **2*(72*gm(3,3)*dgm01(2,2)+108*gm(2,2)*dgm01(3,3)))*dgm10(2,3)&
&   +(72*gm(2,3)**3*dgm01(2,2)+360*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(2,2)&
&   +432*gm(2,2)*gm(2,3)**2*dgm01(2,3)-144*gm(2,2)**2*gm(3,3)*dgm01(2,3))&
&   *dgm10(3,3))/48.d0
   cm(9,32)=3*gm(2,3)**3*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(2,3)**2*(4.5d0*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(3*dgm01(3,3)&
&   *dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)+3*dgm01(2,2)*dgm10(3,3)))&
&   +gm(2,2)*gm(3,3)*(-1.5d0*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)&
&   *(6*dgm01(3,3)*dgm10(2,2)+24*dgm01(2,3)*dgm10(2,3)+6*dgm01(2,2)&
&   *dgm10(3,3)))+gm(2,2)*gm(2,3)*(15*gm(3,3)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3))+6*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3)))
   cm(10,32)=(2*gm(3,3)*(72*gm(3,3)**2*dgm01(2,2)+324*gm(2,3)**2*dgm01(3,3)&
&   +gm(3,3)*(432*gm(2,3)*dgm01(2,3)-108*gm(2,2)*dgm01(3,3)))*dgm10(2,2)&
&   +4*(648*gm(2,3)**2*gm(3,3)*dgm01(2,3)-216*gm(2,2)*gm(3,3)**2*dgm01(2,3)&
&   +180*gm(2,3)**3*dgm01(3,3)+gm(2,3)*gm(3,3)*(216*gm(3,3)*dgm01(2,2)&
&   -108*gm(2,2)*dgm01(3,3)))*dgm10(2,3)+(648*gm(2,3)**2*gm(3,3)&
&   *dgm01(2,2)-216*gm(2,2)*gm(3,3)**2*dgm01(2,2)+720*gm(2,3)**3*dgm01(2,3)&
&   -432*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(2,3))*dgm10(3,3))/144.d0
   cm(1,33)=((8*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)&
&   +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(2,2)+4*(2*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(2,2)+12*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)+6*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(3,3))&
&   *dgm10(2,3)+2*(6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)+12*(90*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(2,3)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3))*dgm10(3,3))/144.d0
   cm(2,33)=((8*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,2)&
&   +4*(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(2,2)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*gm(2,2)*(24*gm(1,3)&
&   *gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)+2*(6*(48*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)&
&   +12*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(2,3)&
&   +72*gm(1,2)*gm(2,2)**2*dgm01(3,3))*dgm10(3,3))/48.d0
   cm(3,33)=(gm(3,3)*(288*gm(1,3)*gm(3,3)*dgm01(2,3)+12*(12*gm(1,3)&
&   *gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(3,3))*dgm10(2,2)+4*(288*gm(1,2)&
&   *gm(3,3)*(gm(3,3)*dgm01(2,3)+gm(2,3)*dgm01(3,3))+gm(1,3)*(72*gm(3,3)&
&   **2*dgm01(2,2)+36*gm(2,3)**2*dgm01(3,3)+gm(3,3)*(144*gm(2,3)&
&   *dgm01(2,3)-108*gm(2,2)*dgm01(3,3))))*dgm10(2,3)+2*(gm(1,3)*(72*gm(2,3)&
&   **2*dgm01(2,3)-216*gm(2,2)*gm(3,3)*dgm01(2,3)+gm(2,3)*(72*gm(3,3)&
&   *dgm01(2,2)-72*gm(2,2)*dgm01(3,3)))+gm(1,2)*(144*gm(3,3)**2*dgm01(2,2)&
&   +180*gm(2,3)**2*dgm01(3,3)+gm(3,3)*(576*gm(2,3)*dgm01(2,3)-36*gm(2,2)&
&   *dgm01(3,3))))*dgm10(3,3))/48.d0
   cm(4,33)=((8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(2,3)+12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,2)+4*(gm(1,3)*(72*gm(2,3)&
&   **2*dgm01(2,3)+288*gm(2,2)*gm(3,3)*dgm01(2,3)+gm(2,3)*(108*gm(3,3)&
&   *dgm01(2,2)+36*gm(2,2)*dgm01(3,3)))+gm(1,2)*(-36*gm(3,3)**2*dgm01(2,2)&
&   +36*gm(2,3)**2*dgm01(3,3)+gm(3,3)*(72*gm(2,3)*dgm01(2,3)+144*gm(2,2)&
&   *dgm01(3,3))))*dgm10(2,3)+2*(gm(1,3)*(36*gm(2,3)**2*dgm01(2,2)&
&   +72*gm(2,2)*gm(2,3)*dgm01(2,3)+gm(2,2)*(144*gm(3,3)*dgm01(2,2)&
&   -36*gm(2,2)*dgm01(3,3)))+gm(1,2)*(72*gm(2,3)**2*dgm01(2,3)+288*gm(2,2)&
&   *gm(3,3)*dgm01(2,3)+gm(2,3)*(36*gm(3,3)*dgm01(2,2)+108*gm(2,2)&
&   *dgm01(3,3))))*dgm10(3,3))/24.d0
   cm(5,33)=((8*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)&
&   +12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(2,2)+4*(2*gm(3,3)*(54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,3)&
&   +2*(6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(2,2)+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(2,3)+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&   **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(3,3))&
&   /48.d0
   cm(6,33)=((8*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+12*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,2)+4*(2*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,2)+12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3))*dgm10(2,3)+2*(6*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,2)+12*(48*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(2,3)+2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3))&
&   *dgm10(3,3))/48.d0
   cm(7,33)=((8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   +12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3))*dgm10(2,2)&
&   +4*(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)+12*gm(2,2)&
&   *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)+216*gm(2,2)**2*gm(2,3)&
&   *dgm01(3,3))*dgm10(2,3)+2*gm(2,2)*(6*(54*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3))*dgm01(2,2)+432*gm(2,2)*gm(2,3)*dgm01(2,3)+72*gm(2,2)&
&   **2*dgm01(3,3))*dgm10(3,3))/144.d0
   cm(8,33)=(gm(3,3)*(288*gm(2,3)*gm(3,3)*dgm01(2,3)+12*(12*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3))*dgm01(3,3))*dgm10(2,2)+4*(144*gm(2,3)&
&   **2*gm(3,3)*dgm01(2,3)+288*gm(2,2)*gm(3,3)**2*dgm01(2,3)+36*gm(2,3)&
&   **3*dgm01(3,3)+gm(2,3)*gm(3,3)*(72*gm(3,3)*dgm01(2,2)+180*gm(2,2)&
&   *dgm01(3,3)))*dgm10(2,3)+2*(72*gm(2,3)**3*dgm01(2,3)+360*gm(2,2)&
&   *gm(2,3)*gm(3,3)*dgm01(2,3)+gm(2,2)*gm(3,3)*(144*gm(3,3)*dgm01(2,2)&
&   -36*gm(2,2)*dgm01(3,3))+gm(2,3)**2*(72*gm(3,3)*dgm01(2,2)+108*gm(2,2)&
&   *dgm01(3,3)))*dgm10(3,3))/48.d0
   cm(9,33)=((8*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)&
&   +12*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(2,2)&
&   +4*(72*gm(2,3)**3*dgm01(2,3)+360*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(2,3)&
&   +gm(2,3)**2*(108*gm(3,3)*dgm01(2,2)+72*gm(2,2)*dgm01(3,3))+gm(2,2)&
&   *gm(3,3)*(-36*gm(3,3)*dgm01(2,2)+144*gm(2,2)*dgm01(3,3)))*dgm10(2,3)&
&   +2*(36*gm(2,3)**3*dgm01(2,2)+144*gm(2,2)*gm(2,3)**2*dgm01(2,3)&
&   +288*gm(2,2)**2*gm(3,3)*dgm01(2,3)+gm(2,2)*gm(2,3)*(180*gm(3,3)&
&   *dgm01(2,2)+72*gm(2,2)*dgm01(3,3)))*dgm10(3,3))/48.d0
   cm(10,33)=(gm(3,3)**2*(288*gm(3,3)*dgm01(2,3)+432*gm(2,3)*dgm01(3,3))&
&   *dgm10(2,2)+4*gm(3,3)*(72*gm(3,3)**2*dgm01(2,2)+324*gm(2,3)**2*dgm01(3,3)&
&   +gm(3,3)*(432*gm(2,3)*dgm01(2,3)-108*gm(2,2)*dgm01(3,3)))*dgm10(2,3)&
&   +2*(648*gm(2,3)**2*gm(3,3)*dgm01(2,3)-216*gm(2,2)*gm(3,3)**2*dgm01(2,3)&
&   +180*gm(2,3)**3*dgm01(3,3)+gm(2,3)*gm(3,3)*(216*gm(3,3)*dgm01(2,2)&
&   -108*gm(2,2)*dgm01(3,3)))*dgm10(3,3))/144.d0
   cm(1,34)=(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,2)+(8*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)&
&   +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm01(3,3))*dgm10(2,3)+(2*(90*gm(1,3)**3-54*gm(1,1)&
&   *gm(1,3)*gm(3,3))*dgm01(2,2)+12*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)+6*(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(3,3))&
&   *dgm10(3,3))/72.d0
   cm(2,34)=(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(2,2)+(8*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,3)+(2*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,2)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&   +12*gm(1,2)*gm(2,3))*dgm01(3,3))*dgm10(3,3))/24.d0
   cm(3,34)=(1440*gm(1,3)*gm(3,3)**2*dgm01(3,3)*dgm10(2,2)+20*gm(3,3)&
&   *(288*gm(1,3)*gm(3,3)*dgm01(2,3)+12*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&   *gm(3,3))*dgm01(3,3))*dgm10(2,3)+20*(288*gm(1,2)*gm(3,3)*(gm(3,3)&
&   *dgm01(2,3)+gm(2,3)*dgm01(3,3))+gm(1,3)*(72*gm(3,3)**2*dgm01(2,2)&
&   +36*gm(2,3)**2*dgm01(3,3)+gm(3,3)*(144*gm(2,3)*dgm01(2,3)-108*gm(2,2)&
&   *dgm01(3,3))))*dgm10(3,3))/480.d0
   cm(4,34)=(2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(3,3)*dgm10(2,2)+(8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(2,3)+12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(2,3)+(gm(1,3)*(72*gm(2,3)&
&   **2*dgm01(2,3)+288*gm(2,2)*gm(3,3)*dgm01(2,3)+gm(2,3)*(108*gm(3,3)&
&   *dgm01(2,2)+36*gm(2,2)*dgm01(3,3)))+gm(1,2)*(-36*gm(3,3)**2*dgm01(2,2)&
&   +36*gm(2,3)**2*dgm01(3,3)+gm(3,3)*(72*gm(2,3)*dgm01(2,3)+144*gm(2,2)&
&   *dgm01(3,3))))*dgm10(3,3))/12.d0
   cm(5,34)=(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,2)+(8*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)&
&   +12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(2,3)+(2*gm(3,3)*(54*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3))*dgm01(2,2)+12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)+6*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3))*dgm10(3,3))&
&   /24.d0
   cm(6,34)=(2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(2,2)+(8*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,3)+12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm01(3,3))*dgm10(2,3)+(2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)+12*(30*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)+6*(48*gm(1,2)&
&   *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *dgm01(3,3))*dgm10(3,3))/24.d0
   cm(7,34)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,2)+(8*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   +12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3))*dgm10(2,3)&
&   +(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)+12*gm(2,2)&
&   *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)+216*gm(2,2)**2*gm(2,3)&
&   *dgm01(3,3))*dgm10(3,3))/72.d0
   cm(8,34)=(1440*gm(2,3)*gm(3,3)**2*dgm01(3,3)*dgm10(2,2)+20*gm(3,3)&
&   *(288*gm(2,3)*gm(3,3)*dgm01(2,3)+12*(12*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3))*dgm01(3,3))*dgm10(2,3)+20*(144*gm(2,3)**2*gm(3,3)*dgm01(2,3)&
&   +288*gm(2,2)*gm(3,3)**2*dgm01(2,3)+36*gm(2,3)**3*dgm01(3,3)+gm(2,3)&
&   *gm(3,3)*(72*gm(3,3)*dgm01(2,2)+180*gm(2,2)*dgm01(3,3)))*dgm10(3,3))&
&   /480.d0
   cm(9,34)=(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,2)+(8*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)&
&   +12*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3))*dgm10(2,3)&
&   +(72*gm(2,3)**3*dgm01(2,3)+360*gm(2,2)*gm(2,3)*gm(3,3)*dgm01(2,3)&
&   +gm(2,3)**2*(108*gm(3,3)*dgm01(2,2)+72*gm(2,2)*dgm01(3,3))+gm(2,2)&
&   *gm(3,3)*(-36*gm(3,3)*dgm01(2,2)+144*gm(2,2)*dgm01(3,3)))*dgm10(3,3))&
&   /24.d0
   cm(10,34)=(gm(3,3)*(1440*gm(3,3)**2*dgm01(3,3)*dgm10(2,2)+20*gm(3,3)&
&   *(288*gm(3,3)*dgm01(2,3)+432*gm(2,3)*dgm01(3,3))*dgm10(2,3)+20*(72*gm(3,3)&
&   **2*dgm01(2,2)+324*gm(2,3)**2*dgm01(3,3)+gm(3,3)*(432*gm(2,3)&
&   *dgm01(2,3)-108*gm(2,2)*dgm01(3,3)))*dgm10(3,3)))/1440.d0
   cm(1,35)=(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,3)+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)&
&   *dgm10(3,3)+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm01(3,3)*dgm10(3,3))/36.d0
   cm(2,35)=(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(2,3)+2*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *dgm01(2,3)*dgm10(3,3)+3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&   *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3))/12.d0
   cm(3,35)=(gm(3,3)*(8640*gm(1,2)*gm(3,3)*dgm01(3,3)*dgm10(3,3)&
&   +gm(1,3)*(4320*gm(2,3)*dgm01(3,3)*dgm10(3,3)+8640*gm(3,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))))/1440.d0
   cm(4,35)=(2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&   *dgm01(3,3)*dgm10(2,3)+2*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&   *gm(3,3))*dgm01(2,3)*dgm10(3,3)+3*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3))&
&   /6.d0
   cm(5,35)=(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,3)+2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)&
&   *dgm10(3,3)+3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(3,3))/12.d0
   cm(6,35)=(2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(2,3)+2*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *dgm01(2,3)*dgm10(3,3)+3*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(3,3))/12.d0
   cm(7,35)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,3)+2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&   *dgm10(3,3)+3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(3,3))/36.d0
   cm(8,35)=(gm(3,3)*(8640*gm(2,3)*gm(3,3)*dgm01(3,3)*dgm10(2,3)&
&   +8640*gm(2,3)*gm(3,3)*dgm01(2,3)*dgm10(3,3)+360*(12*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3))*dgm01(3,3)*dgm10(3,3)))/1440.d0
   cm(9,35)=(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(2,3)+2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,3)&
&   *dgm10(3,3)+3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(3,3))/12.d0
   cm(10,35)=gm(3,3)**2*(3*gm(2,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
   cm(1,36)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(3,3))/36.d0
   cm(2,36)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm01(3,3)*dgm10(3,3))/12.d0
   cm(3,36)=3*gm(1,3)*gm(3,3)**2*dgm01(3,3)*dgm10(3,3)
   cm(4,36)=(gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(3,3))/6.d0
   cm(5,36)=(gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)&
&   *dgm10(3,3))/12.d0
   cm(6,36)=((90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(3,3))/12.d0
   cm(7,36)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&   *dgm10(3,3))/36.d0
   cm(8,36)=3*gm(2,3)*gm(3,3)**2*dgm01(3,3)*dgm10(3,3)
   cm(9,36)=(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&   *dgm10(3,3))/12.d0
   cm(10,36)=gm(3,3)**3*dgm01(3,3)*dgm10(3,3)
 end if
!
!contraction to output scalar
!
 e2nl=0.d0
 do jj=1,((rank+5)*(rank+6))/2
   tmp(:)=0.d0
   do ii=1,((rank+1)*(rank+2))/2
     tmp(:)=tmp(:)+aa(:,ii)*cm(ii,jj)
   end do
   e2nl=e2nl+tmp(1)*bb(1,jj)+tmp(2)*bb(2,jj)
 end do
!factor of 2 multiplied in along with dropping call to conjugate contraction
 e2nl=0.5d0*e2nl

 ABI_FREE(cm)

end subroutine contstr21
!!***

end module m_contstr21
!!***
