// ---------------------------------------------------------------------------
// - Blob.cpp                                                                -
// - afnix:csm service - registered blob class implementation                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2019 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Blob.hpp"
#include "Csmsid.hxx"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the blob request plist
  static const String PN_BLB_TYPE = "PN-BLB-TYPE";
  static const String PI_BLB_TYPE = "BLOB TYPE";
  static const String PN_BLB_XRID = "PN-BLB-XRID";
  static const String PI_BLB_XRID = "BLOB RID";
  // the performance plist
  static const String PN_PRF_RSTM = "PN-PRF-RSTM";
  static const String PI_PRF_RSTM = "BLOB RDSTREAM COUNTER";
  static const String PN_PRF_WSTM = "PN-PRF-WSTM";
  static const String PI_PRF_WSTM = "BLOB WRSTREAM COUNTER";
  static const String PN_PRF_HEAD = "PN-PRF-HEAD";
  static const String PI_PRF_HEAD = "BLOB GETHEAD COUNTER";
  static const String PN_PRF_PLST = "PN-PRF-PLST";
  static const String PI_PRF_PLST = "BLOB GETPLST COUNTER";
  
  // the performance counters
  struct s_perf {
    // the rdstream counter
    long d_rstm;
    // the wrstream counter
    long d_wstm;
    // the gethead counter
    long d_head;
    // the getplst counter
    long d_plst;
    s_perf (void) {
      d_rstm = 0L;
      d_wstm = 0L;
      d_head = 0L;
      d_plst = 0L;
    }
    // get a performance plist
    Plist getplst (void) const {
      Plist result;
      result.add (PN_PRF_RSTM, PI_PRF_RSTM, (t_long) d_rstm);
      result.add (PN_PRF_WSTM, PI_PRF_WSTM, (t_long) d_wstm);
      result.add (PN_PRF_HEAD, PI_PRF_HEAD, (t_long) d_head);
      result.add (PN_PRF_PLST, PI_PRF_PLST, (t_long) d_plst);
      return result;
    }
  };

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default blob

  Blob::Blob (void) {
    d_rid.clear ();
    p_perf = new s_perf;
  }

  // create an blob by name

  Blob::Blob (const String& name) : Part (name) {
    d_rid.clear ();
    p_perf = new s_perf;
  }

  // create an blob by name and info

  Blob::Blob (const String& name, const String& info) : Part (name, info) {
    d_rid.clear ();
    p_perf = new s_perf;
  }

  // create an blob by rid, name and info

  Blob::Blob (const String& rid,
	      const String& name, const String& info) : Part (name, info) {
    setrid (rid);
    p_perf = new s_perf;
  }

  // copy construct this blob

  Blob::Blob (const Blob& that) {
    that.rdlock ();
    try {
      // copy base part
      Part::operator = (that);
      // copy locally
      d_rid = that.d_rid;
      // allocate locally
      p_perf = new s_perf;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // destroy this blob

  Blob::~Blob (void) {
    delete p_perf;
  }
  
  // assign an blob to this one

  Blob& Blob::operator = (const Blob& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      // copy base part
      Part::operator = (that);
      // copy locally
      d_rid = that.d_rid;
      // allocate locally
      p_perf = new s_perf;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // return the blob class name

  String Blob::repr (void) const {
    return "Blob";
  }

  // return a clone of this object

  Object* Blob::clone (void) const {
    return new Blob (*this);
  }

  // return the serial did

  t_word Blob::getdid (void) const {
    return SRL_DEOD_CSM;
  }

  // return the serial sid

  t_word Blob::getsid (void) const {
    return SRL_BLOB_SID;
  }
  
  // serialize this blob

  void Blob::wrstream (OutputStream& os) const {
    rdlock ();
    try {
      // serialize the part
      Part::wrstream (os);
      // serialize locally
      d_rid.wrstream (os);
      // update counters
      p_perf->d_wstm++;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this blob

  void Blob::rdstream (InputStream& is) {
    wrlock ();
    try {
      // deserialize the part
      Part::rdstream (is);
      // deserialize locally
      d_rid.rdstream (is);
      // update counters
      p_perf->d_rstm++;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // request a blob header

  Plist Blob::gethead (void) const {
    rdlock ();
    try {
      Plist result = Part::gethead ();
      result.add (PN_BLB_TYPE, PI_BLB_TYPE, repr().toupper());
      result.add (PN_BLB_XRID, PI_BLB_XRID, d_rid);
      // add performance counters
      result+= p_perf->getplst();
      // update counters
      p_perf->d_head++;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // request an information list

  Plist Blob::getplst (void) const {
    rdlock ();
    try {
      Plist result = Part::getplst ();
      result.add (PN_BLB_TYPE, PI_BLB_TYPE, repr().toupper());
      result.add (PN_BLB_XRID, PI_BLB_XRID, d_rid);
      // update counters
      p_perf->d_plst++;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // validate a blob registration id

  bool Blob::isrid (void) const {
    rdlock ();
    try {
      bool result = d_rid.isnil () ? false : true;
      unlock ();
      return result;
    } catch  (...) {
      unlock ();
      throw;
    }
  }
  
  // validate a blob registration id

  bool Blob::isrid (const String& rid) const {
    rdlock ();
    try {
      bool result = (d_rid == rid);
      unlock ();
      return result;
    } catch  (...) {
      unlock ();
      throw;
    }
  }

  // set the blob registration id

  void Blob::setrid (const String& rid) {
    wrlock ();
    try {
      d_rid = rid;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // get the blob registration id

  String Blob::getrid (void) const {
    rdlock ();
    try {
      String result = d_rid;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 3;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_ISRIDP  = zone.intern ("rid-p");
  static const long QUARK_SETRID  = zone.intern ("set-rid");
  static const long QUARK_GETRID  = zone.intern ("get-rid");

  // create a new object in a generic way

  Object* Blob::mknew (Vector* argv) {
    long argc = (argv == nullptr) ? 0 : argv->length ();
    // create a default blob
    if (argc == 0) return new Blob;
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new Blob (name);
    }
    // check for 2 arguments
    if (argc == 2) {
      String name = argv->getstring (0);
      String info = argv->getstring (1);
      return new Blob (name, info);
    }
    // check for 3 arguments
    if (argc == 3) {
      String  rid = argv->getstring (0);
      String name = argv->getstring (1);
      String info = argv->getstring (2);
      return new Blob (rid, name, info);
    }
    throw Exception ("argument-error",
                     "too many argument with blob constructor");
  }

  // return true if the given quark is defined

  bool Blob::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? Part::isquark (quark, hflg) : false;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* Blob::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nullptr) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETRID)  return new String (getrid  ());
      if (quark == QUARK_ISRIDP)  return new Boolean (isrid  ());
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_ISRIDP) {
	String rid = argv->getstring (0);
	return new Boolean (isrid (rid));
      }
      if (quark == QUARK_SETRID) {
	String rid = argv->getstring (0);
	setrid (rid);
	return nullptr;
      }
    }
    // call the part method
    return Part::apply (robj, nset, quark, argv);
  }
}
