/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "SmoothFilter.h"

#include <QString>
#include <QMessageBox>
#include <QTextStream>
#include <Application.h>

#include <vtkSmoothPolyDataFilter.h>
#include <vtkSmartPointer.h>

using namespace camitk;


// --------------- Constructor -------------------
SmoothFilter::SmoothFilter(ActionExtension * extension) : Action(extension) {
    // Setting name, description and input component 
    setName("SmoothFilter");
    setDescription("This action is a filter that adjusts point coordinates using Laplacian smoothing. It makes the cells better shaped and the vertices more evenly distributed.");
    setComponent("MeshComponent");

    // Setting classification family and tags 
	setFamily("Mesh Processing");
	addTag("SMO");
	
	// Setting the action's parameters
	setProperty("boundary smoothing", QVariant(true));
	setProperty("edge angle", QVariant(15));
	setProperty("edge smoothing", QVariant(false));
	setProperty("feature angle", QVariant(45));
	setProperty("iteration", QVariant(20));
	setProperty("relaxation", QVariant(0.1));

}

// --------------- destructor -------------------
SmoothFilter::~SmoothFilter() {
    // Do not do anything yet. 
	// Delete stuff if you create stuff 
	// (except if you use smart pointers of course !!)
}

// --------------- apply -------------------
Action::ApplyStatus SmoothFilter::apply() {
    foreach (Component *comp, getTargets()) {
        MeshComponent * input = dynamic_cast<MeshComponent *> ( comp );
        process(input);
    }
    return SUCCESS;
}

void SmoothFilter::process(MeshComponent * comp) {
    // Get the parameters
	bool boundarySmoothing = property("boundary smoothing").toBool();
	double edgeAngle = property("edge angle").toDouble();
	bool edgeSmoothing = property("edge smoothing").toBool();
	double featureAngle = property("feature angle").toDouble();
	int iteration = property("iteration").toInt();
	double relaxation = property("relaxation").toDouble();
	
	vtkSmartPointer<vtkPolyData> poly = vtkPolyData::SafeDownCast(comp->getPointSet());

	// create the filter and set the polydata
	vtkSmartPointer<vtkSmoothPolyDataFilter> smoothFilter = vtkSmartPointer<vtkSmoothPolyDataFilter>::New();
	smoothFilter->SetInput(poly);
	// set the number of iterations
	smoothFilter->SetNumberOfIterations(iteration);
	// set the relaxation factor
	smoothFilter->SetRelaxationFactor(relaxation);
	// a small relaxation and a  large number of iterations are more stable than a large relaxation factor and a small number of iterations

	// turn on/off the smoothing of vertices on the boundary of the mesh. 
	if (!boundarySmoothing)
		smoothFilter->SetBoundarySmoothing(0);
	// set angles
	smoothFilter->SetFeatureAngle(featureAngle);
	smoothFilter->SetEdgeAngle(edgeAngle);
	// turn on/off smoothing along sharp interior edges. 
	if (edgeSmoothing)
		smoothFilter->SetFeatureEdgeSmoothing(1);
	smoothFilter->Update();
	
	comp->setPointSet(vtkPointSet::SafeDownCast(smoothFilter->GetOutput()));
	comp->refresh();

}


