/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


#ifndef INTERACTIVE_VIEWER_H
#define INTERACTIVE_VIEWER_H

// -- Core stuff
#include "CamiTKAPI.h"
#include "Component.h"
#include "Viewer.h"

//-- QT stuff
#include <QFrame>
#include <QPushButton>

//-- vtk stuff
#include <vtkType.h>
#include <vtkSmartPointer.h>

//-- vtk stuff classes
class vtkActor;
class vtkPicker;
class vtkProp;
class vtkObject;
class vtkCamera;

namespace camitk {
// -- Core stuff classes
class RendererWidget;
class SliderSpinBoxWidget;
class GeometricObject;
class InterfaceGeometry;
class InterfaceBitMap;
class InteractiveViewer;

typedef vtkSmartPointer<vtkCamera> vtkSmartPointerCamera;

/// InteractiveViewerFrame is just a QFrame that delegates all key events to its InteractiveViewer
class InteractiveViewerFrame : public QFrame {
public:
    /// default constructor
    InteractiveViewerFrame ( QWidget* parent, InteractiveViewer* s3D ) : QFrame ( parent ), myInteractiveViewer ( s3D ) {};

    /// Handle keyboard events in the scene frame, just send everything to InteractiveViewer!
    void keyPressEvent ( QKeyEvent* );

protected:
    InteractiveViewer *myInteractiveViewer;
};

/**
  * InteractiveViewer is used to view 3D objects and slices (anything that provides either a InterfaceBitMap
  * or a InterfaceGeometry).
  *
  * It contains a renderer (class RendererWidget) that combines VTK and QT.
  * The RendererWidget instance manage all the display at the VTK level.
  * InteractiveViewer delegates all pure VTK level stuff to the renderer.
  * All things that needs InterfaceBitMap/InterfaceGeometry interaction/knowledge/access are
  * manage at this level.
  * The keyboard events are all managed in InteractiveViewer as well.
  * Keyboard/Mouse interactions: check "what's this?" on the scene 3D to get all interaction shortcuts.
  *
  * There are five default InteractiveViewer "singleton" instances named:
  * - "3DViewer" classical 3D geometry viewer (GEOMETRY_VIEWER)
  * - "axialViewer" to display axial slice (SLICE_VIEWER)
  * - "coronalViewer" to display axial slice (SLICE_VIEWER)
  * - "sagittalViewer" to display sagittal slice (SLICE_VIEWER)
  * - "arbitraryViewer" to display an arbitrary slice (SLICE_VIEWER)
  * They can be accessed using the corresponding get methods.
  * This is why there are no public constructor for this class.
  * To create a new custom instance of this class, use the getNewViewer(..) method.
  * Of course you can create as many InteractiveViewer instance as you like, but if you try to create
  * a InteractiveViewer with a name that match one of the default InteractiveViewer name, getNewViewer(..) automatically
  * will return the corresponding default InteractiveViewer instance.
  * The list of InteractiveViewer instances is kept in this class as well.
  *
  * InteractiveViewer class is completely independant from MainWindow application skeleton (and please keep it that way!)
  *
  * InteractiveViewer manages a list of cameras. Each camera has a name. Use getCamera(QString) to
  * create a new camera, or get the default camera and setActiveCamera(QString) to change the active
  * camera.
  * The default camera is called "default".
  *
  * InteractiveViewer manages picking session.
  * A picking session starts when the control key is pressed and the left mouse button is clicked and ends when
  * the mouse button is released. A picking session is a nice/quick way to do a lot of picking by
  * simply moving the mouse, without the need to click for each picking.
  * At the start of a picking session, the picking action is determined: it is either selection
  * or unselection. If one of the first picked components was already selected, then the user wants to unselect,
  * therefore picking action is "unselection", and all subsequently calls to pickPoint(..) or pickCell(..)
  * will use pickingIsSelecting=false (2nd parameter of the method). If the first picked component was not
  * selected, then the picking session is going to select any picked component.
  * @see InterfaceGeometry::pickPoint() InterfaceGeometry::pickCell()
  *
  * <hr>
  *
  * The following help is for InteractiveViewer/RendererWidget developers, please read if you want/need
  * to change anything in one of this two class. It should help you in your coding decisions (hopefully!).
  *
  * 1. Keyboard shortcut policy: for keyboard shortcuts do not use the Ctrl key modifier for letters/digit (e.g Alt-P is ok,
  * Ctrl+Arrows is ok, but Ctrl-P is not).
  * Ctrl-Letter/Digit shortcuts should only be used for the application shortcuts.
  *
  * 2. That's it for now!
  * 
  */

class CAMITK_API InteractiveViewer : public Viewer {
    Q_OBJECT
    Q_ENUMS ( HighlightMode RendererWidget::ControlMode RendererWidget::CameraOrientation ) // so that it can be used in property editor
    Q_PROPERTY ( HighlightMode highlightMode READ getHighlightMode WRITE setHighlightMode )
    Q_PROPERTY ( QColor backgroundColor READ getBackgroundColor WRITE setBackgroundColor )
    Q_PROPERTY ( bool gradientBackground READ getGradientBackground WRITE setGradientBackground )
    Q_PROPERTY ( bool linesAsTubes READ getLinesAsTubes WRITE setLinesAsTubes )
    Q_PROPERTY ( bool backfaceCulling READ getBackfaceCulling WRITE setBackfaceCulling )
    Q_PROPERTY ( double pointSize READ getPointSize WRITE setPointSize )

public:
    /// \enum ViewerType there is two possibilities: this InteractiveViewer is used to display slices or geometry
    enum ViewerType {
        SLICE_VIEWER,     ///< display slices (the view is blocked in 2D and the slider is available)
        GEOMETRY_VIEWER  ///< display 3D stuff (geometry, etc...)
    };

    /** \enum PickingMode Different kind of picking must be available: pixel in slice, a point, a cell, ...
     *    So far, only pixel picking is implemented. */
    enum PickingMode {
        PIXEL_PICKING,  ///< pick a pixel on a Slice
        POINT_PICKING,  ///< pick a point in the VTK representation of an Geometry
        CELL_PICKING,   ///< pick a cell in the VTK representation of an Geometry
        NO_PICKING     ///< no picking possible
    };

    /** \enum HighlightMode describes the current mode of display. It is usefull to change the
      * way the currently selected Components look compared to the unselected ones.
      * In the default mode a Component is not highlighted, not shaded and not hidden.
      */
    enum HighlightMode {
        OFF,            ///< both selected and non-selected Components are in default mode
        SELECTION,      ///< the selected Components are in default mode, the non-selected Components are shaded
        SELECTION_ONLY ///< the selected Components are in default mode, the non-selected are hidden
    };

    /** @name General/Singleton
     *  There is no public constructor for this class, but 5 default instances.
     */
    /// @{
    /** Use this method to create a new custom instance of this class.
      * Of course you can create as many InteractiveViewer instance as you like, but if you try to create
      * a InteractiveViewer with a name that match one of the default InteractiveViewer name, getNewViewer(..) automatically
      * will return the corresponding default InteractiveViewer instance.
      */
    static InteractiveViewer * getNewViewer ( QString , ViewerType type );

    /// get a viewer by its name (return NULL if no instance has the given name, use getNewViewer to create one viewer)
    static InteractiveViewer *getViewer ( QString );

    /// get the 3D viewer default InteractiveViewer (provided for convenience, equivalent to getViewer("3DViewer")
    static InteractiveViewer * get3DViewer();

    /// get the axial default InteractiveViewer (provided for convenience, equivalent to getViewer("axialViewer")
    static InteractiveViewer * getAxialViewer();

    /// get the coronal default InteractiveViewer (provided for convenience, equivalent to getViewer("coronalViewer")
    static InteractiveViewer * getCoronalViewer();

    /// get the sagittal default InteractiveViewer (provided for convenience, equivalent to getViewer("sagittalViewer")
    static InteractiveViewer * getSagittalViewer();

    /// get the arbitrary default InteractiveViewer (provided for convenience, equivalent to getViewer("arbitraryViewer")
    static InteractiveViewer * getArbitraryViewer();

    /** Destructor */
    virtual ~InteractiveViewer();

    /// get the scene name
    QString getName() const;
    /// @}

    /** @name Inherited from Viewer
      */
    ///@{
    /// returns the number of Component that are displayed by this viewer
    virtual unsigned int numberOfViewedComponent();

    /// Refresh the display.
    virtual void refresh ( Viewer *whoIsAsking = NULL );

    /// get the InteractiveViewer widget (QTreeWidget). @param parent the parent widget for the viewer widget
    virtual QWidget * getWidget ( QWidget * parent );

    /// get the  InteractiveViewer propertyObject (only non-null for GEOMETRY_VIEWER)
    virtual QObject * getPropertyObject();

    /// get the explorer menu
    virtual QMenu * getMenu();

    /// get the viewer toolbar
    virtual QToolBar * getToolBar();
    ///@}

    /** @name Refresh/screenshot
     */
    /// @{

    /// just refresh the renderer
    void refreshRenderer();

    /** Reset scene camera. Use a trick (when this is a SLICE_VIEWER) for scaling up to max size in the viewer */
    void resetCamera();

    /// Set the active virtual camera
    void setActiveCamera ( QString cameraName );

    /** get a camera by its name, creates one if it does not exist already.
      * This method does not activate the given camera, please use setActiveCamera for this.
      */
    vtkSmartPointer<vtkCamera> getCamera ( QString cameraName = "default" );
    
    /// call this method to take a screenshot using the given filename (the extension must be a supported format extension, see class RendererWindow)
    void screenshot ( QString );

public slots:
    /// call this method to take a screenshot in various format and write the resulting image to a file
    void screenshot();    
    /// @}

    /** @name Viewing/Interaction Property
     */
    ///@{

public:
    /// set gradient background on/off
    virtual void setGradientBackground ( bool );

    /// get the current background gradient mode.
    bool getGradientBackground() const;

    /// set background color
    virtual void setBackgroundColor ( QColor );

    /// get the background color
    QColor getBackgroundColor() const;

public slots:
    /// set the backface culling mode (default is true).
    void setBackfaceCulling ( bool );

public:
    /// get the backface current culling mode.
    bool getBackfaceCulling() const;

public slots:
    /** Update the visualization of lines (for all the InterfaceGeometry of the scene).
      *
      * \note it is only possible to transform lines to tubes if
      * you build an Geometry using lines.
      * 
      * @param tubes if true, then the lines have to be displayed as tube
      */
    void setLinesAsTubes ( bool tubes );

public:
    /// Handle keyboard events in the scene, let to the parent widget if not processed here. This method is a friend of class InteractiveViewerFrame
    void keyPressEvent ( QKeyEvent* e );

    /// get the boolean indicating if the lines are currently set to be displayed as tubes.
    bool getLinesAsTubes() const;

    /** update the visualization of points (for all object3D in the scene).
      *
      * This property acts directly on the actor, thus it can be changed outside the InterfaceGeometry,
      * thus the InterfaceGeometry should not be aware of its value (no method should be concerned with
      * this property)
      *
      */
    void setPointSize ( double size );

    /// get the current point size value
    double getPointSize() const;

    /// Set the current highlighting mode.
    virtual void setHighlightMode ( InteractiveViewer::HighlightMode );

    /// Return the current highlighting mode.
    InteractiveViewer::HighlightMode getHighlightMode() const;

    /// return interactiveViewer RendererWidget
    inline RendererWidget * getRendererWidget() {
        return rendererWidget;
    }


    ///@}

    /** @name Misc
    */
    /// @{
    /// set the color scale in the viewport, use setColorScaleMinMax / setColorScaleTitle to change the displayed values.
    void setColorScale ( bool );

    /// get the current value of the color scale property.
    bool getColorScale() const;

    /** set the min and max values.
     *  Automatically turns setColorScale to true.
     *  @param m minimum value (blue)
     *  @param M maximum value (red)
     */
    void setColorScaleMinMax ( double m, double M );

    /** set the color scale title.
     *  @param t title of the color scale
     */
    void setColorScaleTitle ( QString t );

    /** Init the picker with a given picking mode. */
    void initPicking ( PickingMode );

    /// Compute the bounding box of the selected elements [xmin,xmax, ymin,ymax, zmin,zmax]
    void getBoundsOfSelected ( double bound[6] );

    /// Compute the bounding box of all displayed Component
    void getBounds ( double bound[6] );

    /// set the slice viewer side bar+screenshot button visibility
    void setSideFrameVisible(bool);
    ///@}

public slots:

    /** Slot activated when something was picked in the 3D interactor.
     * Process the picker accordingly to the current pickingMode state.
     * It is connected to the RendererWidget actorPicked signal
     */
    void actorPicked ( vtkSmartPointer<vtkPicker> );

    /** Slot called when the InteractiveViewer slider has been changed. If there is a InterfaceBitMap in the scene,
     *    set the slice index with the new slider value. */
    void sliderChanged ( int );

    /** Slot called when the InteractiveViewer x angle update has been changed. */
    void xAngleChanged ( double angle );

    /** Slot called when the InteractiveViewer y angle update has been changed. */
    void yAngleChanged ( double angle );

    /** Slot called when the InteractiveViewer z angle update has been changed. */
    void zAngleChanged ( double angle );

    /// show/hide the copyright in 3D
    void toggleCopyright ( bool );

protected:
    /** @name General/Singleton
     */
    /// @{
    /** Construtor.
     *  @param name    the name of the scene is mandatory, it is used as an identifier (e.g. in MedicalImageViewer)
     *  @param type    type of the InteractiveViewer, depending on which it will behave as slice viewer, i.e. with no rotation interactions are possible, or 3D viewer
     */
    InteractiveViewer ( QString & name, ViewerType type );

    /// the map containing all the InteractiveViewer instances
    static QMap<QString, InteractiveViewer*> viewers;

    /// the QString array containing all InteractiveViewer instance default names
    static QString defaultNames[5];

    /// @}


    /** @name Display properties
      *
      * Properties that can be managed without the knowledge/intervention of the InterfaceGeometry:
      * - pointSize (the user preferred value is stored here, the access to the actor needs InterfaceGeometry knowledge)
      * - interpolation (toggle, this is a boolean state, kept by each vtkImageActor interpolate property)
      *
      * Properties that need to be managed by the InterfaceGeometry itself (not boolean state managed somewhere
      * by vtk, not integer/float value manage in InteractiveViewer as a user-preference)
      * - linesAsTubes (the InterfaceGeometry needs to add/manage a filter before the mapper)
      */
    /// @{

    /// initialize the property object and state using the user settings (user preferences system files .config/.ini)
    void initSettings();

    /// type of InteractiveViewer (display slice or geometry)
    ViewerType myType;

    /// Are lines currently displayed as tubes (the actors have to add a vtkTubeFilter between the source and the mapper)
    bool linesAsTubes;

    /// for InterfaceBitMap, toggle the interpolation mode (intern method, not a property because it can only be modified by the keyboard interaction)
    void toggleInterpolation();

    /// keep the value of the hightlight mode
    HighlightMode highlightMode;

    /// Update the display of the given Component, according to its selection state and the current HighlightMode.
    void updateSelectionDisplay ( Component * );

    /// the map containing all the actors in the InteractiveViewer
    QMultiMap<Component*, vtkSmartPointer<vtkProp> > actorMap;

    /// add the given actor of the given Component to the renderer and insert it in the map
    void addActor ( Component *, vtkSmartPointer<vtkProp> );

    /// remove all the given Component actors from the renderer and delete comp from the map
    void removeAllActors ( Component * );

    /// number of top-level component that are currently displayed
    unsigned int displayedTopLevelComponents;

    /// all the available camera
    QMap<QString, vtkSmartPointerCamera> cameraMap;
    ///@}

    /** @name Widget/Action management
      */
    /// @{
    /// The 3D scene itself, wrapping VTK render window, renderer and interactor in a single Qt widget
    RendererWidget * rendererWidget;

    /** Slider used to control the slice index in a InteractiveViewer. This slider is visible only when
     *    the scene a 2D viewer (see constructor). */
    SliderSpinBoxWidget * sliceSlider;

    /// the InteractiveViewer frame
    InteractiveViewerFrame * frame;

    /// the right side frame (this is where the slider and screenshot buttons are shown)
    QFrame *sideFrame;
    
    /// the InteractiveViewerFrame keyPressEvent is a good friend of InteractiveViewer
    friend void InteractiveViewerFrame::keyPressEvent ( QKeyEvent* e );

    /// the QMenu for the InteractiveViewer
    QMenu* viewerMenu;

    /// the QToolBar for the InteractiveViewer
    QToolBar *viewerToolBar;

    /// init all the actions (called only once in the getWidget() method)
    void initActions();

    /// update the viewer menu depending on the selection,...
    void updateActions();

    /// Screenshot
    QAction *screenshotAction;

    /// Rendering
    QMenu *renderingMenu;
    QAction *surfaceAction;
    QAction *wireframeAction;
    QAction *pointsAction;
    QAction *colorAction;
    QAction *glyphAction;
    
    /// display mode
    QAction *highlightSelectionAction;
    QAction *highlightSelectionOnlyAction;
    QAction *highlightOffAction;

    /// to change the camera control mode
    QAction *controlModeTrackballAction;
    QAction *controlModeJoystickAction;

    /// to change the axes view mode
    QAction *cameraOrientationRightDownAction;
    QAction *cameraOrientationLeftUpAction;
    QAction *cameraOrientationRightUpAction;

    /// background color
    QAction *backgroundColorAction;

    /// button allows to display the Axes in the InteractiveViewer
    QAction *toggleAxesAction;

    /// button to remove the copyright
    QAction *toggleCopyrightAction;

    /// button allows to display the labels of the object3D
    QAction *toggleLabelAction;

    /// button allows to display the lines as tubes (the lines are to be in vtkPolyData)
    QAction *toggleLinesAsTubesAction;

    /// back face culling
    QAction *toggleBackfaceCullingAction;

    /// action of the picking menu
    QAction *pickPointAction;
    QAction *pickCellAction;
    //TODO QAction *pickRegionAction; /// to pick a region by selecting top left / bottom right corners
    
    ///@}

    /** @name Picking management
      */
    /// @{
    /** list of Component that are currently picked, correctly displayed in the InteractiveViewer,
      * but for speed optimization that are not yet selected in the explorer.
      * They will all be selected in the explorer when the user release the mouse button.
      */
    std::vector <Component *> pickedComponent;

    /** Current picking mode, NO_PICKING be default. */
    PickingMode pickingMode;

    /// picking effect while mouse button is kept pressed is selecting (depends on the selection state of the first component picked)
    bool pickingEffectIsSelecting;

    /// was the picking effect updated (it has to be updated with the first picking for a given button down session)
    bool pickingEffectUpdated;
    ///@}


    /** @name Help Whats This Utility
      */
    /// @{
    /// The what's html string
    QString whatsThis;

    /// are we currently in a odd table line
    bool oddWhatsThis;

    /// initialize the what's this html string
    void initWhatsThis();

    /// start a table (section) in the what's this message
    void startWhatsThisSection ( const QString & title = "" );

    /// end a table (section) in the what's this message
    void endWhatsThisSection();

    /// add an item (row) in the the what's this message (to describe a shortcut)
    void addWhatsThisItem ( const QString & key, const QString & description );
    /// @}

protected slots:
    /** @name All the slots called by the menu actions
      */
    ///@{
    void renderingActorsChanged();

    void highlightModeChanged ( QAction *selectedAction );

    void cameraOrientationChanged ( QAction *selectedAction );

    void viewControlModeChanged ( QAction* );

    void backgroundColor();

    void toggleAxes ( bool );

    void pickingModeChanged ( QAction* );

    void rightClick();

    /// if true currently selected Components label will have their label on (shown)
    void setLabel (bool);

    void setGlyph(bool);
    
    ///@}
};

}


#endif

//**************************************************************************
