/*
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
*/

#include <QtCore>
#include <QKeyEvent>
#include <QNetworkConfigurationManager>

#include "events.h"
#include "../cordova.h"

Events::Events(Cordova *cordova) : CPlugin(cordova) {
    m_previousPercent = 100;
    m_batteryInfo = new QBatteryInfo(this);

    connect(m_batteryInfo, SIGNAL(remainingCapacityChanged(int,int)), this, SLOT(remainingCapacityChanged(int,int)));
    connect(m_batteryInfo, SIGNAL(chargerTypeChanged(QBatteryInfo::ChargerType)), this, SLOT(chargerTypeChanged(QBatteryInfo::ChargerType)));

    if (m_cordova->topLevelEventsReceiver())
        m_cordova->topLevelEventsReceiver()->installEventFilter(this);

    m_networkConfigurationManager = new QNetworkConfigurationManager(this);
    connect(m_networkConfigurationManager, SIGNAL(onlineStateChanged(bool)), this, SLOT(onlineStatusChanged(bool)));
}

bool Events::eventFilter(QObject *obj, QEvent *ev) {
    if (obj == m_cordova->topLevelEventsReceiver()) {
        if (ev->type() == QEvent::KeyRelease) {
            QKeyEvent *keyEvent = dynamic_cast<QKeyEvent *>(ev);
            if (!keyEvent)
                return false;
            switch (keyEvent->key()) {
            case Qt::Key_Menu:
                m_cordova->execJS(QString("Cordova.menuKeyPressed();"));
                break;
            case Qt::Key_Back:
                m_cordova->execJS(QString("Cordova.backKeyPressed();"));
                break;
            case Qt::Key_Search:
                m_cordova->execJS(QString("Cordova.searchKeyPressed();"));
                break;
            case Qt::Key_Call:
                m_cordova->execJS(QString("Cordova.callKeyPressed();"));
                break;
            case Qt::Key_Hangup:
                m_cordova->execJS(QString("Cordova.hangupKeyPressed();"));
                break;
            default:
                break;
            }
        } else if (ev->type() == QEvent::KeyPress) {
            QKeyEvent *keyEvent = dynamic_cast<QKeyEvent *>(ev);
            if (!keyEvent)
                return false;
            switch (keyEvent->key()) {
            case Qt::Key_VolumeUp:
                m_cordova->execJS(QString("Cordova.volumeUpKeyPressed();"));
                break;
            case Qt::Key_VolumeDown:
                m_cordova->execJS(QString("Cordova.volumeDownKeyPressed();"));
                break;
            default:
                break;
            }
        } else if (ev->type() == QEvent::WindowActivate) {
            m_cordova->execJS(QString("Cordova.resumeOccured();"));
        } else if (ev->type() == QEvent::WindowDeactivate) {
            m_cordova->execJS(QString("Cordova.pauseOccured();"));
        }
    }
    return false;
}

void Events::remainingCapacityChanged(int battery, int capacity) {
    int newPercent;

    if (m_batteryInfo->maximumCapacity(battery) < 1)
        newPercent = 0;
    newPercent = capacity/((double)m_batteryInfo->maximumCapacity(battery)/100);
    if (m_previousPercent == newPercent)
        return;
    m_previousPercent = newPercent;

    bool onBattery = m_batteryInfo->chargerType() == QBatteryInfo::UnknownCharger ||
            m_batteryInfo->chargerType() == QBatteryInfo::VariableCurrentCharger;

    m_cordova->execJS(QString("Cordova.batteryStatusChanged(%1, %2, false);")
                      .arg(m_previousPercent)
                      .arg(!onBattery));
}

void Events::chargerTypeChanged(QBatteryInfo::ChargerType type) {
    Q_UNUSED(type);
    qDebug() << Q_FUNC_INFO;

    bool isPlugged = m_batteryInfo->chargerType() == QBatteryInfo::UnknownCharger ||
            m_batteryInfo->chargerType() == QBatteryInfo::VariableCurrentCharger;

    m_cordova->execJS(QString("Cordova.batteryStatusChanged(%1, %2, true);")
                      .arg(m_previousPercent)
                      .arg(isPlugged));
}

void Events::onlineStatusChanged(bool isOnline) {
    if (isOnline)
        m_cordova->execJS(QString("Cordova.onlineOccured();"));
    else
        m_cordova->execJS(QString("Cordova.offlineOccured();"));
}
