/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file address-dialog.c
 * \brief Main address dialog
 */

#include <ffgtk.h>

/** Common widgets */
static GtkWidget *psAddressBook = NULL;
static struct sPerson *psSelectedPerson = NULL;
static GtkTreeStore *psStore = NULL;
static gboolean bChanged = FALSE;
static gboolean bSave = FALSE;
static gboolean bUpdate = FALSE;

/**
 * \brief Save address book
 * \return error code, 0=success else error
 */
gint SaveBook( void ) {
	struct sAddressBook *psBook = NULL;

	psBook = getDefaultBookPlugin( getActiveProfile() );
	if ( psBook && psBook -> SaveBook ) {
		return psBook -> SaveBook();
	}

	return -1;
}

/**
 * \brief Set scaled person image or "missing-image" to addressbook image widget
 * \param psPerson person information structure
 */
static void addrShowImage( struct sPerson *psPerson ) {
	GtkWidget *psImage = g_object_get_data( G_OBJECT( psAddressBook ), "image" );
	GdkPixbuf *psScaled = NULL;

	psScaled = getScaledImage( psPerson, 1 );
	if ( psScaled != NULL ) {
		gtk_image_set_from_pixbuf( GTK_IMAGE( psImage ), psScaled );
	} else {
		gtk_image_set_from_stock( GTK_IMAGE( psImage ), GTK_STOCK_MISSING_IMAGE, GTK_ICON_SIZE_DIALOG );
	}
}

/**
 * \brief Update person address information
 * \param psBook address book widget
 * \param psPerson person structure
 */
static void addrUpdateInformation( GtkWidget *psBook, struct sPerson *psPerson ) {
	GtkWidget *psTitle = g_object_get_data( G_OBJECT( psAddressBook ), "title" );
	GtkWidget *psFirstName = g_object_get_data( G_OBJECT( psAddressBook ), "first_name" );
	GtkWidget *psLastName = g_object_get_data( G_OBJECT( psAddressBook ), "last_name" );
	GtkWidget *psCompany = g_object_get_data( G_OBJECT( psAddressBook ), "company" );
	GtkWidget *psPrivatePhone = g_object_get_data( G_OBJECT( psAddressBook ), "private_phone" );
	GtkWidget *psPrivateFax = g_object_get_data( G_OBJECT( psAddressBook ), "private_fax" );
	GtkWidget *psPrivateMobile = g_object_get_data( G_OBJECT( psAddressBook ), "private_mobile" );
	GtkWidget *psPrivateStreet = g_object_get_data( G_OBJECT( psAddressBook ), "private_street" );
	GtkWidget *psPrivateZipCode = g_object_get_data( G_OBJECT( psAddressBook ), "private_zipcode" );
	GtkWidget *psPrivateCity = g_object_get_data( G_OBJECT( psAddressBook ), "private_city" );
	GtkWidget *psPrivateCountry = g_object_get_data( G_OBJECT( psAddressBook ), "private_country" );
	GtkWidget *psBusinessStreet = g_object_get_data( G_OBJECT( psAddressBook ), "business_street" );
	GtkWidget *psBusinessZipCode = g_object_get_data( G_OBJECT( psAddressBook ), "business_zipcode" );
	GtkWidget *psBusinessCity = g_object_get_data( G_OBJECT( psAddressBook ), "business_city" );
	GtkWidget *psBusinessCountry = g_object_get_data( G_OBJECT( psAddressBook ), "business_country" );
	GtkWidget *psBusinessPhone = g_object_get_data( G_OBJECT( psAddressBook ), "business_phone" );
	GtkWidget *psBusinessFax = g_object_get_data( G_OBJECT( psAddressBook ), "business_fax" );

	if ( psPerson == NULL ) {
		return;
	}

	bUpdate = TRUE;
	addrShowImage( psPerson );

	gtk_entry_set_text( GTK_ENTRY( psTitle ), psPerson -> pnTitle );
	gtk_entry_set_text( GTK_ENTRY( psFirstName ), psPerson -> pnFirstName );
	gtk_entry_set_text( GTK_ENTRY( psLastName ), psPerson -> pnLastName );
	gtk_entry_set_text( GTK_ENTRY( psPrivatePhone ), psPerson -> pnPrivatePhone );
	gtk_entry_set_text( GTK_ENTRY( psPrivateFax ), psPerson -> pnPrivateFax );
	gtk_entry_set_text( GTK_ENTRY( psPrivateMobile ), psPerson -> pnPrivateMobile );
	gtk_entry_set_text( GTK_ENTRY( psPrivateStreet ), psPerson -> pnPrivateStreet );
	gtk_entry_set_text( GTK_ENTRY( psPrivateZipCode ), psPerson -> pnPrivateZipCode );
	gtk_entry_set_text( GTK_ENTRY( psPrivateCity ), psPerson -> pnPrivateCity );
	gtk_entry_set_text( GTK_ENTRY( psPrivateCountry ), psPerson -> pnPrivateCountry );
	gtk_entry_set_text( GTK_ENTRY( psCompany ), psPerson -> pnCompany );
	gtk_entry_set_text( GTK_ENTRY( psBusinessStreet ), psPerson -> pnBusinessStreet );
	gtk_entry_set_text( GTK_ENTRY( psBusinessZipCode ), psPerson -> pnBusinessZipCode );
	gtk_entry_set_text( GTK_ENTRY( psBusinessCity ), psPerson -> pnBusinessCity );
	gtk_entry_set_text( GTK_ENTRY( psBusinessCountry ), psPerson -> pnBusinessCountry );
	gtk_entry_set_text( GTK_ENTRY( psBusinessPhone ), psPerson -> pnBusinessPhone );
	gtk_entry_set_text( GTK_ENTRY( psBusinessFax ), psPerson -> pnBusinessFax );

	bUpdate = FALSE;
}

/**
 * \brief Save person information if needed
 * \param psPerson person structure
 */
static void savePersonInformation( struct sPerson *psPerson ) {
	GtkWidget *psTitle = g_object_get_data( G_OBJECT( psAddressBook ), "title" );
	GtkWidget *psFirstName = g_object_get_data( G_OBJECT( psAddressBook ), "first_name" );
	GtkWidget *psLastName = g_object_get_data( G_OBJECT( psAddressBook ), "last_name" );
	GtkWidget *psCompany = g_object_get_data( G_OBJECT( psAddressBook ), "company" );
	GtkWidget *psPrivatePhone = g_object_get_data( G_OBJECT( psAddressBook ), "private_phone" );
	GtkWidget *psPrivateFax = g_object_get_data( G_OBJECT( psAddressBook ), "private_fax" );
	GtkWidget *psPrivateMobile = g_object_get_data( G_OBJECT( psAddressBook ), "private_mobile" );
	GtkWidget *psPrivateStreet = g_object_get_data( G_OBJECT( psAddressBook ), "private_street" );
	GtkWidget *psPrivateZipCode = g_object_get_data( G_OBJECT( psAddressBook ), "private_zipcode" );
	GtkWidget *psPrivateCity = g_object_get_data( G_OBJECT( psAddressBook ), "private_city" );
	GtkWidget *psPrivateCountry = g_object_get_data( G_OBJECT( psAddressBook ), "private_country" );
	GtkWidget *psBusinessStreet = g_object_get_data( G_OBJECT( psAddressBook ), "business_street" );
	GtkWidget *psBusinessZipCode = g_object_get_data( G_OBJECT( psAddressBook ), "business_zipcode" );
	GtkWidget *psBusinessCity = g_object_get_data( G_OBJECT( psAddressBook ), "business_city" );
	GtkWidget *psBusinessCountry = g_object_get_data( G_OBJECT( psAddressBook ), "business_country" );
	GtkWidget *psBusinessPhone = g_object_get_data( G_OBJECT( psAddressBook ), "business_phone" );
	GtkWidget *psBusinessFax = g_object_get_data( G_OBJECT( psAddressBook ), "business_fax" );

	if ( psPerson == NULL || bChanged == FALSE ) {
		return;
	}

	if ( psPerson -> pnDisplayName != NULL ) {
		Debug( KERN_DEBUG, "display name '%s'\n", psPerson -> pnDisplayName );
	}

	/* Title */
	if ( psPerson -> pnTitle != NULL ) {
		g_free( psPerson -> pnTitle );
	}
	psPerson -> pnTitle = g_strdup( gtk_entry_get_text( GTK_ENTRY( psTitle ) ) );

	/* First name */
	if ( psPerson -> pnFirstName != NULL ) {
		g_free( psPerson -> pnFirstName );
	}
	psPerson -> pnFirstName = g_strdup( gtk_entry_get_text( GTK_ENTRY( psFirstName ) ) );

	/* Last name */
	if ( psPerson -> pnLastName != NULL ) {
		g_free( psPerson -> pnLastName );
	}
	psPerson -> pnLastName = g_strdup( gtk_entry_get_text( GTK_ENTRY( psLastName ) ) );

	/* Company */
	if ( psPerson -> pnCompany != NULL ) {
		g_free( psPerson -> pnCompany );
	}
	psPerson -> pnCompany = g_strdup( gtk_entry_get_text( GTK_ENTRY( psCompany ) ) );

	/* Private phone */
	if ( psPerson -> pnPrivatePhone != NULL ) {
		g_free( psPerson -> pnPrivatePhone );
	}
	psPerson -> pnPrivatePhone = g_strdup( gtk_entry_get_text( GTK_ENTRY( psPrivatePhone ) ) );

	/* Private fax */
	if ( psPerson -> pnPrivateFax != NULL ) {
		g_free( psPerson -> pnPrivateFax );
	}
	psPerson -> pnPrivateFax = g_strdup( gtk_entry_get_text( GTK_ENTRY( psPrivateFax ) ) );

	/* Private mobile */
	if ( psPerson -> pnPrivateMobile != NULL ) {
		g_free( psPerson -> pnPrivateMobile );
	}
	psPerson -> pnPrivateMobile = g_strdup( gtk_entry_get_text( GTK_ENTRY( psPrivateMobile ) ) );

	/* Private street */
	if ( psPerson -> pnPrivateStreet != NULL ) {
		g_free( psPerson -> pnPrivateStreet );
	}
	psPerson -> pnPrivateStreet = g_strdup( gtk_entry_get_text( GTK_ENTRY( psPrivateStreet ) ) );

	/* Private zip code */
	if ( psPerson -> pnPrivateZipCode != NULL ) {
		g_free( psPerson -> pnPrivateZipCode );
	}
	psPerson -> pnPrivateZipCode = g_strdup( gtk_entry_get_text( GTK_ENTRY( psPrivateZipCode ) ) );

	/* Private city */
	if ( psPerson -> pnPrivateCity != NULL ) {
		g_free( psPerson -> pnPrivateCity );
	}
	psPerson -> pnPrivateCity = g_strdup( gtk_entry_get_text( GTK_ENTRY( psPrivateCity ) ) );

	/* Private country */
	if ( psPerson -> pnPrivateCountry != NULL ) {
		g_free( psPerson -> pnPrivateCountry );
	}
	psPerson -> pnPrivateCountry = g_strdup( gtk_entry_get_text( GTK_ENTRY( psPrivateCountry ) ) );

	/* Business street */
	if ( psPerson -> pnBusinessStreet != NULL ) {
		g_free( psPerson -> pnBusinessStreet );
	}
	psPerson -> pnBusinessStreet = g_strdup( gtk_entry_get_text( GTK_ENTRY( psBusinessStreet ) ) );

	/* Business zip code */
	if ( psPerson -> pnBusinessZipCode != NULL ) {
		g_free( psPerson -> pnBusinessZipCode );
	}
	psPerson -> pnBusinessZipCode = g_strdup( gtk_entry_get_text( GTK_ENTRY( psBusinessZipCode ) ) );

	/* Business city */
	if ( psPerson -> pnBusinessCity != NULL ) {
		g_free( psPerson -> pnBusinessCity );
	}
	psPerson -> pnBusinessCity = g_strdup( gtk_entry_get_text( GTK_ENTRY( psBusinessCity ) ) );

	/* Business country */
	if ( psPerson -> pnBusinessCountry != NULL ) {
		g_free( psPerson -> pnBusinessCountry );
	}
	psPerson -> pnBusinessCountry = g_strdup( gtk_entry_get_text( GTK_ENTRY( psBusinessCountry ) ) );

	/* Business phone */
	if ( psPerson -> pnBusinessPhone != NULL ) {
		g_free( psPerson -> pnBusinessPhone );
	}
	psPerson -> pnBusinessPhone = g_strdup( gtk_entry_get_text( GTK_ENTRY( psBusinessPhone ) ) );

	/* Business fax */
	if ( psPerson -> pnBusinessFax != NULL ) {
		g_free( psPerson -> pnBusinessFax );
	}
	psPerson -> pnBusinessFax = g_strdup( gtk_entry_get_text( GTK_ENTRY( psBusinessFax ) ) );

	bSave = TRUE;
}

/**
 * \brief Handle ok button clicked event
 * \param psTreeView treeview widget
 * \param psEvent event button structure
 * \param pUserData window pointer user data
 * \return TRUE if event handled, else FALSE
 */
static gboolean addrSelected( GtkTreeView *psTreeView, GdkEventButton *psEvent, gpointer pUserData ) {
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( psTreeView );
	GtkTreeModel *psModel = NULL;
	GtkTreeIter sIter;
	struct sPerson *psPerson = NULL;

	if ( !( ( psEvent -> type == GDK_BUTTON_PRESS && psEvent -> button == 1 ) || psEvent -> type == GDK_KEY_PRESS ) ) {
		return FALSE;
	}

	if ( gtk_tree_selection_count_selected_rows( psSelection ) <= 1 ) {
		GtkTreePath *psPath;
		gchar *pnName;

		/* Get tree path for row that was clicked */
		if ( gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW( psTreeView ), (gint) psEvent -> x, (gint) psEvent -> y, &psPath, NULL, NULL, NULL ) ) {
			gtk_tree_selection_unselect_all( psSelection );
			gtk_tree_selection_select_path( psSelection, psPath );

			psModel = gtk_tree_view_get_model( psTreeView );
			if ( gtk_tree_model_get_iter( psModel, &sIter, psPath ) ) {
				gtk_tree_model_get( psModel, &sIter, COL_DISPLAY_NAME, &pnName, -1 );
				if ( pnName != NULL ) {
					if ( psSelectedPerson != NULL ) {
						savePersonInformation( psSelectedPerson );
					}
					bChanged = FALSE;

					psPerson = findPerson( pnName );
					addrUpdateInformation( pUserData, psPerson );
					psSelectedPerson = psPerson;

					gtk_tree_path_free( psPath );
				}
			} else {
				gtk_tree_path_free( psPath );
			}
		}
	}

	return FALSE;
}

/**
 * \brief Move cursor up/down
 * \param psTreeView address book tree view widget
 * \param nStep movement step value
 * \param nDirection direction of step
 * \param pUserData addresbook pointer
 * \return TRUE if event handled, else FALSE
 */
static gboolean cursorMoved( GtkTreeView *psTreeView, GtkMovementStep nStep, gint nDirection, gpointer pUserData ) {
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( psTreeView );
	GtkTreeModel *psModel = NULL;
	GtkTreeIter sIter;
	struct sPerson *psPerson = NULL;

	if ( gtk_tree_selection_count_selected_rows( psSelection ) <= 1 ) {
		GtkTreePath *psPath;
		gchar *pnName;

		if ( nStep == GTK_MOVEMENT_DISPLAY_LINES ) {
			gtk_tree_view_get_cursor( psTreeView, &psPath, NULL );

			if ( psPath == NULL ) {
				return FALSE;
			}

			if ( nDirection == 1 ) {
				gtk_tree_path_next( psPath );
			} else {
				gtk_tree_path_prev( psPath );
			}

			psModel = gtk_tree_view_get_model( psTreeView );
			if ( gtk_tree_model_get_iter( psModel, &sIter, psPath ) ) {
				gtk_tree_model_get( psModel, &sIter, COL_DISPLAY_NAME, &pnName, -1 );
				if ( pnName != NULL ) {
					if ( psSelectedPerson != NULL ) {
						savePersonInformation( psSelectedPerson );
					}
					bChanged = FALSE;

					psPerson = findPerson( pnName );
					addrUpdateInformation( pUserData, psPerson );
					psSelectedPerson = psPerson;

					gtk_tree_path_free( psPath );
				}
			} else {
				gtk_tree_path_free( psPath );
			}
		}
	}

	return FALSE;
}

gchar *convert_utf8( const gchar *pnText ) {
	gchar *pnStr = NULL;
	GError *psError = NULL;
	gsize nReadBytes, nWrittenBytes;
	guchar *pnChar;
	const char *pnF;
	gchar *pnFromCodeSet = NULL;

	if ( !pnText ) {
		g_assert_not_reached();
	}

	if ( g_utf8_validate( pnText, -1, NULL ) ) {
		return g_strdup( pnText );
	}

	g_get_charset( &pnF );
	if ( pnF ) {
		pnFromCodeSet = g_strdup( pnF );
	} else {
		pnFromCodeSet = g_strdup( "ISO-8859-1" );
	}

	if ( !strcmp( pnFromCodeSet, "ISO-" ) ) {
		g_free( pnFromCodeSet );
		pnFromCodeSet = g_strdup( "ISO-8859-1" );
	}

	if ( pnStr ) {
		g_free( pnStr );
	}

	for ( pnChar = ( guchar * ) pnText; *pnChar != 0; pnChar++ ) {
		if ( *pnChar < 32 && *pnChar != '\n' ) {
			*pnChar = ' ';
		}
	}

	pnStr = g_convert( pnText, strlen( pnText ), "UTF-8", pnFromCodeSet, &nReadBytes, &nWrittenBytes, &psError );
	if ( pnStr == NULL ){
		pnStr = g_strdup( pnText );

		Debug( KERN_INFO, "case 1\n" );
		for ( pnChar = ( guchar * ) pnStr; *pnChar != 0; pnChar++ ) {
			if ( *pnChar > 128 ) {
				*pnChar = '?';
			}
		}
	}

	if ( psError ) {
		Debug( KERN_INFO, "case 2\n" );
		g_error_free( psError );
	}

	return pnStr;
}

/**
 * \brief Fill persons into list store
 * \param psStore tree store pointer
 */
static void fillPersons( GtkTreeStore *psStore ) {
	GList *psList;
	GtkTreeIter sParent;

	for ( psList = psPersonsList; psList != NULL; psList = psList -> next ) {
		struct sPerson *psPerson = psList -> data;

		if ( psPerson != NULL ) {
			gtk_tree_store_append( psStore, &sParent, NULL );
			gtk_tree_store_set( psStore, &sParent, COL_DISPLAY_NAME, psPerson -> pnDisplayName, -1 );
			gtk_tree_store_set( psStore, &sParent, COL_FIRST_NAME, psPerson -> pnFirstName, -1 );
			gtk_tree_store_set( psStore, &sParent, COL_LAST_NAME, psPerson -> pnLastName, -1 );
			gtk_tree_store_set( psStore, &sParent, COL_COMPANY, psPerson -> pnCompany, -1 );
		}
	}
}

void updateAddressBook( void ) {
		if ( psStore != NULL ) {
			gtk_tree_store_clear( psStore );
			fillPersons( psStore );
		}
}

/**
 * \brief Handle refresh button clicked event
 * \param psWidget widget
 * \param pData optional data
 */
void addrRefreshButtonClicked( GtkWidget *psWidget, void *pData ) {
	struct sAddressBook *psBook = getDefaultBookPlugin( getActiveProfile() );
	
	if ( psBook != NULL ) {
		if ( pData != NULL ) {
			freePersons();
			psBook -> ReadBook();
		}

		updateAddressBook();
	}
}

/**
 * \brief Handle delete button clicked event
 * \param psWidget widget
 * \param pData pointer to treeview
 */
static void addrDeleteButtonClicked( GtkWidget *psWidget, void *pData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( GTK_TREE_VIEW( pData ) );
	struct sPerson *psPerson = NULL;

	if ( gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		GValue sName = { 0 };

		gtk_tree_model_get_value( psModel, &sSelectedIter, COL_DISPLAY_NAME, &sName );

		GtkWidget *psDialog = gtk_message_dialog_new( NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION,
																	GTK_BUTTONS_OK_CANCEL, _( "Do you want to delete the entry '%s'?" ), g_value_get_string( &sName ) );
		gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Delete entry" ) );
		gint nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );
		gtk_widget_destroy( psDialog );

		if ( nResult != GTK_RESPONSE_OK ) {
			return;
		}

		psPerson = findPerson( g_value_get_string( &sName ) );
		if ( psPerson != NULL ) {
			psPerson -> nFlags = PERSON_FLAG_DELETED;
		}

		g_value_unset( &sName );
		
		SaveBook();
		updateAddressBook();
	}
}

/**
 * \brief Handle add button clicked event
 * \param psWidget widget
 * \param pData optional data
 */
static void addrAddButtonClicked( GtkWidget *psWidget, void *pData ) {
	if ( psSelectedPerson != NULL ) {
		savePersonInformation( psSelectedPerson );
		psSelectedPerson = NULL;
	}

	if ( AddAddressDialog( NULL, NULL, 1 ) ) {
		return;
	}

	addrRefreshButtonClicked( NULL, NULL );
}

/**
 * \brief Sort tree model by string
 * \param psModel tree model widget
 * \param psA first entry tree iterator
 * \param psB second entry tree iterator
 * \param pData sort column id
 * \return result, equivalent to strcmp
 */
static gint sortByString( GtkTreeModel *psModel, GtkTreeIter *psA, GtkTreeIter *psB, gpointer pData ) {
	gint nRet = 0;
	gint nSortCol = GPOINTER_TO_INT( pData );

	switch ( nSortCol ) {
		case COL_DISPLAY_NAME:
		case COL_FIRST_NAME:
		case COL_LAST_NAME:
		case COL_COMPANY: {
			gchar *pnStringA;
			gchar *pnStringB;

			gtk_tree_model_get( psModel, psA, nSortCol, &pnStringA, -1 );
			gtk_tree_model_get( psModel, psB, nSortCol, &pnStringB, -1 );

			if ( pnStringA == NULL || pnStringB == NULL ) {
				if ( pnStringA == NULL && pnStringB == NULL ) {
					nRet = 0;
				} else {
					nRet = ( pnStringA == NULL ) ? -1 : 1;
				}
			} else {
				nRet = g_utf8_collate( pnStringA, pnStringB );
			}

			g_free( pnStringA );
			g_free( pnStringB );

			break;
		}
	}

	return nRet;
}

/**
 * \brief Address window response callback
 * \param psDialog dialog window
 * \param nResult response id
 * \param pData user data
 */
static void addressWindowResponse( GtkDialog *psDialog, gint nResult, gpointer pData ) {
	if ( nResult == 1 ) {
		if ( psSelectedPerson != NULL ) {
			savePersonInformation( psSelectedPerson );
		}

		if ( bChanged == TRUE || bSave == TRUE ) {
			Debug( KERN_DEBUG, "Writing address book\n" );
			SaveBook();
		}
	}

	gtk_widget_destroy( GTK_WIDGET( psDialog ) );
	psAddressBook = NULL;
}

/**
 * \brief Import addressbook
 * \param psWidget button widget
 * \param pUserData parent window
 */
static void importCallback( GtkWidget *psWidget, gpointer pUserData ) {
	GtkWidget *psDialog;
	GtkFileFilter *psFilter;

	psDialog = gtk_file_chooser_dialog_new( _( "Import addressbook" ), pUserData, GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL );
	gtk_file_chooser_set_current_folder( GTK_FILE_CHOOSER( psDialog ), getHomeDir() );

	psFilter = gtk_file_filter_new();
	gtk_file_filter_add_pattern( psFilter, "*.csv" );
	gtk_file_filter_add_pattern( psFilter, "*.vcf" );
	gtk_file_chooser_set_filter( GTK_FILE_CHOOSER( psDialog ), psFilter );

	if ( gtk_dialog_run( GTK_DIALOG( psDialog ) ) == GTK_RESPONSE_ACCEPT ) {
		char *pnFileName;

		pnFileName = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( psDialog ) );

		Debug( KERN_DEBUG, "import from '%s'\n", pnFileName );
		if ( strstr( pnFileName, ".csv" ) ) {
			parseCsvFile( pnFileName, getActiveProfile() -> pnName );
		} else {
			loadCardFile( pnFileName );
		}
		g_free( pnFileName );
		gtk_tree_store_clear( psStore );
		fillPersons( psStore );
		SaveBook();
	}

	gtk_widget_destroy( psDialog );
}

/**
 * \brief Export addressbook as csv
 * \param psWidget button widget
 * \param pUserData parent window
 */
static void exportCallback( GtkWidget *psWidget, gpointer pUserData ) {
	GtkWidget *psDialog;

	psDialog = gtk_file_chooser_dialog_new( _( "Export addressbook" ), pUserData, GTK_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL );
	gtk_file_chooser_set_do_overwrite_confirmation( GTK_FILE_CHOOSER( psDialog ), TRUE );

	gtk_file_chooser_set_current_folder( GTK_FILE_CHOOSER( psDialog ), getHomeDir() );
	gtk_file_chooser_set_current_name( GTK_FILE_CHOOSER( psDialog ), "abook.vcf" );

	if ( gtk_dialog_run( GTK_DIALOG( psDialog ) ) == GTK_RESPONSE_ACCEPT ) {
		char *pnFileName;
		GList *psList = NULL;
		struct sPerson *psPerson = NULL;

		pnFileName = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( psDialog ) );

		for ( psList = psPersonsList; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
			psPerson = psList -> data;

			psPerson -> nFlags = PERSON_FLAG_NEW;
		}
		writeCardFile( pnFileName );
		Debug( KERN_DEBUG, "export to '%s'\n", pnFileName );
		g_free( pnFileName );
	}

	gtk_widget_destroy( psDialog );
}

/**
 * \brief Export addressbook to fritzbox
 * \param psWidget button widget
 * \param pUserData parent window
 */
static void exportFritzBoxCallback( GtkWidget *psWidget, gpointer pUserData ) {
	struct sPlugin *psPlugin = NULL;
	struct sAddressBook *psBook = NULL;

	psPlugin = findPlugin( PLUGIN_TYPE_BOOK, _( "FritzFon Addressbook" ) );
	if ( psPlugin != NULL ) {
		Debug( KERN_DEBUG, "found!!!\n" );
		psBook = psPlugin -> pData;
		if ( psBook != NULL ) {
			GtkWidget *psDialog = NULL;
			gint nResult = 0;

			psDialog = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL,
				GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, _( "Do you want to export your addressbook to the fritzbox?" ) );
			gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Question" ) );
	
			nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );
			gtk_widget_destroy( psDialog );

			switch ( nResult ) {
				case GTK_RESPONSE_YES:
					break;
				default:
					return;
			}
			psBook -> SaveBook();
		}
	} else {
		Debug( KERN_DEBUG, "NOT found!!!\n" );
	}
}
 
/**
 * \brief Dial button clicked - callback
 * \param pData entry widget
 * \param psWidget button widget
 */
void addressDialClicked( gpointer pData, GtkWidget *psWidget ) {
	GtkWidget *psEntry = GTK_WIDGET( pData );
	gint nType = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( psAddressBook ), "type" ) );
	GtkWidget *psParent = g_object_get_data( G_OBJECT( psAddressBook ), "parent" );
	const gchar *pnNumber = gtk_entry_get_text( GTK_ENTRY( psEntry ) );
	struct sPerson *psPerson = findPersonByNumber( pnNumber );

	if ( psPerson != NULL ) {
		if ( psParent != NULL && nType == 1 ) {
			setDialNameNumber( psParent, psPerson -> pnDisplayName, pnNumber );
			gtk_widget_destroy( GTK_WIDGET( psAddressBook ) );
			psAddressBook = NULL;
		} else {
			dialNumberDialog( psPerson, pnNumber );
		}
	}
}

/**
 * \brief Entry changed- callback
 * \param pData entry widget
 * \param psWidget treeview widget
 */
static void addressEntryChanged( gpointer pData, GtkWidget *psWidget ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( GTK_TREE_VIEW( psWidget ) );
	struct sPerson *psPerson = NULL;

	if ( bUpdate == TRUE ) {
		return;
	}

	if ( gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		GValue sName = { 0 };

		gtk_tree_model_get_value( psModel, &sSelectedIter, COL_DISPLAY_NAME, &sName );

		psPerson = findPerson( g_value_get_string( &sName ) );

		g_value_unset( &sName );

		if ( psPerson != NULL ) {
			psPerson -> nFlags = PERSON_FLAG_CHANGED;
			bChanged = TRUE;
		}		
	}
}

/**
 * \brief Load image
 * \param psPerson person structure
 * \param psButton image button widget
 */
static void addrLoadImage( struct sPerson *psPerson, GtkWidget *psButton ) {
	GtkWidget *psFileChooser;
	GtkFileFilter *psFilter;
	gint nResult;
	
	psFileChooser = gtk_file_chooser_dialog_new( _( "Open image" ), ( GtkWindow * ) gtk_widget_get_ancestor(  psButton, GTK_TYPE_WINDOW ),
		GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, _( "No image" ), 1, NULL );

	psFilter = gtk_file_filter_new();

	gtk_file_filter_add_mime_type( psFilter, "image/gif" );
	gtk_file_filter_add_mime_type( psFilter, "image/jpeg" );
	gtk_file_filter_add_mime_type( psFilter, "image/png" );
	gtk_file_filter_add_mime_type( psFilter, "image/tiff" );
	gtk_file_filter_add_mime_type( psFilter, "image/ief" );
	gtk_file_filter_add_mime_type( psFilter, "image/cgm" );
	gtk_file_chooser_set_filter( GTK_FILE_CHOOSER( psFileChooser ), psFilter );

	nResult = gtk_dialog_run( GTK_DIALOG( psFileChooser ) );

	if ( nResult == GTK_RESPONSE_ACCEPT ) {
		if ( psPerson -> pnNewImage != NULL ) {
			g_free( psPerson -> pnNewImage );
			psPerson -> pnNewImage = NULL;
		}
		psPerson -> pnNewImage = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( psFileChooser ) );
		if ( psPerson -> pnNewImage != NULL ) {
			psPerson -> psImage = gdk_pixbuf_new_from_file( psPerson -> pnNewImage, NULL );
			addrShowImage( psPerson );
			psPerson -> nFlags = PERSON_FLAG_CHANGED;
			bChanged = TRUE;
		}
	} else if ( nResult == 1 ) {
		if ( psPerson -> psImage != NULL ) {
			psPerson -> psImage = NULL;
			psPerson -> nFlags = PERSON_FLAG_CHANGED;
			bChanged = TRUE;
		}
	}

	gtk_widget_destroy( psFileChooser );
}

/**
 * \brief Entry changed- callback
 * \param psWidget entry widget
 * \param pData treeview widget
 */
static void addressChangePhoto( GtkWidget *psWidget, gpointer pData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( GTK_TREE_VIEW( pData ) );
	struct sPerson *psPerson = NULL;

	if ( bUpdate == TRUE ) {
		return;
	}

	if ( gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		GValue sName = { 0 };

		gtk_tree_model_get_value( psModel, &sSelectedIter, COL_DISPLAY_NAME, &sName );

		psPerson = findPerson( g_value_get_string( &sName ) );

		g_value_unset( &sName );

		if ( psPerson != NULL ) {
			Debug( KERN_DEBUG, "'%s' found\n", psPerson -> pnDisplayName );
			addrLoadImage( psPerson, psWidget );
		}		
	}
}

/**
 * \brief Address Book
 * \param nType 0 = standard window, 1 = dial number window, 2 = find address
 * \return address book widget
 */
GtkWidget *AddressBook( int nType ) {
	GtkBuilder *psBuilder = NULL;
	GError *psError = NULL;
	GtkWidget *psTreeView = NULL;
	GtkWidget *psAddButton = NULL;
	GtkWidget *psDeleteButton = NULL;
	GtkWidget *psRefreshButton = NULL;
	GtkWidget *psImportButton = NULL;
	GtkWidget *psExportButton = NULL;
	GtkWidget *psImageButton = NULL;
	GtkWidget *psImage = NULL;
	GtkWidget *psTitle = NULL;
	GtkWidget *psFirstNameEntry = NULL;
	GtkWidget *psLastNameEntry = NULL;
	GtkWidget *psCompanyEntry = NULL;
	GtkWidget *psPrivatePhoneEntry = NULL;
	GtkWidget *psPrivateFaxEntry = NULL;
	GtkWidget *psPrivateMobileEntry = NULL;
	GtkWidget *psPrivateStreetEntry = NULL;
	GtkWidget *psPrivateCityEntry = NULL;
	GtkWidget *psPrivateZipCodeEntry = NULL;
	GtkWidget *psPrivateCountryEntry = NULL;
	GtkWidget *psBusinessPhoneEntry = NULL;
	GtkWidget *psBusinessFaxEntry = NULL;
	GtkWidget *psBusinessStreetEntry = NULL;
	GtkWidget *psBusinessCityEntry = NULL;
	GtkWidget *psBusinessZipCodeEntry = NULL;
	GtkWidget *psBusinessCountryEntry = NULL;
	GtkWidget *psExportFritzBoxButton = NULL;
	GtkWidget *psTmp = NULL;
	GtkTreeModel *psListStore = NULL;
	GtkTreeSortable *psSortable = NULL;
	GtkCellRenderer *psRenderer = NULL;
	GtkTreeViewColumn *psColumn = NULL;
	GtkTreeIter sIter;
	struct sAddressBook *psBook = NULL;
	gchar *pnUiFile;

	psBook = getDefaultBookPlugin( getActiveProfile() );
	if ( psBook == NULL ) {
		Debug( KERN_WARNING, "No addressbook plugin, abort!\n" );
		ShowError( FALSE, _( "Abort" ), _( "No addressbook plugin selected" ) );
		return NULL;
	}

	if ( psAddressBook != NULL ) {
		gtk_window_present( GTK_WINDOW( psAddressBook ) );
		return psAddressBook;
	}

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "book.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
	    Debug( KERN_WARNING, "Error: %s\n", psError -> message );
	    g_error_free( psError );
		g_free( pnUiFile );
	    return NULL;
	}
	g_free( pnUiFile );

	psAddressBook = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psAddressBook" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "type", GINT_TO_POINTER( nType ) );
	psTreeView = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psTreeView" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "treeview", psTreeView );
	psAddButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psAddButton" ) );
	psDeleteButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDelButton" ) );
	psRefreshButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRefreshButton" ) );
	psImportButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psImportButton" ) );
	g_signal_connect( G_OBJECT( psImportButton ), "clicked", G_CALLBACK( importCallback ), psAddressBook );
	psExportButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psExportButton" ) );
	g_signal_connect( G_OBJECT( psExportButton ), "clicked", G_CALLBACK( exportCallback ), psAddressBook );

	psExportFritzBoxButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psExportFritzBox" ) );
	g_signal_connect( G_OBJECT( psExportFritzBoxButton ), "clicked", G_CALLBACK( exportFritzBoxCallback ), psAddressBook );

	/* Image */
	psImage = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psImage" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "image", psImage );
	if ( !( psBook -> nReadFlags & PERSON_IMAGE ) ) {
		gtk_widget_set_sensitive( psImage, FALSE );
	}
	if ( psBook -> nWriteFlags & PERSON_IMAGE ) {
		psImageButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psImageButton" ) );
		g_signal_connect( G_OBJECT( psImageButton ), "clicked", G_CALLBACK( addressChangePhoto ), psTreeView );
	}

	/* Title */
	psTitle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psTitleEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "title", psTitle );
	g_signal_connect( G_OBJECT( psTitle ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_TITLE ) ) {
		gtk_widget_set_sensitive( psTitle, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_TITLE ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psTitle ), FALSE );
	}

	/* First name */
	psFirstNameEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFirstNameEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "first_name", psFirstNameEntry );
	g_signal_connect( G_OBJECT( psFirstNameEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_FIRST_NAME ) ) {
		gtk_widget_set_sensitive( psFirstNameEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_FIRST_NAME ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psFirstNameEntry ), FALSE );
	}

	/* Last name */
	psLastNameEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psLastNameEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "last_name", psLastNameEntry );
	g_signal_connect( G_OBJECT( psLastNameEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_LAST_NAME ) ) {
		gtk_widget_set_sensitive( psLastNameEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_LAST_NAME ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psLastNameEntry ), FALSE );
	}

	/* Company */
	psCompanyEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCompanyEntry" ) );
	g_signal_connect( G_OBJECT( psCompanyEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	g_object_set_data( G_OBJECT( psAddressBook ), "company", psCompanyEntry );
	if ( !( psBook -> nReadFlags & PERSON_COMPANY ) ) {
		gtk_widget_set_sensitive( psCompanyEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_COMPANY ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psCompanyEntry ), FALSE );
	}

	/* Private phone */
	psPrivatePhoneEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivatePhoneEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "private_phone", psPrivatePhoneEntry );
	g_signal_connect( G_OBJECT( psPrivatePhoneEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_PRIVATE_PHONE ) ) {
		gtk_widget_set_sensitive( psPrivatePhoneEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_PRIVATE_PHONE ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psPrivatePhoneEntry ), FALSE );
	}

	psTmp = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivatePhoneButton" ) );
	gtk_widget_set_sensitive( psTmp, routerHasDial( getActiveProfile() ) );

	/* Private fax */
	psPrivateFaxEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivateFaxEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "private_fax", psPrivateFaxEntry );
	g_signal_connect( G_OBJECT( psPrivateFaxEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_PRIVATE_FAX ) ) {
		gtk_widget_set_sensitive( psPrivateFaxEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_PRIVATE_FAX ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psPrivateFaxEntry ), FALSE );
	}

	psTmp = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivateFaxButton" ) );
	gtk_widget_set_sensitive( psTmp, routerHasDial( getActiveProfile() ) );

	/* Private mobile */
	psPrivateMobileEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivateMobileEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "private_mobile", psPrivateMobileEntry );
	g_signal_connect( G_OBJECT( psPrivateMobileEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_PRIVATE_MOBILE ) ) {
		gtk_widget_set_sensitive( psPrivateMobileEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_PRIVATE_MOBILE ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psPrivateMobileEntry ), FALSE );
	}

	psTmp = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivateMobileButton" ) );
	gtk_widget_set_sensitive( psTmp, routerHasDial( getActiveProfile() ) );

	/* Private street */
	psPrivateStreetEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivateStreetEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "private_street", psPrivateStreetEntry );
	g_signal_connect( G_OBJECT( psPrivateStreetEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_PRIVATE_STREET ) ) {
		gtk_widget_set_sensitive( psPrivateStreetEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_PRIVATE_STREET ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psPrivateStreetEntry ), FALSE );
	}

	/* Private city */
	psPrivateCityEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivateCityEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "private_city", psPrivateCityEntry );
	g_signal_connect( G_OBJECT( psPrivateCityEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_PRIVATE_CITY ) ) {
		gtk_widget_set_sensitive( psPrivateCityEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_PRIVATE_CITY ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psPrivateCityEntry ), FALSE );
	}

	/* Private zip code */
	psPrivateZipCodeEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivateZipCodeEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "private_zipcode", psPrivateZipCodeEntry );
	g_signal_connect( G_OBJECT( psPrivateZipCodeEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_PRIVATE_ZIPCODE ) ) {
		gtk_widget_set_sensitive( psPrivateZipCodeEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_PRIVATE_ZIPCODE ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psPrivateZipCodeEntry ), FALSE );
	}

	/* Private country */
	psPrivateCountryEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPrivateCountryEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "private_country", psPrivateCountryEntry );
	g_signal_connect( G_OBJECT( psPrivateCountryEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_PRIVATE_COUNTRY ) ) {
		gtk_widget_set_sensitive( psPrivateCountryEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_PRIVATE_COUNTRY ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psPrivateCountryEntry ), FALSE );
	}

	/* Business street */
	psBusinessStreetEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBusinessStreetEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "business_street", psBusinessStreetEntry );
	g_signal_connect( G_OBJECT( psBusinessStreetEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_BUSINESS_STREET ) ) {
		gtk_widget_set_sensitive( psBusinessStreetEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_BUSINESS_STREET ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psBusinessStreetEntry ), FALSE );
	}

	/* Business city */
	psBusinessCityEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBusinessCityEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "business_city", psBusinessCityEntry );
	g_signal_connect( G_OBJECT( psBusinessCityEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_BUSINESS_CITY ) ) {
		gtk_widget_set_sensitive( psBusinessCityEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_BUSINESS_CITY ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psBusinessCityEntry ), FALSE );
	}

	/* Business zip code */
	psBusinessZipCodeEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBusinessZipCodeEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "business_zipcode", psBusinessZipCodeEntry );
	g_signal_connect( G_OBJECT( psBusinessZipCodeEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_BUSINESS_ZIPCODE ) ) {
		gtk_widget_set_sensitive( psBusinessZipCodeEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_BUSINESS_ZIPCODE ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psBusinessZipCodeEntry ), FALSE );
	}

	/* Business country */
	psBusinessCountryEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBusinessCountryEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "business_country", psBusinessCountryEntry );
	g_signal_connect( G_OBJECT( psBusinessCountryEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_BUSINESS_COUNTRY ) ) {
		gtk_widget_set_sensitive( psBusinessCountryEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_BUSINESS_COUNTRY ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psBusinessCountryEntry ), FALSE );
	}

	/* Business phone */
	psBusinessPhoneEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBusinessPhoneEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "business_phone", psBusinessPhoneEntry );
	g_signal_connect( G_OBJECT( psBusinessPhoneEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_BUSINESS_PHONE ) ) {
		gtk_widget_set_sensitive( psBusinessPhoneEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_BUSINESS_PHONE ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psBusinessPhoneEntry ), FALSE );
	}

	psTmp = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBusinessPhoneButton" ) );
	gtk_widget_set_sensitive( psTmp, routerHasDial( getActiveProfile() ) );

	/* Business fax */
	psBusinessFaxEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBusinessFaxEntry" ) );
	g_object_set_data( G_OBJECT( psAddressBook ), "business_fax", psBusinessFaxEntry );
	g_signal_connect( G_OBJECT( psBusinessFaxEntry ), "changed", G_CALLBACK( addressEntryChanged ), psTreeView );
	if ( !( psBook -> nReadFlags & PERSON_BUSINESS_FAX ) ) {
		gtk_widget_set_sensitive( psBusinessFaxEntry, FALSE );
	}
	if ( !( psBook -> nWriteFlags & PERSON_BUSINESS_FAX ) ) {
		gtk_editable_set_editable( GTK_EDITABLE( psBusinessFaxEntry ), FALSE );
	}

	psTmp = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBusinessFaxButton" ) );
	gtk_widget_set_sensitive( psTmp, routerHasDial( getActiveProfile() ) );

	psStore = gtk_tree_store_new( NUM_COLS, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING );

	psListStore = GTK_TREE_MODEL( psStore );
	if ( psListStore == NULL ) {
		return NULL;
	}

	fillPersons( psStore );
	gtk_tree_view_set_model( GTK_TREE_VIEW( psTreeView ), GTK_TREE_MODEL( psListStore ) );

	g_signal_connect( psTreeView, "button-press-event", (GCallback) addrSelected, psAddressBook );
	g_signal_connect( psTreeView, "move-cursor", (GCallback) cursorMoved, psAddressBook );

	psSortable = GTK_TREE_SORTABLE( psListStore );

	psRenderer = gtk_cell_renderer_text_new();
	psColumn = gtk_tree_view_column_new_with_attributes( _( "Display Name" ), psRenderer, "text", COL_DISPLAY_NAME, NULL );
	gtk_tree_view_column_set_sort_column_id( psColumn, COL_DISPLAY_NAME );
	gtk_tree_view_column_set_resizable( psColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psTreeView ), psColumn );

	psRenderer = gtk_cell_renderer_text_new();
	psColumn = gtk_tree_view_column_new_with_attributes( _( "First Name" ), psRenderer, "text", COL_FIRST_NAME, NULL );
	gtk_tree_view_column_set_sort_column_id( psColumn, COL_FIRST_NAME );
	gtk_tree_view_column_set_resizable( psColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psTreeView ), psColumn );

	psRenderer = gtk_cell_renderer_text_new();
	psColumn = gtk_tree_view_column_new_with_attributes( _( "Last Name" ), psRenderer, "text", COL_LAST_NAME, NULL );
	gtk_tree_view_column_set_sort_column_id( psColumn, COL_LAST_NAME );
	gtk_tree_view_column_set_resizable( psColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psTreeView ), psColumn );

	psRenderer = gtk_cell_renderer_text_new();
	psColumn = gtk_tree_view_column_new_with_attributes( _( "Company" ), psRenderer, "text", COL_COMPANY, NULL );
	gtk_tree_view_column_set_sort_column_id( psColumn, COL_COMPANY );
	gtk_tree_view_column_set_resizable( psColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psTreeView ), psColumn );

	GtkTreeSelection *selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( psTreeView ) );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_SINGLE );

	gtk_tree_view_set_rules_hint( GTK_TREE_VIEW( psTreeView ), true );

	gtk_tree_sortable_set_sort_func( psSortable, COL_DISPLAY_NAME, sortByString, GINT_TO_POINTER( COL_DISPLAY_NAME ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_FIRST_NAME, sortByString, GINT_TO_POINTER( COL_FIRST_NAME ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_LAST_NAME, sortByString, GINT_TO_POINTER( COL_LAST_NAME ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_COMPANY, sortByString, GINT_TO_POINTER( COL_COMPANY ), NULL );
	gtk_tree_sortable_set_sort_column_id( psSortable, COL_DISPLAY_NAME, GTK_SORT_ASCENDING );

	g_signal_connect(G_OBJECT( psAddButton ), "clicked", G_CALLBACK( addrAddButtonClicked ), GINT_TO_POINTER( nType ) );
	g_signal_connect(G_OBJECT( psDeleteButton ), "clicked", G_CALLBACK( addrDeleteButtonClicked ), psTreeView );
	g_signal_connect(G_OBJECT( psRefreshButton ), "clicked", G_CALLBACK( addrRefreshButtonClicked ), psStore );

	if ( psBook != NULL && psBook -> SaveBook == NULL ) {
		gtk_widget_set_sensitive( psAddButton, false );
		gtk_widget_set_sensitive( psDeleteButton, false );
	}

	if ( nType == 2 ) {
		GtkWidget *psNoteLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNoteLabel" ) );
		gtk_label_set_markup( GTK_LABEL( psNoteLabel ), _( "<b>Insert number via clipboard</b>" ) );
	}

	gtk_builder_connect_signals( psBuilder, NULL );

	g_object_unref( G_OBJECT( psBuilder ) );

	bChanged = FALSE;
	bSave = FALSE;

	if ( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( psListStore ), &sIter ) == TRUE ) {
		GValue sName = { 0 };

		gtk_tree_selection_select_iter( gtk_tree_view_get_selection( GTK_TREE_VIEW( psTreeView ) ), &sIter );
		gtk_tree_model_get_value( psListStore, &sIter, COL_DISPLAY_NAME, &sName );
		psSelectedPerson = findPerson( g_value_get_string( &sName ) );
		addrUpdateInformation( NULL, psSelectedPerson );

		g_value_unset( &sName );
	}	

	g_signal_connect(G_OBJECT( psAddressBook ), "response", G_CALLBACK( addressWindowResponse ), NULL );
	//g_signal_connect(G_OBJECT( psAddressBook ), "notify::update", G_CALLBACK( updateAddressBook ), NULL );

	gtk_widget_show_all( psAddressBook );

	return psAddressBook;
}
