/*******************************************************************
 * libfaxophone                                                    *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file isdn-convert.c
 * \brief Audio conversion function
 */

#include "isdn-convert.h"
#include "phone.h"

static unsigned char *pnLutIn = NULL;
static unsigned char *pnLutOut = NULL;
static unsigned char *pnLutAnalyze = NULL;
static short *pnLutA2S = NULL;
signed char linear16_2_law[ 65536 ];
unsigned short law_2_linear16[ 256 ];

static double fLineLevelInState = 0.0f;
static double fLineLevelOutState = 0.0f;

/**
 * \brief Get sign and magnitude
 * \param nSample alaw sample
 * \param pnSign sign
 * \param pnMag magnitude
 */
static inline void alaw_get_sign_mag( short nSample, unsigned *pnSign, unsigned *pnMag ) {
	if ( nSample < 0 ) {
		*pnMag = -nSample;
		*pnSign = 0;
	} else {
		*pnMag = nSample;
		*pnSign = 0x80;
	}
}

/**
 * \brief Convert linear to alaw value
 * \param nSample linear sample value
 * \return alaw sample value
 */
static unsigned char linear2alaw( short nSample ) {
	unsigned nSign, nExponent, nMantissa, nMag;
	unsigned char nAlawByte;
	static const unsigned anExpLut[ 128 ] = {
		1, 1, 2, 2, 3, 3, 3, 3,
		4, 4, 4, 4, 4, 4, 4, 4,
		5, 5, 5, 5, 5, 5, 5, 5,
		5, 5, 5, 5, 5, 5, 5, 5,
		6, 6, 6, 6, 6, 6, 6, 6,
		6, 6, 6, 6, 6, 6, 6, 6,
		6, 6, 6, 6, 6, 6, 6, 6,
		6, 6, 6, 6, 6, 6, 6, 6,
		7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7
	};

	alaw_get_sign_mag( nSample, &nSign, &nMag );
	if ( nMag > 32767 ) {
		nMag = 32767;
	}

	nExponent = anExpLut[ ( nMag >> 8 ) & 0x7F ];
	nMantissa = ( nMag >> ( nExponent + 3 ) ) & 0x0F;

	if ( nMag < 0x100 ) {
		nExponent = 0;
	}

	nAlawByte = ( unsigned char )( nSign | ( nExponent << 4 ) | nMantissa );
	nAlawByte ^= 0x55;

	return nAlawByte;
}

/**
 * \brief Convert alaw value to linear value
 * \param nAlawByte alaw value
 * \return linear value
 */
static short alaw2linear( unsigned char nAlawByte ) {
	int nT;
	int nSeg;

	nAlawByte ^= 0x55;
	nT = nAlawByte & 0x7F;

	if ( nT < 16 ) {
		nT = ( nT << 4 ) + 8;
	} else {
		nSeg = ( nT >> 4 ) & 0x07;
		nT = ( ( nT & 0x0F ) << 4 ) + 0x108;
		nT <<= nSeg - 1;
	}

	return ( ( nAlawByte & 0x80 ) ? nT : - nT );
}

/**
 * \brief Bit-inverse the given value
 * \param nC char value
 * \return bit-inversed value
 */
inline unsigned char bitinverse( unsigned char nC ) {
	return ( ( nC >> 7 ) & 0x1 ) |
		( ( nC >> 5 ) & 0x2 ) |
		( ( nC >> 3 ) & 0x4 ) |
		( ( nC >> 1 ) & 0x8 ) |
		( ( nC << 1 ) & 0x10 ) |
		( ( nC << 3 ) & 0x20 ) |
		( ( nC << 5 ) & 0x40 ) |
		( ( nC << 7 ) & 0x80 );
}

/**
 * \brief Create lookup table buffer
 */
void createTableBuffer( void ) {
	signed char *_linear16_2_law = &linear16_2_law[ 32768 ];
	long nIndex;
	int nBufSizeIn = 0;
	int nBufSizeOut = 0;
	int nSample;
	unsigned int nAudioSampleSizeIn = 2;
	unsigned int nAudioSampleSizeOut = 2;

	for ( nIndex = 0; nIndex < 65535; nIndex++ ) {
		_linear16_2_law[ nIndex - 32768 ] = bitinverse( linear2alaw( ( short ) nIndex - 32768 ) );
	}

	for ( nIndex = 0; nIndex < 256; nIndex++ ) {
		law_2_linear16[ nIndex ] = alaw2linear( bitinverse( nIndex ) ) & 0xFFFF;
	}

	nBufSizeIn = nAudioSampleSizeIn * 256;
	pnLutIn = malloc( nBufSizeIn );

	for ( nIndex = 0; nIndex < nBufSizeIn; nIndex += nAudioSampleSizeIn ) {
		nSample = alaw2linear( bitinverse( ( unsigned char )( nIndex / 2 ) ) );
		pnLutIn[ nIndex + 0 ] = ( unsigned char )( nSample & 0xFF );
		pnLutIn[ nIndex + 1 ] = ( unsigned char )( nSample >> 8 & 0xFF );
	}

	nBufSizeOut = ( 1 + ( nAudioSampleSizeOut - 1 ) * 255 ) * 256;
	pnLutOut = malloc( nBufSizeOut );

	for ( nIndex = 0; nIndex < nBufSizeOut; nIndex++ ) {
		pnLutOut[ nIndex ] = bitinverse( linear2alaw( ( int )( signed char )( nIndex >> 8 ) << 8 | ( int )( nIndex & 0xFF ) ) );
	}

	pnLutAnalyze = malloc( 256 );
	pnLutA2S = malloc( 256 * sizeof( short ) );

	for ( nIndex = 0; nIndex < 256; nIndex++ ) {
		pnLutAnalyze[ nIndex ] = ( unsigned char )( ( alaw2linear( ( unsigned char )bitinverse( nIndex ) ) / 256 & 0xFF ) ^ 0x80 );

		pnLutA2S[ nIndex ] = alaw2linear( bitinverse( nIndex ) );
	}
}

/**
 * \brief Get line level input value
 * \return line level input state
 */
double getLineLevelIn( void ) {
	return fLineLevelInState;
}

/**
 * \brief Get line level output value
 * \return line level output state
 */
double getLineLevelOut( void ) {
	return fLineLevelOutState;
}

/**
 * \brief Convert isdn format to audio format
 * \param pnInBuf input buffer
 * \param nInBufLen length of input buffer
 * \param pnOutBuffer output buffer
 * \param pnOutBufLen pointer to output buffer len
 */
void convertIsdnToAudio( struct sRecorder *psRecorder, unsigned char *pnInBuf, unsigned int nInBufLen, unsigned char *pnOutBuf, unsigned int *pnOutBufLen, short *pnRecBuf ) {
	unsigned int nIndex;
	unsigned int nToProcess;
	unsigned int nOutPtr = 0;
	unsigned int nJ;
	unsigned char nInByte;
	int nSample;
	double fRatioIn = 1.0f;
	double fLlRatio;
	int nMax = 0;

	nOutPtr = 0;

	for ( nIndex = 0; nIndex < nInBufLen; nIndex++ ) {
		nInByte = pnInBuf[ nIndex ];

		if ( psRecorder != NULL && pnRecBuf != NULL ) {
			pnRecBuf[ nIndex ] = psRecorder -> psFile ? pnLutA2S[ nInByte ] : 0;
		}

		nSample = pnLutAnalyze[ nInByte ];
		if ( abs( ( int ) nSample - 128 ) > nMax ) {
			nMax = abs( ( int ) nSample - 128 );
		}

		nToProcess = ( int ) floor( ( double )( nIndex + 1 ) * fRatioIn ) - ( int ) floor( ( double ) nIndex * fRatioIn );

		for ( nJ = 0; nJ < nToProcess; nJ++ ) {
			pnOutBuf[ nOutPtr++ ] = pnLutIn[ ( int ) nInByte * 2 ];
			pnOutBuf[ nOutPtr++ ] = pnLutIn[ ( int ) nInByte * 2 + 1 ];
		}
	}

	/* Record data */
	if ( psRecorder != NULL && pnRecBuf != NULL ) {
		recordingWrite( psRecorder, pnRecBuf, nInBufLen, RECORDING_REMOTE );
	}

	fLlRatio = nInBufLen / 400.0f;
	if ( fLlRatio > 1.0 ) {
		fLlRatio = 1.0;
	}

	fLineLevelInState = fLineLevelInState * ( 1.0 - fLlRatio ) + ( ( double ) nMax / 128 ) * fLlRatio;

	*pnOutBufLen = nOutPtr;
}

/**
 * \brief Convert audio format to isdn format
 * \param psConnection active capi connection
 * \param pmInBuf input buffer
 * \param nInBufLen length of input buffer
 * \param pnOutBuffer output buffer
 * \param pnOutBufLen pointer to output buffer len
 */
void convertAudioToIsdn( struct sCapiConnection *psConnection, unsigned char *pnInBuf, unsigned int nInBufLen, unsigned char *pnOutBuf, unsigned int *pnOutBufLen, short *pnRecBuf ) {
	unsigned int nIndex;
	unsigned int nToProcess;
	unsigned int nOutPtr = 0;
	unsigned int nJ;
	unsigned char nSample;
	double fRatioOut = 1.0f;
	double fLlRatio;
	int nMax = 0;
	unsigned char nSampleU8;

	nOutPtr = 0;

	for ( nIndex = 0; nIndex < nInBufLen; nIndex += 2 ) {
		nToProcess = ( int ) floor( ( double )( nOutPtr + 1 ) * fRatioOut ) - ( int ) floor( ( double ) nOutPtr * fRatioOut );

		for ( nJ = 0; nJ < nToProcess; nJ++ ) {
			int nTmp = ( int )( pnInBuf[ nIndex ] ) | ( ( int )( pnInBuf[ nIndex + 1 ] ) << 8 );
			nSample = pnLutOut[ nTmp ];

			if ( psConnection != NULL && psConnection -> nMute != 0 ) {
				nSample = pnLutOut[ 0 ];
			}

			nSampleU8 = pnLutAnalyze[ nSample ];
			if ( abs( ( int ) nSampleU8 - 128 ) > nMax ) {
				nMax = abs( ( int ) nSampleU8 - 128 );
			}

			pnRecBuf[ nOutPtr ] = psConnection -> sRecorder.psFile ? pnLutA2S[ nSample ] : 0;

			pnOutBuf[ nOutPtr ] = nSample;
			nOutPtr++;
		}
	}

	/* Record data */
	if ( psConnection -> sRecorder.psFile != NULL && pnRecBuf != NULL ) {
		recordingWrite( &psConnection -> sRecorder, pnRecBuf, nOutPtr, RECORDING_LOCAL );
	}

	fLlRatio = nOutPtr / 400.0f;
	if ( fLlRatio > 1.0 ) {
		fLlRatio = 1.0;
	}

	fLineLevelOutState = fLineLevelOutState * ( 1.0 - fLlRatio ) + ( ( double ) nMax / 128 ) * fLlRatio;

	*pnOutBufLen = nOutPtr;
}
