/* Quad-precision floating point trigonometric functions on <-pi/4,pi/4>.
   Copyright (C) 1999, 2006-2007, 2009-2014 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Jakub Jelinek <jj@ultra.linux.cz>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#include <config.h>

/* Specification.  */
#include "trigl.h"

#include <float.h>
#include <math.h>

/* Code based on glibc/sysdeps/ieee754/ldbl-128/k_sincosl.c
   or            glibc/sysdeps/ieee754/ldbl-128/k_{sin,cos}l.c
   and           glibc/sysdeps/ieee754/ldbl-128/t_sincosl.c.  */

static const long double sin_c[] = {
#define ONE sin_c[0]
  1.00000000000000000000000000000000000E+00L,   /* 3fff0000000000000000000000000000 */

/* cos x ~ ONE + x^2 ( SCOS1 + SCOS2 * x^2 + ... + SCOS4 * x^6 + SCOS5 * x^8 )
   x in <0,1/256>  */
#define SCOS1 sin_c[1]
#define SCOS2 sin_c[2]
#define SCOS3 sin_c[3]
#define SCOS4 sin_c[4]
#define SCOS5 sin_c[5]
  -5.00000000000000000000000000000000000E-01L,  /* bffe0000000000000000000000000000 */
  4.16666666666666666666666666556146073E-02L,   /* 3ffa5555555555555555555555395023 */
  -1.38888888888888888888309442601939728E-03L,  /* bff56c16c16c16c16c16a566e42c0375 */
  2.48015873015862382987049502531095061E-05L,   /* 3fefa01a01a019ee02dcf7da2d6d5444 */
  -2.75573112601362126593516899592158083E-07L,  /* bfe927e4f5dce637cb0b54908754bde0 */

/* sin x ~ ONE * x + x^3 ( SIN1 + SIN2 * x^2 + ... + SIN7 * x^12 + SIN8 * x^14 )
   x in <0,0.1484375>  */
#define SIN1 sin_c[6]
#define SIN2 sin_c[7]
#define SIN3 sin_c[8]
#define SIN4 sin_c[9]
#define SIN5 sin_c[10]
#define SIN6 sin_c[11]
#define SIN7 sin_c[12]
#define SIN8 sin_c[13]
  -1.66666666666666666666666666666666538e-01L,  /* bffc5555555555555555555555555550 */
  8.33333333333333333333333333307532934e-03L,   /* 3ff811111111111111111111110e7340 */
  -1.98412698412698412698412534478712057e-04L,  /* bff2a01a01a01a01a01a019e7a626296 */
  2.75573192239858906520896496653095890e-06L,   /* 3fec71de3a556c7338fa38527474b8f5 */
  -2.50521083854417116999224301266655662e-08L,  /* bfe5ae64567f544e16c7de65c2ea551f */
  1.60590438367608957516841576404938118e-10L,   /* 3fde6124613a811480538a9a41957115 */
  -7.64716343504264506714019494041582610e-13L,  /* bfd6ae7f3d5aef30c7bc660b060ef365 */
  2.81068754939739570236322404393398135e-15L,   /* 3fce9510115aabf87aceb2022a9a9180 */

/* sin x ~ ONE * x + x^3 ( SSIN1 + SSIN2 * x^2 + ... + SSIN4 * x^6 + SSIN5 * x^8 )
   x in <0,1/256>  */
#define SSIN1 sin_c[14]
#define SSIN2 sin_c[15]
#define SSIN3 sin_c[16]
#define SSIN4 sin_c[17]
#define SSIN5 sin_c[18]
  -1.66666666666666666666666666666666659E-01L,  /* bffc5555555555555555555555555555 */
  8.33333333333333333333333333146298442E-03L,   /* 3ff81111111111111111111110fe195d */
  -1.98412698412698412697726277416810661E-04L,  /* bff2a01a01a01a01a019e7121e080d88 */
  2.75573192239848624174178393552189149E-06L,   /* 3fec71de3a556c640c6aaa51aa02ab41 */
  -2.50521016467996193495359189395805639E-08L,  /* bfe5ae644ee90c47dc71839de75b2787 */
};

static const long double cos_c[] = {
/* cos x ~ ONE + x^2 ( COS1 + COS2 * x^2 + ... + COS7 * x^12 + COS8 * x^14 )
   x in <0,0.1484375>  */
#define COS1 cos_c[0]
#define COS2 cos_c[1]
#define COS3 cos_c[2]
#define COS4 cos_c[3]
#define COS5 cos_c[4]
#define COS6 cos_c[5]
#define COS7 cos_c[6]
#define COS8 cos_c[7]
  -4.99999999999999999999999999999999759E-01L,  /* bffdfffffffffffffffffffffffffffb */
  4.16666666666666666666666666651287795E-02L,   /* 3ffa5555555555555555555555516f30 */
  -1.38888888888888888888888742314300284E-03L,  /* bff56c16c16c16c16c16c16a463dfd0d */
  2.48015873015873015867694002851118210E-05L,   /* 3fefa01a01a01a01a0195cebe6f3d3a5 */
  -2.75573192239858811636614709689300351E-07L,  /* bfe927e4fb7789f5aa8142a22044b51f */
  2.08767569877762248667431926878073669E-09L,   /* 3fe21eed8eff881d1e9262d7adff4373 */
  -1.14707451049343817400420280514614892E-11L,  /* bfda9397496922a9601ed3d4ca48944b */
  4.77810092804389587579843296923533297E-14L,   /* 3fd2ae5f8197cbcdcaf7c3fb4523414c */

};

#define SINCOSL_COS_HI 0
#define SINCOSL_COS_LO 1
#define SINCOSL_SIN_HI 2
#define SINCOSL_SIN_LO 3
static const long double sincosl_table[];

long double
kernel_sinl (long double x, long double y, int iy)
{
  long double h, l, z, sin_l, cos_l_m1;
  int index, sign;

  sign = 1;
  if (x < 0)
    {
      x = -x;
      sign = -1;
    }

  if (x < 0.1484375L)           /* |x| < 0.1484375 */
    {
      /* Argument is small enough to approximate it by a Chebyshev
         polynomial of degree 17.  */
      if (x < 0.000000000000000006938893903907228377647697925567626953125L)     /* |x| < 2^-57 */
        if (!((int) x))
          return x;             /* generate inexact */

      z = x * x;
      return x + (x * (z * (SIN1 + z * (SIN2 + z * (SIN3 + z * (SIN4 +
                                                                z * (SIN5 +
                                                                     z *
                                                                     (SIN6 +
                                                                      z *
                                                                      (SIN7 +
                                                                       z *
                                                                       SIN8)))))))));
    }
  else
    {
      /* So that we don't have to use too large polynomial,  we find
         k and l such that x = k + l,  where fabsl(l) <= 1.0/256 with 83
         possible values for k.  We look up cosl(k) and sinl(k) in
         pre-computed tables,  compute cosl(l) and sinl(l) using a
         Chebyshev polynomial of degree 10(11) and compute
         sinl(k+l) = sinl(k)cosl(l) + cosl(k)sinl(l).
         Furthermore write k = 0.1484375 + h.  */
      x -= 0.1484375L;
      index = (int) (x * 128L + 0.5L);
      h = index / 128.0L;
      if (iy)
        l = y - (h - x);
      else
        l = x - h;

      z = l * l;
      sin_l =
        l * (ONE +
             z * (SSIN1 +
                  z * (SSIN2 + z * (SSIN3 + z * (SSIN4 + z * SSIN5)))));
      cos_l_m1 =
        z * (SCOS1 + z * (SCOS2 + z * (SCOS3 + z * (SCOS4 + z * SCOS5))));

      index *= 4;
      /* We rely on this expression not being "contracted" by the compiler
         (cf. ISO C 99 section 6.5 paragraph 8).  */
      z =
        sincosl_table[index + SINCOSL_SIN_HI]
        + (sincosl_table[index + SINCOSL_COS_HI] * sin_l
           + (sincosl_table[index + SINCOSL_SIN_HI] * cos_l_m1
              + (sincosl_table[index + SINCOSL_SIN_LO] * (1 + cos_l_m1)
                 + sincosl_table[index + SINCOSL_COS_LO] * sin_l)));
      return z * sign;
    }
}

long double
kernel_cosl (long double x, long double y)
{
  long double h, l, z, sin_l, cos_l_m1;
  int index;

  if (x < 0)
    x = -x;

  if (x < 0.1484375L)           /* |x| < 0.1484375 */
    {
      /* Argument is small enough to approximate it by a Chebyshev
         polynomial of degree 16.  */
      if (x < 0.000000000000000006938893903907228377647697925567626953125L)     /* |x| < 2^-57 */
        if (!((int) x))
          return ONE;           /* generate inexact */
      z = x * x;
      return ONE + (z * (COS1 + z * (COS2 + z * (COS3 + z * (COS4 +
                                                             z * (COS5 +
                                                                  z * (COS6 +
                                                                       z *
                                                                       (COS7 +
                                                                        z *
                                                                        COS8))))))));
    }
  else
    {
      /* So that we don't have to use too large polynomial,  we find
         k and l such that x = k + l,  where fabsl(l) <= 1.0/256 with 83
         possible values for k.  We look up cosl(k) and sinl(k) in
         pre-computed tables,  compute cosl(l) and sinl(l) using a
         Chebyshev polynomial of degree 10(11) and compute
         cosl(k+l) = cosl(k)cosl(l) - sinl(k)sinl(l).
         Furthermore write k = 0.1484375 + h.  */
      x -= 0.1484375L;
      index = (int) (x * 128L + 0.5L);
      h = index / 128.0L;
      l = y - (h - x);
      z = l * l;
      sin_l =
        l * (ONE +
             z * (SSIN1 +
                  z * (SSIN2 + z * (SSIN3 + z * (SSIN4 + z * SSIN5)))));
      cos_l_m1 =
        z * (SCOS1 + z * (SCOS2 + z * (SCOS3 + z * (SCOS4 + z * SCOS5))));

      index *= 4;
      /* We rely on this expression not being "contracted" by the compiler
         (cf. ISO C 99 section 6.5 paragraph 8).  */
      z =
        sincosl_table [index + SINCOSL_COS_HI]
        - (sincosl_table [index + SINCOSL_SIN_HI] * sin_l
           - (sincosl_table [index + SINCOSL_COS_HI] * cos_l_m1
              + (sincosl_table [index + SINCOSL_COS_LO] * (1 + cos_l_m1)
                 - sincosl_table [index + SINCOSL_SIN_LO] * sin_l)));
      return z;
    }
}


/* For 0.1484375 + n/128.0, n=0..82 this table contains
   first 113 bits of cosine, then at least 113 additional
   bits and the same for sine.
   0.1484375+82.0/128.0 is the smallest number among above defined numbers
   larger than pi/4.
   Computed using gmp.
 */

static const long double sincosl_table[] = {

/* x =  1.48437500000000000000000000000000000e-01L 3ffc3000000000000000000000000000 */
/* cos(x) = 0.fd2f5320e1b790209b4dda2f98f79caaa7b873aff1014b0fbc5243766d03cb006bc837c4358 */
  9.89003367927322909016887196069562069e-01L,   /* 3ffefa5ea641c36f2041369bb45f31ef */
  2.15663692029265697782289400027743703e-35L,   /* 3f8bcaaa7b873aff1014b0fbc5243767 */
/* sin(x) = 0.25dc50bc95711d0d9787d108fd438cf5959ee0bfb7a1e36e8b1a112968f356657420e9cc9ea */
  1.47892995873409608580026675734609314e-01L,   /* 3ffc2ee285e4ab88e86cbc3e8847ea1c */
  9.74950446464233268291647449768590886e-36L,   /* 3f8a9eb2b3dc17f6f43c6dd16342252d */

/* x = 1.56250000000000000000000000000000000e-01 3ffc4000000000000000000000000000 */
/* cos(x) = 0.fce1a053e621438b6d60c76e8c45bf0a9dc71aa16f922acc10e95144ec796a249813c9cb649 */
  9.87817783816471944100503034363211317e-01L,   /* 3ffef9c340a7cc428716dac18edd188b */
  4.74271307836705897892468107620526395e-35L,   /* 3f8cf854ee38d50b7c915660874a8a27 */
/* sin(x) = 0.27d66258bacd96a3eb335b365c87d59438c5142bb56a489e9b8db9d36234ffdebb6bdc22d8e */
  1.55614992773556041209920643203516258e-01L,   /* 3ffc3eb312c5d66cb51f599ad9b2e43f */
  -7.83989563419287980121718050629497270e-36L,  /* bf8a4d78e75d7a8952b6ec2c8e48c594 */

/* x = 1.64062500000000000000000000000000000e-01 3ffc5000000000000000000000000000 */
/* cos(x) = 0.fc8ffa01ba6807417e05962b0d9fdf1fddb0cc4c07d22e19e08019bffa50a6c7acdb40307a3 */
  9.86571908399497588757337407495308409e-01L,   /* 3ffef91ff40374d00e82fc0b2c561b40 */
  -2.47327949936985362476252401212720725e-35L,  /* bf8c070112799d9fc16e8f30fbff3200 */
/* sin(x) = 0.29cfd49b8be4f665276cab01cbf0426934906c3dd105473b226e410b1450f62e53ff7c6cce1 */
  1.63327491736612850846866172454354370e-01L,   /* 3ffc4e7ea4dc5f27b3293b65580e5f82 */
  1.81380344301155485770367902300754350e-36L,   /* 3f88349a48361ee882a39d913720858a */

/* x = 1.71875000000000000000000000000000000e-01 3ffc6000000000000000000000000000 */
/* cos(x) = 0.fc3a6170f767ac735d63d99a9d439e1db5e59d3ef153a4265d5855850ed82b536bf361b80e3 */
  9.85265817718213816204294709759578994e-01L,   /* 3ffef874c2e1eecf58e6bac7b3353a87 */
  2.26568029505818066141517497778527952e-35L,   /* 3f8be1db5e59d3ef153a4265d5855851 */
/* sin(x) = 0.2bc89f9f424de5485de7ce03b2514952b9faf5648c3244d4736feb95dbb9da49f3b58a9253b */
  1.71030022031395019281347969239834331e-01L,   /* 3ffc5e44fcfa126f2a42ef3e701d928a */
  7.01395875187487608875416030203241317e-36L,   /* 3f8a2a573f5eac9186489a8e6dfd72bb */

/* x = 1.79687500000000000000000000000000000e-01 3ffc7000000000000000000000000000 */
/* cos(x) = 0.fbe0d7f7fef11e70aa43b8abf4f6a457cea20c8f3f676b47781f9821bbe9ce04b3c7b981c0b */
  9.83899591489663972178309351416487245e-01L,   /* 3ffef7c1afeffde23ce154877157e9ed */
  2.73414318948066207810486330723761265e-35L,   /* 3f8c22be75106479fb3b5a3bc0fcc10e */
/* sin(x) = 0.2dc0bb80b49a97ffb34e8dd1f8db9df7af47ed2dcf58b12c8e7827e048cae929da02c04ecac */
  1.78722113535153659375356241864180724e-01L,   /* 3ffc6e05dc05a4d4bffd9a746e8fc6dd */
  -1.52906926517265103202547561260594148e-36L,  /* bf8804285c09691853a769b8c3ec0fdc */

/* x = 1.87500000000000000000000000000000000e-01 3ffc8000000000000000000000000000 */
/* cos(x) = 0.fb835efcf670dd2ce6fe7924697eea13ea358867e9cdb3899b783f4f9f43aa5626e8b67b3bc */
  9.82473313101255257487327683243622495e-01L,   /* 3ffef706bdf9ece1ba59cdfcf248d2fe */
  -1.64924358891557584625463868014230342e-35L,  /* bf8b5ec15ca779816324c766487c0b06 */
/* sin(x) = 0.2fb8205f75e56a2b56a1c4792f856258769af396e0189ef72c05e4df59a6b00e4b44a6ea515 */
  1.86403296762269884552379983103205261e-01L,   /* 3ffc7dc102fbaf2b515ab50e23c97c2b */
  1.76460304806826780010586715975331753e-36L,   /* 3f882c3b4d79cb700c4f7b9602f26fad */

/* x = 1.95312500000000000000000000000000000e-01 3ffc9000000000000000000000000000 */
/* cos(x) = 0.fb21f7f5c156696b00ac1fe28ac5fd76674a92b4df80d9c8a46c684399005deccc41386257c */
  9.80987069605669190469329896435309665e-01L,   /* 3ffef643efeb82acd2d601583fc5158c */
  -1.90899259410096419886996331536278461e-36L,  /* bf8844cc5ab6a5903f931badc9cbde34 */
/* sin(x) = 0.31aec65df552876f82ece9a2356713246eba6799983d7011b0b3698d6e1da919c15d57c30c1 */
  1.94073102892909791156055200214145404e-01L,   /* 3ffc8d7632efaa943b7c17674d11ab39 */
  -9.67304741051998267208945242944928999e-36L,  /* bf8a9b7228b30cccf851fdc9e992ce52 */

/* x = 2.03125000000000000000000000000000000e-01 3ffca000000000000000000000000000 */
/* cos(x) = 0.fabca467fb3cb8f1d069f01d8ea33ade5bfd68296ecd1cc9f7b7609bbcf3676e726c3301334 */
  9.79440951715548359998530954502987493e-01L,   /* 3ffef57948cff67971e3a0d3e03b1d46 */
  4.42878056591560757066844797290067990e-35L,   /* 3f8cd6f2dfeb414b7668e64fbdbb04de */
/* sin(x) = 0.33a4a5a19d86246710f602c44df4fa513f4639ce938477aeeabb82e8e0a7ed583a188879fd4 */
  2.01731063801638804725038151164000971e-01L,   /* 3ffc9d252d0cec31233887b016226fa8 */
  -4.27513434754966978435151290617384120e-36L,  /* bf896bb02e718c5b1ee21445511f45c8 */

/* x = 2.10937500000000000000000000000000000e-01 3ffcb000000000000000000000000000 */
/* cos(x) = 0.fa5365e8f1d3ca27be1db5d76ae64d983d7470a4ab0f4ccf65a2b8c67a380df949953a09bc1 */
  9.77835053797959793331971572944454549e-01L,   /* 3ffef4a6cbd1e3a7944f7c3b6baed5cd */
  -3.79207422905180416937210853779192702e-35L,  /* bf8c933e145c7adaa7859984d2ea39cc */
/* sin(x) = 0.3599b652f40ec999df12a0a4c8561de159c98d4e54555de518b97f48886f715d8df5f4f093e */
  2.09376712085993643711890752724881652e-01L,   /* 3ffcaccdb297a0764ccef895052642b1 */
  -1.59470287344329449965314638482515925e-36L,  /* bf880f531b3958d5d5510d73a3405bbc */

/* x = 2.18750000000000000000000000000000000e-01 3ffcc000000000000000000000000000 */
/* cos(x) = 0.f9e63e1d9e8b6f6f2e296bae5b5ed9c11fd7fa2fe11e09fc7bde901abed24b6365e72f7db4e */
  9.76169473868635276723989035435135534e-01L,   /* 3ffef3cc7c3b3d16dede5c52d75cb6be */
  -2.87727974249481583047944860626985460e-35L,  /* bf8c31f701402e80f70fb01c210b7f2a */
/* sin(x) = 0.378df09db8c332ce0d2b53d865582e4526ea336c768f68c32b496c6d11c1cd241bb9f1da523 */
  2.17009581095010156760578095826055396e-01L,   /* 3ffcbc6f84edc6199670695a9ec32ac1 */
  1.07356488794216831812829549198201194e-35L,   /* 3f8ac8a4dd466d8ed1ed1865692d8da2 */

/* x = 2.26562500000000000000000000000000000e-01 3ffcd000000000000000000000000000 */
/* cos(x) = 0.f9752eba9fff6b98842beadab054a932fb0f8d5b875ae63d6b2288d09b148921aeb6e52f61b */
  9.74444313585988980349711056045434344e-01L,   /* 3ffef2ea5d753ffed7310857d5b560a9 */
  3.09947905955053419304514538592548333e-35L,   /* 3f8c4997d87c6adc3ad731eb59144685 */
/* sin(x) = 0.39814cb10513453cb97b21bc1ca6a337b150c21a675ab85503bc09a436a10ab1473934e20c8 */
  2.24629204957705292350428549796424820e-01L,   /* 3ffccc0a6588289a29e5cbd90de0e535 */
  2.42061510849297469844695751870058679e-36L,   /* 3f889bd8a8610d33ad5c2a81de04d21b */

/* x = 2.34375000000000000000000000000000000e-01 3ffce000000000000000000000000000 */
/* cos(x) = 0.f90039843324f9b940416c1984b6cbed1fc733d97354d4265788a86150493ce657cae032674 */
  9.72659678244912752670913058267565260e-01L,   /* 3ffef20073086649f3728082d833096e */
  -3.91759231819314904966076958560252735e-35L,  /* bf8ca09701c6613465595ecd43babcf5 */
/* sin(x) = 0.3b73c2bf6b4b9f668ef9499c81f0d965087f1753fa64b086e58cb8470515c18c1412f8c2e02 */
  2.32235118611511462413930877746235872e-01L,   /* 3ffcdb9e15fb5a5cfb3477ca4ce40f87 */
  -4.96930483364191020075024624332928910e-36L,  /* bf89a6bde03a2b0166d3de469cd1ee3f */

/* x = 2.42187500000000000000000000000000000e-01 3ffcf000000000000000000000000000 */
/* cos(x) = 0.f887604e2c39dbb20e4ec5825059a789ffc95b275ad9954078ba8a28d3fcfe9cc2c1d49697b */
  9.70815676770349462947490545785046027e-01L,   /* 3ffef10ec09c5873b7641c9d8b04a0b3 */
  2.97458820972393859125277682021202860e-35L,   /* 3f8c3c4ffe4ad93ad6ccaa03c5d45147 */
/* sin(x) = 0.3d654aff15cb457a0fca854698aba33039a8a40626609204472d9d40309b626eccc6dff0ffa */
  2.39826857830661564441369251810886574e-01L,   /* 3ffceb2a57f8ae5a2bd07e542a34c55d */
  2.39867036569896287240938444445071448e-36L,   /* 3f88981cd45203133049022396cea018 */

/* x = 2.50000000000000000000000000000000000e-01 3ffd0000000000000000000000000000 */
/* cos(x) = 0.f80aa4fbef750ba783d33cb95f94f8a41426dbe79edc4a023ef9ec13c944551c0795b84fee1 */
  9.68912421710644784144595449494189205e-01L,   /* 3ffef01549f7deea174f07a67972bf2a */
  -5.53634706113461989398873287749326500e-36L,  /* bf89d6faf649061848ed7f704184fb0e */
/* sin(x) = 0.3f55dda9e62aed7513bd7b8e6a3d1635dd5676648d7db525898d7086af9330f03c7f285442a */
  2.47403959254522929596848704849389203e-01L,   /* 3ffcfaaeed4f31576ba89debdc7351e9 */
  -7.36487001108599532943597115275811618e-36L,  /* bf8a39445531336e50495b4ece51ef2a */

/* x = 2.57812500000000000000000000000000000e-01 3ffd0800000000000000000000000000 */
/* cos(x) = 0.f78a098069792daabc9ee42591b7c5a68cb1ab822aeb446b3311b4ba5371b8970e2c1547ad7 */
  9.66950029230677822008341623610531503e-01L,   /* 3ffeef141300d2f25b55793dc84b2370 */
  -4.38972214432792412062088059990480514e-35L,  /* bf8cd2cb9a72a3eea8a5dca667725a2d */
/* sin(x) = 0.414572fd94556e6473d620271388dd47c0ba050cdb5270112e3e370e8c4705ae006426fb5d5 */
  2.54965960415878467487556574864872628e-01L,   /* 3ffd0515cbf65155b991cf58809c4e23 */
  2.20280377918534721005071688328074154e-35L,   /* 3f8bd47c0ba050cdb5270112e3e370e9 */

/* x = 2.65625000000000000000000000000000000e-01 3ffd1000000000000000000000000000 */
/* cos(x) = 0.f7058fde0788dfc805b8fe88789e4f4253e3c50afe8b22f41159620ab5940ff7df9557c0d1f */
  9.64928619104771009581074665315748371e-01L,   /* 3ffeee0b1fbc0f11bf900b71fd10f13d */
  -3.66685832670820775002475545602761113e-35L,  /* bf8c85ed60e1d7a80ba6e85f7534efaa */
/* sin(x) = 0.4334033bcd90d6604f5f36c1d4b84451a87150438275b77470b50e5b968fa7962b5ffb379b7 */
  2.62512399769153281450949626395692931e-01L,   /* 3ffd0cd00cef364359813d7cdb0752e1 */
  3.24923677072031064673177178571821843e-36L,   /* 3f89146a1c5410e09d6ddd1c2d4396e6 */

/* x = 2.73437500000000000000000000000000000e-01 3ffd1800000000000000000000000000 */
/* cos(x) = 0.f67d3a26af7d07aa4bd6d42af8c0067fefb96d5b46c031eff53627f215ea3242edc3f2e13eb */
  9.62848314709379699899701093480214365e-01L,   /* 3ffeecfa744d5efa0f5497ada855f180 */
  4.88986966383343450799422013051821394e-36L,   /* 3f899ffbee5b56d1b00c7bfd4d89fc85 */
/* sin(x) = 0.452186aa5377ab20bbf2524f52e3a06a969f47166ab88cf88c111ad12c55941021ef3317a1a */
  2.70042816718585031552755063618827102e-01L,   /* 3ffd14861aa94ddeac82efc9493d4b8f */
  -2.37608892440611310321138680065803162e-35L,  /* bf8bf956960b8e99547730773eee52ed */

/* x = 2.81250000000000000000000000000000000e-01 3ffd2000000000000000000000000000 */
/* cos(x) = 0.f5f10a7bb77d3dfa0c1da8b57842783280d01ce3c0f82bae3b9d623c168d2e7c29977994451 */
  9.60709243015561903066659350581313472e-01L,   /* 3ffeebe214f76efa7bf4183b516af085 */
  -5.87011558231583960712013351601221840e-36L,  /* bf89f35fcbf8c70fc1f5147118a770fa */
/* sin(x) = 0.470df5931ae1d946076fe0dcff47fe31bb2ede618ebc607821f8462b639e1f4298b5ae87fd3 */
  2.77556751646336325922023446828128568e-01L,   /* 3ffd1c37d64c6b8765181dbf8373fd20 */
  -1.35848595468998128214344668770082997e-36L,  /* bf87ce44d1219e71439f87de07b9d49c */

/* x = 2.89062500000000000000000000000000000e-01 3ffd2800000000000000000000000000 */
/* cos(x) = 0.f561030ddd7a78960ea9f4a32c6521554995667f5547bafee9ec48b3155cdb0f7fd00509713 */
  9.58511534581228627301969408154919822e-01L,   /* 3ffeeac2061bbaf4f12c1d53e94658ca */
  2.50770779371636481145735089393154404e-35L,   /* 3f8c0aaa4cab33faaa3dd7f74f624599 */
/* sin(x) = 0.48f948446abcd6b0f7fccb100e7a1b26eccad880b0d24b59948c7cdd49514d44b933e6985c2 */
  2.85053745940547424587763033323252561e-01L,   /* 3ffd23e52111aaf35ac3dff32c4039e8 */
  2.04269325885902918802700123680403749e-35L,   /* 3f8bb26eccad880b0d24b59948c7cdd5 */

/* x = 2.96875000000000000000000000000000000e-01 3ffd3000000000000000000000000000 */
/* cos(x) = 0.f4cd261d3e6c15bb369c8758630d2ac00b7ace2a51c0631bfeb39ed158ba924cc91e259c195 */
  9.56255323543175296975599942263028361e-01L,   /* 3ffee99a4c3a7cd82b766d390eb0c61a */
  3.21616572190865997051103645135837207e-35L,   /* 3f8c56005bd671528e0318dff59cf68b */
/* sin(x) = 0.4ae37710fad27c8aa9c4cf96c03519b9ce07dc08a1471775499f05c29f86190aaebaeb9716e */
  2.92533342023327543624702326493913423e-01L,   /* 3ffd2b8ddc43eb49f22aa7133e5b00d4 */
  1.93539408668704450308003687950685128e-35L,   /* 3f8b9b9ce07dc08a1471775499f05c2a */

/* x = 3.04687500000000000000000000000000000e-01 3ffd3800000000000000000000000000 */
/* cos(x) = 0.f43575f94d4f6b272f5fb76b14d2a64ab52df1ee8ddf7c651034e5b2889305a9ea9015d758a */
  9.53940747608894733981324795987611623e-01L,   /* 3ffee86aebf29a9ed64e5ebf6ed629a5 */
  2.88075689052478602008395972924657164e-35L,   /* 3f8c3255a96f8f746efbe32881a72d94 */
/* sin(x) = 0.4ccc7a50127e1de0cb6b40c302c651f7bded4f9e7702b0471ae0288d091a37391950907202f */
  2.99995083378683051163248282011699944e-01L,   /* 3ffd3331e94049f877832dad030c0b19 */
  1.35174265535697850139283361475571050e-35L,   /* 3f8b1f7bded4f9e7702b0471ae0288d1 */

/* x = 3.12500000000000000000000000000000000e-01 3ffd4000000000000000000000000000 */
/* cos(x) = 0.f399f500c9e9fd37ae9957263dab8877102beb569f101ee4495350868e5847d181d50d3cca2 */
  9.51567948048172202145488217364270962e-01L,   /* 3ffee733ea0193d3fa6f5d32ae4c7b57 */
  6.36842628598115658308749288799884606e-36L,   /* 3f8a0ee2057d6ad3e203dc892a6a10d2 */
/* sin(x) = 0.4eb44a5da74f600207aaa090f0734e288603ffadb3eb2542a46977b105f8547128036dcf7f0 */
  3.07438514580380850670502958201982091e-01L,   /* 3ffd3ad129769d3d80081eaa8243c1cd */
  1.06515172423204645839241099453417152e-35L,   /* 3f8ac510c07ff5b67d64a8548d2ef621 */

/* x = 3.20312500000000000000000000000000000e-01 3ffd4800000000000000000000000000 */
/* cos(x) = 0.f2faa5a1b74e82fd61fa05f9177380e8e69b7b15a945e8e5ae1124bf3d12b0617e03af4fab5 */
  9.49137069684463027665847421762105623e-01L,   /* 3ffee5f54b436e9d05fac3f40bf22ee7 */
  6.84433965991637152250309190468859701e-37L,   /* 3f86d1cd36f62b528bd1cb5c22497e7a */
/* sin(x) = 0.509adf9a7b9a5a0f638a8fa3a60a199418859f18b37169a644fdb986c21ecb00133853bc35b */
  3.14863181319745250865036315126939016e-01L,   /* 3ffd426b7e69ee69683d8e2a3e8e9828 */
  1.92431240212432926993057705062834160e-35L,   /* 3f8b99418859f18b37169a644fdb986c */

/* x = 3.28125000000000000000000000000000000e-01 3ffd5000000000000000000000000000 */
/* cos(x) = 0.f2578a595224dd2e6bfa2eb2f99cc674f5ea6f479eae2eb580186897ae3f893df1113ca06b8 */
  9.46648260886053321846099507295532976e-01L,   /* 3ffee4af14b2a449ba5cd7f45d65f33a */
  -4.32906339663000890941529420498824645e-35L,  /* bf8ccc5850ac85c30a8e8a53ff3cbb43 */
/* sin(x) = 0.5280326c3cf481823ba6bb08eac82c2093f2bce3c4eb4ee3dec7df41c92c8a4226098616075 */
  3.22268630433386625687745919893188031e-01L,   /* 3ffd4a00c9b0f3d20608ee9aec23ab21 */
  -1.49505897804759263483853908335500228e-35L,  /* bf8b3df6c0d431c3b14b11c213820be3 */

/* x = 3.35937500000000000000000000000000000e-01 3ffd5800000000000000000000000000 */
/* cos(x) = 0.f1b0a5b406b526d886c55feadc8d0dcc8eb9ae2ac707051771b48e05b25b000009660bdb3e3 */
  9.44101673557004345630017691253124860e-01L,   /* 3ffee3614b680d6a4db10d8abfd5b91a */
  1.03812535240120229609822461172145584e-35L,   /* 3f8ab991d735c558e0e0a2ee3691c0b6 */
/* sin(x) = 0.54643b3da29de9b357155eef0f332fb3e66c83bf4dddd9491c5eb8e103ccd92d6175220ed51 */
  3.29654409930860171914317725126463176e-01L,   /* 3ffd5190ecf68a77a6cd5c557bbc3ccd */
  -1.22606996784743214973082192294232854e-35L,  /* bf8b04c19937c40b22226b6e3a1471f0 */

/* x = 3.43750000000000000000000000000000000e-01 3ffd6000000000000000000000000000 */
/* cos(x) = 0.f105fa4d66b607a67d44e042725204435142ac8ad54dfb0907a4f6b56b06d98ee60f19e557a */
  9.41497463127881068644511236053670815e-01L,   /* 3ffee20bf49acd6c0f4cfa89c084e4a4 */
  3.20709366603165602071590241054884900e-36L,   /* 3f8910d450ab22b5537ec241e93dad5b */
/* sin(x) = 0.5646f27e8bd65cbe3a5d61ff06572290ee826d9674a00246b05ae26753cdfc90d9ce81a7d02 */
  3.37020069022253076261281754173810024e-01L,   /* 3ffd591bc9fa2f5972f8e97587fc195d */
  -2.21435756148839473677777545049890664e-35L,  /* bf8bd6f117d92698b5ffdb94fa51d98b */

/* x = 3.51562500000000000000000000000000000e-01 3ffd6800000000000000000000000000 */
/* cos(x) = 0.f0578ad01ede707fa39c09dc6b984afef74f3dc8d0efb0f4c5a6b13771145b3e0446fe33887 */
  9.38835788546265488632578305984712554e-01L,   /* 3ffee0af15a03dbce0ff473813b8d731 */
  -3.98758068773974031348585072752245458e-35L,  /* bf8ca808458611b978827859d2ca7644 */
/* sin(x) = 0.582850a41e1dd46c7f602ea244cdbbbfcdfa8f3189be794dda427ce090b5f85164f1f80ac13 */
  3.44365158145698408207172046472223747e-01L,   /* 3ffd60a14290787751b1fd80ba891337 */
  -3.19791885005480924937758467594051927e-36L,  /* bf89100c815c339d9061ac896f60c7dc */

/* x = 3.59375000000000000000000000000000000e-01 3ffd7000000000000000000000000000 */
/* cos(x) = 0.efa559f5ec3aec3a4eb03319278a2d41fcf9189462261125fe6147b078f1daa0b06750a1654 */
  9.36116812267055290294237411019508588e-01L,   /* 3ffedf4ab3ebd875d8749d6066324f14 */
  3.40481591236710658435409862439032162e-35L,   /* 3f8c6a0fe7c8c4a31130892ff30a3d84 */
/* sin(x) = 0.5a084e28e35fda2776dfdbbb5531d74ced2b5d17c0b1afc4647529d50c295e36d8ceec126c1 */
  3.51689228994814059222584896955547016e-01L,   /* 3ffd682138a38d7f689ddb7f6eed54c7 */
  1.75293433418270210567525412802083294e-35L,   /* 3f8b74ced2b5d17c0b1afc4647529d51 */

/* x = 3.67187500000000000000000000000000000e-01 3ffd7800000000000000000000000000 */
/* cos(x) = 0.eeef6a879146af0bf9b95ea2ea0ac0d3e2e4d7e15d93f48cbd41bf8e4fded40bef69e19eafa */
  9.33340700242548435655299229469995527e-01L,   /* 3ffeddded50f228d5e17f372bd45d416 */
  -4.75255707251679831124800898831382223e-35L,  /* bf8cf960e8d940f513605b9a15f2038e */
/* sin(x) = 0.5be6e38ce8095542bc14ee9da0d36483e6734bcab2e07624188af5653f114eeb46738fa899d */
  3.58991834546065053677710299152868941e-01L,   /* 3ffd6f9b8e33a025550af053ba76834e */
  -2.06772389262723368139416970257112089e-35L,  /* bf8bb7c198cb4354d1f89dbe7750a9ac */

/* x = 3.75000000000000000000000000000000000e-01 3ffd8000000000000000000000000000 */
/* cos(x) = 0.ee35bf5ccac89052cd91ddb734d3a47e262e3b609db604e217053803be0091e76daf28a89b7 */
  9.30507621912314291149476792229555481e-01L,   /* 3ffedc6b7eb9959120a59b23bb6e69a7 */
  2.74541088551732982573335285685416092e-35L,   /* 3f8c23f13171db04edb02710b829c01e */
/* sin(x) = 0.5dc40955d9084f48a94675a2498de5d851320ff5528a6afb3f2e24de240fce6cbed1ba0ccd6 */
  3.66272529086047561372909351716264177e-01L,   /* 3ffd7710255764213d22a519d6892638 */
  -1.96768433534936592675897818253108989e-35L,  /* bf8ba27aecdf00aad759504c0d1db21e */

/* x = 3.82812500000000000000000000000000000e-01 3ffd8800000000000000000000000000 */
/* cos(x) = 0.ed785b5c44741b4493c56bcb9d338a151c6f6b85d8f8aca658b28572c162b199680eb9304da */
  9.27617750192851909628030798799961350e-01L,   /* 3ffedaf0b6b888e83689278ad7973a67 */
  7.58520371916345756281201167126854712e-36L,   /* 3f8a42a38ded70bb1f1594cb1650ae58 */
/* sin(x) = 0.5f9fb80f21b53649c432540a50e22c53057ff42ae0fdf1307760dc0093f99c8efeb2fbd7073 */
  3.73530868238692946416839752660848112e-01L,   /* 3ffd7e7ee03c86d4d92710c950294389 */
  -1.48023494778986556048879113411517128e-35L,  /* bf8b3acfa800bd51f020ecf889f23ff7 */

/* x = 3.90625000000000000000000000000000000e-01 3ffd9000000000000000000000000000 */
/* cos(x) = 0.ecb7417b8d4ee3fec37aba4073aa48f1f14666006fb431d9671303c8100d10190ec8179c41d */
  9.24671261467036098502113014560138771e-01L,   /* 3ffed96e82f71a9dc7fd86f57480e755 */
  -4.14187124860031825108649347251175815e-35L,  /* bf8cb87075cccffc825e7134c767e1bf */
/* sin(x) = 0.6179e84a09a5258a40e9b5face03e525f8b5753cd0105d93fe6298010c3458e84d75fe420e9 */
  3.80766408992390192057200703388896675e-01L,   /* 3ffd85e7a1282694962903a6d7eb3810 */
  -2.02009541175208636336924533372496107e-35L,  /* bf8bada074a8ac32fefa26c019d67fef */

/* x = 3.98437500000000000000000000000000000e-01 3ffd9800000000000000000000000000 */
/* cos(x) = 0.ebf274bf0bda4f62447e56a093626798d3013b5942b1abfd155aacc9dc5c6d0806a20d6b9c1 */
  9.21668335573351918175411368202712714e-01L,   /* 3ffed7e4e97e17b49ec488fcad4126c5 */
  -1.83587995433957622948710263541479322e-35L,  /* bf8b8672cfec4a6bd4e5402eaa553362 */
/* sin(x) = 0.6352929dd264bd44a02ea766325d8aa8bd9695fc8def3caefba5b94c9a3c873f7b2d3776ead */
  3.87978709727025046051079690813741960e-01L,   /* 3ffd8d4a4a774992f51280ba9d98c976 */
  8.01904783870935075844443278617586301e-36L,   /* 3f8a5517b2d2bf91bde795df74b72993 */

/* x = 4.06250000000000000000000000000000000e-01 3ffda000000000000000000000000000 */
/* cos(x) = 0.eb29f839f201fd13b93796827916a78f15c85230a4e8ea4b21558265a14367e1abb4c30695a */
  9.18609155794918267837824977718549863e-01L,   /* 3ffed653f073e403fa27726f2d04f22d */
  2.97608282778274433460057745798409849e-35L,   /* 3f8c3c78ae429185274752590aac132d */
/* sin(x) = 0.6529afa7d51b129631ec197c0a840a11d7dc5368b0a47956feb285caa8371c4637ef17ef01b */
  3.95167330240934236244832640419653657e-01L,   /* 3ffd94a6be9f546c4a58c7b065f02a10 */
  7.57560031388312550940040194042627704e-36L,   /* 3f8a423afb8a6d16148f2adfd650b955 */

/* x = 4.14062500000000000000000000000000000e-01 3ffda800000000000000000000000000 */
/* cos(x) = 0.ea5dcf0e30cf03e6976ef0b1ec26515fba47383855c3b4055a99b5e86824b2cd1a691fdca7b */
  9.15493908848301228563917732180221882e-01L,   /* 3ffed4bb9e1c619e07cd2edde163d84d */
  -3.50775517955306954815090901168305659e-35L,  /* bf8c75022dc63e3d51e25fd52b3250bd */
/* sin(x) = 0.66ff380ba0144109e39a320b0a3fa5fd65ea0585bcbf9b1a769a9b0334576c658139e1a1cbe */
  4.02331831777773111217105598880982387e-01L,   /* 3ffd9bfce02e805104278e68c82c28ff */
  -1.95678722882848174723569916504871563e-35L,  /* bf8ba029a15fa7a434064e5896564fcd */

/* x = 4.21875000000000000000000000000000000e-01 3ffdb000000000000000000000000000 */
/* cos(x) = 0.e98dfc6c6be031e60dd3089cbdd18a75b1f6b2c1e97f79225202f03dbea45b07a5ec4efc062 */
  9.12322784872117846492029542047341734e-01L,   /* 3ffed31bf8d8d7c063cc1ba611397ba3 */
  7.86903886556373674267948132178845568e-36L,   /* 3f8a4eb63ed6583d2fef244a405e07b8 */
/* sin(x) = 0.68d32473143327973bc712bcc4ccddc47630d755850c0655243b205934dc49ffed8eb76adcb */
  4.09471777053295066122694027011452236e-01L,   /* 3ffda34c91cc50cc9e5cef1c4af31333 */
  2.23945241468457597921655785729821354e-35L,   /* 3f8bdc47630d755850c0655243b20593 */

/* x = 4.29687500000000000000000000000000000e-01 3ffdb800000000000000000000000000 */
/* cos(x) = 0.e8ba8393eca7821aa563d83491b6101189b3b101c3677f73d7bad7c10f9ee02b7ab4009739a */
  9.09095977415431051650381735684476417e-01L,   /* 3ffed1750727d94f04354ac7b069236c */
  1.20886014028444155733776025085677953e-35L,   /* 3f8b01189b3b101c3677f73d7bad7c11 */
/* sin(x) = 0.6aa56d8e8249db4eb60a761fe3f9e559be456b9e13349ca99b0bfb787f22b95db3b70179615 */
  4.16586730282041119259112448831069657e-01L,   /* 3ffdaa95b63a09276d3ad829d87f8fe8 */
  -2.00488106831998813675438269796963612e-35L,  /* bf8baa641ba9461eccb635664f404878 */

/* x = 4.37500000000000000000000000000000000e-01 3ffdc000000000000000000000000000 */
/* cos(x) = 0.e7e367d2956cfb16b6aa11e5419cd0057f5c132a6455bf064297e6a76fe2b72bb630d6d50ff */
  9.05813683425936420744516660652700258e-01L,   /* 3ffecfc6cfa52ad9f62d6d5423ca833a */
  -3.60950307605941169775676563004467163e-35L,  /* bf8c7fd4051f66acdd5207cdeb40cac5 */
/* sin(x) = 0.6c760c14c8585a51dbd34660ae6c52ac7036a0b40887a0b63724f8b4414348c3063a637f457 */
  4.23676257203938010361683988031102480e-01L,   /* 3ffdb1d83053216169476f4d1982b9b1 */
  1.40484456388654470329473096579312595e-35L,   /* 3f8b2ac7036a0b40887a0b63724f8b44 */

/* x = 4.45312500000000000000000000000000000e-01 3ffdc800000000000000000000000000 */
/* cos(x) = 0.e708ac84d4172a3e2737662213429e14021074d7e702e77d72a8f1101a7e70410df8273e9aa */
  9.02476103237941504925183272675895999e-01L,   /* 3ffece115909a82e547c4e6ecc442685 */
  2.26282899501344419018306295680210602e-35L,   /* 3f8be14021074d7e702e77d72a8f1102 */
/* sin(x) = 0.6e44f8c36eb10a1c752d093c00f4d47ba446ac4c215d26b0316442f168459e677d06e7249e3 */
  4.30739925110803197216321517850849190e-01L,   /* 3ffdb913e30dbac42871d4b424f003d3 */
  1.54096780001629398850891218396761548e-35L,   /* 3f8b47ba446ac4c215d26b0316442f17 */

/* x = 4.53125000000000000000000000000000000e-01 3ffdd000000000000000000000000000 */
/* cos(x) = 0.e62a551594b970a770b15d41d4c0e483e47aca550111df6966f9e7ac3a94ae49e6a71eb031e */
  8.99083440560138456216544929209379307e-01L,   /* 3ffecc54aa2b2972e14ee162ba83a982 */
  -2.06772615490904370666670275154751976e-35L,  /* bf8bb7c1b8535aafeee209699061853c */
/* sin(x) = 0.70122c5ec5028c8cff33abf4fd340ccc382e038379b09cf04f9a52692b10b72586060cbb001 */
  4.37777302872755132861618974702796680e-01L,   /* 3ffdc048b17b140a3233fcceafd3f4d0 */
  9.62794364503442612477117426033922467e-36L,   /* 3f8a998705c0706f36139e09f34a4d25 */

/* x = 4.60937500000000000000000000000000000e-01 3ffdd800000000000000000000000000 */
/* cos(x) = 0.e54864fe33e8575cabf5bd0e5cf1b1a8bc7c0d5f61702450fa6b6539735820dd2603ae355d5 */
  8.95635902463170698900570000446256350e-01L,   /* 3ffeca90c9fc67d0aeb957eb7a1cb9e3 */
  3.73593741659866883088620495542311808e-35L,   /* 3f8c8d45e3e06afb0b812287d35b29cc */
/* sin(x) = 0.71dd9fb1ff4677853acb970a9f6729c6e3aac247b1c57cea66c77413f1f98e8b9e98e49d851 */
  4.44787960964527211433056012529525211e-01L,   /* 3ffdc7767ec7fd19de14eb2e5c2a7d9d */
  -1.67187936511493678007508371613954899e-35L,  /* bf8b6391c553db84e3a831599388bec1 */

/* x = 4.68750000000000000000000000000000000e-01 3ffde000000000000000000000000000 */
/* cos(x) = 0.e462dfc670d421ab3d1a15901228f146a0547011202bf5ab01f914431859aef577966bc4fa4 */
  8.92133699366994404723900253723788575e-01L,   /* 3ffec8c5bf8ce1a843567a342b202452 */
  -1.10771937602567314732693079264692504e-35L,  /* bf8ad72bf571fddbfa814a9fc0dd779d */
/* sin(x) = 0.73a74b8f52947b681baf6928eb3fb021769bf4779bad0e3aa9b1cdb75ec60aad9fc63ff19d5 */
  4.51771471491683776581688750134062870e-01L,   /* 3ffdce9d2e3d4a51eda06ebda4a3acff */
  -1.19387223016472295893794387275284505e-35L,  /* bf8afbd12c81710c8a5e38aac9c64914 */

/* x = 4.76562500000000000000000000000000000e-01 3ffde800000000000000000000000000 */
/* cos(x) = 0.e379c9045f29d517c4808aa497c2057b2b3d109e76c0dc302d4d0698b36e3f0bdbf33d8e952 */
  8.88577045028035543317609023116020980e-01L,   /* 3ffec6f39208be53aa2f890115492f84 */
  4.12354278954664731443813655177022170e-36L,   /* 3f895ecacf44279db0370c0b5341a62d */
/* sin(x) = 0.756f28d011d98528a44a75fc29c779bd734ecdfb582fdb74b68a4c4c4be54cfd0b2d3ad292f */
  4.58727408216736592377295028972874773e-01L,   /* 3ffdd5bca340476614a29129d7f0a71e */
  -4.70946994194182908929251719575431779e-36L,  /* bf8990a32c4c8129f40922d25d6ceced */

/* x = 4.84375000000000000000000000000000000e-01 3ffdf000000000000000000000000000 */
/* cos(x) = 0.e28d245c58baef72225e232abc003c4366acd9eb4fc2808c2ab7fe7676cf512ac7f945ae5fb */
  8.84966156526143291697296536966647926e-01L,   /* 3ffec51a48b8b175dee444bc46557800 */
  4.53370570288325630442037826313462165e-35L,   /* 3f8ce21b3566cf5a7e14046155bff3b4 */
/* sin(x) = 0.77353054ca72690d4c6e171fd99e6b39fa8e1ede5f052fd2964534c75340970a3a9cd3c5c32 */
  4.65655346585160182681199512507546779e-01L,   /* 3ffddcd4c15329c9a43531b85c7f667a */
  -1.56282598978971872478619772155305961e-35L,  /* bf8b4c60571e121a0fad02d69bacb38b */

/* x = 4.92187500000000000000000000000000000e-01 3ffdf800000000000000000000000000 */
/* cos(x) = 0.e19cf580eeec046aa1422fa74807ecefb2a1911c94e7b5f20a00f70022d940193691e5bd790 */
  8.81301254251340599140161908298100173e-01L,   /* 3ffec339eb01ddd808d542845f4e9010 */
  -1.43419192312116687783945619009629445e-35L,  /* bf8b3104d5e6ee36b184a0df5ff08ffe */
/* sin(x) = 0.78f95b0560a9a3bd6df7bd981dc38c61224d08bc20631ea932e605e53b579e9e0767dfcbbcb */
  4.72554863751304451146551317808516942e-01L,   /* 3ffde3e56c1582a68ef5b7def660770e */
  9.31324774957768018850224267625371204e-36L,   /* 3f8a8c2449a117840c63d5265cc0bca7 */

/* x = 5.00000000000000000000000000000000000e-01 3ffe0000000000000000000000000000 */
/* cos(x) = 0.e0a94032dbea7cedbddd9da2fafad98556566b3a89f43eabd72350af3e8b19e801204d8fe2e */
  8.77582561890372716116281582603829681e-01L,   /* 3ffec1528065b7d4f9db7bbb3b45f5f6 */
  -2.89484960181363924855192538540698851e-35L,  /* bf8c33d54d4ca62bb05e0aa146e57a86 */
/* sin(x) = 0.7abba1d12c17bfa1d92f0d93f60ded9992f45b4fcaf13cd58b303693d2a0db47db35ae8a3a9 */
  4.79425538604203000273287935215571402e-01L,   /* 3ffdeaee8744b05efe8764bc364fd838 */
  -1.38426977616718318950175848639381926e-35L,  /* bf8b2666d0ba4b0350ec32a74cfc96c3 */

/* x = 5.07812500000000000000000000000000000e-01 3ffe0400000000000000000000000000 */
/* cos(x) = 0.dfb20840f3a9b36f7ae2c515342890b5ec583b8366cc2b55029e95094d31112383f2553498b */
  8.73810306413054508282556837071377159e-01L,   /* 3ffebf641081e75366def5c58a2a6851 */
  1.25716864497849302237218128599994785e-35L,   /* 3f8b0b5ec583b8366cc2b55029e95095 */
/* sin(x) = 0.7c7bfdaf13e5ed17212f8a7525bfb113aba6c0741b5362bb8d59282a850b63716bca0c910f0 */
  4.86266951793275574311011306895834993e-01L,   /* 3ffdf1eff6bc4f97b45c84be29d496ff */
  -1.12269393250914752644352376448094271e-35L,  /* bf8add8a8b27f17c9593a88e54dafaaf */

/* x = 5.15625000000000000000000000000000000e-01 3ffe0800000000000000000000000000 */
/* cos(x) = 0.deb7518814a7a931bbcc88c109cd41c50bf8bb48f20ae8c36628d1d3d57574f7dc58f27d91c */
  8.69984718058417388828915599901466243e-01L,   /* 3ffebd6ea310294f526377991182139b */
  -4.68168638300575626782741319792183837e-35L,  /* bf8cf1d7a03a25b86fa8b9e4ceb97161 */
/* sin(x) = 0.7e3a679daaf25c676542bcb4028d0964172961c921823a4ef0c3a9070d886dbd073f6283699 */
  4.93078685753923057265136552753487121e-01L,   /* 3ffdf8e99e76abc9719d950af2d00a34 */
  7.06498693112535056352301101088624950e-36L,   /* 3f8a2c82e52c3924304749de187520e2 */

/* x = 5.23437500000000000000000000000000000e-01 3ffe0c00000000000000000000000000 */
/* cos(x) = 0.ddb91ff318799172bd2452d0a3889f5169c64a0094bcf0b8aa7dcf0d7640a2eba68955a80be */
  8.66106030320656714696616831654267220e-01L,   /* 3ffebb723fe630f322e57a48a5a14711 */
  2.35610597588322493119667003904687628e-35L,   /* 3f8bf5169c64a0094bcf0b8aa7dcf0d7 */
/* sin(x) = 0.7ff6d8a34bd5e8fa54c97482db5159df1f24e8038419c0b448b9eea8939b5d4dfcf40900257 */
  4.99860324733013463819556536946425724e-01L,   /* 3ffdffdb628d2f57a3e95325d20b6d45 */
  1.94636052312235297538564591686645139e-35L,   /* 3f8b9df1f24e8038419c0b448b9eea89 */

/* x = 5.31250000000000000000000000000000000e-01 3ffe1000000000000000000000000000 */
/* cos(x) = 0.dcb7777ac420705168f31e3eb780ce9c939ecada62843b54522f5407eb7f21e556059fcd734 */
  8.62174479934880504367162510253324274e-01L,   /* 3ffeb96eeef58840e0a2d1e63c7d6f02 */
  -3.71556818317533582234562471835771823e-35L,  /* bf8c8b1b6309a92cebde255d6e855fc1 */
/* sin(x) = 0.81b149ce34caa5a4e650f8d09fd4d6aa74206c32ca951a93074c83b2d294d25dbb0f7fdfad2 */
  5.06611454814257367642296000893867192e-01L,   /* 3ffe0362939c69954b49cca1f1a13faa */
  -3.10963699824274155702706043065967062e-35L,  /* bf8c4aac5efc9e69ab572b67c59be269 */

/* x = 5.39062500000000000000000000000000000e-01 3ffe1400000000000000000000000000 */
/* cos(x) = 0.dbb25c25b8260c14f6e7bc98ec991b70c65335198b0ab628bad20cc7b229d4dd62183cfa055 */
  8.58190306862660347046629564970494649e-01L,   /* 3ffeb764b84b704c1829edcf7931d932 */
  2.06439574601190798155563653000684861e-35L,   /* 3f8bb70c65335198b0ab628bad20cc7b */
/* sin(x) = 0.8369b434a372da7eb5c8a71fe36ce1e0b2b493f6f5cb2e38bcaec2a556b3678c401940d1c3c */
  5.13331663943471218288801270215706878e-01L,   /* 3ffe06d3686946e5b4fd6b914e3fc6da */
  -2.26614796466671970772244932848067224e-35L,  /* bf8be1f4d4b6c090a34d1c743513d5ab */

/* x = 5.46875000000000000000000000000000000e-01 3ffe1800000000000000000000000000 */
/* cos(x) = 0.daa9d20860827063fde51c09e855e9932e1b17143e7244fd267a899d41ae1f3bc6a0ec42e27 */
  8.54153754277385385143451785105103176e-01L,   /* 3ffeb553a410c104e0c7fbca3813d0ac */
  -1.68707534013095152873222061722573172e-35L,  /* bf8b66cd1e4e8ebc18dbb02d9857662c */
/* sin(x) = 0.852010f4f0800521378bd8dd614753d080c2e9e0775ffc609947b9132f5357404f464f06a58 */
  5.20020541953727004760213699874674730e-01L,   /* 3ffe0a4021e9e1000a426f17b1bac28f */
  -3.32415021330884924833711842866896734e-35L,  /* bf8c617bf9e8b0fc45001cfb35c23767 */

/* x = 5.54687500000000000000000000000000000e-01 3ffe1c00000000000000000000000000 */
/* cos(x) = 0.d99ddd44e44a43d4d4a3a3ed95204106fd54d78e8c7684545c0da0b7c2c72be7a89b7c182ad */
  8.50065068549420263957072899177793617e-01L,   /* 3ffeb33bba89c89487a9a94747db2a41 */
  -4.73753917078785974356016104842568442e-35L,  /* bf8cf7c81559438b9c4bdd5d1f92fa42 */
/* sin(x) = 0.86d45935ab396cb4e421e822dee54f3562dfcefeaa782184c23401d231f5ad981a1cc195b18 */
  5.26677680590386730710789410624833901e-01L,   /* 3ffe0da8b26b5672d969c843d045bdcb */
  -3.67066148195515214077582496518566735e-35L,  /* bf8c8654e901880aac3ef3d9ee5ff16e */

/* x = 5.62500000000000000000000000000000000e-01 3ffe2000000000000000000000000000 */
/* cos(x) = 0.d88e820b1526311dd561efbc0c1a9a5375eb26f65d246c5744b13ca26a7e0fd42556da843c8 */
  8.45924499231067954459723078597493262e-01L,   /* 3ffeb11d04162a4c623baac3df781835 */
  1.98054947141989878179164342925274053e-35L,   /* 3f8ba5375eb26f65d246c5744b13ca27 */
/* sin(x) = 0.88868625b4e1dbb2313310133022527200c143a5cb16637cb7daf8ade82459ff2e98511f40f */
  5.33302673536020173329131103308161529e-01L,   /* 3ffe110d0c4b69c3b764626620266045 */
  -3.42715291319551615996993795226755157e-35L,  /* bf8c6c6ff9f5e2d1a74ce41a41283a91 */

/* x = 5.70312500000000000000000000000000000e-01 3ffe2400000000000000000000000000 */
/* cos(x) = 0.d77bc4985e93a607c9d868b906bbc6bbe3a04258814acb0358468b826fc91bd4d814827f65e */
  8.41732299041338366963111794309701085e-01L,   /* 3ffeaef78930bd274c0f93b0d1720d78 */
  -4.30821936750410026005408345400225948e-35L,  /* bf8cca20e2fded3bf5a9a7e53dcba3ed */
/* sin(x) = 0.8a3690fc5bfc11bf9535e2739a8512f448a41251514bbed7fc18d530f9b4650fcbb2861b0aa */
  5.39895116435204405041660709903993340e-01L,   /* 3ffe146d21f8b7f8237f2a6bc4e7350a */
  1.42595803521626714477253741404712093e-35L,   /* 3f8b2f448a41251514bbed7fc18d5310 */

/* x = 5.78125000000000000000000000000000000e-01 3ffe2800000000000000000000000000 */
/* cos(x) = 0.d665a937b4ef2b1f6d51bad6d988a4419c1d7051faf31a9efa151d7631117efac03713f950a */
  8.37488723850523685315353348917240617e-01L,   /* 3ffeaccb526f69de563edaa375adb311 */
  2.72761997872084533045777718677326179e-35L,   /* 3f8c220ce0eb828fd798d4f7d0a8ebb2 */
/* sin(x) = 0.8be472f9776d809af2b88171243d63d66dfceeeb739cc894e023fbc165a0e3f26ff729c5d57 */
  5.46454606919203564403349553749411001e-01L,   /* 3ffe17c8e5f2eedb0135e57102e2487b */
  -2.11870230730160315420936523771864858e-35L,  /* bf8bc29920311148c63376b1fdc043ea */

/* x = 5.85937500000000000000000000000000000e-01 3ffe2c00000000000000000000000000 */
/* cos(x) = 0.d54c3441844897fc8f853f0655f1ba695eba9fbfd7439dbb1171d862d9d9146ca5136f825ac */
  8.33194032664581363070224042208032321e-01L,   /* 3ffeaa98688308912ff91f0a7e0cabe3 */
  4.39440050052045486567668031751259899e-35L,   /* 3f8cd34af5d4fdfeba1cedd88b8ec317 */
/* sin(x) = 0.8d902565817ee7839bce3cd128060119492cd36d42d82ada30d7f8bde91324808377ddbf5d4 */
  5.52980744630527369849695082681623667e-01L,   /* 3ffe1b204acb02fdcf07379c79a2500c */
  8.26624790417342895897164123189984127e-37L,   /* 3f8719492cd36d42d82ada30d7f8bde9 */

/* x = 5.93750000000000000000000000000000000e-01 3ffe3000000000000000000000000000 */
/* cos(x) = 0.d42f6a1b9f0168cdf031c2f63c8d9304d86f8d34cb1d5fccb68ca0f2241427fc18d1fd5bbdf */
  8.28848487609325734810171790119116638e-01L,   /* 3ffea85ed4373e02d19be06385ec791b */
  1.43082508100496581719048175506239770e-35L,   /* 3f8b304d86f8d34cb1d5fccb68ca0f22 */
/* sin(x) = 0.8f39a191b2ba6122a3fa4f41d5a3ffd421417d46f19a22230a14f7fcc8fce5c75b4b28b29d1 */
  5.59473131247366877384844006003116688e-01L,   /* 3ffe1e7343236574c24547f49e83ab48 */
  -1.28922620524163922306886952100992796e-37L,  /* bf845ef5f415c8732eeee7af584019b8 */

/* x = 6.01562500000000000000000000000000000e-01 3ffe3400000000000000000000000000 */
/* cos(x) = 0.d30f4f392c357ab0661c5fa8a7d9b26627846fef214b1d19a22379ff9eddba087cf410eb097 */
  8.24452353914429207485643598212356053e-01L,   /* 3ffea61e9e72586af560cc38bf514fb3 */
  3.79160239225080026987031418939026741e-35L,   /* 3f8c93313c237f790a58e8cd111bcffd */
/* sin(x) = 0.90e0e0d81ca678796cc92c8ea8c2815bc72ca78abe571bfa8576aacc571e096a33237e0e830 */
  5.65931370507905990773159095689276114e-01L,   /* 3ffe21c1c1b0394cf0f2d992591d5185 */
  1.02202775968053982310991962521535027e-36L,   /* 3f875bc72ca78abe571bfa8576aacc57 */

/* x = 6.09375000000000000000000000000000000e-01 3ffe3800000000000000000000000000 */
/* cos(x) = 0.d1ebe81a95ee752e48a26bcd32d6e922d7eb44b8ad2232f6930795e84b56317269b9dd1dfa6 */
  8.20005899897234008255550633876556043e-01L,   /* 3ffea3d7d0352bdcea5c9144d79a65ae */
  -1.72008811955230823416724332297991247e-35L,  /* bf8b6dd2814bb4752ddcd096cf86a17b */
/* sin(x) = 0.9285dc9bc45dd9ea3d02457bcce59c4175aab6ff7929a8d287195525fdace200dba032874fb */
  5.72355068234507240384953706824503608e-01L,   /* 3ffe250bb93788bbb3d47a048af799cb */
  2.12572273479933123944580199464514529e-35L,   /* 3f8bc4175aab6ff7929a8d2871955260 */

/* x = 6.17187500000000000000000000000000000e-01 3ffe3c00000000000000000000000000 */
/* cos(x) = 0.d0c5394d772228195e25736c03574707de0af1ca344b13bd3914bfe27518e9e426f5deff1e1 */
  8.15509396946375476876345384201386217e-01L,   /* 3ffea18a729aee445032bc4ae6d806af */
  -4.28589138410712954051679139949341961e-35L,  /* bf8cc7c10fa871ae5da76216375a00ec */
/* sin(x) = 0.94288e48bd0335fc41c4cbd2920497a8f5d1d8185c99fa0081f90c27e2a53ffdd208a0dbe69 */
  5.78743832357770354521111378581385347e-01L,   /* 3ffe28511c917a066bf8838997a52409 */
  1.77998063432551282609698670002456093e-35L,   /* 3f8b7a8f5d1d8185c99fa0081f90c27e */

/* x = 6.25000000000000000000000000000000000e-01 3ffe4000000000000000000000000000 */
/* cos(x) = 0.cf9b476c897c25c5bfe750dd3f308eaf7bcc1ed00179a256870f4200445043dcdb1974b5878 */
  8.10963119505217902189534803941080724e-01L,   /* 3ffe9f368ed912f84b8b7fcea1ba7e61 */
  1.10481292856794436426051402418804358e-35L,   /* 3f8ad5ef7983da002f344ad0e1e84009 */
/* sin(x) = 0.95c8ef544210ec0b91c49bd2aa09e8515fa61a156ebb10f5f8c232a6445b61ebf3c2ec268f9 */
  5.85097272940462154805399314150080459e-01L,   /* 3ffe2b91dea88421d817238937a55414 */
  -1.78164576278056195136525335403380464e-35L,  /* bf8b7aea059e5ea9144ef0a073dcd59c */

/* x = 6.32812500000000000000000000000000000e-01 3ffe4400000000000000000000000000 */
/* cos(x) = 0.ce6e171f92f2e27f32225327ec440ddaefae248413efc0e58ceee1ae369aabe73f88c87ed1a */
  8.06367345055103913698795406077297399e-01L,   /* 3ffe9cdc2e3f25e5c4fe6444a64fd888 */
  1.04235088143133625463876245029180850e-35L,   /* 3f8abb5df5c490827df81cb19ddc35c7 */
/* sin(x) = 0.9766f93cd18413a6aafc1cfc6fc28abb6817bf94ce349901ae3f48c3215d3eb60acc5f78903 */
  5.91415002201316315087000225758031236e-01L,   /* 3ffe2ecdf279a308274d55f839f8df85 */
  8.07390238063560077355762466502569603e-36L,   /* 3f8a576d02f7f299c6932035c7e91864 */

/* x = 6.40625000000000000000000000000000000e-01 3ffe4800000000000000000000000000 */
/* cos(x) = 0.cd3dad1b5328a2e459f993f4f5108819faccbc4eeba9604e81c7adad51cc8a2561631a06826 */
  8.01722354098418450607492605652964208e-01L,   /* 3ffe9a7b5a36a65145c8b3f327e9ea21 */
  6.09487851305233089325627939458963741e-36L,   /* 3f8a033f599789dd752c09d038f5b5aa */
/* sin(x) = 0.9902a58a45e27bed68412b426b675ed503f54d14c8172e0d373f42cadf04daf67319a7f94be */
  5.97696634538701531238647618967334337e-01L,   /* 3ffe32054b148bc4f7dad0825684d6cf */
  -2.49527608940873714527427941350461554e-35L,  /* bf8c0957e0559759bf468f964605e9a9 */

/* x = 6.48437500000000000000000000000000000e-01 3ffe4c00000000000000000000000000 */
/* cos(x) = 0.cc0a0e21709883a3ff00911e11a07ee3bd7ea2b04e081be99be0264791170761ae64b8b744a */
  7.97028430141468342004642741431945296e-01L,   /* 3ffe98141c42e1310747fe01223c2341 */
  -8.35364432831812599727083251866305534e-37L,  /* bf871c42815d4fb1f7e416641fd9b86f */
/* sin(x) = 0.9a9bedcdf01b38d993f3d7820781de292033ead73b89e28f39313dbe3a6e463f845b5fa8490 */
  6.03941786554156657267270287527367726e-01L,   /* 3ffe3537db9be03671b327e7af040f04 */
  -2.54578992328947177770363936132309779e-35L,  /* bf8c0eb6fe60a94623b0eb863676120e */

/* x = 6.56250000000000000000000000000000000e-01 3ffe5000000000000000000000000000 */
/* cos(x) = 0.cad33f00658fe5e8204bbc0f3a66a0e6a773f87987a780b243d7be83b3db1448ca0e0e62787 */
  7.92285859677178543141501323781709399e-01L,   /* 3ffe95a67e00cb1fcbd04097781e74cd */
  2.47519558228473167879248891673807645e-35L,   /* 3f8c07353b9fc3cc3d3c05921ebdf41e */
/* sin(x) = 0.9c32cba2b14156ef05256c4f857991ca6a547cd7ceb1ac8a8e62a282bd7b9183648a462bd04 */
  6.10150077075791371273742393566183220e-01L,   /* 3ffe386597456282adde0a4ad89f0af3 */
  1.33842237929938963780969418369150532e-35L,   /* 3f8b1ca6a547cd7ceb1ac8a8e62a282c */

/* x = 6.64062500000000000000000000000000000e-01 3ffe5400000000000000000000000000 */
/* cos(x) = 0.c99944936cf48c8911ff93fe64b3ddb7981e414bdaf6aae1203577de44878c62bc3bc9cf7b9 */
  7.87494932167606083931328295965533034e-01L,   /* 3ffe93328926d9e9191223ff27fcc968 */
  -2.57915385618070637156514241185180920e-35L,  /* bf8c12433f0df5a1284aa8f6fe54410e */
/* sin(x) = 0.9dc738ad14204e689ac582d0f85826590feece34886cfefe2e08cf2bb8488d55424dc9d3525 */
  6.16321127181550943005700433761731837e-01L,   /* 3ffe3b8e715a28409cd1358b05a1f0b0 */
  2.88497530050197716298085892460478666e-35L,   /* 3f8c32c87f7671a44367f7f17046795e */

/* x = 6.71875000000000000000000000000000000e-01 3ffe5800000000000000000000000000 */
/* cos(x) = 0.c85c23c26ed7b6f014ef546c47929682122876bfbf157de0aff3c4247d820c746e32cd4174f */
  7.82655940026272796930787447428139026e-01L,   /* 3ffe90b84784ddaf6de029dea8d88f25 */
  1.69332045679237919427807771288506254e-35L,   /* 3f8b682122876bfbf157de0aff3c4248 */
/* sin(x) = 0.9f592e9b66a9cf906a3c7aa3c10199849040c45ec3f0a747597311038101780c5f266059dbf */
  6.22454560222343683041926705090443330e-01L,   /* 3ffe3eb25d36cd539f20d478f5478203 */
  1.91974786921147072717621236192269859e-35L,   /* 3f8b9849040c45ec3f0a747597311038 */

/* x = 6.79687500000000000000000000000000000e-01 3ffe5c00000000000000000000000000 */
/* cos(x) = 0.c71be181ecd6875ce2da5615a03cca207d9adcb9dfb0a1d6c40a4f0056437f1a59ccddd06ee */
  7.77769178600317903122203513685412863e-01L,   /* 3ffe8e37c303d9ad0eb9c5b4ac2b407a */
  -4.05296033424632846931240580239929672e-35L,  /* bf8caefc13291a31027af149dfad87fd */
/* sin(x) = 0.a0e8a725d33c828c11fa50fd9e9a15ffecfad43f3e534358076b9b0f6865694842b1e8c67dc */
  6.28550001845029662028004327939032867e-01L,   /* 3ffe41d14e4ba679051823f4a1fb3d34 */
  1.65507421184028099672784511397428852e-35L,   /* 3f8b5ffecfad43f3e534358076b9b0f7 */

/* x = 6.87500000000000000000000000000000000e-01 3ffe6000000000000000000000000000 */
/* cos(x) = 0.c5d882d2ee48030c7c07d28e981e34804f82ed4cf93655d2365389b716de6ad44676a1cc5da */
  7.72834946152471544810851845913425178e-01L,   /* 3ffe8bb105a5dc900618f80fa51d303c */
  3.94975229341211664237241534741146939e-35L,   /* 3f8ca4027c176a67c9b2ae91b29c4db9 */
/* sin(x) = 0.a2759c0e79c35582527c32b55f5405c182c66160cb1d9eb7bb0b7cdf4ad66f317bda4332914 */
  6.34607080015269296850309914203671436e-01L,   /* 3ffe44eb381cf386ab04a4f8656abea8 */
  4.33025916939968369326060156455927002e-36L,   /* 3f897060b1985832c767adeec2df37d3 */

/* x = 6.95312500000000000000000000000000000e-01 3ffe6400000000000000000000000000 */
/* cos(x) = 0.c4920cc2ec38fb891b38827db08884fc66371ac4c2052ca8885b981bbcfd3bb7b093ee31515 */
  7.67853543842850365879920759114193964e-01L,   /* 3ffe89241985d871f712367104fb6111 */
  3.75100035267325597157244776081706979e-36L,   /* 3f893f198dc6b130814b2a2216e606ef */
/* sin(x) = 0.a400072188acf49cd6b173825e038346f105e1301afe642bcc364cea455e21e506e3e927ed8 */
  6.40625425040230409188409779413961021e-01L,   /* 3ffe48000e431159e939ad62e704bc07 */
  2.46542747294664049615806500747173281e-36L,   /* 3f88a37882f0980d7f3215e61b267523 */

/* x = 7.03125000000000000000000000000000000e-01 3ffe6800000000000000000000000000 */
/* cos(x) = 0.c348846bbd3631338ffe2bfe9dd1381a35b4e9c0c51b4c13fe376bad1bf5caacc4542be0aa9 */
  7.62825275710576250507098753625429792e-01L,   /* 3ffe869108d77a6c62671ffc57fd3ba2 */
  4.22067411888601505004748939382325080e-35L,   /* 3f8cc0d1ada74e0628da609ff1bb5d69 */
/* sin(x) = 0.a587e23555bb08086d02b9c662cdd29316c3e9bd08d93793634a21b1810cce73bdb97a99b9e */
  6.46604669591152370524042159882800763e-01L,   /* 3ffe4b0fc46aab761010da05738cc59c */
  -3.41742981816219412415674365946079826e-35L,  /* bf8c6b6749e0b217b9364364e5aef274 */

/* x = 7.10937500000000000000000000000000000e-01 3ffe6c00000000000000000000000000 */
/* cos(x) = 0.c1fbeef380e4ffdd5a613ec8722f643ffe814ec2343e53adb549627224fdc9f2a7b77d3d69f */
  7.57750448655219342240234832230493361e-01L,   /* 3ffe83f7dde701c9ffbab4c27d90e45f */
  -2.08767968311222650582659938787920125e-35L,  /* bf8bbc0017eb13dcbc1ac524ab69d8de */
/* sin(x) = 0.a70d272a76a8d4b6da0ec90712bb748b96dabf88c3079246f3db7eea6e58ead4ed0e2843303 */
  6.52544448725765956407573982284767763e-01L,   /* 3ffe4e1a4e54ed51a96db41d920e2577 */
  -8.61758060284379660697102362141557170e-36L,  /* bf8a6e8d24a80ee79f0db721849022b2 */

/* x = 7.18750000000000000000000000000000000e-01 3ffe7000000000000000000000000000 */
/* cos(x) = 0.c0ac518c8b6ae710ba37a3eeb90cb15aebcb8bed4356fb507a48a6e97de9aa6d9660116b436 */
  7.52629372418066476054541324847143116e-01L,   /* 3ffe8158a31916d5ce21746f47dd7219 */
  3.71306958657663189665450864311104571e-35L,   /* 3f8c8ad75e5c5f6a1ab7da83d245374c */
/* sin(x) = 0.a88fcfebd9a8dd47e2f3c76ef9e2439920f7e7fbe735f8bcc985491ec6f12a2d4214f8cfa99 */
  6.58444399910567541589583954884041989e-01L,   /* 3ffe511f9fd7b351ba8fc5e78eddf3c5 */
  -4.54412944084300330523721391865787219e-35L,  /* bf8ce336f840c020c6503a19b3d5b70a */

/* x = 7.26562500000000000000000000000000000e-01 3ffe7400000000000000000000000000 */
/* cos(x) = 0.bf59b17550a4406875969296567cf3e3b4e483061877c02811c6cae85fad5a6c3da58f49292 */
  7.47462359563216166669700384714767552e-01L,   /* 3ffe7eb362eaa14880d0eb2d252cacfa */
  -9.11094340926220027288083639048016945e-36L,  /* bf8a8389636f9f3cf107fafdc726a2f4 */
/* sin(x) = 0.aa0fd66eddb921232c28520d3911b8a03193b47f187f1471ac216fbcd5bb81029294d3a73f1 */
  6.64304163042946276515506587432846246e-01L,   /* 3ffe541facddbb7242465850a41a7223 */
  4.26004843895378210155889028714676019e-35L,   /* 3f8cc5018c9da3f8c3f8a38d610b7de7 */

/* x = 7.34375000000000000000000000000000000e-01 3ffe7800000000000000000000000000 */
/* cos(x) = 0.be0413f84f2a771c614946a88cbf4da1d75a5560243de8f2283fefa0ea4a48468a52d51d8b3 */
  7.42249725458501306991347253449610537e-01L,   /* 3ffe7c0827f09e54ee38c2928d51197f */
  -3.78925270049800913539923473871287550e-35L,  /* bf8c92f1452d54fede10b86ebe0082f9 */
/* sin(x) = 0.ab8d34b36acd987210ed343ec65d7e3adc2e7109fce43d55c8d57dfdf55b9e01d2cc1f1b9ec */
  6.70123380473162894654531583500648495e-01L,   /* 3ffe571a6966d59b30e421da687d8cbb */
  -1.33165852952743729897634069393684656e-36L,  /* bf87c523d18ef6031bc2aa372a82020b */

/* x = 7.42187500000000000000000000000000000e-01 3ffe7c00000000000000000000000000 */
/* cos(x) = 0.bcab7e6bfb2a14a9b122c574a376bec98ab14808c64a4e731b34047e217611013ac99c0f25d */
  7.36991788256240741057089385586450844e-01L,   /* 3ffe7956fcd7f654295362458ae946ed */
  4.72358938637974850573747497460125519e-35L,   /* 3f8cf64c558a404632527398d9a023f1 */
/* sin(x) = 0.ad07e4c409d08c4fa3a9057bb0ac24b8636e74e76f51e09bd6b2319707cbd9f5e254643897a */
  6.75901697026178809189642203142423973e-01L,   /* 3ffe5a0fc98813a1189f47520af76158 */
  2.76252586616364878801928456702948857e-35L,   /* 3f8c25c31b73a73b7a8f04deb5918cb8 */

/* x = 7.50000000000000000000000000000000000e-01 3ffe8000000000000000000000000000 */
/* cos(x) = 0.bb4ff632a908f73ec151839cb9d993b4e0bfb8f20e7e44e6e4aee845e35575c3106dbe6fd06 */
  7.31688868873820886311838753000084529e-01L,   /* 3ffe769fec655211ee7d82a3073973b3 */
  1.48255637548931697184991710293198620e-35L,   /* 3f8b3b4e0bfb8f20e7e44e6e4aee845e */
/* sin(x) = 0.ae7fe0b5fc786b2d966e1d6af140a488476747c2646425fc7533f532cd044cb10a971a49a6a */
  6.81638760023334166733241952779893908e-01L,   /* 3ffe5cffc16bf8f0d65b2cdc3ad5e281 */
  2.74838775935027549024224114338667371e-35L,   /* 3f8c24423b3a3e1323212fe3a99fa996 */

/* x = 7.57812500000000000000000000000000000e-01 3ffe8400000000000000000000000000 */
/* cos(x) = 0.b9f180ba77dd0751628e135a9508299012230f14becacdd14c3f8862d122de5b56d55b53360 */
  7.26341290974108590410147630237598973e-01L,   /* 3ffe73e30174efba0ea2c51c26b52a10 */
  3.12683579338351123545814364980658990e-35L,   /* 3f8c4c80911878a5f6566e8a61fc4317 */
/* sin(x) = 0.aff522a954f2ba16d9defdc416e33f5e9a5dfd5a6c228e0abc4d521327ff6e2517a7b3851dd */
  6.87334219303873534951703613035647220e-01L,   /* 3ffe5fea4552a9e5742db3bdfb882dc6 */
  4.76739454455410744997012795035529128e-35L,   /* 3f8cfaf4d2efead361147055e26a9099 */

/* x = 7.65625000000000000000000000000000000e-01 3ffe8800000000000000000000000000 */
/* cos(x) = 0.b890237d3bb3c284b614a0539016bfa1053730bbdf940fa895e185f8e58884d3dda15e63371 */
  7.20949380945696418043812784148447688e-01L,   /* 3ffe712046fa776785096c2940a7202d */
  4.78691285733673379499536326050811832e-35L,   /* 3f8cfd0829b985defca07d44af0c2fc7 */
/* sin(x) = 0.b167a4c90d63c4244cf5493b7cc23bd3c3c1225e078baa0c53d6d400b926281f537a1a260e6 */
  6.92987727246317910281815490823048210e-01L,   /* 3ffe62cf49921ac7884899ea9276f984 */
  4.50089871077663557180849219529189918e-35L,   /* 3f8cde9e1e0912f03c5d50629eb6a006 */

/* x = 7.73437500000000000000000000000000000e-01 3ffe8c00000000000000000000000000 */
/* cos(x) = 0.b72be40067aaf2c050dbdb7a14c3d7d4f203f6b3f0224a4afe55d6ec8e92b508fd5c5984b3b */
  7.15513467882981573520620561289896903e-01L,   /* 3ffe6e57c800cf55e580a1b7b6f42988 */
  -3.02191815581445336509438104625489192e-35L,  /* bf8c41586fe04a607eedada80d51489c */
/* sin(x) = 0.b2d7614b1f3aaa24df2d6e20a77e1ca3e6d838c03e29c1bcb026e6733324815fadc9eb89674 */
  6.98598938789681741301929277107891591e-01L,   /* 3ffe65aec2963e755449be5adc414efc */
  2.15465226809256290914423429408722521e-35L,   /* 3f8bca3e6d838c03e29c1bcb026e6733 */

/* x = 7.81250000000000000000000000000000000e-01 3ffe9000000000000000000000000000 */
/* cos(x) = 0.b5c4c7d4f7dae915ac786ccf4b1a498d3e73b6e5e74fe7519d9c53ee6d6b90e881bddfc33e1 */
  7.10033883566079674974121643959490219e-01L,   /* 3ffe6b898fa9efb5d22b58f0d99e9635 */
  -4.09623224763692443220896752907902465e-35L,  /* bf8cb3960c6248d0c580c573131d608d */
/* sin(x) = 0.b44452709a59752905913765434a59d111f0433eb2b133f7d103207e2aeb4aae111ddc385b3 */
  7.04167511454533672780059509973942844e-01L,   /* 3ffe6888a4e134b2ea520b226eca8695 */
  -2.87259372740393348676633610275598640e-35L,  /* bf8c3177707de60a6a76604177e6fc0f */

/* x = 7.89062500000000000000000000000000000e-01 3ffe9400000000000000000000000000 */
/* cos(x) = 0.b45ad4975b1294cadca4cf40ec8f22a68cd14b175835239a37e63acb85e8e9505215df18140 */
  7.04510962440574606164129481545916976e-01L,   /* 3ffe68b5a92eb6252995b9499e81d91e */
  2.60682037357042658395360726992048803e-35L,   /* 3f8c1534668a58bac1a91cd1bf31d65c */
/* sin(x) = 0.b5ae7285bc10cf515753847e8f8b7a30e0a580d929d770103509880680f7b8b0e8ad23b65d8 */
  7.09693105363899724959669028139035515e-01L,   /* 3ffe6b5ce50b78219ea2aea708fd1f17 */
  -4.37026016974122945368562319136420097e-36L,  /* bf8973c7d69fc9b58a23fbf2bd9dfe60 */
};
