/*
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package main

import (
	"testing"
)

func TestVerifyBoilerPlate(t *testing.T) {
	testcases := []struct {
		name          string
		bp            string
		expectedError bool
	}{
		{
			name: "valid: boilerplate is valid",
			bp: `\/*
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
		*/`,
			expectedError: false,
		},
		{
			name: "invalid: missing lines",
			bp: `
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
`,
			expectedError: true,
		},
		{
			name:          "invalid: bad year",
			bp:            "Copyright 1019 The Kubernetes Authors.",
			expectedError: true,
		},
	}

	for _, tc := range testcases {
		t.Run(tc.name, func(t *testing.T) {
			if err := verifyBoilerplate(tc.bp); err != nil != tc.expectedError {
				t.Errorf("expected error: %v, got: %v, error: %v", tc.expectedError, err != nil, err)
			}
		})
	}
}

func TestTrimLeadingComment(t *testing.T) {
	testcases := []struct {
		name           string
		comment        string
		line           string
		expectedResult string
	}{
		{
			name:           "trim leading comment",
			comment:        "#",
			line:           "# test",
			expectedResult: "test",
		},
		{
			name:           "empty line",
			comment:        "#",
			line:           "#",
			expectedResult: "",
		},
		{
			name:           "trim leading comment and space",
			comment:        "//",
			line:           "// test",
			expectedResult: "test",
		},
		{
			name:           "no comment",
			comment:        "//",
			line:           "test",
			expectedResult: "test",
		},
	}

	for _, tc := range testcases {
		t.Run(tc.name, func(t *testing.T) {
			if res := trimLeadingComment(tc.line, tc.comment); res != tc.expectedResult {
				t.Errorf("expected: %q, got: %q", tc.expectedResult, res)
			}
		})
	}
}
