### Copyright (C) 2010 Peter Williams <peter_ono@users.sourceforge.net>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os
import os.path
import gobject
import subprocess
import signal

from gquilt_pkg import urlops
from gquilt_pkg import cmd_result

def path_relative_to_dir(fdir, path):
    if not os.path.isdir(fdir):
        return None
    if fdir == path:
        return os.curdir
    lcwd = len(fdir)
    if len(path) <= lcwd + 1 or fdir != path[0:lcwd] or path[lcwd] != os.sep:
        return None
    return path[lcwd + 1:]

def path_relative_to_playground(path):
    return path_relative_to_dir(os.getcwd(), path)

HOME = os.path.expanduser("~")

def path_rel_home(path):
    """Return the given path as a path relative to user's home directory."""
    if urlops.parse_url(path).scheme:
        return path
    path = os.path.abspath(path)
    len_home = len(HOME)
    if len(path) >= len_home and HOME == path[:len_home]:
        path = "~" + path[len_home:]
    return path

def cwd_rel_home():
    """Return path of current working directory relative to user's home
    directory.
    """
    return path_rel_home(os.getcwd())


def file_list_to_string(file_list):
    """Return the given list of file names as a single string:
    - using a single space as a separator, and
    - placing double quotes around file names that contain spaces.
    """
    mod_file_list = []
    for fname in file_list:
        if fname.count(' ') == 0:
            mod_file_list.append(fname)
        else:
            mod_file_list.append('"%s"' % fname)
    return ' '.join(mod_file_list)


def string_to_file_list(string):
    """Return a list of the file names in the given string:
    - assuming names are separated by spaces, and
    - file names that contain spaces are inside double quotes.
    """
    if string.count('"') == 0:
        return string.split()
    file_list = []
    index = 0
    lqi = string.rfind('"')
    while index < lqi:
        qib = string.find('"', index)
        file_list += string[index:qib].split()
        index = string.find('"', qib + 1) + 1
        file_list.append(string[qib+1:index-1])
    if index < len(string):
        file_list += string[index:].split()
    return file_list


# handle the fact os.path.samefile is not available on all operating systems
def samefile(filename1, filename2):
    """Return whether the given paths refer to the same file or not."""
    try:
        return os.path.samefile(filename1, filename2)
    except AttributeError:
        return os.path.abspath(filename1) == os.path.abspath(filename2)

def run_cmd(cmd, input_text=None):
    """Run the given command and report the outcome as a cmd_result tuple.
    If input_text is not None pas it to the command as standard input.
    """
    try:
        oldterm = os.environ['TERM']
        os.environ['TERM'] = "dumb"
    except LookupError:
        oldterm = None
    is_posix = os.name == 'posix'
    if is_posix:
        savedsh = signal.getsignal(signal.SIGPIPE)
        signal.signal(signal.SIGPIPE, signal.SIG_DFL)
    sub = subprocess.Popen(cmd, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
          stderr=subprocess.PIPE, shell=True, close_fds=is_posix, bufsize=-1)
    outd, errd = sub.communicate(input_text)
    if is_posix:
        signal.signal(signal.SIGPIPE, savedsh)
    if oldterm:
        os.environ['TERM'] = oldterm
    return cmd_result.Result(sub.returncode, outd, errd)

def _wait_for_bgnd_cmd_timeout(pid):
    """Callback to clean up after background tasks complete"""
    try:
        if os.name == 'nt' or os.name == 'dos':
            rpid, _ = os.waitpid(pid, 0)
        else:
            rpid, _ = os.waitpid(pid, os.WNOHANG)
        return rpid != pid
    except OSError:
        return False


def run_cmd_in_bgnd(cmd):
    """Run the given command in the background and poll for its exit using
    _wait_for_bgnd_timeout() as a callback.
    """
    if not cmd:
        return False
    pid = subprocess.Popen(string_to_file_list(cmd)).pid
    if not pid:
        return False
    gobject.timeout_add(2000, _wait_for_bgnd_cmd_timeout, pid)
    return True

if os.name == 'nt' or os.name == 'dos':
    def _which(cmd):
        """Return the path of the executable for the given command"""
        for dirpath in os.environ['PATH'].split(os.pathsep):
            potential_path = os.path.join(dirpath, cmd)
            if os.path.isfile(potential_path) and \
               os.access(potential_path, os.X_OK):
                return potential_path
        return None


    NT_EXTS = ['.bat', '.bin', '.exe']


    def which(cmd):
        """Return the path of the executable for the given command"""
        path = _which(cmd)
        if path:
            return path
        _, ext = os.path.splitext(cmd)
        if ext in NT_EXTS:
            return None
        for ext in NT_EXTS:
            path = _which(cmd + ext)
            if path is not None:
                return path
        return None
else:
    def which(cmd):
        """Return the path of the executable for the given command"""
        for dirpath in os.environ['PATH'].split(os.pathsep):
            potential_path = os.path.join(dirpath, cmd)
            if os.path.isfile(potential_path) and \
               os.access(potential_path, os.X_OK):
                return potential_path
        return None
