<?php
/*
 *  $Id: XmlLintTask.php 1001 2010-11-24 10:40:05Z mrook $
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the LGPL. For more information please see
 * <http://phing.info>.
 */

require_once 'phing/Task.php';

/**
 * A XML lint task. Checking syntax of one or more XML files against an XML Schema using the DOM extension.
 *
 * @author   Knut Urdalen <knut.urdalen@telio.no>
 * @version  $Id: XmlLintTask.php 1001 2010-11-24 10:40:05Z mrook $
 * @package  phing.tasks.ext
 */
class XmlLintTask extends Task {

  protected $file;  // the source file (from xml attribute)
  protected $schema; // the schema file (from xml attribute)
  protected $filesets = array(); // all fileset objects assigned to this task
  
  protected $haltonfailure = true;

  /**
   * File to be performed syntax check on
   *
   * @param PhingFile $file
   */
  public function setFile(PhingFile $file) {
    $this->file = $file;
  }

  /**
   * XML Schema Description file to validate against
   *
   * @param PhingFile $schema
   */
  public function setSchema(PhingFile $schema) {
    $this->schema = $schema;
  }
  
  /**
   * Nested creator, creates a FileSet for this task
   *
   * @return FileSet The created fileset object
   */
  public function createFileSet() {
    $num = array_push($this->filesets, new FileSet());
    return $this->filesets[$num-1];
  }
  
  /**
   * Sets the haltonfailure attribute
   *
   * @param bool $haltonfailure
   */
  public function setHaltonfailure($haltonfailure) {
    $this->haltonfailure = (bool) $haltonfailure;
  }

  /**
   * Execute lint check against PhingFile or a FileSet
   */
  public function main() {
    if(isset($this->schema) && !file_exists($this->schema->getPath())) {
      throw new BuildException("Schema file not found: ".$this->schema->getPath());
    }
    if(!isset($this->file) and count($this->filesets) == 0) {
      throw new BuildException("Missing either a nested fileset or attribute 'file' set");
    }

    set_error_handler(array($this, 'errorHandler'));
    if($this->file instanceof PhingFile) {
      $this->lint($this->file->getPath());
    } else { // process filesets
      $project = $this->getProject();
      foreach($this->filesets as $fs) {
    $ds = $fs->getDirectoryScanner($project);
    $files = $ds->getIncludedFiles();
    $dir = $fs->getDir($this->project)->getPath();
    foreach($files as $file) {
      $this->lint($dir.DIRECTORY_SEPARATOR.$file);
    }
      }
    }
    restore_error_handler();
  }
  
  protected function logError($message) {
    if ($this->haltonfailure) {
      throw new BuildException($message);
    } else {
      $this->log($message, Project::MSG_ERR);
    }
  }

  /**
   * Performs validation
   *
   * @param string $file
   * @return void
   */
  protected function lint($file) {
    if(file_exists($file)) {
      if(is_readable($file)) {
        $dom = new DOMDocument();
        if ($dom->load($file) === false) {
          $error = libxml_get_last_error();
          $this->logError($file.' is not well-formed (See messages above)');
        } else {
          if(isset($this->schema)) {
            if($dom->schemaValidate($this->schema->getPath())) {
              $this->log($file.' validated', Project::MSG_INFO);
            } else {
              $this->logError($file.' fails to validate (See messages above)');
            }
          } else {
            $this->log($file.' is well-formed', Project::MSG_INFO);
          }
        }
      } else {
        $this->logError('Permission denied: '.$file);
      }
    } else {
      $this->logError('File not found: '.$file);
    }
  }

  /**
   * Local error handler to catch validation errors and log them through Phing
   *
   * @param int    $level
   * @param string $message
   * @param string $file
   * @param int    $line
   */
  public function errorHandler($level, $message, $file, $line, $context) {
    $matches = array();
    preg_match('/^.*\(\): (.*)$/', $message, $matches);
    $this->log($matches[1], Project::MSG_ERR);
  }

}

