
import os
import subprocess
import logging

from juju.errors import ProviderInteractionError

from twisted.internet.defer import inlineCallbacks, returnValue
from twisted.internet.threads import deferToThread

DEFAULT_LXC_ADDR = '10.0.3.1'
DEFAULT_LXC_BRIDGE = 'lxcbr0'

log = logging.getLogger("juju.local-dev")

_SHELL_GET_LXC_DEFAULTS=""". /etc/default/lxc; export LXC_BRIDGE; export LXC_ADDR; env|grep ^LXC_"""

class Network(object):
    """ Setup a bridge network with forwarding and dnsmasq for the environment.

    Utilizes lxc's networking subsystem to actualize configuration.
    """

    def start(self):
        """Start the network.
        """
        return deferToThread(start_network)

    def stop(self):
        """Stop the network.
        """
        return deferToThread(stop_network)

    @inlineCallbacks
    def is_running(self):
        """Returns True if the network is currently active, False otherwise.
        """
        (goal, state) = yield deferToThread(_status_lxc_net)
        returnValue(bool(state == 'running'))

    def get_attributes(self):
        """Return attributes of the network as a dictionary.

        The network name, starting ip address, and bridge name are returned
        """
        return deferToThread(get_network_attributes)


def _status_lxc_net():
    try:
        status_lxc_net = subprocess.check_output(['status','lxc-net'])
    except subprocess.CalledProcessError:
        raise ProviderInteractionError(
            'Problem checking status of lxc-net upstart job.')
    try:
        (_, state) = status_lxc_net.split(' ')
        (goal, state) = state.split('/')
    except ValueError:
        raise ProviderInteractionError(
            'status lxc-net returned unexpected output (%s)' % status_lxc_net)
    return (goal, state)


def _set_lxc_net_state(desired_state):
    (goal, state) = _status_lxc_net()

    if goal != desired_state:
        try:
            subprocess.check_call([desired_state,'lxc-net'])
        except subprocess.CalledProcessError, e:
            log.warn('Problem %sing lxc-net' % desired_state)
            log.warn(e)


def start_network():
    _set_lxc_net_state('start')
        

def get_network_attributes():
    env = {}
    if os.path.exists('/etc/default/lxc'):
        try:
            output = subprocess.check_output(
                ["sh", "-c", _SHELL_GET_LXC_DEFAULTS])
            for l in output.split("\n"):
                if '=' in l:
                    (name, value) = l.split('=', 1)
                    env[name] = value
        except subprocess.CalledProcessError, e:
            log.warn('Problem reading values from /etc/default/lxc')
            log.warn(e)

    attrs = {}
    attrs["ip"] = env.get('LXC_ADDR', DEFAULT_LXC_ADDR)
    attrs["bridge"] = env.get('LXC_BRIDGE', DEFAULT_LXC_BRIDGE)

    return attrs


def stop_network():
    _set_lxc_net_state('stop')
