/* ----- encode/1277x3, derived from supercop/crypto_encode/try.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ntruprime.h>
#include "ntruprime_test.h"

#define fail ((ok = 0),printf)
static const char *encode_1277x3_checksums[] = {
  "5f8a22de6e4e3893b53577b6606bc3ccdb111ee025d609d21cb06592468e29af",
  "b11455450ab60cf8c24ae5a25003e85e650fffe8bc5b29cd76439f154905de75",
} ;

static void (*crypto_encode)(unsigned char *,const void *);
#define crypto_encode_STRBYTES ntruprime_encode_1277x3_STRBYTES
#define crypto_encode_ITEMS ntruprime_encode_1277x3_ITEMS
#define crypto_encode_ITEMBYTES ntruprime_encode_1277x3_ITEMBYTES

static void *storage_encode_1277x3_s;
static unsigned char *test_encode_1277x3_s;
static void *storage_encode_1277x3_x;
static unsigned char *test_encode_1277x3_x;
static void *storage_encode_1277x3_s2;
static unsigned char *test_encode_1277x3_s2;
static void *storage_encode_1277x3_x2;
static unsigned char *test_encode_1277x3_x2;

#define precomputed_encode_1277x3_NUM 4

static const unsigned char precomputed_encode_1277x3_s[precomputed_encode_1277x3_NUM][crypto_encode_STRBYTES] = {
  {4,36,10,100,115,236,133,187,28,141,159,245,24,8,88,185,32,176,73,208,165,203,1,202,155,233,100,68,212,222,254,13,47,55,242,11,100,240,21,4,225,70,112,145,38,96,100,77,21,20,118,55,117,118,211,230,116,13,157,46,78,8,216,223,19,70,84,114,183,61,51,190,45,80,156,39,61,0,51,9,124,11,236,229,54,117,217,2,249,253,201,141,226,72,67,143,80,21,197,216,162,241,205,159,44,134,198,166,173,130,36,225,224,7,81,206,145,242,96,71,89,15,27,33,192,121,232,46,141,1,57,151,134,85,29,125,149,60,71,60,246,160,85,39,206,105,211,92,128,78,4,232,73,229,118,91,157,242,194,224,143,170,56,218,133,117,123,97,47,243,181,255,113,181,24,10,208,21,59,35,31,174,50,67,101,168,222,193,207,11,162,106,12,1,21,101,101,73,43,85,132,231,176,41,201,144,78,239,199,121,22,85,250,60,155,197,214,59,99,66,198,184,236,235,181,67,75,218,204,245,116,15,41,130,117,38,14,132,84,233,135,11,14,141,153,57,62,200,128,4,199,175,79,11,26,106,161,139,7,221,175,200,148,226,218,50,214,222,83,111,130,171,94,145,79,252,168,84,234,208,109,127,132,236,159,81,209,13,44,12,242,206,115,57,54,216,125,50,9,86,200,170,57,181,216,199,134,71,23,205,207,169,187,56,157,243,66,131,218,248},
  {51,68,177,248,180,163,206,78,62,223,245,94,230,190,37,54,36,176,119,129,76,18,16,18,9,212,116,77,15,94,30,85,68,91,160,58,198,207,16,44,166,131,12,43,207,25,168,238,89,118,234,113,109,254,238,209,136,91,133,81,193,158,29,60,4,142,245,85,238,40,63,245,63,90,146,242,238,202,50,224,222,57,21,236,53,144,8,189,205,24,182,184,128,16,47,24,92,1,111,34,68,73,117,2,186,92,77,252,167,219,187,245,217,208,7,183,17,33,125,182,242,157,175,194,55,47,145,20,105,47,41,159,87,250,12,207,21,48,69,196,206,130,136,43,79,247,7,213,0,208,180,95,59,109,14,3,15,224,124,250,253,149,206,160,167,219,57,26,91,145,45,130,9,123,6,136,163,65,76,103,137,56,123,52,90,198,103,213,109,183,124,132,21,172,224,244,91,212,32,117,5,167,233,181,213,244,189,235,177,134,243,101,202,29,133,182,9,146,44,136,149,28,201,105,37,182,120,28,46,172,3,181,50,128,223,181,97,43,136,168,35,204,223,66,159,170,43,173,108,91,202,52,190,204,254,195,162,120,122,220,90,118,97,173,70,242,50,108,129,31,116,158,111,57,152,183,59,189,58,178,203,165,83,151,216,111,143,10,175,138,241,96,213,50,237,55,123,226,66,228,140,202,126,76,229,140,191,231,250,55,151,79,124,23,71,122,220,11,207,125},
  {219,8,45,241,57,253,56,198,52,31,37,22,155,202,2,182,83,73,100,133,48,90,11,34,2,166,37,178,58,220,251,179,188,214,90,50,217,165,147,76,55,13,216,98,188,62,45,35,18,29,119,196,162,21,121,44,46,223,175,187,240,93,67,236,155,185,97,80,82,141,219,74,199,46,221,237,135,0,4,149,20,122,214,51,208,68,67,127,188,66,100,39,232,197,178,58,113,39,187,154,203,179,174,237,15,2,213,70,222,208,115,43,179,243,101,45,157,145,25,35,232,87,92,104,192,22,175,0,211,67,67,181,191,58,26,180,209,218,163,222,175,65,162,189,15,163,29,239,175,31,233,171,219,187,11,247,115,33,196,81,195,117,34,160,77,117,193,35,130,236,230,251,138,126,117,1,51,7,189,243,129,0,84,215,50,215,103,102,11,25,195,43,196,58,100,214,64,223,245,76,30,180,33,102,127,229,110,8,55,98,28,38,221,218,157,196,87,183,56,91,185,239,117,75,208,190,155,253,51,43,63,123,24,5,196,60,144,64,239,228,74,173,107,252,42,108,105,170,184,244,241,89,3,35,230,142,150,141,26,189,134,231,144,139,244,69,55,200,153,127,37,170,1,176,122,80,226,188,154,42,213,54,98,65,91,8,9,145,125,83,62,36,250,234,142,106,122,53,244,214,44,153,218,60,111,99,93,237,228,101,207,87,19,73,133,78,134,64,149,243},
  {66,222,129,98,25,14,240,113,182,196,15,72,198,234,11,53,46,83,194,143,11,23,5,236,210,210,58,193,182,56,125,29,223,125,193,162,72,124,58,34,77,115,93,155,64,96,83,207,131,147,177,32,225,181,161,218,9,222,37,214,206,16,249,230,229,94,31,148,9,138,59,131,123,247,93,120,230,132,172,179,125,162,222,162,122,56,61,242,139,139,92,134,177,81,78,114,233,216,221,225,14,144,142,73,218,193,209,34,20,87,79,249,27,4,196,15,15,190,92,7,142,18,61,210,86,48,61,104,115,45,107,26,162,250,172,239,1,25,46,179,139,229,141,197,162,35,3,244,150,194,196,245,138,33,17,255,79,137,248,156,202,188,163,106,103,248,74,131,247,141,192,184,152,215,131,187,29,78,139,114,30,146,19,193,161,111,17,61,61,123,255,223,190,218,57,110,226,22,43,36,64,73,133,55,188,180,223,164,137,243,52,173,249,193,74,16,224,41,163,29,110,211,244,108,106,166,88,90,46,245,19,157,178,39,6,247,44,8,149,143,254,29,216,31,146,28,78,110,123,161,144,175,14,195,207,190,171,48,144,202,89,216,190,117,248,204,149,86,102,220,247,36,182,151,139,127,212,130,248,175,34,42,153,231,179,63,55,44,137,221,152,185,140,214,30,63,109,76,210,13,136,82,136,93,43,200,112,180,170,232,96,112,38,164,155,150,37,234,136,164},
} ;

static const unsigned char precomputed_encode_1277x3_x[precomputed_encode_1277x3_NUM][crypto_encode_ITEMS] = {
  {235,17,76,123,139,81,172,217,177,21,147,86,131,51,20,18,238,168,201,232,15,170,142,184,20,207,26,33,150,244,184,102,251,55,239,60,36,17,77,212,210,222,224,228,144,44,122,11,219,114,78,229,3,32,75,2,227,120,24,67,8,163,253,168,143,199,230,231,123,176,102,7,124,182,186,20,127,191,56,238,80,68,111,240,198,224,167,79,116,90,241,63,161,49,89,42,102,76,251,188,120,239,218,191,43,37,181,236,39,34,196,0,39,34,181,174,97,230,133,221,217,64,93,164,176,162,104,228,42,236,56,102,50,84,74,119,13,172,162,179,122,239,64,254,123,233,163,235,219,188,193,19,64,220,197,151,227,163,188,62,208,231,190,105,13,97,98,201,91,164,71,107,204,240,239,103,245,43,47,185,251,32,177,238,15,72,118,146,168,4,40,3,184,66,44,205,143,116,150,240,157,145,28,178,14,121,47,140,184,86,177,56,237,17,83,227,222,52,25,133,105,242,190,70,183,122,32,210,140,75,244,43,140,227,247,91,21,213,207,130,133,229,190,12,143,57,52,188,119,215,227,82,110,99,65,146,30,172,187,209,149,47,242,122,103,170,251,81,5,182,128,113,222,201,214,208,124,199,157,38,242,147,14,119,149,165,196,252,184,152,156,244,211,62,83,40,147,129,99,118,22,91,109,150,216,88,19,14,215,113,129,80,94,28,205,149,68,160,103,62,207,130,197,172,98,13,94,169,175,178,86,19,52,7,164,248,5,7,40,117,76,25,43,35,136,103,18,69,109,144,204,137,24,231,223,76,56,48,179,130,216,207,182,252,224,218,143,124,153,197,182,82,151,91,71,254,66,47,159,72,38,237,30,150,235,36,116,217,72,250,197,245,132,11,84,170,151,123,56,182,205,184,34,150,0,199,116,49,84,105,92,83,22,253,68,248,163,253,180,104,185,66,95,82,161,176,137,66,161,92,116,128,180,161,210,0,53,254,144,68,31,44,16,16,156,104,37,28,199,105,50,192,30,113,217,45,36,82,189,147,185,220,57,131,20,206,251,5,173,92,12,179,211,42,153,132,14,53,177,232,0,57,30,41,246,153,246,124,234,55,28,113,8,125,145,127,63,247,221,242,128,117,225,167,79,85,67,119,245,57,84,119,4,165,226,138,240,131,147,206,216,51,156,210,206,169,117,34,213,163,93,44,4,119,30,221,224,238,251,246,200,164,97,219,172,45,254,120,167,124,129,161,2,36,122,99,207,134,108,129,25,234,90,181,107,120,40,162,28,245,45,137,106,250,96,246,165,181,212,67,196,224,21,14,78,172,212,241,151,152,5,199,191,47,203,116,205,67,138,58,151,86,0,19,15,49,68,170,216,215,4,150,120,134,124,169,33,8,34,243,38,204,255,251,192,198,119,172,221,196,239,11,29,196,188,110,7,33,0,148,214,81,6,115,81,61,209,116,39,3,239,51,73,51,199,152,20,223,98,10,60,229,41,119,170,23,130,136,92,84,146,117,234,156,84,153,30,232,174,12,228,207,80,45,134,189,243,236,244,118,77,98,232,158,36,239,155,174,67,13,105,55,151,4,55,105,170,44,20,251,248,185,242,13,176,183,14,212,91,219,210,130,47,184,65,34,69,33,129,51,213,77,158,96,237,64,136,30,205,213,95,65,147,223,253,19,76,38,220,76,230,132,78,219,88,29,150,212,45,184,177,219,59,90,137,91,146,160,71,40,73,37,60,136,9,163,180,135,71,2,92,57,161,59,80,180,31,120,136,223,171,29,230,164,82,225,128,116,31,11,251,253,220,14,158,253,180,47,123,180,29,133,32,77,200,236,172,186,210,19,63,139,249,57,233,36,172,39,6,221,86,31,30,229,64,59,0,133,49,76,209,172,156,238,24,138,164,158,125,86,41,197,23,84,228,45,58,67,128,216,156,13,74,168,137,254,224,196,38,60,201,205,202,77,109,37,1,119,101,31,241,208,120,159,215,207,173,66,255,90,151,147,50,9,104,63,116,23,201,112,222,176,201,107,14,10,160,123,57,143,197,179,11,15,115,176,100,47,20,29,63,8,56,77,82,214,209,168,32,137,5,169,97,187,82,103,136,2,109,139,65,0,101,5,185,48,64,197,115,139,9,20,123,205,193,45,0,9,115,217,194,48,239,123,18,225,222,97,37,209,17,25,28,15,226,91,116,0,133,18,208,37,200,76,169,16,200,110,239,205,145,77,119,197,150,29,83,119,228,33,130,92,28,43,190,111,130,107,130,50,223,95,218,31,6,132,125,6,191,137,165,216,173,221,208,119,202,76,194,217,33,220,190,245,114,4,176,238,246,11,178,110,191,36,55,127,20,4,207,204,176,157,213,150,82,177,189,142,153,8,65,235,79,248,245,7,251,119,21,93,240,76,138,225,187,209,2,110,230,92,89,89,211,117,6,152,199,82,169,4,113,145,192,111,88,5,230,185,147,0,115,121,253,34,171,238,211,127,103,106,191,144,53,220,175,48,27,164,137,229,168,192,183,240,146,30,74,154,19,83,249,113,127,6,248,134,246,121,131,117,148,74,172,34,104,149,252,104,72,176,39,41,175,242,18,201,35,121,147,142,81,23,131,35,165,49,109,245,26,73,226,47,20,144,53,205,253,237,67,36,187,42,174,129,188,163,254,72,45,12,248,0,212,44,255,241,221,161,106,44,183,235,245,27,225,221,252,197,200,191,38,119,216,150,215,123,247,231,151,178,146,48,146,252,21,163,125,99,170,142,1,40,210,240,103,43,20,161,142,0,142,255,155,177,44,122,208,41,158,230,171,166,243,240,79,225,68,161,8,76,122,193,230,135,217,93,254,11,114,83,191,134,53,172,186,244,247},
  {210,155,122,152,71,238,163,3,16,147,144,160,239,153,173,182,159,96,232,35,102,26,44,55,33,38,252,244,41,100,68,140,245,69,250,117,154,196,222,104,120,10,164,171,21,213,34,254,169,146,86,193,201,132,49,42,180,211,33,231,77,5,184,4,203,181,187,102,35,74,165,207,138,14,126,214,64,79,239,135,87,60,175,139,113,199,75,58,127,159,20,14,33,131,213,229,108,222,101,142,119,126,201,254,191,88,216,30,40,8,139,108,90,144,146,8,249,204,250,137,25,96,147,108,124,25,22,3,199,18,58,137,206,254,24,211,115,142,234,160,237,30,68,65,21,95,82,111,122,80,36,114,39,25,143,253,47,242,2,115,73,10,222,156,226,135,131,160,23,64,82,182,42,27,132,100,110,11,66,99,4,100,227,64,111,209,190,219,137,132,204,21,128,201,210,165,177,156,132,191,101,180,110,218,72,241,21,59,88,148,47,98,249,184,49,243,49,150,109,13,200,233,9,70,211,68,201,95,70,236,141,49,0,68,94,161,124,4,239,30,24,57,235,239,121,248,71,202,156,139,60,37,167,28,29,224,251,205,192,182,197,173,20,82,108,101,90,104,212,51,198,148,1,34,169,214,183,251,139,166,170,212,239,60,36,227,167,174,106,104,14,88,245,124,138,234,253,192,228,228,53,174,43,24,169,200,17,79,89,87,188,48,217,29,113,234,139,152,234,148,189,111,121,202,236,46,172,222,80,4,22,248,235,115,110,196,160,44,157,211,243,110,213,129,175,49,188,70,156,215,223,34,196,17,135,184,95,113,222,115,237,49,75,140,195,172,195,239,155,140,6,176,183,197,131,171,142,107,174,91,91,82,110,117,3,137,66,211,176,215,58,172,142,119,147,70,57,89,79,193,175,40,206,143,40,67,68,206,64,8,28,140,185,101,18,61,73,243,229,40,115,21,244,212,100,225,105,70,227,161,148,168,242,220,119,230,154,163,158,122,70,36,141,11,176,88,69,185,56,75,166,123,23,69,29,62,78,53,229,117,50,196,142,89,104,77,190,225,240,199,204,92,204,95,174,132,177,168,229,43,105,21,238,92,112,242,205,181,170,144,151,40,197,226,6,147,46,165,86,187,218,204,85,62,152,253,159,123,55,242,48,3,180,252,199,196,46,67,98,158,32,201,49,130,46,123,87,87,58,122,110,52,57,15,79,73,91,111,234,188,70,73,237,221,68,195,159,134,83,110,201,141,216,222,126,51,231,178,88,53,173,247,63,184,25,53,164,52,143,89,60,166,102,84,38,67,62,195,19,46,94,185,26,179,142,133,57,58,44,121,91,187,143,131,157,53,227,42,31,172,63,0,250,158,142,251,205,91,22,224,57,151,48,50,98,252,197,161,103,68,14,148,29,58,162,90,75,144,171,152,96,130,51,21,226,134,73,203,87,167,160,162,200,40,88,162,178,245,205,23,13,10,39,149,221,140,210,200,46,206,150,88,153,100,144,185,63,210,185,15,193,67,250,91,34,138,244,238,137,108,88,148,183,251,13,224,114,106,24,147,9,103,238,214,46,72,185,70,146,31,191,93,188,117,18,223,240,111,192,75,240,12,231,4,116,147,166,207,187,202,40,251,74,178,87,107,78,80,110,110,80,248,247,214,9,240,241,241,25,183,189,45,224,192,226,108,152,80,120,26,246,29,100,81,219,87,46,201,245,31,147,137,243,233,175,184,2,86,180,47,121,203,159,206,116,41,255,211,208,205,175,28,56,192,110,214,8,127,243,119,15,103,159,23,117,210,144,248,87,33,96,144,249,86,190,49,197,170,128,177,225,217,128,184,60,77,144,132,254,248,23,114,136,45,92,155,114,134,214,104,106,30,208,175,173,184,201,138,236,98,154,117,7,115,144,232,50,83,237,134,15,190,92,103,41,217,92,229,180,184,181,35,66,88,36,172,82,39,101,162,229,84,19,201,10,244,199,147,102,255,208,80,103,147,247,72,55,153,112,9,146,195,148,144,196,104,112,116,249,12,225,176,76,136,107,102,194,88,111,186,19,181,213,9,226,243,91,195,219,89,178,255,224,236,36,7,234,46,209,191,45,245,203,216,4,139,210,66,251,16,232,94,60,125,216,181,202,171,214,80,20,14,7,227,90,116,243,72,56,203,142,60,221,186,103,40,123,231,230,229,137,214,17,15,198,17,46,175,198,48,141,82,24,88,130,81,49,60,96,190,12,137,71,184,27,209,210,9,125,49,21,48,18,130,227,207,188,200,125,83,122,152,211,225,178,116,189,199,137,189,250,173,224,247,97,255,91,121,135,191,42,76,69,116,109,237,47,146,125,189,25,155,216,244,29,101,159,126,4,6,171,117,40,36,206,23,213,167,116,77,241,211,110,86,33,91,105,83,191,34,61,92,44,204,125,68,210,186,145,192,187,165,57,85,181,101,72,106,238,247,133,32,108,218,73,6,39,71,172,17,150,219,94,166,94,146,180,156,88,244,248,187,173,34,167,201,101,70,142,108,102,20,80,7,8,194,188,46,238,108,42,79,234,138,211,44,203,206,0,254,10,244,8,219,218,180,221,135,65,157,80,96,46,207,43,69,209,141,43,246,176,215,45,151,107,64,114,166,40,18,97,24,69,34,105,6,192,154,35,229,106,210,7,187,58,83,62,231,133,18,208,135,145,67,125,110,43,109,207,151,203,91,140,209,193,253,248,249,97,174,105,190,71,164,98,68,236,37,13,197,59,75,209,83,26,148,208,28,170,106,44,226,25,251,138,176,82,224,49,92,242,96,245,118,158,11,143,173,95,210,88,4,66,144,24,155,194,0,131,36,105,195,159,111,147,29,192,222,242,205,73,124,250,240,224,159,124},
  {214,236,44,157,207,249,76,57,200,192,165,255,164,50,123,242,136,27,163,163,0,102,33,220,147,4,108,66,29,33,173,36,195,219,59,156,2,98,64,13,16,40,10,22,65,184,98,50,6,156,241,115,181,164,151,243,161,167,99,13,193,52,117,238,234,141,126,160,84,132,233,136,35,163,74,167,60,245,150,127,99,18,19,28,61,174,181,151,106,27,110,124,193,163,144,154,249,249,173,196,221,73,253,50,224,228,30,154,41,101,126,91,101,140,61,202,159,34,42,175,242,173,228,111,18,71,207,77,147,69,232,73,117,123,78,249,217,235,188,167,113,247,53,53,204,194,163,118,72,10,246,194,94,144,90,244,31,178,101,255,54,227,26,154,208,58,64,200,203,138,129,58,29,0,47,224,251,23,97,236,105,92,113,48,132,105,99,90,222,108,212,232,97,223,190,96,88,108,48,167,230,187,233,191,3,219,180,179,17,51,115,53,208,4,170,157,120,227,126,77,183,172,239,102,113,170,40,233,114,42,247,136,98,130,59,137,102,12,69,90,3,102,208,252,156,67,142,166,150,250,91,127,7,200,203,175,170,7,92,39,112,17,255,27,76,240,176,84,71,229,218,102,221,64,78,145,172,51,160,107,146,29,212,85,81,221,239,213,174,81,104,0,149,117,175,70,248,164,216,18,64,30,178,143,238,17,52,63,51,134,190,214,59,89,37,107,120,246,47,68,135,66,39,203,153,115,124,240,249,178,184,149,46,4,186,200,67,246,30,250,183,38,46,159,34,79,5,69,254,95,155,76,207,26,27,218,69,190,243,130,67,175,133,211,34,16,31,158,219,134,182,253,60,81,107,209,189,108,251,49,21,4,92,212,200,149,124,21,150,79,12,221,90,195,84,255,241,89,175,144,134,104,89,7,218,77,213,172,61,3,5,179,92,183,30,142,58,136,246,112,69,30,67,147,248,202,58,103,213,83,213,22,113,138,103,222,111,35,88,124,186,153,176,8,110,150,217,190,50,255,119,135,114,151,225,127,52,59,19,39,196,157,134,81,32,78,211,191,240,99,8,249,40,3,45,207,126,100,24,45,31,229,166,170,46,193,5,90,163,105,140,100,131,80,165,148,151,189,6,38,45,85,132,252,174,179,158,60,199,45,151,19,238,152,31,233,214,146,93,224,50,119,26,149,172,5,54,53,6,93,251,213,45,98,47,177,89,43,0,199,127,120,52,39,216,53,65,139,85,84,190,68,122,241,173,21,9,100,3,131,218,40,214,146,200,129,206,165,85,166,14,14,64,130,246,3,62,175,240,67,94,144,239,99,156,83,74,209,22,253,221,43,222,67,219,195,206,3,191,192,48,121,12,219,10,223,209,179,62,234,118,245,110,162,31,122,150,92,231,33,118,195,130,180,34,203,57,157,212,254,132,99,191,0,159,27,84,134,213,173,46,48,180,185,244,139,236,69,149,2,207,197,27,112,139,91,64,26,165,104,220,29,213,218,252,44,68,62,246,190,66,255,157,164,160,172,171,47,99,180,209,112,108,249,130,57,172,242,113,61,93,68,121,156,208,121,12,77,122,121,40,65,180,105,94,248,158,127,158,25,163,54,24,236,210,222,154,201,102,166,64,235,28,66,155,144,197,110,239,164,52,233,166,155,11,47,217,17,0,223,158,53,165,159,50,64,194,243,73,214,119,76,38,228,192,128,84,108,188,160,34,163,212,10,214,168,114,65,123,193,99,243,177,185,17,121,19,199,238,204,181,7,67,10,75,240,182,174,254,59,138,224,36,203,221,138,247,4,231,106,41,144,134,196,111,112,207,253,36,174,55,94,137,202,18,145,102,77,225,26,52,147,85,147,85,165,115,0,127,165,242,178,206,253,18,84,225,23,33,167,147,1,55,55,45,37,217,110,136,24,98,102,169,203,251,24,0,162,116,138,243,239,196,102,122,26,50,240,114,200,89,245,167,135,98,48,214,144,15,64,220,154,218,114,250,16,206,86,188,33,142,4,46,106,186,218,247,225,48,132,1,66,186,131,66,129,228,235,48,90,9,138,54,234,148,195,226,65,183,50,130,198,33,61,154,187,10,208,19,56,25,189,232,238,69,23,215,162,209,196,215,48,29,174,179,110,213,246,251,156,116,93,174,107,92,38,227,191,27,112,137,175,39,29,76,123,134,99,70,228,164,24,79,206,182,15,73,75,238,202,71,60,170,217,20,208,176,211,61,102,160,78,175,217,65,237,43,145,223,41,28,196,35,41,132,12,37,4,118,138,229,215,74,237,105,57,201,118,51,247,141,47,107,105,237,145,53,31,216,92,223,188,221,110,119,227,184,232,61,89,65,196,218,105,22,199,234,147,147,42,100,123,6,52,250,183,131,163,1,106,152,150,252,197,134,23,116,87,107,222,109,211,232,163,5,190,170,52,231,149,12,45,105,22,153,127,108,12,112,103,142,215,231,237,136,5,157,166,163,119,175,171,150,37,109,168,228,221,143,78,118,149,29,26,24,93,69,168,47,56,80,188,16,77,253,177,110,61,217,57,61,190,96,247,219,12,106,59,63,67,3,16,219,19,52,240,212,190,20,250,64,29,60,187,140,133,54,62,105,81,45,63,24,61,27,121,201,161,25,203,50,5,137,175,53,140,253,91,1,163,93,34,19,32,125,230,105,98,156,217,150,178,67,51,57,28,37,51,217,4,95,222,220,81,184,231,11,53,153,215,1,238,195,65,122,198,54,177,166,139,198,14,61,97,192,30,173,248,128,238,214,120,143,88,194,194,184,58,235,15,10,52,154,160,222,41,100,133,134,78,204,225,140,162,174,112,16,188,175,93,101,25,203,238,217,174,106,252,15,139,227,230,204,25,57,198,242},
  {97,223,89,150,253,247,123,136,52,230,214,108,29,76,144,135,156,86,157,240,217,228,1,221,87,29,245,230,188,140,255,121,173,137,81,42,31,0,149,212,210,42,168,99,83,84,205,38,37,243,252,119,149,148,207,203,118,44,189,235,224,116,155,10,33,162,47,241,206,208,59,81,225,227,168,110,134,85,214,32,30,74,107,15,102,59,59,11,160,228,236,135,111,58,200,67,29,232,72,141,221,72,200,72,129,61,55,172,136,81,234,204,29,201,246,235,155,130,48,29,152,64,185,140,160,190,23,183,98,150,105,141,80,122,75,205,88,189,218,138,229,146,114,96,135,191,63,210,131,125,203,200,157,170,130,170,17,223,244,72,124,151,102,55,146,135,183,48,12,51,87,247,38,0,62,237,255,147,190,151,167,169,156,64,102,62,34,126,10,76,80,49,70,50,74,242,90,185,76,185,204,20,80,149,159,167,209,10,212,46,92,165,212,147,28,62,188,224,169,142,137,231,26,155,140,234,176,138,33,98,154,229,124,29,42,77,153,146,130,26,193,122,113,99,19,222,243,112,4,156,151,187,193,108,246,43,56,138,15,157,65,230,43,158,246,225,249,179,43,9,127,152,164,188,210,200,69,16,99,202,226,213,27,212,114,79,192,2,78,234,7,199,182,191,187,121,140,95,204,213,3,212,161,115,85,87,102,6,27,69,104,66,167,164,46,169,246,150,225,88,188,163,98,251,25,165,58,168,201,68,251,152,61,12,98,31,221,38,231,161,75,170,227,195,100,93,13,249,213,198,51,62,194,137,9,143,246,228,211,154,71,28,93,102,133,159,23,79,208,143,253,246,232,185,103,3,193,238,80,158,53,198,53,114,8,123,6,213,55,227,8,245,96,86,41,197,16,111,76,84,189,35,219,243,231,93,122,137,225,90,99,179,248,167,198,148,105,199,56,250,52,178,83,229,228,78,94,46,209,203,181,86,83,115,118,189,42,234,67,136,118,197,51,97,48,181,18,50,202,251,249,138,115,123,89,215,167,98,12,86,70,25,53,35,182,201,162,127,65,94,47,67,83,65,95,95,110,77,41,150,245,73,210,200,197,98,139,3,52,109,172,73,109,24,231,197,57,218,130,81,15,83,224,42,200,24,16,143,195,184,119,72,34,195,63,64,208,250,62,25,142,206,149,44,125,114,88,216,253,244,80,246,241,161,35,80,235,179,146,118,110,159,131,186,8,89,108,106,24,103,53,87,29,223,40,127,186,125,159,103,182,96,156,249,252,153,111,61,232,28,146,127,132,59,212,87,225,103,217,137,18,171,213,207,134,114,22,12,3,179,245,218,169,118,52,110,137,225,150,4,95,240,49,61,112,108,112,250,9,91,168,181,108,120,100,113,192,3,187,117,218,128,45,228,42,136,43,48,100,124,210,151,239,45,160,48,195,69,91,47,69,180,210,213,139,115,13,17,198,218,226,220,81,229,179,16,234,22,13,160,11,230,228,139,41,71,87,197,90,141,146,150,98,108,125,123,140,75,148,253,243,222,148,15,179,164,126,129,47,93,178,102,170,142,178,102,122,17,147,173,110,130,223,160,168,172,231,168,13,23,81,50,182,40,146,3,197,166,24,65,185,120,75,62,85,182,5,167,198,50,251,174,132,198,97,151,224,235,124,236,82,222,161,93,216,57,240,252,180,25,49,167,76,35,25,118,126,82,134,88,14,115,149,14,234,24,168,19,125,187,152,22,169,7,228,190,168,99,95,127,189,215,216,30,173,84,109,195,129,192,176,209,186,203,207,136,35,231,37,59,63,179,126,52,180,216,94,232,52,99,227,141,166,95,60,208,255,58,200,108,11,9,47,29,230,97,238,57,7,90,186,133,160,57,255,199,154,209,196,82,239,144,55,233,20,253,61,190,148,116,156,166,120,145,211,146,253,22,6,101,171,172,190,2,7,233,250,137,156,198,34,160,194,99,132,71,68,49,200,61,133,181,44,28,226,139,171,55,39,150,106,102,31,254,236,85,201,71,43,218,165,203,124,107,183,67,60,218,21,208,167,101,49,74,80,2,80,80,37,55,82,107,96,211,40,28,107,148,229,138,181,96,50,132,133,41,189,41,49,105,34,48,128,103,27,71,186,152,119,47,188,163,212,103,177,171,30,3,37,31,45,31,92,61,40,132,97,18,43,226,65,179,146,46,168,228,165,98,21,243,91,67,190,74,141,23,249,226,141,251,94,123,58,123,164,3,16,151,166,150,103,17,181,181,18,110,49,194,229,65,153,81,38,186,234,67,138,188,132,227,65,242,142,14,50,5,117,231,239,27,244,165,131,242,83,27,233,59,18,142,108,82,9,151,167,163,93,166,161,6,91,120,76,201,111,239,231,35,55,19,91,43,15,106,200,242,27,224,165,95,238,60,45,25,120,0,183,244,200,162,106,186,117,116,187,113,21,136,241,136,45,78,210,204,96,101,130,45,36,211,242,62,228,136,207,8,129,246,81,219,123,11,223,45,109,81,234,176,223,143,113,71,152,123,57,47,238,172,104,103,20,208,54,119,92,131,114,11,172,228,82,42,71,6,54,55,121,161,79,98,236,17,156,226,181,155,56,252,226,253,227,44,67,10,212,84,212,100,203,195,234,99,97,140,187,13,81,70,102,120,110,23,210,32,240,194,178,12,147,157,151,14,189,56,230,242,120,44,237,107,127,141,144,150,109,12,142,98,163,96,149,123,68,69,235,8,186,91,251,180,147,44,183,175,11,208,105,109,15,208,105,250,109,190,99,191,7,83,84,184,71,117,7,250,239,131,182,167,109,29,27,201,119,138,227,146,102,244,61,133,105,14,246,113,180,223,198,69,161,173,208,77,231,103,143,163,163},
} ;

static void test_encode_1277x3_impl(long long impl)
{
  unsigned char *s = test_encode_1277x3_s;
  unsigned char *x = test_encode_1277x3_x;
  unsigned char *s2 = test_encode_1277x3_s2;
  unsigned char *x2 = test_encode_1277x3_x2;
  long long slen = crypto_encode_STRBYTES;
  long long xwords = crypto_encode_ITEMS;
  long long xlen;

  if (targeti && strcmp(targeti,".") && strcmp(targeti,ntruprime_dispatch_encode_1277x3_implementation(impl))) return;
  if (targetn && atol(targetn) != impl) return;
  if (impl >= 0) {
    crypto_encode = ntruprime_dispatch_encode_1277x3(impl);
    printf("encode_1277x3 %lld implementation %s compiler %s\n",impl,ntruprime_dispatch_encode_1277x3_implementation(impl),ntruprime_dispatch_encode_1277x3_compiler(impl));
  } else {
    crypto_encode = ntruprime_encode_1277x3;
    printf("encode_1277x3 selected implementation %s compiler %s\n",ntruprime_encode_1277x3_implementation(),ntruprime_encode_1277x3_compiler());
  }
  for (long long checksumbig = 0;checksumbig < 2;++checksumbig) {
    long long loops = checksumbig ? 4096 : 1024;

    checksum_clear();

    for (long long loop = 0;loop < loops;++loop) {
      xlen = xwords*crypto_encode_ITEMBYTES;

      output_prepare(s2,s,slen);
      input_prepare(x2,x,xlen);
      endianness(x,xwords,crypto_encode_ITEMBYTES);
      secret(x,xlen);
      crypto_encode(s,x);
      public(x,xlen);
      endianness(x,xwords,crypto_encode_ITEMBYTES);
      public(s,slen);
      checksum(s,slen);
      output_compare(s2,s,slen,"crypto_encode");
      input_compare(x2,x,xlen,"crypto_encode");

      double_canary(s2,s,slen);
      double_canary(x2,x,xlen);
      endianness(x2,xwords,crypto_encode_ITEMBYTES);
      secret(x2,xlen);
      crypto_encode(s2,x2);
      public(x2,xlen);
      endianness(x2,xwords,crypto_encode_ITEMBYTES);
      public(s2,slen);
      if (memcmp(s2,s,slen) != 0) fail("failure: crypto_encode is nondeterministic\n");
    }
    checksum_expected(encode_1277x3_checksums[checksumbig]);
  }
  for (long long precomp = 0;precomp < precomputed_encode_1277x3_NUM;++precomp) {
    output_prepare(s2,s,crypto_encode_STRBYTES);
    input_prepare(x2,x,crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
    memcpy(x,precomputed_encode_1277x3_x[precomp],crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
    memcpy(x2,precomputed_encode_1277x3_x[precomp],crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
    crypto_encode(s,x);
    if (memcmp(s,precomputed_encode_1277x3_s[precomp],crypto_encode_STRBYTES)) {
      fail("failure: crypto_encode fails precomputed test vectors\n");
      printf("expected s: ");
      for (long long pos = 0;pos < crypto_encode_STRBYTES;++pos) printf("%02x",((unsigned char *) precomputed_encode_1277x3_s[precomp])[pos]);
      printf("\n");
      printf("received s: ");
      for (long long pos = 0;pos < crypto_encode_STRBYTES;++pos) printf("%02x",s[pos]);
      printf("\n");
    }
    output_compare(s2,s,crypto_encode_STRBYTES,"crypto_encode");
    input_compare(x2,x,crypto_encode_ITEMS*crypto_encode_ITEMBYTES,"crypto_encode");
  }
}

void test_encode_1277x3(void)
{
  long long maxalloc = 0;
  if (targeto && strcmp(targeto,"encode")) return;
  if (targetp && strcmp(targetp,"1277x3")) return;
  storage_encode_1277x3_s = callocplus(crypto_encode_STRBYTES);
  test_encode_1277x3_s = aligned(storage_encode_1277x3_s,crypto_encode_STRBYTES);
  if (crypto_encode_STRBYTES > maxalloc) maxalloc = crypto_encode_STRBYTES;
  storage_encode_1277x3_x = callocplus(crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
  test_encode_1277x3_x = aligned(storage_encode_1277x3_x,crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
  if (crypto_encode_ITEMS*crypto_encode_ITEMBYTES > maxalloc) maxalloc = crypto_encode_ITEMS*crypto_encode_ITEMBYTES;
  storage_encode_1277x3_s2 = callocplus(maxalloc);
  test_encode_1277x3_s2 = aligned(storage_encode_1277x3_s2,crypto_encode_STRBYTES);
  storage_encode_1277x3_x2 = callocplus(maxalloc);
  test_encode_1277x3_x2 = aligned(storage_encode_1277x3_x2,crypto_encode_ITEMS*crypto_encode_ITEMBYTES);

  for (long long offset = 0;offset < 1;++offset) {
    if (targetoffset && atol(targetoffset) != offset) continue;
    if (offset && valgrind) break;
    printf("encode_1277x3 offset %lld\n",offset);
    for (long long impl = -1;impl < ntruprime_numimpl_encode_1277x3();++impl)
      forked(test_encode_1277x3_impl,impl);
    ++test_encode_1277x3_s;
    ++test_encode_1277x3_x;
    ++test_encode_1277x3_s2;
    ++test_encode_1277x3_x2;
  }
  free(storage_encode_1277x3_x2);
  free(storage_encode_1277x3_s2);
  free(storage_encode_1277x3_x);
  free(storage_encode_1277x3_s);
}
#undef crypto_encode_STRBYTES
#undef crypto_encode_ITEMS
#undef crypto_encode_ITEMBYTES

