#include "osl/container/pieceMask.h"
#include "osl/piece.h"
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

using namespace osl;

class PieceMaskTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(PieceMaskTest);
  CPPUNIT_TEST(testIsEmpty);
  CPPUNIT_TEST(testSetGet);
  CPPUNIT_TEST(testMultiple);
  CPPUNIT_TEST(testCount2);
  CPPUNIT_TEST(testSelectBit);
  CPPUNIT_TEST(testClearBit);
  CPPUNIT_TEST_SUITE_END();
 public:
  void testIsEmpty();
  void testSetGet();
  void testMultiple();
  void testCount2();
  void testSelectBit();
  void testClearBit();
};

CPPUNIT_TEST_SUITE_REGISTRATION(PieceMaskTest);

void PieceMaskTest::testCount2()
{
    PieceMask piece_mask;
    CPPUNIT_ASSERT_EQUAL(0, piece_mask.countBit2());
    piece_mask.set(0);
    CPPUNIT_ASSERT_EQUAL(1, piece_mask.countBit2());
    piece_mask.set(1);
    CPPUNIT_ASSERT_EQUAL(2, piece_mask.countBit2());
}

void PieceMaskTest::testIsEmpty()
{
  {
    PieceMask piece_mask;
    CPPUNIT_ASSERT(piece_mask.none());
    piece_mask.setAll();
    CPPUNIT_ASSERT(!piece_mask.none());
    piece_mask.resetAll();
    CPPUNIT_ASSERT(piece_mask.none());
    piece_mask.set(10);
    CPPUNIT_ASSERT(!piece_mask.none());
    piece_mask.reset(10);
    CPPUNIT_ASSERT(piece_mask.none());
    piece_mask.set(37);
    CPPUNIT_ASSERT(!piece_mask.none());
    piece_mask.reset(37);
    CPPUNIT_ASSERT(piece_mask.none());
  }
  {
#if OSL_WORDSIZE == 64
    CPPUNIT_ASSERT(PieceMask64(mask_t::makeDirect(0x0uLL)).none());
    CPPUNIT_ASSERT(!PieceMask64(mask_t::makeDirect(0x8uLL)).none());
    CPPUNIT_ASSERT(!PieceMask64(mask_t::makeDirect(0x800000000uLL)).none());
#elif OSL_WORDSIZE == 32
    CPPUNIT_ASSERT(PieceMask32(mask_t::makeDirect(0u),mask_t::makeDirect(0x0u)).none());
    CPPUNIT_ASSERT(!PieceMask32(mask_t::makeDirect(0u),mask_t::makeDirect(0x8u)).none());
    CPPUNIT_ASSERT(!PieceMask32(mask_t::makeDirect(0x8u),mask_t::makeDirect(0u)).none());
#endif
  }
}

void PieceMaskTest::testSetGet()
{
  {
    PieceMask piece_mask;
#if OSL_WORDSIZE == 64
    CPPUNIT_ASSERT(piece_mask.getMask(0).value()==0uLL);
    piece_mask.set(10);
    CPPUNIT_ASSERT(piece_mask.getMask(0).value()==(1uLL<<10));
    piece_mask.set(37);
    CPPUNIT_ASSERT(piece_mask.getMask(0).value() == ((1uLL<<10)|(1uLL<<37)));
#elif OSL_WORDSIZE == 32
    CPPUNIT_ASSERT(piece_mask.getMask(0).value()==0u 
		   && piece_mask.getMask(1).value()==0u);
    piece_mask.set(10);
    CPPUNIT_ASSERT(piece_mask.getMask(0).value()==(1u<<10)
		   && piece_mask.getMask(1).value()==0u);
    piece_mask.set(37);
    CPPUNIT_ASSERT(piece_mask.getMask(0).value()==(1u<<10) &&
		   piece_mask.getMask(1).value()==(1u<<5));
#endif
  }
  {
    PieceMask piece_mask;
#if OSL_WORDSIZE == 64
    piece_mask.setMask(0, mask_t::makeDirect(0xf00000000fuLL));
    CPPUNIT_ASSERT(piece_mask.test(3));
    CPPUNIT_ASSERT(!piece_mask.test(4));
    CPPUNIT_ASSERT(!piece_mask.test(35));
    CPPUNIT_ASSERT(piece_mask.test(36));
#elif OSL_WORDSIZE == 32
    piece_mask.setMask(0,mask_t::makeDirect(0x0000000fu));
    piece_mask.setMask(1,mask_t::makeDirect(0xf0u));
    CPPUNIT_ASSERT(piece_mask.test(3));
    CPPUNIT_ASSERT(!piece_mask.test(4));
    CPPUNIT_ASSERT(!piece_mask.test(35));
    CPPUNIT_ASSERT(piece_mask.test(36));
#endif
  }
}

void PieceMaskTest::testMultiple()
{
  {
    PieceMask piece_mask;
    CPPUNIT_ASSERT(!piece_mask.hasMultipleBit());
    piece_mask.set(10);
    CPPUNIT_ASSERT(!piece_mask.hasMultipleBit());
    piece_mask.set(37);
    CPPUNIT_ASSERT(piece_mask.hasMultipleBit());
    piece_mask.reset(10);
    CPPUNIT_ASSERT(!piece_mask.hasMultipleBit());
    piece_mask.set(32);
    CPPUNIT_ASSERT(piece_mask.hasMultipleBit());
    piece_mask.set(12);
    CPPUNIT_ASSERT(piece_mask.hasMultipleBit());
    piece_mask.set(31);
    CPPUNIT_ASSERT(piece_mask.hasMultipleBit());
    piece_mask.reset(32);
    CPPUNIT_ASSERT(piece_mask.hasMultipleBit());
    piece_mask.reset(37);
    CPPUNIT_ASSERT(piece_mask.hasMultipleBit());
    piece_mask.reset(12);
    CPPUNIT_ASSERT(!piece_mask.hasMultipleBit());
  }
}

void PieceMaskTest::testSelectBit()
{
  PieceMask m;
  for (int i=0; i<Piece::SIZE; ++i) {
    m.set(i);
  }

  CPPUNIT_ASSERT_EQUAL(m.selectBit<PAWN>(),   mask_t::makeDirect(PtypeFuns<PAWN>::indexMask));
  CPPUNIT_ASSERT_EQUAL(m.selectBit<LANCE>(),  mask_t::makeDirect(PtypeFuns<LANCE>::indexMask));
  CPPUNIT_ASSERT_EQUAL(m.selectBit<KNIGHT>(), mask_t::makeDirect(PtypeFuns<KNIGHT>::indexMask));
  CPPUNIT_ASSERT_EQUAL(m.selectBit<SILVER>(), mask_t::makeDirect(PtypeFuns<SILVER>::indexMask));
  CPPUNIT_ASSERT_EQUAL(m.selectBit<GOLD>(),   mask_t::makeDirect(PtypeFuns<GOLD>::indexMask));
  CPPUNIT_ASSERT_EQUAL(m.selectBit<BISHOP>(), mask_t::makeDirect(PtypeFuns<BISHOP>::indexMask));
  CPPUNIT_ASSERT_EQUAL(m.selectBit<ROOK>(),   mask_t::makeDirect(PtypeFuns<ROOK>::indexMask));
}

void PieceMaskTest::testClearBit()
{
  PieceMask m;
  for (int i=0; i<Piece::SIZE; ++i) {
    m.set(i);
  }
  m.clearBit<KNIGHT>();
  CPPUNIT_ASSERT(m.selectBit<KNIGHT>().none());
  CPPUNIT_ASSERT(~(m.selectBit<PAWN>()).none());
  CPPUNIT_ASSERT(~(m.selectBit<LANCE>()).none());
  CPPUNIT_ASSERT(~(m.selectBit<SILVER>()).none());
  CPPUNIT_ASSERT(~(m.selectBit<GOLD>()).none());
  CPPUNIT_ASSERT(~(m.selectBit<BISHOP>()).none());
  CPPUNIT_ASSERT(~(m.selectBit<ROOK>()).none());
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
