//***************************************************************************
//
//  Copyright (c) 1999 - 2006 Intel Corporation
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
//***************************************************************************
//
//	IFXTimeManager.h
//
//	DESCRIPTION
//		This module defines the IFXTimeManager class.  It is used to...
//
//	NOTES
//
//***************************************************************************

/**
	@file	IFXTimeManager.h

			This header file defines the IFXTimeManager interface and its functionality.
*/

#ifndef __IFXTimeManager_H__
#define __IFXTimeManager_H__

#include "IFXUnknown.h"
#include "IFXTask.h"
#include "IFXClock.h"

// {AD4A0942-B21A-11d5-9AE4-00D0B73FB755}
IFXDEFINE_GUID(IID_IFXTimeManager,
0xad4a0942, 0xb21a, 0x11d5, 0x9a, 0xe4, 0x0, 0xd0, 0xb7, 0x3f, 0xb7, 0x55);

//class IFXClock;

/**
	This is the main interface for IFXTimeManager.

	@note	The associated IID is named IID_IFXTimeManager.
*/

class IFXTimeManager : public IFXTask {
public:
	/**
		This method initializes the Time Manager.

		@param	pCoreServices	A pointer to the CoreServices object.\n

		@return	Returns an IFXRESULT code..\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_ALREADY_INITIALIZED if the Time Manager has already been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pCoreServices is invalid.\n
	*/
	virtual IFXRESULT IFXAPI  Initialize(IFXCoreServices * pCoreServices) = 0;

	/**
		This method removes all registered jobs from the Time Manager.

		@return	Returns an IFXRESULT code..\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the Time Manager has not been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  Cleanup() = 0;

	/**
		This method retrieves the TaskManager for this component.

		@param	ppTaskManager	pointer to recieve a pointer to the TaskManager for this component.\n

		@return	Returns an IFXRESULT code..\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the Time Manager has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if ppTaskManager is invalid.\n
	*/
	virtual IFXRESULT IFXAPI  GetTaskManager(IFXTaskManager ** ppTaskManager) = 0;

	/**
		This method creates a new timer. This does not create an associated Notification for the timer;
		the notification to be generated must be specified by the id parameter.

		@param	id	The IFXNotificationId for this timer. This id will be supplied to the
					NotificationManager when this timer is triggered, and may be caught
					by notification handlers.\n
		@param	start	starting time for this timer in milliseconds, relative to the present.\n
		@param	period	period for this timer in milliseconds.\n
		@param	duration	duration for this timer in milliseconds. A duration of zero means infinity.\n
		@param	pTimerId	Receives a unique id for this timer. This id can be used to kill the timer prematurely.\n

		@return	Returns an IFXRESULT code..\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the Time Manager has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pTimerId is invalid.\n
		- @b IFX_E_OUT_OF_MEMORY is returned if the new timer cannot be allocated.\n
		- @b IFX_E_OUT_OF_RANGE indicates that id is an invalid IFXNotificationId.\n
	*/
	virtual IFXRESULT IFXAPI  SetTimer(IFXNotificationId id, U32 start, U32 period, U32 duration, U32 * pTimerId) = 0;

	/**
		This method deletes an existing timer.

		@param	timerId	The id of the timer to be deleted.\n

		@return	Returns an IFXRESULT code..\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the Time Manager has not been initialized.\n
		- @b IFX_E_INVALID_HANDLE is returned if timerId is invalid.\n
		- @b IFX_E_INVALID_POINTER indicates an internal error has occurred.
	*/
	virtual IFXRESULT IFXAPI  KillTimer(U32 timerId) = 0;

	// convenience methods

	/**
		Creates a new notification that is controlled by a timer. This is similar to the SetTimer method,
		except that the notification is auto-generated rather than being specified by the client.
		Instead of returning the TimerId, this method returns the taskhandle of the notification handler.

		@param	pTask	The IFXTask which is to be handle events generated by this timer.\n
		@param	start	starting time for this timer in milliseconds, relative to the present.\n
		@param	period	period for this timer in milliseconds.\n
		@param	duration	duration for this timer in milliseconds. A duration of zero means infinity.\n
		@param	pTaskHandle	Receives the task handle that uniquely identifies this timer/notification pair.\n

		@return	Returns an IFXRESULT code..\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the Time Manager has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pTask or pTaskHandle is invalid.\n
		- @b IFX_E_OUT_OF_MEMORY is returned if the new timer cannot be allocated.\n

		@note	Timers created by this method should be deleted using the UnregisterTask method.
				Unregister task properly kills the timer and deletes the associated notification.
	*/
	// The notification id is is auto-generated and stored in NM->TM->Node->Info->Id
	// The timer id is also auto-generated and stored in NM->TM->Node->Info->UserData
	virtual IFXRESULT IFXAPI  RegisterTask(IFXTask *pTask, U32 start, U32 period, U32 duration, IFXTaskHandle * pTaskHandle) = 0;

	/**
		Removes a timer and notification created by RegisterTask. This should only be used
		on notifications created by IFXTimeManager->RegisterTask, since it interprets the
		UserData field of registered NotificationInfo as a valid timer and attempts to kill it.

		@param	taskHandle	The handle generated by RegisterTask.\n

		@return	Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the Time Manager has not been initialized.\n
		- @b IFX_E_INVALID_HANDLE is returned if the taskHandle is invalid.\n
	*/
	// unregister an auto-generated timer notification
	// this assumes the timer is stored in NM->TM->Node->Info->UserData
	virtual IFXRESULT IFXAPI  UnregisterTask(IFXTaskHandle taskHandle) = 0;
};


#endif


