(*
 * Copyright 2003-2010 Savonet team
 *
 * This file is part of Ocaml-flac.
 *
 * Ocaml-flac is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Ocaml-flac is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Ocaml-flac; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)

(* Author; Romain Beauxis <toots@rastageeks.org> *)

(** {1 Ogg/flac encoder/decoder modules for OCaml} *)

(** Decode ogg/flac data *)
module Decoder : sig
  (** {3 Usage} *)

  (** Usage is similar to the case
    * of the native FLAC decoder, using
    * the appropriate ogg/flac decoding
    * callbacks. 
    *
    * The main difference is that in the 
    * case of the ogg/flac decoding, the
    * exception [Ogg.Not_enough_data] may 
    * be raised if the ogg stream used to
    * create the decoder does not contain 
    * enough data. In this case, you should 
    * feed more data into the ogg stream and 
    * call the decoding function again.
    * 
    * This remark is valid for both the
    * [Flac.Decoder.init] and [Flac.Decoder.process] 
    * functions. *)

  (** {3 Types} *)

  (** Variant type for ogg/flac decoder *)
  type ogg

  (** Check if an ogg packet is the first
    * packet of an ogg/flac stream. *)
  val check_packet : Ogg.Stream.packet -> bool

  (** Create a set of callbacks to decode an ogg/flac stream *)
  val get_callbacks :
    Ogg.Stream.stream -> Flac.Decoder.write -> ogg Flac.Decoder.callbacks
end

(** Encode ogg/flac data *)
module Encoder : sig
  (** {3 Usage} *)

  (** Usage is similar to the case
    * of the native FLAC encoder, using
    * the appropriate ogg/flac encoding
    * callbacks. *)

  (** {3 Types} *)

  (** Variant type for ogg/flac encoder *)
  type ogg

  type t = {
    encoder : ogg Flac.Encoder.t;
    callbacks : ogg Flac.Encoder.callbacks;
    first_pages : Ogg.Page.t list;
  }

  (** Create an ogg/flac encoder.
    * 
    * The returned value contains an encoder value
    * that can be used with the functions from the 
    * [Flac.Encoder] module, as well as the
    * corresponding callbacks to use with the various
    * encoding functions. *)
  val create :
    ?comments:(string * string) list ->
    serialno:Nativeint.t ->
    Flac.Encoder.params ->
    (Ogg.Page.t -> unit) ->
    t
end

(** Ogg/flac skeleton module *)
module Skeleton : sig
  (** Generate a flac fisbone packet with
    * these parameters, to use in an ogg skeleton.
    * Default value for [start_granule] is [Int64.zero],
    * Default value for [headers] is ["Content-type","audio/x-flac"]
    *
    * See: http://xiph.org/ogg/doc/skeleton.html. *)
  val fisbone :
    ?start_granule:Int64.t ->
    ?headers:(string * string) list ->
    serialno:Nativeint.t ->
    samplerate:Int64.t ->
    unit ->
    Ogg.Stream.packet
end
