///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __CALC_DISPLACEMENTS_MODIFIER_H
#define __CALC_DISPLACEMENTS_MODIFIER_H

#include <core/Core.h>
#include <core/gui/properties/SubObjectParameterUI.h>

#include <atomviz/AtomViz.h>
#include "../AtomsObjectModifierBase.h"
#include <atomviz/atoms/datachannels/DisplacementDataChannel.h>

namespace AtomViz {

/******************************************************************************
* Compares the input atoms to a reference set of atoms and calculates the
* displacement vectors.
******************************************************************************/
class ATOMVIZ_DLLEXPORT CalcDisplacementsModifier : public AtomsObjectModifierBase
{
public:

	/// Default constructor.
	CalcDisplacementsModifier(bool isLoading = false);

	/// Asks the modifier for its validity interval at the given time.
	virtual TimeInterval modifierValidity(TimeTicks time);

	/// Returns the object that contains the reference configuration of the atoms
	/// used for calculating the displacement vectors.
	SceneObject* referenceConfiguration() const { return _referenceObject; }
	/// Sets the object that contains the reference configuration of the atoms
	/// used for calculating the displacement vectors.
	void setReferenceConfiguration(const SceneObject::SmartPtr& refAtoms) { _referenceObject = refAtoms; }

	/// Returns whether the reference configuration is shown instead of the current configuration.
	bool referenceShown() const { return _referenceShown; }
	/// Sets whether the reference configuration is shown instead of the current configuration.
	void setReferenceShown(bool show) { _referenceShown = show; }

	/// \brief Returns the prototype of the displacement data channel.
	/// \return The internal channel prototype. A copy of this object is made each time the modifier is evaluated.
	DisplacementDataChannel* displacementChannelPrototype() const { return _displacementChannelPrototype; }

	/// \brief Returns the path to the reference configuration file.
	/// \return The path to the input file or an empty string if no input file has been selected.
	/// \sa AtomsImportObject::inputFile()
	/// \sa showSelectionDialog()
	QString inputFile() const;

public Q_SLOTS:

	/// \brief Displays the file selection dialog and lets the user select the file with the reference configuration.
	/// \psaram parent The optional parent widget for the dialog.
	void showSelectionDialog(QWidget* parent = NULL);

public:

	Q_PROPERTY(bool referenceShown READ referenceShown WRITE setReferenceShown)
	Q_PROPERTY(QString inputFile READ inputFile)

protected:

	/// Modifies the atoms object. The time interval passed
	/// to the function is reduced to the interval where the modified object is valid/constant.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

	/// The reference configuration of the atoms.
	ReferenceField<SceneObject> _referenceObject;

	/// The prototype of the displacement data channel.
	/// A copy of this object will be made each time the modifier is evaluated.
	ReferenceField<DisplacementDataChannel> _displacementChannelPrototype;

	/// Controls the whether the reference configuration is shown instead of the current configuration.
	PropertyField<bool> _referenceShown;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(CalcDisplacementsModifier)
	DECLARE_REFERENCE_FIELD(_referenceObject)
	DECLARE_REFERENCE_FIELD(_displacementChannelPrototype)
	DECLARE_PROPERTY_FIELD(_referenceShown)
};

/******************************************************************************
* A properties editor for the CalcDisplacementsModifier class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT CalcDisplacementsModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	SubObjectParameterUI* subObjectUI;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CalcDisplacementsModifierEditor)
};

};	// End of namespace AtomViz

#endif // __CALC_DISPLACEMENTS_MODIFIER_H
