///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ApplicationManager.h
 * \brief Contains the definition of the Core::ApplicationManager class.
 */

#ifndef __OVITO_APPLICATION_MANAGER_H
#define __OVITO_APPLICATION_MANAGER_H

#include <core/Core.h>
#include <core/plugins/branding/UIEventFilter.h>
#include <core/plugins/autostart/AutoStart.h>

namespace Core {

/// \def APPLICATION_MANAGER
/// \brief The predefined instance of the Core::ApplicationManager class.
///
/// Always use this macro to access the Core::ApplicationManager class instance.
#define APPLICATION_MANAGER		(*(ApplicationManager*)qApp)

/**
 * \brief The main application object.
 *
 * A instance of this class is created at program startup.
 * It can be accessed via the APPLICATION_MANAGER macro.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT ApplicationManager : public QApplication
{
	Q_OBJECT

public:

	/// \brief This creates the application object and is called from the program's main() method.
	/// \note This is an internal method not for public use.
	static int main(int argc, char** argv);

private:

	/// \brief Constructor.
	/// \param argc The number of command line arguments.
	/// \param argv The command line arguments.
	/// \param noGUI Specifies whether the application is started with or without a graphical user interface.
	ApplicationManager(int& argc, char** argv, bool noGUI) : QApplication(argc, argv, !noGUI),
		_consoleMode(noGUI), _verboseMode(false), _experimentalMode(false), _exitCode(0) {}

	/// \brief Initializes the application.
	/// \return \c true if the application was initialized successfully;
	///         \c false if an error occurred and the program should be terminated.
	///
	/// This is called on program startup. The method
	/// creates all other global objects and the main window.
	bool initialize();

	/// \brief Enters the main event loop.
	/// \return The program exit code.
	///
	/// If the application has been started in non graphical mode then
	/// this method does nothing.
	int runApplication();

	/// \brief Releases everything.
	///
	/// This is called before the application exits.
	void shutdown();

	/// \brief Handler method for Qt error messages.
	///
	/// This can be used to set a debugger breakpoint for the OVITO_ASSERT macros.
	static void qtMessageOutput(QtMsgType type, const char *msg);

public:

	/// \brief Return whether the application has been started in graphical mode.
	/// \return \c true if the application should use a graphical user interface;
	///         \c false if the application has been started in the non-graphical console mode.
	bool guiMode() const { return !_consoleMode; }

	/// \brief Return whether the application has been started in console mode.
	/// \return \c true if the application has been started in the non-graphical console mode;
	///         \c false if the application should use a graphical user interface.
	bool consoleMode() const { return _consoleMode; }

	/// \brief Indicates whether the application should show more messages in the terminal.
	/// \return \c true if the application has been started with the \c --verbose command line switch.
	bool verboseMode() const { return _verboseMode; }

	/// \brief Indicates whether the user has started the application in experimental mode.
	/// \return \c true if the application has been started with the \c --experimental command line switch.
	///
	/// In experimental mode, experimental program features are enabled.
	bool experimentalMode() const { return _experimentalMode; }

	/// \brief Returns the path to the scene file loaded on startup.
	/// \return The path of the scene to be loaded at program startup or an empty string
	///         if no file has been specified on the command line.
	const QString& startupSceneFile() const { return _startupSceneFile; }

	/// \brief Returns the path to the script file executed on startup.
	/// \return The path to the script file to be run on program startup or an empty string
	///         if no script file has been specified using the \c --script command line switch.
	const QString& startupScriptFile() const { return _startupScriptFile; }

	/// \brief Returns the list of command line arguments.
	/// \return All arguments except those that are passed on to the startup script.
	const QStringList& arguments() const { return _arguments; }

	/// \brief When in console mode, this specifies the exit code that will be returned by the application on shutdown.
	void setExitCode(int code) { _exitCode = code; }

#if defined(Q_WS_X11)

	/// Handles X11 GUI events.
	virtual bool x11EventFilter(XEvent* event);

#elif defined(Q_WS_WIN)

	/// Handles Windows GUI events.
	virtual bool winEventFilter(MSG * msg, long * result);

#elif defined(Q_WS_MAC)

	/// Handles Carbon GUI events.
	virtual bool macEventFilter(EventHandlerCallRef caller, EventRef event);

#endif

	/// \brief Handles events from the event queue.
	virtual bool notify(QObject* receiver, QEvent* event);

private:

	/// Parses the command line parameters.
	bool parseCommandLine();

	/// Loads all installed GUI event filters.
	void loadGUIEventFilters();

	/// Loads all installed auto start objects on application startup.
	void loadAutoStartObjects();

	/// Calls the AutoStart::startEvent() method on all installed auto start objects.
	void callAutoStartObjects(AutoStart::InitializationStage stage);

private:

	/// The path to the startup scene file to load.
	QString _startupSceneFile;

	/// The path to the script file to execute on startup.
	QString _startupScriptFile;

	/// The installed GUI event filter objects.
	QVector<UIEventFilter::SmartPtr> guiEventFilters;

	/// The auto start objects.
	QVector<AutoStart::SmartPtr> autostartObjects;

	/// Indicates that the application is running in console mode.
	bool _consoleMode;

	/// Indicates that the application should show more messages in the terminal.
	bool _verboseMode;

	/// In console mode, this is the exit code returned by the application on shutdown.
	int _exitCode;

	/// Indicates whether experimental program features are enabled.
	bool _experimentalMode;

	/// The list of command line arguments.
	QStringList _arguments;

	/// The default message handler method of Qt.
	static QtMsgHandler defaultQtMessageHandler;

	/// Handler function for exceptions used in GUI mode.
	static void guiExceptionHandler(const Exception& exception);

	/// Handler function for exceptions used in console mode.
	static void consoleExceptionHandler(const Exception& exception);
};

};

#endif // __OVITO_APPLICATION_MANAGER_H
