/*
 * Picviz - Parallel coordinates ploter
 * Copyright (C) 2008 Sebastien Tricaud <toady@gscore.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $Id: axis.c 374 2009-01-20 18:08:24Z toady $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <pcimage.h>
#include <linuxlist.h>

#include "axis.h"
#include "line.h"

/**
 * \ingroup PicvizMain
 * @{
 */

/** \file axis.c
 * \brief Create and manipulate axes
 */

static unsigned int id = 0;


PicvizAxis *picviz_axis_new(void)
{
	PicvizAxis *axis = NULL;

	axis = malloc(sizeof(*axis));
	if ( ! axis ) {
		fprintf(stderr, "Cannot initialize axis: memory exhausted.\n");
		return NULL;
	}
	INIT_LLIST_HEAD(&axis->list);
	axis->id = id++;
	axis->type = DATATYPE_EMPTY;
	axis->xpos = 0;
	axis->ymin = ~0; /* FIXME: that must be auto-adaptative */
	axis->ymax = 0;

	picviz_properties_new(&axis->props);
	picviz_properties_set(axis->props, "label", "");
	picviz_properties_set(axis->props, "color", "#000000");

	return axis;
}

void picviz_axis_destroy(PicvizAxis *axis)
{
		picviz_properties_destroy(axis->props);
		free(axis);
}

PicvizAxis *picviz_axis_get(PicvizImage *i, unsigned int id)
{
	PicvizAxis *a;

	llist_for_each_entry(a, &i->axes, list) {
		if (a->id == id)
			return a;
	}

	return NULL;
}

void picviz_axis_set_type(PicvizAxis *axis, PicvizDataType type)
{
	axis->type = type;
}

void picviz_axis_set_type_from_string(PicvizAxis *axis, char *string)
{
	if (!strcmp(string, "timeline")) {
		axis->type = DATATYPE_TIMELINE;
		return;
	}
	if (!strcmp(string, "integer")) {
		axis->type = DATATYPE_INTEGER;
		return;
	}
	if (!strcmp(string, "string")) {
		axis->type = DATATYPE_STRING;
		return;
	}
	if (!strcmp(string, "float")) {
		axis->type = DATATYPE_FLOAT;
		return;
	}
	if (!strcmp(string, "short")) {
		axis->type = DATATYPE_SHORT;
		return;
	}
	if (!strcmp(string, "ipv4")) {
		axis->type = DATATYPE_IPV4;
		return;
	}
	if (!strcmp(string, "char")) {
		axis->type = DATATYPE_CHAR;
		return;
	}
	if (!strcmp(string, "gold")) {
		axis->type = DATATYPE_GOLD;
		return;
	}
	if (!strcmp(string, "years")) {
		axis->type = DATATYPE_YEARS;
		return;
	}
	if (!strcmp(string, "enum")) {
		axis->type = DATATYPE_ENUM;
		return;
	}
	if (!strcmp(string, "ln")) {
		axis->type = DATATYPE_LN;
		return;
	}
	if (!strcmp(string, "port")) {
		axis->type = DATATYPE_PORT;
		return;
	}

	axis->type = DATATYPE_EMPTY;
}

char *picviz_axis_get_string_from_type(PicvizAxis *axis)
{
	switch(axis->type) {
		case DATATYPE_TIMELINE:
			return "timeline";
			break;
		case DATATYPE_INTEGER:
			return "integer";
			break;
		case DATATYPE_STRING:
			return "string";
			break;
		case DATATYPE_FLOAT:
			return "float";
			break;
		case DATATYPE_SHORT:
			return "short";
			break;
		case DATATYPE_IPV4:
			return "ipv4";
			break;
		case DATATYPE_CHAR:
			return "char";
			break;
		case DATATYPE_GOLD:
			return "gold";
			break;
		case DATATYPE_YEARS:
			return "years";
			break;
		case DATATYPE_ENUM:
			return "enum";
			break;
		case DATATYPE_LN:
			return "ln";
			break;
		default:
			return "*** error ***";
			break;
	}
}


PicvizAxisPlot *picviz_axisplot_new(void)
{
	PicvizAxisPlot *axisplot;

	axisplot = malloc(sizeof(*axisplot));
	if ( ! axisplot ) {
		fprintf(stderr, "Cannot initalize axisplot: memory exhaustred.\n");
		return NULL;
	}

	axisplot->strval  = NULL;
	axisplot->y       = 0;
	axisplot->axis_id = 0;
	picviz_properties_new(&axisplot->props);

	return axisplot;
}

void picviz_axisplot_destroy(PicvizAxisPlot *axisplot)
{
	picviz_properties_destroy(axisplot->props);
	free(axisplot);
}
