/*	Update_DB

PIRL CVS ID: Update_DB.java,v 3.27 2012/04/16 06:08:57 castalia Exp

Copyright (C) 2003-2007  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Database;

import	PIRL.Configuration.Configuration;
import	PIRL.Configuration.Configuration_Exception;
import	PIRL.Strings.String_Buffer;
import	PIRL.Strings.Words;
import	PIRL.Strings.String_Utilities;

import	java.util.Vector;
import	java.util.Stack;
import	java.io.File;
import	java.io.FileReader;
import	java.io.BufferedReader;
import	java.io.InputStreamReader;
import	java.io.IOException;
import	java.io.FileNotFoundException;


/**	<i>Update_DB</i> updates an existing database table record or inserts a
	new record in a table.
<p>
	Database update operations are based on references to one or more
	tables to be affected plus, for each table reference, a list of
	table field names paired with the values that they are to receive.
	By attaching a key (SQL conditional expression) to a table
	reference one or more records from the table can be selected to
	have the specified field values updated. Without a key the field
	values will be put in a new record inserted at the end of the
	table.
<p>
	An application {@link #main(String[]) main} is provided using a
	{@link #Usage(boolean) command line user inteface} that will invoke the
	update operations of this class.
<p>
@author		Bradford Castalia, Christian Schaller - UA/PIRL
@version	3.27
@see		PIRL.Database
@see		PIRL.Configuration
*/
public class Update_DB
{
/**	The Class identification.
*/
public static final String
	ID = "PIRL.Database.Update_DB (3.27 2012/04/16 06:08:57)";


/**	The default Configuration filename.
*/
public static final String
	DEFAULT_CONFIGURATION_FILENAME		= "Database.conf";

/**	Key delimiter for table/field references.
*/
public static final char
	KEY_DELIMITER						= ':';

/**	Delimiter for input field assignments' values.
*/
public static final char
	VALUE_DELIMITER						= '=';


/**	The Database to be updated.
*/
protected Database
	The_Database						= null;

/**	Flag for verbose operation.
*/
protected boolean
	Verbose								= false;

/**	Flag for limiting updates to one record.
*/
protected boolean
	Update_One_Only						= false;

/**	Flag for ignoring multiple updates, rather than throwing an exception,
	when {@link #Update_One_Only(boolean) Update_One_Only} is enabled.
*/
protected boolean
	Ignore_Multiple_Updates				= false;

/**	Flag for {@link #Delete(boolean) deleting} selected records instead
	of updating them.
*/
protected boolean
	Delete								= false;

/**	Application exit status when invalid command line syntax was encounterd.
*/
public static final int
	EXIT_INVALID_COMMAND_LINE_SYNTAX	= 255;

/**	Application exit status when there is a Configuration file problem.
*/
public static final int
	EXIT_CONFIGURATION_PROBLEM			= 254;

/**	Application exit status when there is a problem accessing the Database.
*/
public static final int
	EXIT_DATABASE_ERROR					= 253;

/**	Application exit status when an illegal argument is encountered.
*/
public static final int
	EXIT_ILLEGAL_ARGUMENT				= 252;

/**	Application exit status when an I/O error occured on an arguments file.
*/
public static final int
	EXIT_IO_ERROR						= 251;

/**	Application exit status limit indicating the number of records affected.
	This is the maxium number of records that can be specified through
	the 8-bit exit status value.
*/
public static final int
	EXIT_STATUS_LIMIT					= 250;


//	Filesystem pathname separator.
private static final char
	FILE_SEPARATOR = System.getProperty ("file.separator").charAt (0);

/**	System new line sequence.
*/
protected static final String
	NL									= Database.NL;


//	Debug control
private static final int
	DEBUG_OFF							= 0,
	DEBUG_CONSTRUCTOR					= 1 << 0,
	DEBUG_DATABASE						= 1 << 1,
	DEBUG_MAIN							= 1 << 2,
	DEBUG_ALL							= -1,

	DEBUG								= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs an Update_DB object from a Configuration with a
	connection to a database server.
<p>
	The Configuration object is expected to contain all the necessary
	information Update_DB needs to connect to the database.
<p>
	@param	database	The Database to be updated.
*/
public Update_DB
	(
	Database	database
	)
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">>> Update_DB");
The_Database = database;
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("<<< Update_DB");
}

//	La construccion de un PVL_to_DB vacio se prohibe.
private Update_DB () {}

/*==============================================================================
	Accessors
*/
/**	Enables or disables verbose mode.
<p>
	When verbose mode is enabled all operations are logged. Verbose
	mode is disabled by default.
<p>
	@param	enable	true to enable the mode, false to disable.
	@return	This Update_DB object.
*/
public Update_DB Verbose
	(
	boolean	enable
	)
{Verbose = enable; return this;}

/**	Tests if verbose mode is enabled.
<p>
	@return	true if verbose mode is enabled, false if disabled.
	@see #Verbose(boolean)
*/
public boolean Verbose ()
{return Verbose;}

/**	Enables or disables the single record update only rule.
<p>
	When enabled only one record will be allowed to be updated for any
	table reference/field assignments set. The table reference key is
	first used to select the records specified by the key and, if more
	that one record is selected, the update operation is aborted. When
	disabled (the default) more than one record may be updated by each
	update set.
<p>
	<b>N.B.</b>: Enforcing the single record update rule by doing an
	initial select is not very efficient, and would be better
	implemented using transaction rollbacks. However, since not all
	databases support transaction rollbacks the select first method is
	guaranteed to work with all databases. The initial select is not
	done if the single record update rule is disabled.
<p>
	@param	enable	true to enable the rule, false to disable.
	@return	This Update_DB object.
	@see	#Ignore_Multiple_Updates(boolean)
*/
public Update_DB Update_One_Only
	(
	boolean	enable
	)
{Update_One_Only = enable; return this;}

/**	Tests if the single record update rule is enabled.
<p>
	@return	true if the rule is enabled, false if disabled.
	@see #Update_One_Only(boolean)
*/
public boolean Update_One_Only ()
{return Update_One_Only;}

/**	Enables or disables ignoring multiple record updates, rather than
	throwing an exception, when only one record is allowed.
<p>
	When enabled and {@link #Update_One_Only(boolean) Update_One_Only}
	is also enabled if a table reference key selects more than one
	record the update will not be done. When disabled and Update_One_Only
	is enabled an IllegalArgumentException will be thrown with a message
	describing the situation.
<p>
	@param	enable	If true multiple record updates will be ignored when
		only one record is allowed; otherwise an IllegalArgumentException
		will be thrown in this case.
	@return	This Update_DB object.
	@see	#Update_One_Only(boolean)
*/
public Update_DB Ignore_Multiple_Updates
	(
	boolean	enable
	)
{Ignore_Multiple_Updates = enable; return this;}

/**	Tests if multiple record updates will be ignored when only one
	record is allowed.
<p>
	@return	true if multiple record updates will be ignored when
		only one record is allowed; otherwise an IllegalArgumentException
		will be thrown in this case.
	@see #Ignore_Multiple_Updates(boolean)
*/
public boolean Ignore_Multiple_Updates ()
{return Ignore_Multiple_Updates;}

/**	Enables or disables record deletion.
<p>
	When delete is enabled and the table reference includes a key that
	selects one or more records, the records will be deleted rather than
	updated. If {@link #Update_One_Only(boolean) Update_One_Only} is also
	enabled the check for a single record selection will be applied.
<p>
	If no key is provided
<p>
	@param	enable	If true selected records will be deleted; otherwise
		selected records will be updated.
	@return	This Update_DB object.
*/
public Update_DB Delete
	(
	boolean	enable
	)
{Delete = enable; return this;}

/**	Tests if record deletion is enabled.
<p>
	@return	true if record deletion is enabled; false otherwise.
	@see #Delete(boolean)
*/
public boolean Delete ()
{return Delete;}

/**	Gets the Database being updated.
<p>
	@return	The Database currently being used.
*/
public Database Database ()
{return The_Database;}

/*==============================================================================
	Database Operations
*/
/**	Updates the database using a table reference and field assignments.
<p>
	A key is split off from the table_reference (on the first occurance
	of a <code>{@link #KEY_DELIMITER KEY_DELIMITER} character)</code>,
	if possible. The field_assignments are split (on the first
	occurance of a <code>{@link #VALUE_DELIMITER VALUE_DELIMITER}
	character) into a list of field names and their corresponding
	values. Then the alternate <code>{@link #Update_Database(String,
	String, Vector, Vector) Update_Database} </code> method is invoked.
<p>
	@param	table_reference		The table reference, with optional key.
	@param	field_assignments	The Vector of fields with assigned values.
	@return	The number of records affected.
	@throws	IllegalArgumentException	If the table_reference is null, or
		a field assignment is missing either a name or value.
	@see #Update_Database(String, String, Vector, Vector)
*/
public int Update_Database
	(
	String			table_reference,
	Vector<String>	field_assignments
	)
throws Database_Exception, IllegalArgumentException
{
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println
		(">>> Update_Database:" + NL
		+"    table_reference - " + table_reference + NL
		+"    field_assignments - " + field_assignments + NL);
String
	table,
	key,
	assignment,
	field_name,
	field_value;
if (table_reference == null)
	throw new IllegalArgumentException (Error_Message (
		"Missing table reference for Update_Database."));
if ((table = String_Utilities.Preceeding (table_reference, KEY_DELIMITER))
		== null)
	table = table_reference;
key = String_Utilities.Following (table_reference, KEY_DELIMITER);
Vector<String>
	field_names = new Vector<String> (),
	field_values = new Vector<String> ();
int
	index = -1,
	size = field_assignments.size ();
while (++index < size)
	{
	assignment = field_assignments.get (index);
	if ((field_name = String_Utilities.Preceeding (assignment, VALUE_DELIMITER))
			== null)
		throw new IllegalArgumentException (Error_Message (
			"Missing field name in " + assignment));
	if ((field_value = String_Utilities.Following (assignment, VALUE_DELIMITER))
			== null)
		throw new IllegalArgumentException (Error_Message (
			"Missing field value in " + assignment));
	field_names.add (field_name);
	field_values.add (field_value);
	}
int
	records = 
		Update_Database (table, key, field_names, field_values);
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println
		("<<< Update_Database (table_reference): " + records + " records");
return records;
}

/**	Updates the database.
<p>
	The details of for the arguments is described in the command line
	<code>{@link #Usage(boolean) Usage}</code>..
<p>
	@param	table_reference	A Database table reference, without any key.
	@param	key	The SQL conditional to select records for an update
		operation; null if an insert operation is to be done.
	@param	field_names	The names of the fields into which data
		will be placed.
	@param	field_values	The values of the fields for the
		corresponding field_names.
	@return	The number of records affected.
	@throws	IllegalArgumentException	If {@link
		#Update_One_Only(boolean) only one} record is to be updated (not
		inserted) and the key selects more than one record. Also, if the
		table_reference is null, or a field assignment is missing either
		a name or value.
	@throws	Database_Exception	If there is a problem accessing the database.
	@see	Database#Insert(String, Vector, Vector)
	@see	Database#Update(String, Vector, Vector, String)
*/
public int Update_Database
	(
	String			table_reference,
	String			key,
	Vector<String>	field_names,
	Vector<String>	field_values
	)
throws Database_Exception, IllegalArgumentException
{
String
	full_table_reference =
		The_Database.Table_Reference
			(The_Database.Catalog_Name (table_reference),
			 The_Database.Table_Name (table_reference));
if ((DEBUG & DEBUG_DATABASE) != 0)
	{
	System.out.println
		(">>> Update_Database" + NL
		+"    table_reference - " + table_reference
			+ " (" + full_table_reference + ')' + NL
		+"                key - " + key + NL
		+"    field names - " + field_names + NL
		+"    field values - " + field_values);
	Verbose = true;
	}
int
	records = 0;
if (key == null)	//	INSERT
	{
	if (Delete)
		{
		if (Verbose)
			System.out.println ("For " + Database_Reference () + NL
				+"No key for selecting records to delete.");
		if ((DEBUG & DEBUG_DATABASE) != 0)
			System.out.println ("<<< Update_Database: 0");
		return 0;
		}
	records = The_Database.Insert
				(table_reference, field_names, field_values);
	if (Verbose)
		System.out.println ("For " + Database_Reference () + NL
			+ records + " record" + ((records == 1) ? "" : "s")
				+ " inserted in table " + full_table_reference
				+ " with fields:");
	}
else				//	UPDATE
	{
	if (Update_One_Only)
		{
		Vector<Vector<String>>
			selected_records = The_Database.Select (table_reference, key);
		if ((records = selected_records.size () - 1) != 1)
			{
			String
				report =
				 "For " + Database_Reference () + NL
				+"key " + key + NL
				+"selected " + records + " record" + ((records == 1) ? "" : "s")
				+" from the " + full_table_reference + " table" + NL
				+"but only one record update is allowed." + NL
				+"No " + (Delete ? "delete" : "update") + " performed.";
			if (! Ignore_Multiple_Updates)
				throw new IllegalArgumentException (Error_Message (report));
			if (Verbose)
				System.out.println (report + NL);
			if ((DEBUG & DEBUG_DATABASE) != 0)
				System.out.println ("<<< Update_Database: 0");
			return 0;
			}
		}
	if (Delete)
		records = The_Database.Delete (table_reference, key);
	else
		records = The_Database.Update
					(table_reference, field_names, field_values, key);
	if (Verbose)
		System.out.println ("For " + Database_Reference () + NL
			+"using key " + key + NL
			+ records + " record" + ((records == 1) ? "" : "s")
				+ (Delete ? " deleted" : " updated")
				+ " in the " + full_table_reference + " table"
				+ (Delete ? "." : " with fields:"));
	}
if (Verbose &&
	! Delete)
	{
	int
		index = -1,
		size = field_names.size ();
	while (++index < size)
		System.out.println
			("    " + field_names.get (index)
				+ " = " + field_values.get (index));
	}
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println ("<<< Update_Database: " + records + " records");
return records;
}

/*==============================================================================
	Utilties
*/
/**	Produces an error message with the application's <code>ID</code>.
<p>
	The message string will be prepended with the class (@link #ID ID}
	and a new-line, if the ID is not already present in the message. If
	the message is null, it will be replaced with the ID.
<p>
	@param	message	The message String.
	@return	The modified message String.
*/
private static String Error_Message
	(
	String			message
	)
{
if (message == null)
	message = ID;
else if (message.indexOf (ID) < 0)
	message = ID + NL
		+ message;
return message;
}

/**	Produces a Database_Exception containing a message.
<p>
	The message will be prepended with a description of the Database
	from the Configuration and then modified by the {@link
	#Error_Message(String) Error_Message} method before it is used to
	construct a new Database_Exception.
<p>
	@param	message	The message String.
	@return	A Database_Exception.
*/
private Database_Exception Database_Error
	(
	String	message
	)
{
return new Database_Exception (Error_Message
	("Problem accessing " + Database_Reference () + ":" + NL
	+ message));
}

/**	Produces a one line description of the Database.
<p>
	The description is of the form:
<blockquote><p>
	<i>TYPE</i> database on host <i>HOST</i>
</blockquote>
	The Configuration is used to obtain the {@link Database#TYPE TYPE}
	and {@link Database#HOST HOST} parameters.
<p>
	@return	A database reference string.
*/
private String Database_Reference ()
{
if (The_Database == null ||
	The_Database.Data_Port () == null)
	return "No Database";
return
	The_Database.Data_Port ().Configuration ().Get_One (Database.TYPE) +
	" database on host " +
	The_Database.Data_Port ().Configuration ().Get_One (Configuration.HOST);
}

/*==============================================================================
	Application main
*/
/**	Runs the Update_DB application.
<p>
	<b>N.B.</b>: If a Database connection is establshed it is always
	Disconnected before the application exits for any reason.
<p>
	Exit status values:
<p>
	>=0 - Success: the number of records affected (to a maximum of 250).<br>
	255 - Invalid command line syntax.<br>
	254 - Configuration problem.<br>
	253 - Database access error.<br>
	252 - An illegal argument was encountered.<br>
	251 - An I/O error occured while attempting to read an argument file.
<p>
	@param	args	The {@link #Usage(boolean) command line} arguments.
*/
public static void main
	(
	String[]		args
	)
{
if ((DEBUG & DEBUG_MAIN) != 0)
	System.out.println
		("*** " + ID + " ***");
if (args.length == 0)
	Usage (true);

for (int
		count = args.length;
		--count >= 0;)
	if (args[count].toUpperCase ().startsWith ("-H"))
		Usage (true);

Update_DB
	update_DB = null;
String
	configuration_filename = null,
	database_server = null,
	table_reference = null,
	name;
Vector<String>
	field_assignments = new Vector<String> ();
int
	records = 0;
boolean
	verbose = false,
	update_one_only = false,
	ignore_multiple_updates = false,
	delete = false;

//	Argument processing.
String
	arguments[] = args;
String
	arguments_source = null;
BufferedReader
	arguments_reader = null;
Words
	argument_words = new Words ().Delimit_at_Quote (false);
Stack<String>
	sources_stack = new Stack<String> ();
Stack<String[]>
	arguments_stack = new Stack<String[]> ();
Stack<BufferedReader>
	reader_stack = new Stack<BufferedReader> ();
Stack<Integer>
	index_stack = new Stack<Integer> ();
int
	index = 0;
while (index < arguments.length)
	{
	//	Start a new update set.
	table_reference = null;
	field_assignments.clear ();

	Arguments:
	while (true)
		{
		if ((DEBUG & DEBUG_MAIN) != 0)
			System.out.println
				("    Argument " + index + ": " + arguments[index]);
		if (arguments[index].length () > 1 &&
			arguments[index].charAt (0) == '-')
			{
			switch (arguments[index].charAt (1))
				{
				case 'C':	//	-Configuration
				case 'c':
					if (++index == arguments.length ||
						arguments[index].charAt (0) == '-')
						{
						//	Use the default configuration file.
						name = DEFAULT_CONFIGURATION_FILENAME;
						--index;
						}
					else
						name = arguments[index];
					if ((DEBUG & DEBUG_MAIN) != 0)
						System.out.println
							("    Config file: " + name);
					if (configuration_filename == null)
						configuration_filename = name;
					else if (! name.equals (configuration_filename))
						{
						System.out.println
							("Additional configuration file " + name
								+ " can not be used;" + NL
							+ configuration_filename + " is already in use.");
						Disconnect (update_DB);
						System.exit (EXIT_ILLEGAL_ARGUMENT);
						}
					break;
				case 'N':	//	-No_delete
				case 'n':
					delete = false;
					break;
				case 'D':
				case 'd':
					if (arguments[index].length () > 2 &&
						Character.toUpperCase (arguments[index].charAt(2)) == 'E')
						{
							//	-Delete
						delete = true;
						break;
						}
							//	-Database
				case 'S':	//	-Server
				case 's':
					if (++index == arguments.length ||
						arguments[index].charAt (0) == '-')
						{
						System.err.println (ID + NL
							+ "Missing database server name for argument "
							+ arguments[--index] + '.');
						Disconnect (update_DB);
						Usage (verbose);
						}
					name = arguments[index];
					if ((DEBUG & DEBUG_MAIN) != 0)
						System.out.println
							("    Database server: " + name);
					if (database_server == null)
						{
						if (update_DB != null)
							{
							System.out.println
								("Additional database server name " + name
									+ " can not be used;" + NL
								+ "configuration default is already in use.");
							Disconnect (update_DB);
							System.exit (EXIT_ILLEGAL_ARGUMENT);
							}
						database_server = name;
						}
					else if (! name .equals (database_server))
						{
						System.out.println
							("Additional database server name " + name
								+ " can not be used;" + NL
							+ database_server + " is already in use.");
						Disconnect (update_DB);
						System.exit (EXIT_ILLEGAL_ARGUMENT);
						}
					break;
				case 'T':	//	-Table
				case 't':
				case 'R':	//	-Reference
				case 'r':
					if (table_reference != null)
						//	Process the previous table and fields set.
						break Arguments;

					if (++index == arguments.length ||
						arguments[index].charAt (0) == '-')
						{
						System.err.println (ID + NL
							+ "Missing table reference for argument "
							+ arguments[--index] + '.');
						Disconnect (update_DB);
						Usage (verbose);
						}
					table_reference = arguments[index];
					if ((DEBUG & DEBUG_MAIN) != 0)
						System.out.println
							("    Table reference: " + table_reference);
					break;
				case '1':	//	-1, Only single record updates allowd.
					update_one_only = true;
					if (arguments[index].length () > 2 &&
						arguments[index].charAt(2) == '+')
						ignore_multiple_updates = true;
					else
						ignore_multiple_updates = false;
					break;
				case '+':	//	-+, Multiple updates allowed.
					update_one_only = false;
					break;
				case 'V':	//	-Verbose
				case 'v':
					verbose = true;
					break;
				case 'Q':	//	-Quiet
				case 'q':
					verbose = false;
					break;
				default:
					System.err.println (ID + NL
						+ "Unrecognized switch: " + arguments[index]);
				}
			}
		else
			{
			if (arguments[index].length () > 0 &&
				arguments[index].charAt (0) == '@')
				{
				//	Push the current arguments context.
				sources_stack.push (arguments_source);
				reader_stack.push (arguments_reader);
				arguments_stack.push (arguments);
				index_stack.push (new Integer (index));
				if ((DEBUG & DEBUG_MAIN) != 0)
					System.out.println
						("    Argument context pushed.");

				//	Arguments source.
				if (arguments[index].length () == 1)
					{
					System.err.println (ID + NL
						+ "Missing filename for '@' argument.");
					Disconnect (update_DB);
					Usage (verbose);
					}
				arguments_source = arguments[index].substring (1);
				File
					file = null;
				if (! arguments_source.equals ("-"))
					{
					file = new File (arguments_source);
					if (! file.exists ())
						{
						System.err.println (ID + NL
							+ "No such file: " + arguments[index]);
						Disconnect (update_DB);
						System.exit (EXIT_ILLEGAL_ARGUMENT);
						}
					if (! file.isFile ())
						{
						System.err.println (ID + NL
							+ "Not a regular file: " + arguments[index]);
						Disconnect (update_DB);
						System.exit (EXIT_ILLEGAL_ARGUMENT);
						}
					if (! file.canRead ())
						{
						System.err.println (ID + NL
							+ "Can not read file: " + arguments[index]);
						Disconnect (update_DB);
						System.exit (EXIT_ILLEGAL_ARGUMENT);
						}
					try {arguments_source = file.getCanonicalPath ();}
					catch (IOException exception)
						{
						System.err.println (ID + NL
							+ "Unable to obtain cononical pathname for file: "
								+ arguments_source + NL
							+ exception.getMessage ());
						Disconnect (update_DB);
						System.exit (EXIT_IO_ERROR);
						}
					}
				if (sources_stack.contains (arguments_source))
					{
					System.err.println (ID + NL
						+ "Circular reference file: " + arguments[index] + NL
						+ "     " + arguments_source);
					while (! sources_stack.isEmpty ())
						System.err.println
							("from " + sources_stack.pop ());
					Disconnect (update_DB);
					System.exit (EXIT_ILLEGAL_ARGUMENT);
					}

				//	Arguments reader.
				if (file == null)
					//	stdin.
					arguments_reader = new BufferedReader
						(new InputStreamReader (System.in));
				else
					{
					try {arguments_reader =
							new BufferedReader (new FileReader (file));}
					catch (FileNotFoundException exception)
						{
						//	Shouldn't happen.
						System.err.println (ID + NL
							+ "Could not access file: " + arguments[index]);
						Disconnect (update_DB);
						System.exit (EXIT_ILLEGAL_ARGUMENT);
						}
					}

				//	Force arguments refresh from the new reader.
				index = arguments.length;
				}
			else
				//	Field assignment.
				field_assignments.add (arguments[index]);
			}

		while (++index >= arguments.length)
			{
			if (arguments_reader != null)
				{
				String
					line = null;
				Vector<String>
					words = new Vector<String> ();
				while (words.isEmpty ())
					{
					try {line = arguments_reader.readLine ();}
					catch (IOException exception)
						{
						System.err.println (ID + NL
							+ "Read failed from file: " + arguments_source + NL
							+ exception.getMessage ());
						Disconnect (update_DB);
						System.exit (EXIT_IO_ERROR);
						}
					if (line == null)
						{
						//	EOF.
						if (sources_stack.isEmpty ())
							//	End of arguments.
							break Arguments;

						//	Pop the previous arguments context.
						arguments_source = sources_stack.pop ();
						arguments_reader = reader_stack.pop ();
						arguments = arguments_stack.pop ();
						index = index_stack.pop ().intValue ();
						if ((DEBUG & DEBUG_MAIN) != 0)
							System.out.println
								("    Arguments context popped to "
									+ arguments_source);
						break;
						}
					if ((DEBUG & DEBUG_MAIN) != 0)
						System.out.println
							("    Line: " + line);
					words = argument_words.Characters (line).Split ();
					}
				if (line != null)
					{
					//	Refresh the arguments array.
					arguments = new String[words.size ()];
					if ((DEBUG & DEBUG_MAIN) != 0)
						System.out.print
							("    New arguments: ");
					for (index = 0;
						 index < words.size ();
						 index++)
						{
						String_Buffer
							word = new String_Buffer (words.get (index));
						arguments[index] = word.clean ('"').toString ();
						if ((DEBUG & DEBUG_MAIN) != 0)
							System.out.print (arguments[index]
								+ (((index + 1) == words.size ()) ? "" : ", "));
						}
					if ((DEBUG & DEBUG_MAIN) != 0)
						System.out.println ();
					index = 0;
					break;
					}
				}
			else
				//	End of arguments.
				break Arguments;
			}
		}

	//	Process the current table and fields set.
	if ((DEBUG & DEBUG_MAIN) != 0)
		System.out.println
			("    Processing update set.");

	if (table_reference == null)
		{
		System.err.println (ID + NL
			+ "No table reference.");
		Disconnect (update_DB);
		Usage (verbose);
		}

	if (field_assignments.size () == 0 &&
		! delete)
		{
		System.err.println (ID + NL
			+ "Table reference " + table_reference + NL
			+"  has no field assignment.");
		if (update_DB != null ||
			index < arguments.length ||
			arguments_reader != null)
			//	Not the only table and fields set.
			continue;
		Disconnect (update_DB);
		Usage (verbose);
		}

	if (update_DB == null)
		{
		//	Construct the Configuration.
		Configuration
			configuration = null;
		if (configuration_filename == null)
			configuration_filename = DEFAULT_CONFIGURATION_FILENAME;
		try {configuration = new Configuration
			(configuration_filename);}
		catch (Configuration_Exception exception)
			{
			System.err.println (ID + NL
				+ "Unable to construct a Configuration with file "
				+ configuration_filename + "." + NL
				+ exception.getMessage ());
			System.exit (EXIT_CONFIGURATION_PROBLEM);
			}
		catch (IllegalArgumentException exception)
			{
			System.err.println (ID + NL
				+ "Unable to locate the Configuration file "
				+ configuration_filename + "." + NL
				+ exception.getMessage ());
			System.exit (EXIT_CONFIGURATION_PROBLEM);
			}


		//	Establish the Database connection.
		Database
			database = null;
		try
			{
			database = new Database (configuration);
			database.Connect (database_server);
			}
		catch (Database_Exception exception)
			{
			System.err.println (ID + NL
				+ "Unable to connect to the database." + NL
				+ exception.getMessage ());
			System.exit (EXIT_DATABASE_ERROR);
			}
		catch (Configuration_Exception exception)
			{
			System.err.println (ID + NL
				+ "Unable to connect to the database." + NL
				+ exception.getMessage ());
			System.exit (EXIT_CONFIGURATION_PROBLEM);
			}
		if ((DEBUG & DEBUG_MAIN) != 0)
			System.out.println
				("    The_Database Data_Port Configuration -" + NL
				+ database.Data_Port ().Configuration ().Description ());

		//	Construct the Update_DB object.
		if (verbose)
			System.out.println (ID);
		update_DB = new Update_DB (database);
		}

	//	Set the modes.
	update_DB
		.Verbose (verbose)
		.Update_One_Only (update_one_only)
		.Ignore_Multiple_Updates (ignore_multiple_updates)
		.Delete (delete);

	//	Apply the table and fields set.
	try
		{
		records +=
			update_DB.Update_Database (table_reference, field_assignments);
		}
	catch (IllegalArgumentException exception)
		{
		System.err.println
			(exception.getMessage ());
		Disconnect (update_DB);
		System.exit (EXIT_ILLEGAL_ARGUMENT);
		}
	catch (Database_Exception exception)
		{
		System.err.println
			(exception.getMessage ());
		Disconnect (update_DB);
		System.exit (EXIT_DATABASE_ERROR);
		}
	}
Disconnect (update_DB);
if (records > EXIT_STATUS_LIMIT)
	records = EXIT_STATUS_LIMIT;
System.exit (records);
}

/**	Disconnects the Database of an Update_DB.
<p>
	If an exception is thrown by the Database Disconnect the exception
	message is reported to stderr.
<p>
	@param	update_DD	An Update_DB object. This may be null in which case
		nothing is done.
*/
private static void Disconnect
	(
	Update_DB update_DB
	)
{
if (update_DB == null)
	return;
try
	{
	update_DB.Database ().Disconnect ();
	}
catch (Database_Exception exception)
	{
	System.err.println (ID + NL
		+ "Unable to disconnect from the database." + NL
		+ exception.getMessage ());
	}
}

/**	Command line usage syntax.
<p>
<blockquote><pre>
Usage: <b>Update_DB</b> &lt;<i>Switches</i>&gt;
&nbsp;&nbsp;Switches -
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>C</u>onfiguration</b> [&lt;<i>filename</i>&gt;]]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>D</u>atabase</b>|<b>-<u>S</u>erver</b> &lt;<i>server name</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>T</u>able</b>|<b>-<u>R</u>eference</b> [&lt;<i>catalog</i>&gt;<b>.</b>]&lt;<i>table</i>&gt;[<b>:</b>&lt;<i>key</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;&lt;<i>field_name</i>&gt;<b>=</b>&lt;<i>value</i>&gt; [...]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>1</u></b>[<b><u>+</u></b>]]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>+</u></b>]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-</b>[<b><u>N</u>o_</b>]<b><u>DE</u>lete</b>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>V</u>erbose</b>]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>Q</u>uiet</b>]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>H</u>elp</b>]
</pre></blockquote>
<p>
<h4>
	Configuration:
</h4><p>
	If the name of the {@link PIRL.Configuration.Configuration
	Configuration} file is not specified the {@link
	#DEFAULT_CONFIGURATION_FILENAME} will be used. If the configuration
	file is not in the current working directory, it will be looked for
	in the user's home directory. The configuration file must contain
	the necessary information needed to identify and connect with the
	database server (as required by the {@link
	PIRL.Database.Database#Database(Configuration) Database} constructor
	and its {@link PIRL.Database.Database#Connect() Connect} method).
	These are typically the server "Host" name and database "Type",
	"User" and "Password" access values.
<p>
	Only one configuration file may be used.
<p>
<h4>
	Database server name:
</h4><p>
	The Configuration file may contain connection information for more
	than one database. The information for each database is organized
	by {@link Database#SERVER Server} name, which may be specified. If
	no server name is provided, the Database will attempt to use the
	default (first) Server specified in the Configuration.
<p>
	Only one database server may be used.
<p>
<h4>
	Table reference:
</h4><p>
	A Table Reference specifies a &lt;<i>table</i>&gt;, and its
	containing &lt;<i>catalog</i>&gt;, to be updated. If the catalog is
	not specified the Database will attempt to use the configuration
	Catalog parameter for the Server group, or a default Catalog
	parameter. The optional &lt;<i>key</i>&gt; is an SQL conditional
	expression that is used to select the record(s) to be updated.
	<b>N.B.</b>: If the key specification contains spaces then it must be
	quoted; if it contains quotes then these must be escaped from shell
	interpretation (by a preceeding backslash) or the the specification
	is already quoted then the specication quotes must be different from
	the enclosing quotes (use double quotes to enclose the specification
	and single quotes in the specification content). If the key is not
	present, a new record will be inserted into the table.
<p>
<h4>
	Field value assignments:
</h4><p>
	The list of one or more &lt;<i>field_name</i>&gt;=&lt;<i>value</i>&gt;
	pairs specifies the name of the record fields and the values they are
	to receive. <b>N.B.</b>: There are no spaces around the '='
	character, unless the entire assignment is quoted; if the value
	contains spaces then it must be quoted and quotes in the value
	specification must be escaped from the shell or be different from any
	enclosing quotes. If an update is occuring, only the specified fields
	will be modified. If a record is being inserted, any records of the
	field that are not specified will automatically be given their
	default values by the database server.
<p>
<h4>
	One (1) record update:
</h4><p>
	By default, more than one record may be updated for each table
	reference and field assignment list. By specifying the -1 option,
	only one record will be allowed to be updated for each set. In this
	case, if more than one record would be updated as a result of the
	key selection this will be prevented and processing will abort.
	However, if '+' is appended to this option processing will not
	abort though the update will not be done.
<p>
	The -+ option will enable multiple record updates. This is useful
	when multiple update sets are specified and a previous set disabled
	multiple record updates but the following sets should allow it.
<p>
<h4>
	Deletion:
</h4><p>
	Instead of updating records in the table selected records may be deleted
	from the table. By default, or if -no_delete is specified, table
	records are updated. A record selection key is required when delete
	mode is in effect, and any field value assignments will be ignored.
	One record update checking applies to record deletion.
<p>
<h4>
	Update sets:
</h4><p>
<p>
	A table reference and its following field value assignments, plus
	any preceeding mode switches, up to the next table reference, if
	any, constitutes an update set. Multiple update sets may be
	specified on a command line. Each update set will be processed
	before any following command line arguments are processed. The
	configuration file and modes in effect at that point will apply to
	the update set processing. An error encountered in an update set
	halts all further processing of the command line.
<p>
<h4>
	Arguments files:
</h4><p>
	At any point on the command line an argument of the form
	"@&lt;<i>filename</i>&gt;" may be specified. The file specified will
	be read for command line arguments that are processed in the order
	they occur in the file. When the end of file has been reached
	processing continues with the next argument on the command line. In
	effect, the content of an argument file replaces the
	&#064;&lt;<i>filename</i>&gt; argument on the command line. If the dash
	character ('-') is used for the &lt;<i>filename</i>&gt; the arguments
	will be read from the standard input. Any argument in an argument
	file may be an argument file argument. Circular references are
	detected and cause an error exit.
<p>
<h4>
	Verbose/Quiet:
</h4><p>
	The verbose option provides a log of operations as they occur printed
	to the standard output. The quiet option disables the verbose mode.
	By default quiet mode is in effect.
<p>
<h4>
	Help:
</h4><p>
	The command line syntax usage is listed and the program exits.
	<b>N.B.</b>: If the -Help option occurs anywhere on the command line
	no other command line arguments will be used.
<p>
	<b>N.B.</b>The usage is printed to stderr. This method always
	results in a System.exit with a status of <code>{@link
	#EXIT_INVALID_COMMAND_LINE_SYNTAX
	EXIT_INVALID_COMMAND_LINE_SYNTAX}</code>.
<p>
	@param	verbose	If true the usage is printed, otherwise silence.
*/
public static void Usage
	(
	boolean	verbose
	)
{
if (verbose)
	System.err.println
		("Usage: Update_DB <Switches>" + NL
		+"  Switches -" + NL
		+"    [-Configuration [<filename>]]"
			+ " (default: " + DEFAULT_CONFIGURATION_FILENAME + ")" + NL
		+"    [-Database|-Server <server name>]"
			+ " (default: first configuration Server)" + NL
		+"    -Table | -Reference [<catalog>.]<table>[:<key>]" + NL
		+"    <field_name>=<value> [...]" + NL
		+"    [-1[+]]"
			+ " (default: disabled, multiple record updates OK)" + NL
		+"    [-+]"
			+ " (default: enabled, multiple record updates OK)" + NL
		+"    [-[No_]DElete]"
			+ " (default: no deletion)" + NL
		+"    [-Verbose]"
			+ " (default: false)" + NL
		+"    [-Quiet]"
			+ " (default: true)" + NL
		+"    [-Help]" + NL
		);
System.exit (EXIT_INVALID_COMMAND_LINE_SYNTAX);
}

}
