//simple.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2014
 *  The code (may) include prototypes and comments (and maybe
 *  other code fragements) from libpulse*. They are mostly copyrighted by:
 *  Lennart Poettering <poettering@users.sourceforge.net> and
 *  Pierre Ossman <drzeus@drzeus.cx>
 *
 *  This file is part of libroarpulse a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this libroar
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include <libroarpulse/libroarpulse.h>

#define _seterr() if ( error != NULL ) { *error = roar_pa_raerror2paerror(err); }

/** Create a new connection to the server */
pa_simple* pa_simple_new(
    const char *server,                 /**< Server name, or NULL for default */
    const char *name,                   /**< A descriptive name for this client (application name, ...) */
    pa_stream_direction_t dir,          /**< Open this stream for recording or playback? */
    const char *dev,                    /**< Sink (resp. source) name, or NULL for default */
    const char *stream_name,            /**< A descriptive name for this client (application name, song title, ...) */
    const pa_sample_spec *ss,           /**< The sample type to use */
    const pa_channel_map *map,          /**< The channel map to use, or NULL for default */
    const pa_buffer_attr *attr,         /**< Buffering attributes, or NULL for default */
    int *error                          /**< A pointer where the error code is stored when the routine returns NULL. It is OK to pass NULL here. */
    ) {
 struct roar_keyval kv[1];
 roar_vs_t * vss = NULL;
 struct roar_audio_info info;
 int roar_dir;
 int err = ROAR_ERROR_NONE;

 if ( dev != NULL || map != NULL || attr != NULL )
  return NULL;

 if ( dir == PA_STREAM_PLAYBACK ) {
  roar_dir = ROAR_DIR_PLAY;
 } else if ( dir == PA_STREAM_RECORD ) {
  roar_dir = ROAR_DIR_RECORD;
 } else {
  return NULL;
 }

 if ( roar_pa_sspec2auinfo(&info, ss) == -1 ) {
  return NULL;
 }

 server = roar_pa_find_server((char*)server);

 vss = roar_vs_new(server, name, &err);
 if ( vss == NULL ) {
  _seterr();
  return NULL;
 }

 if ( roar_vs_stream(vss, &info, roar_dir, &err) == -1 ) {
  roar_vs_close(vss, ROAR_VS_TRUE, NULL);
  _seterr();
  return NULL;
 }

 if ( stream_name != NULL && stream_name[0] != 0 ) {
  kv[0].key   = "DESCRIPTION";
  kv[0].value = (char*)stream_name;

  roar_vs_meta(vss, kv, 1, NULL);
 }

 return (pa_simple*) vss;
}

/** Close and free the connection to the server. The connection objects becomes invalid when this is called. */
void pa_simple_free(pa_simple *s) {
 if ( s == NULL )
  return;

 roar_vs_close((roar_vs_t*)s, ROAR_VS_FALSE, NULL);
}

/** Write some data to the server */
int pa_simple_write(pa_simple *s, const void*data, size_t length, int *error) {
 int ret;
 int err = ROAR_ERROR_NONE;

 if ( s == NULL )
  return -1;

 ret = roar_vs_write((roar_vs_t*)s, data, length, &err);

 _seterr();

 return ret;
}

/** Wait until all data already written is played by the daemon */
int pa_simple_drain(pa_simple *s, int *error) {
 int ret;
 int err = ROAR_ERROR_NONE;

 if ( s == NULL )
  return -1;

 ret = roar_vs_sync((roar_vs_t*)s, ROAR_VS_WAIT, &err);

 _seterr();

 return ret;
}

/** Read some data from the server */
int pa_simple_read(pa_simple *s, void*data, size_t length, int *error) {
 int ret;
 int err = ROAR_ERROR_NONE;

 if ( s == NULL )
  return -1;

 ret = roar_vs_read((roar_vs_t*)s, data, length, &err);

 _seterr();

 return ret;
}

/** Return the playback latency. \since 0.5 */
pa_usec_t pa_simple_get_latency(pa_simple *s, int *error) {
 roar_mus_t ret;
 int err = ROAR_ERROR_NONE;

 if ( s == NULL )
  return -1;

 ret = roar_vs_latency((roar_vs_t*)s, ROAR_VS_BACKEND_DEFAULT, ROAR_VS_WAIT, &err);

 if ( ret < 0 )
  ret *= -1;

 if ( ret == 0 && err != ROAR_ERROR_NONE ) {
  _seterr();
  return -1;
 }

 return ret;
}

/** Flush the playback buffer. \since 0.5 */
int pa_simple_flush(pa_simple *s, int *error) {
 int ret;
 int err = ROAR_ERROR_NONE;

 if ( s == NULL )
  return -1;

 ret = roar_vs_sync((roar_vs_t*)s, ROAR_VS_NOWAIT, &err);

 _seterr();

 return ret;
}

//ll
