//filter-slfi-chain.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2012-2019
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>
#include <libroarlight/libroarlight.h>

#define MAX_FILTER  32

struct slfi_chain {
 struct roar_slfi_inst * filter[MAX_FILTER];
 size_t filter_len;
};

static int __cb_event_add(struct roar_slfi_inst * inst, void * userdata, uint8_t event) {
 (void)inst;
 return roar_slfi_event_add(userdata, event);
}

static int __init(struct roar_slfi_inst * inst, const struct roar_keyval * para, ssize_t paralen) {
 struct slfi_chain * self = roar_mm_malloc(sizeof(struct slfi_chain));

 (void)para, (void)paralen;

 if ( self == NULL )
  return -1;

 self->filter_len = 0;
 inst->userdata = self;

 return 0;
}

static int __uninit(struct roar_slfi_inst * inst) {
 struct slfi_chain * self = inst->userdata;
 size_t i;

 for (i = 0; i < self->filter_len; i++) {
  roar_slfi_unref(self->filter[i]);
 }

 return 0;
}

static int __update(struct roar_slfi_inst * inst, uint8_t * universe, ssize_t size_of_universe, int32_t usecspassed, const uint8_t * event, size_t eventlen) {
 struct slfi_chain * self = inst->userdata;
 size_t i;
 int ret;

 for (i = 0; i < self->filter_len; i++) {
  ret = roar_slfi_update(self->filter[i], universe, size_of_universe, usecspassed, event, eventlen);
  if ( ret != 0 )
   return ret;
 }

 return 0;
}

static int __ctl(struct roar_slfi_inst * inst, enum roar_slfi_command command, void * argp) {
 struct slfi_chain * self = inst->userdata;

 switch (command) {
  case ROAR_SLFI_CMD_PUSH:
    if ( self->filter_len == MAX_FILTER ) {
     roar_err_set(ROAR_ERROR_NOSPC);
     return -1;
    }
    if ( roar_slfi_ref(argp) == -1 )
     return -1;
    roar_slfi_cb_set_event_add(argp, __cb_event_add, inst);
    self->filter[self->filter_len++] = argp;
    return 0;
   break;
#ifndef DEBUG
  default:
   break;
#endif
 }

 roar_err_set(ROAR_ERROR_BADRQC);
 return -1;
}

static const struct roar_slfi_filter filter[1] = {
 {
  .name = "chain",
  .description = "Chain SLFI filter",
  .flags = ROAR_SLFI_FLAG_NONE,
  .init = __init,
  .uninit = __uninit,
  .update = __update,
  .ctl = __ctl
 }
};

ROAR_DL_PLUGIN_REG_SLFI(filter);

// This is the plugin control block.
ROAR_DL_PLUGIN_START(filter_slfi_chain) {
 // Here we set the name and vendor of our plugin.
 // If you have no Vendor ID you need to use ROAR_DL_PLUGIN_META_PRODUCT_NV().
 ROAR_DL_PLUGIN_META_PRODUCT_NIV("filter-slfi-chain", ROAR_VID_ROARAUDIO, ROAR_VNAME_ROARAUDIO);

 // This sets the version of your plugin.
 ROAR_DL_PLUGIN_META_VERSION(ROAR_VERSION_STRING);

 // This sets the license of your plugin.
 // If there is no tag for the license you use you can just
 // use ROAR_DL_PLUGIN_META_LICENSE().
 ROAR_DL_PLUGIN_META_LICENSE_TAG(GPLv3_0);

 // This sets the author and contact infos.
 // There are several other macros to do this with other parameters.
 // See ROAR_DL_PLUGIN_META_CONTACT*() in the header or documentation.
 ROAR_DL_PLUGIN_META_CONTACT_FLNE("Philipp", "Schafft", "ph3-der-loewe", "lion@lion.leolix.org");

 // This sets the description for your plugin.
 ROAR_DL_PLUGIN_META_DESC("This is a container plugin for SLFI filter chains.");

 // Load filters.
 ROAR_DL_PLUGIN_REG_FNFUNC(ROAR_DL_FN_FILTER);

// This is the end of the control block.
} ROAR_DL_PLUGIN_END

//ll
