/*
    This file is part of Rocs.
    Copyright 2010-2011  Tomaz Canabrava <tomaz.canabrava@gmail.com>
    Copyright 2010-2012  Wagner Reck <wagner.reck@gmail.com>
    Copyright 2013       Andreas Cord-Landwehr <cola@uni-paderborn.de>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef TOOLSPLUGININTERFACE_H
#define TOOLSPLUGININTERFACE_H

#include "VisualEditorExport.h"
#include "CoreTypes.h"

#include <KPluginInfo>
#include <KXMLGUIClient>

#include <QtCore/QObject>

class Document;
class ToolsPluginInterfacePrivate;

/** \class ToolsPluginInterface
 * This interface must be implemented by all Visual graph editor tools. To create a plugin, it is
 * necessary to provide a desktop file that contains the following information:
 *
\verbatim
[Desktop Entry]
Encoding=UTF-8
Type=Service
#Icon=icon
ServiceTypes=Rocs/ToolPlugin
X-KDE-Library=rocs_mytool                                   # Name of tool library
X-KDE-PluginInfo-Author=Rocs Developers
X-KDE-PluginInfo-Email=rocs-devel@kde.org
X-KDE-PluginInfo-Name=Transform Edges Rocs Tool Plugin
X-KDE-PluginInfo-Version=0.0.1
X-KDE-PluginInfo-Website=http://edu.kde.org/rocs/
X-KDE-PluginInfo-Category=Plugins
X-KDE-PluginInfo-License=GPL
X-KDE-PluginInfo-EnabledByDefault=false
X-Rocs-SupportedDataStructures=Graph,LinkedList             # Comma separated list of supprted backends
Name=My Tool                                                # Tool name for user visible strings
Comment=This tool makes cool things, i.e. ...               # Tool description.
\endverbatim
 *
 * Further, the plugin itself is generated by the following factory method:
 *
\code
    typedef KGenericFactory<MyTool> ToolsPluginFactory;
    static const KAboutData aboutdata("rocs_mytool", 0, ki18n("MyTool") , "0.0.1");
    K_EXPORT_COMPONENT_FACTORY( rocs_mytool, ToolsPluginFactory( &aboutdata )  )

    MyTool::MyTool(QObject *parent, const char *name, const QStringList &args)
        : ToolsPluginInterface(ToolsPluginFactory::componentData(), parent)
    {
        //...
    }
\endcode
 */
class VISUALEDITOR_EXPORT ToolsPluginInterface
:   public QObject, public KXMLGUIClient
{
    Q_OBJECT

public:
    ToolsPluginInterface(const KComponentData &instance, QObject *parent);
    virtual ~ToolsPluginInterface();

    /**
     * Return the KPluginInfo object associated with this plugin
     */
    KPluginInfo pluginInfo();

    /**
     * Returns the display name of this plugin.
     *
     * This is a convenience method that simply calls \ref pluginInfo->name().
     */
    QString displayName();

    /**
     * Return the plugin identifier.
     *
     * \return the plugin's id, which is \ref QObject::metaObject()->className()
     */
    QString pluginId() const;

    /**
     * Gives list of supported data structures of the tool. Use this method to check whether the
     * tool plugin is compatible with the given data structure backend. The list of supported
     * backends consists of data structure backend identifiers as given by
     * \ref DataStructureBackendInterface::internalName().
     *
     * \return list of supported data structure backend identifiers
     */
    QStringList supportedDataStructures();

    /**
     * Implement this method for each plugin. The implementation shall execute the tool at the given
     * document \p document or, if not given, at the active document as given by the
     * \ref DocumentManager.
     *
     * \param document is the graph document at that the tool shall operate
     */
    virtual void run(Document *document = 0) const = 0;

private:
    ToolsPluginInterfacePrivate * const d;
};

#endif
