// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "nrrd.h"
#include "head.h"

FitsNRRD::FitsNRRD(FitsFile* fits)
{
  bitpix_ = fits->pBitpix();
  width_ = fits->pWidth();
  height_ = fits->pHeight();
  depth_ = fits->pDepth();
  size_ = (size_t)width_*height_*depth_;
}

FitsNRRD::~FitsNRRD()
{
  if (data_)
    delete [] (char*)data_;
}

int FitsNRRD::initHeader(FitsFile* fits) 
{
  // simple check
  if (!width_ || !height_ || !bitpix_)
    return 0;

  // create header
  head_ = new FitsHead(width_, height_, depth_, bitpix_);
  if (!head_->isValid())
    return 0;

  // other
  primary_ = fits->primary();
  managePrimary_ = 0;

  inherit_ = head_->inherit();

  return 1;
}

template<class T> FitsNRRDm<T>::FitsNRRDm(FitsFile* fits) 
  : FitsNRRD(fits) {}

template <class T> void FitsNRRDm<T>::uncompress(FitsFile* fits)
{
  if (!initHeader(fits))
    return;

  T* dest = new T[size_];
  if (!dest) {
    internalError("Fitsy++ nrrd unable to allocate memory");
    return;
  }
  memset(dest, 0, size_*sizeof(T));
  compressed(dest, (char*)fits->data(), fits->dataSize()-fits->dataSkip());

  data_ = dest;

  dataSize_ = size_;
  dataSkip_ = 0;

  swapBytes(fits->pArch());

  // all done
  valid_ = 1;
}

template<class T> void FitsNRRDm<T>::swapBytes(FitsFile::ArchType arch) 
{
  switch (arch) {
  case FitsFile::NATIVE:
    break;
  case FitsFile::BIG:
    if (!byteswap_) {
      T* dest = (T*)data_;
      for (int i=0; i<size_; i++)
	dest[i] = swap(dest+i);
    }
    break;
  case FitsFile::LITTLE:
    if (byteswap_) {
      T* dest = (T*)data_;
      for (int i=0; i<size_; i++)
	dest[i] = swap(dest+i);
    }
    break;
  }
}

template <> unsigned char FitsNRRDm<unsigned char>::swap(unsigned char* ptr)
{
  return *ptr;
}

template <> short FitsNRRDm<short>::swap(short* ptr)
{
  const char* p = (const char*)ptr;
  union {
    char c[2];
    short s;
  } u;

  u.c[1] = *p++;
  u.c[0] = *p;

  return u.s;
}

template <> unsigned short FitsNRRDm<unsigned short>::swap(unsigned short* ptr)
{
  const char* p = (const char*)ptr;
  union {
    char c[2];
    unsigned short s;
  } u;

  u.c[1] = *p++;
  u.c[0] = *p;

  return u.s;
}

template <> int FitsNRRDm<int>::swap(int* ptr)
{
  const char* p = (const char*)ptr;
  union {
    char c[4];
    int i;
  } u;

  u.c[3] = *p++;
  u.c[2] = *p++;
  u.c[1] = *p++;
  u.c[0] = *p;

  return u.i;
}

template <> long long FitsNRRDm<long long>::swap(long long* ptr)
{
  const char* p = (const char*)ptr;
  union {
    char c[8];
    long long i;
  } u;

  u.c[7] = *p++;
  u.c[6] = *p++;
  u.c[5] = *p++;
  u.c[4] = *p++;
  u.c[3] = *p++;
  u.c[2] = *p++;
  u.c[1] = *p++;
  u.c[0] = *p;

  return u.i;
}

template <> float FitsNRRDm<float>::swap(float* ptr)
{
  const char* p = (const char*)ptr;
  union {
    char c[4];
    float f;
  } u;

  u.c[3] = *p++;
  u.c[2] = *p++;
  u.c[1] = *p++;
  u.c[0] = *p;

  return u.f;
}

template <> double FitsNRRDm<double>::swap(double* ptr)
{
  const char* p = (const char*)ptr;
  union {
    char c[8];
    double d;
  } u;

  u.c[7] = *p++;
  u.c[6] = *p++;
  u.c[5] = *p++;
  u.c[4] = *p++;
  u.c[3] = *p++;
  u.c[2] = *p++;
  u.c[1] = *p++;
  u.c[0] = *p;

  return u.d;
}

template class FitsNRRDm<unsigned char>;
template class FitsNRRDm<short>;
template class FitsNRRDm<unsigned short>;
template class FitsNRRDm<int>;
template class FitsNRRDm<long long>;
template class FitsNRRDm<float>;
template class FitsNRRDm<double>;

FitsNRRDNext::FitsNRRDNext(FitsFile* p)
{
  FitsNRRD* prev = (FitsNRRD*)p;

  primary_ = prev->primary();
  managePrimary_ = 0;

  head_ = prev->head();
  manageHead_ = 0;

  FitsImageHDU* hdu = (FitsImageHDU*)head_->hdu();
  data_ = (char*)prev->data() + hdu->imgbytes();
  dataSize_ = 0;
  dataSkip_ = 0;

  ext_ = prev->ext();
  inherit_ = prev->inherit();
  byteswap_ = prev->byteswap();
  orgFits_ = prev->orgFits();
  valid_ = 1;

  pWidth_ = prev->pWidth();
  pHeight_ = prev->pHeight();
  pDepth_ = prev->pDepth();
  pBitpix_ = prev->pBitpix();
  pSkip_ = prev->pSkip();
  pArch_ = prev->pArch();

  pNRRDEncoding_ = prev->pNRRDEncoding();
  pNRRDDimension_ = prev->pNRRDDimension();

  return;
}
