// Copyright (C) 2009 Martin Sandve Alnes
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.
//
// First added:  2009-05-05
// Last changed: 2009-05-05
//
// This program assembles linear systems for different definitions
// of forms that are supposed to be equal, for comparing and debugging.

#include <iostream>
#include <dolfin.h>
#include "generated_code/HyperElasticity.h"

using dolfin::Function;
using dolfin::FunctionSpace;
using dolfin::uint;
using dolfin::UnitCube;
using dolfin::UnitSquare;
using dolfin::VariationalProblem;
using dolfin::Vector;
using dolfin::Matrix;
using dolfin::File;

using namespace HyperElasticity;

using std::cout;
using std::endl;

// ---------------------------------------- Functions

class Value: public Function
{
public:
  Value(const FunctionSpace & V, double value):
    Function(V), value(value)
  {}

  void eval(double* values, const double* x) const
  {
    values[0] = value;
  }

  double value;
};

class Fibers: public Function
{
public:
  int d;

  Fibers(const FunctionSpace & V):
    Function(V), d(V.mesh().geometry().dim())
  {}

  void eval(double* values, const double* x) const
  {
    if(d == 1)
    {
      values[0] = 1;
    }
    if(d == 2)
    {
      values[0] = 1;
      values[1] = 0;
      values[2] = 0;
      values[3] = 1;
    }
    if(d == 3)
    {
      values[0] = 1;
      values[1] = 0;
      values[2] = 0;
      values[3] = 0;
      values[4] = 1;
      values[5] = 0;
      values[6] = 0;
      values[7] = 0;
      values[8] = 1;
    }
  }

  double value;
};

double norm(Matrix & A)
{
    double sum = 0.0;
    std::vector<unsigned> columns;
    std::vector<double> values;

    for(unsigned i=0; i<A.size(0); i++)
    {
        double tmp = 0.0;
        A.getrow(i, columns, values);
        for(unsigned j=0; j<values.size(); j++)
            tmp += values[j]*values[j];
        if(tmp > 1e-12)
            sum += tmp;
    }
    if(sum < 1e-20)
        sum = 0.0;
    return sum;
}

// ---------------------------------------- Main program

int main(int argc, char**argv)
{
    // Geometry
    unsigned n = 1;
    UnitCube mesh(n, n, n);
    //UnitSquare mesh(n, n);

    // Function spaces
    Form_a_F::TestSpace V(mesh);
//  CoefficientSpace_c1_nh C(mesh);
//  CoefficientSpace_mu C(mesh);
    CoefficientSpace_K C(mesh);
    CoefficientSpace_A Q(mesh);
    
    // Material parameters
    Value c1_nh(C, 1.0);
    Value c1(C, 1.0);
    Value c2(C, 1.0);
    Value mu(C, 1.0);
    Value lamda(C, 1.0);
    Value K(C, 1.0);
    Value C_compr(C, 1.0);
    Value bff(C, 1.0);
    Value bfx(C, 1.0);
    Value bxx(C, 1.0);
    Fibers A(Q);
    
    // Displacement at current and two previous timesteps
    Function u(V);
    u.vector().zero();

    // Collect coefficients
    CoefficientSet coeffs;
    coeffs.u       = u;
  //coeffs.c1_nh   = c1_nh;
  //coeffs.c1      = c1;
  //coeffs.c2      = c2;
  //coeffs.lamda   = lamda;
  //coeffs.mu      = mu;
    coeffs.K       = K;
    coeffs.C_compr = C_compr;
    coeffs.bff     = bff;
    coeffs.bfx     = bfx;
    coeffs.bxx     = bxx;
    coeffs.A       = A;

    // a version
    Form_a_F L_a(V, coeffs);
    Form_a_J a_a(V, V, coeffs);

    VariationalProblem problem_a(a_a, L_a, true);
    
    Matrix J_a;
    problem_a.J(J_a, u.vector());
    //J_a.disp();

    Vector F_a;
    problem_a.F(F_a, u.vector());
    //F_a.disp();

    // b version
    Form_b_F L_b(V, coeffs);
    Form_b_J a_b(V, V, coeffs);

    VariationalProblem problem_b(a_b, L_b, true);
    
    Matrix J_b;
    problem_b.J(J_b, u.vector());
    //J_b.disp();

    Vector F_b;
    problem_b.F(F_b, u.vector());
    //F_b.disp();

    // c version
    Form_c_F L_c(V, coeffs);
    Form_c_J a_c(V, V, coeffs);

    VariationalProblem problem_c(a_c, L_c, true);
    
    Matrix J_c;
    problem_c.J(J_c, u.vector());
    //J_c.disp();

    Vector F_c;
    problem_c.F(F_c, u.vector());
    //F_c.disp();

    // dump to files
    File F_a_file("F_a.m");
    F_a_file << F_a;
    File J_a_file("J_a.m");
    J_a_file << J_a;
    File F_b_file("F_b.m");
    F_b_file << F_b;
    File J_b_file("J_b.m");
    J_b_file << J_b;
    File F_c_file("F_c.m");
    F_c_file << F_c;
    File J_c_file("J_c.m");
    J_c_file << J_c;

    // vector norms
    cout << "F a = " << F_a.norm() << endl;
    cout << "F b = " << F_b.norm() << endl;
    cout << "F c = " << F_c.norm() << endl;
    // matrix norms
    cout << "J a = " << norm(J_a) << endl;
    cout << "J b = " << norm(J_b) << endl;
    cout << "J c = " << norm(J_c) << endl;

    // vector diffs
    Vector F_diff;

    F_diff = F_a;
    F_diff -= F_b;
    cout << "F a - b = " << F_diff.norm() << endl;

    F_diff = F_a;
    F_diff -= F_c;
    cout << "F a - c = " << F_diff.norm() << endl;

    F_diff = F_b;
    F_diff -= F_c;
    cout << "F b - c = " << F_diff.norm() << endl;
    
    // matrix diffs
    Matrix J_diff;

    J_diff = J_a;
    J_diff -= J_b;
    cout << "J a - b = " << norm(J_diff) << endl;
    //J_diff.disp();

    J_diff = J_a;
    J_diff -= J_c;
    cout << "J a - c = " << norm(J_diff) << endl;
    //J_diff.disp();

    J_diff = J_b;
    J_diff -= J_c;
    cout << "J b - c = " << norm(J_diff) << endl;
    //J_diff.disp();
}

