//-----------------------------------------------------------------------------
//  Copyright (C) 2017 Thomas S. Ullrich
//
//  This file is part of "xyscan".
//
//  This file may be used under the terms of the GNU General Public License.
//  This project is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License.
//
//  Author: Thomas S. Ullrich
//  Last update: December 5, 2017
//-----------------------------------------------------------------------------
#include <QHeaderView>
#include <QPixmap>
#include <QDir>
#include <QTextCursor>
#include <QTextFrame>
#include <QTextTable>
#include <QTextStream>
#include <QFile>
#include <ctime>
#include <cfloat>
#include "xyscanDataTable.h"
#include "xyscanVersion.h"

using namespace std;

xyscanDataTable::xyscanDataTable(QWidget *parent) : QTableWidget(parent)
{
    //
    //   Some general setup
    //
    mDataSaved = true;
    setAutoScroll(true);
    setTextElideMode(Qt::ElideMiddle);
    setShowGrid(true);
    setSortingEnabled(false); // no sorting, done manually
    setEditTriggers(QAbstractItemView::NoEditTriggers);

    //
    //   Setup columns
    //
    setColumnCount(6);
    QTableWidgetItem *colItem = new QTableWidgetItem();
    colItem->setText(tr("x"));
    setHorizontalHeaderItem(0, colItem);
    QTableWidgetItem *colItem1 = new QTableWidgetItem();
    colItem1->setText(tr("y"));
    setHorizontalHeaderItem(1, colItem1);
    QTableWidgetItem *colItem2 = new QTableWidgetItem();
    colItem2->setText(tr("-dx"));
    setHorizontalHeaderItem(2, colItem2);
    QTableWidgetItem *colItem3 = new QTableWidgetItem();
    colItem3->setText(tr("+dx"));
    setHorizontalHeaderItem(3, colItem3);
    QTableWidgetItem *colItem4 = new QTableWidgetItem();
    colItem4->setText(tr("-dy"));
    setHorizontalHeaderItem(4, colItem4);
    QTableWidgetItem *colItem5 = new QTableWidgetItem();
    colItem5->setText(tr("+dy"));
    setHorizontalHeaderItem(5, colItem5);
    
    horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
    verticalHeader()->setSectionResizeMode(QHeaderView::ResizeToContents);
}

void xyscanDataTable::addPoint(double x, double y, double dxl, double dxu, double dyl, double dyu)
{
    //
    //  Store data into new table row
    //
    QString str;
    int nrows = rowCount();
    insertRow(nrows);
    str.setNum(x);
    setItem(nrows, 0, new QTableWidgetItem(str));
    item(nrows, 0)->setTextAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
    str.setNum(y);
    setItem(nrows, 1, new QTableWidgetItem(str));
    item(nrows, 1)->setTextAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
    str.setNum(dxl);
    setItem(nrows, 2, new QTableWidgetItem(str));
    item(nrows, 2)->setTextAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
    str.setNum(dxu);
    setItem(nrows, 3, new QTableWidgetItem(str));
    item(nrows, 3)->setTextAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
    str.setNum(dyl);
    setItem(nrows, 4, new QTableWidgetItem(str));
    item(nrows, 4)->setTextAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
    str.setNum(dyu);
    setItem(nrows, 5, new QTableWidgetItem(str));
    item(nrows, 5)->setTextAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
    scrollToBottom();
    mDataSaved = false;
}

void xyscanDataTable::deleteAll()
{
    //
    //  Delete all rows in table
    //
    clearContents();
    setRowCount(0);
    
    mDataSaved = true; 
}

void xyscanDataTable::deleteLast()
{
    //
    //  Delete last row in table
    //
    int nrows = rowCount();
    if (nrows) {
        removeRow(nrows);
        setRowCount(nrows-1);
    }
}

void xyscanDataTable::writePrinterDocument(QTextDocument& document, const QString& src, const QImage *image)
{
    //
    //   Write table data into the given QTextDocument
    //
    document.setUseDesignMetrics(true);
    QTextCursor cursor(&document);
    
    QTextCharFormat textFormat;
    QFont fontVar("Helvetica",10);
    textFormat.setFont(fontVar);
    
    QFont fontFix("Courier",9);
    QTextCharFormat boldTableTextFormat;
    boldTableTextFormat.setFont(fontFix);
    boldTableTextFormat.setFontWeight(QFont::Bold);
    QTextCharFormat tableTextFormat;
    tableTextFormat.setFont(fontFix);
    
    cursor.movePosition(QTextCursor::Start);
    QTextFrame *topFrame = cursor.currentFrame();

    //
    //  Header
    //
    time_t now = time(0);
    cursor.insertText(tr("xyscan Version %1\n").arg(VERSION), textFormat);
    cursor.insertText(tr("Date: %1").arg(ctime(&now)), textFormat);
    cursor.insertText(tr("Scanned by: %1\n").arg(QDir::home().dirName()), textFormat);
    cursor.insertText(tr("Source: %1\n").arg(src), textFormat);
    cursor.insertText(tr("Comment: %1\n").arg(mUserComment), textFormat);
    cursor.insertBlock();
    cursor.insertBlock();

    //
    //  Insert Plot
    //  (don't scale image first - looks terrible even with Qt::SmoothTransformation)
    //
    if (image) {
        cursor.setPosition(topFrame->lastPosition());
        document.addResource(QTextDocument::ImageResource, QUrl("image"), QVariant(*image));
        int maxSize = 200; // width or height
        double scale = 1;
        if (image->size().width() > maxSize || image->size().height() > maxSize) {
            if (image->size().width() > image->size().height())
                scale = maxSize/static_cast<double>(image->size().width());
            else
                scale = maxSize/static_cast<double>(image->size().height());
        }
        
        QTextImageFormat imageFormat;
        imageFormat.setWidth(static_cast<int>(scale*image->size().width()));
        imageFormat.setHeight(static_cast<int>(scale*image->size().height()));
        imageFormat.setName("image");
        cursor.insertImage(imageFormat);
        cursor.insertBlock();
        cursor.insertBlock();
    }

    //
    //  Table
    //
    cursor.setPosition(topFrame->lastPosition());
    QTextTableFormat tableFormat;
    tableFormat.setBorder(1);
    tableFormat.setCellPadding(3);
    tableFormat.setCellSpacing(0);
    tableFormat.setAlignment(Qt::AlignLeft);
    tableFormat.setHeaderRowCount(1);
    QTextTable *table = cursor.insertTable (rowCount()+1,
                                            columnCount()+1, tableFormat);
    table->cellAt(0, 1).firstCursorPosition().insertText("x", boldTableTextFormat);
    table->cellAt(0, 2).firstCursorPosition().insertText("y", boldTableTextFormat);
    table->cellAt(0, 3).firstCursorPosition().insertText("-dx", boldTableTextFormat);
    table->cellAt(0, 4).firstCursorPosition().insertText("+dx", boldTableTextFormat);
    table->cellAt(0, 5).firstCursorPosition().insertText("-dy", boldTableTextFormat);
    table->cellAt(0, 6).firstCursorPosition().insertText("+dy", boldTableTextFormat);
    for (int irow = 0; irow < rowCount(); irow++) {
        table->cellAt(irow+1, 0).firstCursorPosition().insertText(tr("%1").arg(irow+1), tableTextFormat);
        for (int icol = 0; icol < columnCount(); icol++) {
            table->cellAt(irow+1, icol+1).firstCursorPosition().insertText(item(irow,icol)->text(), tableTextFormat);
        }
    }
}

void xyscanDataTable::writeRootMacro(QFile& file, const QString& src, bool logx, bool logy, double markerCoord[4])
{
    QTextStream out(&file);
    int irow;
    time_t now = time(0);

    out << "//" << endl;
    out << tr("// ROOT macro: ") << QFileInfo(file).fileName().toLatin1() << tr(" (autogenerated by xyscan)") << endl;
    out << tr("// xyscan Version ") << VERSION << endl;
    out << tr("// Date: ") << ctime(&now);
    out << tr("// Scanned by: ") << QDir::home().dirName() << endl;
    out << tr("// Source: ") << src << endl;
    out << tr("// Comment: ") << userComment() << endl;
    out << "//" << endl;
    
    out << "#include \"TROOT.h\"" << endl;
    out << "#include \"TH1D.h\"" << endl;
    out << "#include \"TCanvas.h\"" << endl;
    out << "#include \"TStyle.h\"" << endl;
    out << "#include \"TGraphAsymmErrors.h\"" << endl;
    out << endl;
    
    out << "void " << QFileInfo(file).baseName().toLatin1() << "()" << endl;
    out << "{" << endl;
    out << "    gROOT->SetStyle(\"Plain\");" << endl;
    out << "    gStyle->SetOptFit(0);" << endl;
    out << "    gStyle->SetOptStat(0);" << endl;
    out << endl;
    out << "    TCanvas *c1 = new TCanvas(\"c1\",\"xyscan Data Display\",720,540);" << endl;
    out << "    c1->SetTickx(1);" << endl;
    out << "    c1->SetTicky(1);" << endl;
    out << "    c1->SetBorderSize(1);" << endl;
    out << "    c1->SetFillColor(0);" << endl;
    out << endl;
    
    //
    //  Cannot rely on markers being set at the actual plot
    //  boundaries. Here we calculate a reasonable x and y
    //  range and then compare it to the marker boundaries
    //  using the 'wider' one.
    //
    double xmin = DBL_MAX;
    double xmax = DBL_MIN;
    double ymin = DBL_MAX;
    double ymax = DBL_MIN;
    for (irow=0; irow < rowCount(); irow++) {
        double xx = x(irow);
        double yy = y(irow);
        double dxxlow = dx_lower(irow);
        double dxxup = dx_upper(irow);
        double dyylow = dy_lower(irow);
        double dyyup = dy_upper(irow);
        if (xx-dxxlow < xmin) xmin = xx-dxxlow;
        if (xx+dxxup > xmax) xmax = xx+dxxup;
        if (yy-dyylow < ymin) ymin = yy-dyylow;
        if (yy+dyyup > ymax) ymax = yy+dyyup;
    }
    if (logy) {
        ymax *= 2;
        ymin /= 2;
    }
    else {
        ymax += (ymax-ymin)/10;
        ymin -= (ymax-ymin)/10;
        if (ymin > 0) ymin = 0;
    }
    if (logx) {
        xmax *= 2;
        xmin /= 2;
    }
    else {
        xmax += (xmax-xmin)/10;
        xmin -= (xmax-xmin)/10;
    }
    xmin = min(xmin, markerCoord[0]);
    xmax = max(xmax, markerCoord[1]);
    ymin = min(ymin, markerCoord[2]);
    ymax = max(ymax, markerCoord[3]);
    
    out << "    TH1D *histo = new TH1D(\"histo\",\"xyscan\", 100, " << xmin << ", " << xmax << ");" << endl;
    out << "    histo->SetMinimum(" << ymin << ");" << endl;
    out << "    histo->SetMaximum(" << ymax << ");" << endl;
    out << "    histo->SetStats(false);" << endl;
    out << "    histo->GetXaxis()->SetTitle(\"x\");" << endl;
    out << "    histo->GetYaxis()->SetTitle(\"y\");" << endl;
    out << "    gPad->SetLogy(" << (logy ? 1 : 0) << ");" << endl;
    out << "    gPad->SetLogx(" << (logx ? 1 : 0) << ");" << endl;
    
    out << "    histo->Draw();" << endl;
    out << endl;
    int nrc = rowCount();
    out << "    double x[" << nrc << "];" << endl;
    out << "    double y[" << nrc << "];" << endl;
    out << "    double dxlow[" << nrc << "];" << endl;
    out << "    double dxup[" << nrc << "];" << endl;
    out << "    double dylow[" << nrc << "];" << endl;
    out << "    double dyup[" << nrc << "];" << endl;
    out << "    int n = 0;" << endl;
    for (irow=0; irow < nrc; irow++) {
        out << "    x[n] = " << x(irow) << ";\t";
        out << "y[n] = " << y(irow) << ";\t";
        out << "dxlow[n] = " << dx_lower(irow) << ";\t";
        out << "dxup[n] = " << dx_upper(irow) << ";\t";
        out << "dylow[n] = " << dy_lower(irow) << ";\t";
        out << "dyup[n] = " << dy_upper(irow) << ";\t";
        out << "n++;" << endl;
    }
    out << endl;
    out << "    TGraphAsymmErrors *xyscan = new TGraphAsymmErrors(n, x, y, dxlow, dxup, dylow, dyup);" << endl;
    out << "    xyscan->SetMarkerStyle(20);" << endl;
    out << "    xyscan->SetMarkerColor(1);" << endl;
    out << "    xyscan->Draw(\"PE same\");" << endl;
    out << "}" << endl;

    mDataSaved = true;
}

void xyscanDataTable::writeTextFile(QFile& file, const QString& src)
{
    QTextStream out(&file);
    int irow, icol;
    time_t now = time(0);

    out << tr("# xyscan Version ") << VERSION << endl;
    out << tr("# Date: ") << ctime(&now);
    out << tr("# Scanned by: ") << QDir::home().dirName() << endl;
    out << tr("# Source: ") << src << endl;
    out << tr("# Comment: ") << userComment() << endl;
    out << tr("# Format: x y -dx +dx -dy +dy") << endl;
    for (irow=0; irow < rowCount(); irow++) {
        for (icol=0; icol < columnCount(); icol++) {
            out << item(irow, icol)->text() << "\t";
        }
        out << endl;
    }
    out << "# EoF" << endl;

    mDataSaved = true;
}
