/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "Camera.h"
#include "trackball.h"
#include <glm/gtx/rotate_vector.hpp>
#include <cstring>
#include <cmath>

#define AERIALVIEWLIMIT 89.9 * 14

namespace libgltf
{

const float PI = float(std::atan(1.0) * 4.0);

CPhysicalCamera::CPhysicalCamera()
    : vSpeed(1.0f)
    , fSetTime(0)
    , fTotalMoveTime(0)
    , bMove_camera_flag(false)
    , bAontrolCamera(false)
    , fSensitivity(0.1f)
    , mTrackBallY(1.0f)
    , mTrackBallX(1.0f)
    , AerialViewY(0)
    , flength(0.0)
    , vModelCenterPos(0.0f)
    , mViewMatrix(0.0f)
    , mProjection(1.0f)
    , mbAerialView(false)
{
    memset(curquatY, 0, sizeof(float) * 4);
    memset(curquatX, 0, sizeof(float) * 4);
}

void CPhysicalCamera::buildRotMatrix(glm::mat4& m, float q[4])
{
    m[0][0] = (float)(1.0 - 2.0 * (q[1] * q[1] + q[2] * q[2]));
    m[0][1] = (float)(2.0 * (q[0] * q[1] - q[2] * q[3]));
    m[0][2] = (float)(2.0 * (q[2] * q[0] + q[1] * q[3]));
    m[0][3] = 0.0;

    m[1][0] = (float)(2.0 * (q[0] * q[1] + q[2] * q[3]));
    m[1][1] = (float)(1.0 - 2.0 * (q[2] * q[2] + q[0] * q[0]));
    m[1][2] = (float)(2.0 * (q[1] * q[2] - q[0] * q[3]));
    m[1][3] = 0.0;

    m[2][0] = (float)(2.0 * (q[2] * q[0] - q[1] * q[3]));
    m[2][1] = (float)(2.0 * (q[1] * q[2] + q[0] * q[3]));
    m[2][2] = (float)(1.0 - 2.0 * (q[1] * q[1] + q[0] * q[0]));
    m[2][3] = 0.0;

    m[3][0] = 0.0;
    m[3][1] = 0.0;
    m[3][2] = 0.0;
    m[3][3] = 1.0;
}

void CPhysicalCamera::moveCamera(double x, double y, double z,
                                 double time)
{
    if (std::abs(time) > 0.0001)
    {
        bMove_camera_flag = true;
        fSetTime = time;
        glm::vec3 endView(x, y, z);
        glm::vec3 currentView;
        getCameraPosVectors(0, &currentView, 0);
        glm::vec3 vRotate = endView - currentView;
        vRotate /= time ;
        vSpeed = vRotate;
    }
    else
    {
        if (std::abs(x) > 0.0001 || std::abs(y) > 0.0001 || std::abs(z) > 0.0001)
        {
            mViewMatrix = glm::translate(mViewMatrix, glm::vec3(-x, -y, -z));
        }
    }
}

void CPhysicalCamera::setViewMatrix(const glm::mat4& rViewMatrix)
{
    mViewMatrix = rViewMatrix;
}

const glm::mat4& CPhysicalCamera::getViewMatrix() const
{
    return mViewMatrix;
}

void CPhysicalCamera::setAerialView( bool bAerialView )
{
    mbAerialView = bAerialView;
}

void CPhysicalCamera::getCameraPosVectors(glm::vec3* pEye, glm::vec3* pView, glm::vec3* pUp) const
{
    // Calculate eye, view and up verctor from the view matrix
    glm::mat4 inverseViewMatrix = glm::inverse(mViewMatrix);
    if( pEye || pView )
    {
        glm::vec3 vEye;
        vEye.x = inverseViewMatrix[3][0];
        vEye.y = inverseViewMatrix[3][1];
        vEye.z = inverseViewMatrix[3][2];

        if( pEye )
            *pEye = vEye;

        if( pView )
        {
            if( mbAerialView )
            {
                *pView = vModelCenterPos;
            }
            else
            {
                *pView = vEye + glm::vec3(-inverseViewMatrix[2][0],-inverseViewMatrix[2][1],-inverseViewMatrix[2][2]);
            }
        }
    }

    if( pUp )
    {
        pUp->x = inverseViewMatrix[1][0];
        pUp->y = inverseViewMatrix[1][1];
        pUp->z = inverseViewMatrix[1][2];
        *pUp = glm::normalize(*pUp);
    }
}

void CPhysicalCamera::setPerspective(glm::mat4 Perspective)
{
    mProjection = Perspective;
}

const glm::mat4 CPhysicalCamera::getPerspective()
{
    return mProjection;
}

glm::mat4 CPhysicalCamera::getModelViewMatrix(glm::mat4 lookat,
                                              glm::mat4& matrix,
                                              float fMoveTime,
                                              float fPreTime)
{
    glm::mat4 modelViewMatrix;
    if (fPreTime <= 0)
    {
        modelViewMatrix = lookat * matrix;
    }
    else
    {
        fTotalMoveTime += fMoveTime;
        if (fTotalMoveTime <= fSetTime)
        {
            if (bMove_camera_flag == true)
            {
                mViewMatrix = glm::translate(mViewMatrix, vSpeed);
            }
            modelViewMatrix = lookat * matrix;
        }
        else
        {
            modelViewMatrix = lookat * matrix;
        }
    }
    return modelViewMatrix;
}

void CPhysicalCamera::rotateObjectMouse(double horizontal, double vertical,
                                        double planar)
{
    if (std::abs(horizontal) > planar || std::abs(vertical) > 0.0001)
    {
        float deltaX = (float)(horizontal * fSensitivity * 0.01f);
        float deltaY = (float)(-vertical * fSensitivity * 0.01f);
        AerialViewY += -vertical;
        if (AerialViewY >= -AERIALVIEWLIMIT && AerialViewY <= AERIALVIEWLIMIT)
        {
            float quat[4];
            trackball(quat, 0.0, deltaY, 0.0, 0.0);
            add_quats(quat, curquatY, curquatY);
        }
        else
        {
            AerialViewY -= -vertical;
        }
        float quat[4];
        trackball(quat, deltaX, 0.0, 0.0, 0.0);
        add_quats(quat, curquatX, curquatX);

        mViewMatrix *= glm::inverse(mTrackBallX);
        mViewMatrix *= glm::inverse(mTrackBallY);

        buildRotMatrix(mTrackBallY, curquatY);
        mViewMatrix *= mTrackBallY;

        buildRotMatrix(mTrackBallX, curquatX);
        mViewMatrix *= mTrackBallX;
    }
}

void CPhysicalCamera::rotateCamera(double horizontal, double vertical,
                                   double planar)
{
    if (std::abs(horizontal) > planar || std::abs(vertical) > 0.0001)
    {
        float deltaX = (float)(horizontal * fSensitivity * 0.01f);
        float deltaY = (float)(-vertical * fSensitivity * 0.01f);

        float quat[4];
        glm::mat4 mRotation;
        trackball(quat, deltaX, deltaY, 0.0, 0.0);
        buildRotMatrix(mRotation, quat);
        mViewMatrix = mRotation * mViewMatrix;
    }
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
