// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2022 MediaTek Inc. All rights reserved.
 *
 * Author: Weijie Gao <weijie.gao@mediatek.com>
 */

#include <dm.h>
#include <dm/pinctrl.h>
#include <dm/device_compat.h>
#include <linux/bitops.h>
#include <linux/io.h>

#include "pinctrl-mtmips-common.h"

#define SYSC_MAP_SIZE			0x100

#define PAD_UART1_GPIO0_OFS		0x00
#define PAD_UART3_I2C_OFS		0x04
#define PAD_UART2_JTAG_OFS		0x08
#define PAD_PERST_WDT_OFS		0x0c
#define PAD_RGMII2_MDIO_OFS		0x10
#define PAD_SDXC_SPI_OFS		0x14
#define GPIOMODE_OFS			0x18
#define PAD_BOPT_ESWINT_OFS		0x28

#define ESWINT_SHIFT			20
#define SDXC_SHIFT			18
#define SPI_SHIFT			16
#define RGMII2_SHIFT			15
#define RGMII1_SHIFT			14
#define MDIO_SHIFT			12
#define PERST_SHIFT			10
#define WDT_SHIFT			8
#define JTAG_SHIFT			7
#define UART2_SHIFT			5
#define UART3_SHIFT			3
#define I2C_SHIFT			2
#define UART1_SHIFT			1
#define GPIO0_SHIFT			0 /* Dummy */

#define GM4_MASK			3

#define E4_E2_M				0x03
#define E4_E2_S				4
#define PULL_UP				BIT(3)
#define PULL_DOWN			BIT(2)
#define SMT				BIT(1)
#define SR				BIT(0)

struct mt7621_pinctrl_priv {
	struct mtmips_pinctrl_priv mp;
};

#if CONFIG_IS_ENABLED(PINMUX)
static const struct mtmips_pmx_func esw_int_grp[] = {
	FUNC("gpio", 1),
	FUNC("esw int", 0),
};

static const struct mtmips_pmx_func sdxc_grp[] = {
	FUNC("nand", 2),
	FUNC("gpio", 1),
	FUNC("sdxc", 0),
};

static const struct mtmips_pmx_func spi_grp[] = {
	FUNC("nand", 2),
	FUNC("gpio", 1),
	FUNC("spi", 0),
};

static const struct mtmips_pmx_func rgmii2_grp[] = {
	FUNC("gpio", 1),
	FUNC("rgmii", 0),
};

static const struct mtmips_pmx_func rgmii1_grp[] = {
	FUNC("gpio", 1),
	FUNC("rgmii", 0),
};

static const struct mtmips_pmx_func mdio_grp[] = {
	FUNC("gpio", 1),
	FUNC("mdio", 0),
};

static const struct mtmips_pmx_func perst_grp[] = {
	FUNC("refclk", 2),
	FUNC("gpio", 1),
	FUNC("pcie reset", 0),
};

static const struct mtmips_pmx_func wdt_grp[] = {
	FUNC("refclk", 2),
	FUNC("gpio", 1),
	FUNC("wdt rst", 0),
};

static const struct mtmips_pmx_func jtag_grp[] = {
	FUNC("gpio", 1),
	FUNC("jtag", 0),
};

static const struct mtmips_pmx_func uart2_grp[] = {
	FUNC("spdif", 3),
	FUNC("pcm", 2),
	FUNC("gpio", 1),
	FUNC("uart", 0),
};

static const struct mtmips_pmx_func uart3_grp[] = {
	FUNC("spdif", 3),
	FUNC("i2s", 2),
	FUNC("gpio", 1),
	FUNC("uart", 0),
};

static const struct mtmips_pmx_func i2c_grp[] = {
	FUNC("gpio", 1),
	FUNC("i2c", 0),
};

static const struct mtmips_pmx_func uart1_grp[] = {
	FUNC("gpio", 1),
	FUNC("uart", 0),
};

static const struct mtmips_pmx_func gpio0_grp[] = {
	FUNC("gpio", 0),
};

static const struct mtmips_pmx_group mt7621_pmx_data[] = {
	GRP_PCONF("esw int", esw_int_grp, GPIOMODE_OFS, ESWINT_SHIFT, 1,
		  PAD_BOPT_ESWINT_OFS, 0),
	GRP_PCONF("sdxc", sdxc_grp, GPIOMODE_OFS, SDXC_SHIFT, GM4_MASK,
		  PAD_SDXC_SPI_OFS, 16),
	GRP_PCONF("spi", spi_grp, GPIOMODE_OFS, SPI_SHIFT, GM4_MASK,
		  PAD_SDXC_SPI_OFS, 0),
	GRP_PCONF("rgmii2", rgmii2_grp, GPIOMODE_OFS, RGMII2_SHIFT, 1,
		  PAD_RGMII2_MDIO_OFS, 16),
	GRP("rgmii1", rgmii1_grp, GPIOMODE_OFS, RGMII1_SHIFT, 1),
	GRP_PCONF("mdio", mdio_grp, GPIOMODE_OFS, MDIO_SHIFT, GM4_MASK,
		  PAD_RGMII2_MDIO_OFS, 0),
	GRP_PCONF("pcie reset", perst_grp, GPIOMODE_OFS, PERST_SHIFT, GM4_MASK,
		  PAD_PERST_WDT_OFS, 16),
	GRP_PCONF("wdt", wdt_grp, GPIOMODE_OFS, WDT_SHIFT, GM4_MASK,
		  PAD_PERST_WDT_OFS, 0),
	GRP_PCONF("jtag", jtag_grp, GPIOMODE_OFS, JTAG_SHIFT, 1,
		  PAD_UART2_JTAG_OFS, 16),
	GRP_PCONF("uart2", uart2_grp, GPIOMODE_OFS, UART2_SHIFT, GM4_MASK,
		  PAD_UART2_JTAG_OFS, 0),
	GRP_PCONF("uart3", uart3_grp, GPIOMODE_OFS, UART3_SHIFT, GM4_MASK,
		  PAD_UART3_I2C_OFS, 16),
	GRP_PCONF("i2c", i2c_grp, GPIOMODE_OFS, I2C_SHIFT, 1,
		  PAD_UART3_I2C_OFS, 0),
	GRP_PCONF("uart1", uart1_grp, GPIOMODE_OFS, UART1_SHIFT, 1,
		  PAD_UART1_GPIO0_OFS, 16),
	GRP_PCONF("gpio0", gpio0_grp, GPIOMODE_OFS, GPIO0_SHIFT, 1,
		  PAD_UART1_GPIO0_OFS, 0),
};

static int mt7621_get_groups_count(struct udevice *dev)
{
	return ARRAY_SIZE(mt7621_pmx_data);
}

static const char *mt7621_get_group_name(struct udevice *dev,
					 unsigned int selector)
{
	return mt7621_pmx_data[selector].name;
}
#endif /* CONFIG_IS_ENABLED(PINMUX) */

#if CONFIG_IS_ENABLED(PINCONF)
static const struct pinconf_param mt7621_conf_params[] = {
	{ "bias-disable", PIN_CONFIG_BIAS_DISABLE, 0 },
	{ "bias-pull-up", PIN_CONFIG_BIAS_PULL_UP, 1 },
	{ "bias-pull-down", PIN_CONFIG_BIAS_PULL_DOWN, 1 },
	{ "input-schmitt-enable", PIN_CONFIG_INPUT_SCHMITT_ENABLE, 1 },
	{ "input-schmitt-disable", PIN_CONFIG_INPUT_SCHMITT_ENABLE, 0 },
	{ "drive-strength", PIN_CONFIG_DRIVE_STRENGTH, 0 },
	{ "slew-rate", PIN_CONFIG_SLEW_RATE, 0 },
};

static const u32 mt7621_pconf_drv_strength_tbl[] = {2, 4, 6, 8};

static int mt7621_pinconf_group_set(struct udevice *dev,
				    unsigned int group_selector,
				    unsigned int param, unsigned int arg)
{
	struct mt7621_pinctrl_priv *priv = dev_get_priv(dev);
	const struct mtmips_pmx_group *grp = &mt7621_pmx_data[group_selector];
	u32 clr = 0, set = 0;
	int i;

	if (!grp->pconf_avail)
		return 0;

	switch (param) {
	case PIN_CONFIG_BIAS_DISABLE:
		clr = PULL_UP | PULL_DOWN;
		break;

	case PIN_CONFIG_BIAS_PULL_UP:
		clr = PULL_DOWN;
		set = PULL_UP;
		break;

	case PIN_CONFIG_BIAS_PULL_DOWN:
		clr = PULL_UP;
		set = PULL_DOWN;
		break;

	case PIN_CONFIG_INPUT_SCHMITT_ENABLE:
		if (arg)
			set = SMT;
		else
			clr = SMT;
		break;

	case PIN_CONFIG_DRIVE_STRENGTH:
		for (i = 0; i < ARRAY_SIZE(mt7621_pconf_drv_strength_tbl); i++)
			if (mt7621_pconf_drv_strength_tbl[i] == arg)
				break;

		if (i >= ARRAY_SIZE(mt7621_pconf_drv_strength_tbl))
			return -EINVAL;

		clr = E4_E2_M << E4_E2_S;
		set = i << E4_E2_S;
		break;

	case PIN_CONFIG_SLEW_RATE:
		if (arg)
			set = SR;
		else
			clr = SR;
		break;

	default:
		return -EINVAL;
	}

	mtmips_pinctrl_reg_set(&priv->mp, grp->pconf_reg, grp->pconf_shift,
			       clr, set);

	return 0;
}
#endif

static int mt7621_pinctrl_probe(struct udevice *dev)
{
	struct mt7621_pinctrl_priv *priv = dev_get_priv(dev);
	int ret = 0;

#if CONFIG_IS_ENABLED(PINMUX)
	ret = mtmips_pinctrl_probe(&priv->mp, ARRAY_SIZE(mt7621_pmx_data),
				   mt7621_pmx_data);
#endif /* CONFIG_IS_ENABLED(PINMUX) */

	return ret;
}

static int mt7621_pinctrl_of_to_plat(struct udevice *dev)
{
	struct mt7621_pinctrl_priv *priv = dev_get_priv(dev);

	priv->mp.base = (void __iomem *)dev_remap_addr_index(dev, 0);

	if (!priv->mp.base)
		return -EINVAL;

	return 0;
}

static const struct pinctrl_ops mt7621_pinctrl_ops = {
#if CONFIG_IS_ENABLED(PINMUX)
	.get_groups_count = mt7621_get_groups_count,
	.get_group_name = mt7621_get_group_name,
	.get_functions_count = mtmips_get_functions_count,
	.get_function_name = mtmips_get_function_name,
	.pinmux_group_set = mtmips_pinmux_group_set,
#endif /* CONFIG_IS_ENABLED(PINMUX) */
#if CONFIG_IS_ENABLED(PINCONF)
	.pinconf_num_params = ARRAY_SIZE(mt7621_conf_params),
	.pinconf_params = mt7621_conf_params,
	.pinconf_group_set = mt7621_pinconf_group_set,
#endif /* CONFIG_IS_ENABLED(PINCONF) */
	.set_state = pinctrl_generic_set_state,
};

static const struct udevice_id mt7621_pinctrl_ids[] = {
	{ .compatible = "mediatek,mt7621-pinctrl" },
	{ }
};

U_BOOT_DRIVER(mt7621_pinctrl) = {
	.name = "mt7621-pinctrl",
	.id = UCLASS_PINCTRL,
	.of_match = mt7621_pinctrl_ids,
	.of_to_plat = mt7621_pinctrl_of_to_plat,
	.ops = &mt7621_pinctrl_ops,
	.probe = mt7621_pinctrl_probe,
	.priv_auto = sizeof(struct mt7621_pinctrl_priv),
};
