// Copyright 2012 The Closure Library Authors. All Rights Reserved.
// Use of this source code is governed by the Apache License, Version 2.0.

goog.provide('goog.vec.Vec2Test');
goog.setTestOnly('goog.vec.Vec2Test');

goog.require('goog.vec.Float32Array');
goog.require('goog.vec.Vec2');
goog.require('goog.testing.jsunit');

function testConstructor() {
  var v = goog.vec.Vec2.createFloat32();
  assertElementsEquals(0, v[0]);
  assertEquals(0, v[1]);

  assertElementsEquals([0, 0], goog.vec.Vec2.createFloat32());

  goog.vec.Vec2.setFromValues(v, 1, 2);
  assertElementsEquals([1, 2], v);

  var w = goog.vec.Vec2.createFloat64();
  assertElementsEquals(0, w[0]);
  assertEquals(0, w[1]);

  assertElementsEquals([0, 0], goog.vec.Vec2.createFloat64());

  goog.vec.Vec2.setFromValues(w, 1, 2);
  assertElementsEquals([1, 2], w);
}


function testSet() {
  var v = goog.vec.Vec2.createFloat32();
  goog.vec.Vec2.setFromValues(v, 1, 2);
  assertElementsEquals([1, 2], v);

  goog.vec.Vec2.setFromArray(v, [4, 5]);
  assertElementsEquals([4, 5], v);

  var w = goog.vec.Vec2.createFloat32();
  goog.vec.Vec2.setFromValues(w, 1, 2);
  assertElementsEquals([1, 2], w);

  goog.vec.Vec2.setFromArray(w, [4, 5]);
  assertElementsEquals([4, 5], w);
}

function testAdd() {
  var v0 = goog.vec.Vec2.createFloat32FromArray([1, 2]);
  var v1 = goog.vec.Vec2.createFloat32FromArray([4, 5]);
  var v2 = goog.vec.Vec2.cloneFloat32(v0);

  goog.vec.Vec2.add(v2, v1, v2);
  assertElementsEquals([1, 2], v0);
  assertElementsEquals([4, 5], v1);
  assertElementsEquals([5, 7], v2);

  goog.vec.Vec2.add(goog.vec.Vec2.add(v0, v1, v2), v0, v2);
  assertElementsEquals([6, 9], v2);
}

function testSubtract() {
  var v0 = goog.vec.Vec2.createFloat32FromArray([1, 2]);
  var v1 = goog.vec.Vec2.createFloat32FromArray([4, 5]);
  var v2 = goog.vec.Vec2.cloneFloat32(v0);

  goog.vec.Vec2.subtract(v2, v1, v2);
  assertElementsEquals([1, 2], v0);
  assertElementsEquals([4, 5], v1);
  assertElementsEquals([-3, -3], v2);

  goog.vec.Vec2.setFromValues(v2, 0, 0, 0);
  goog.vec.Vec2.subtract(v1, v0, v2);
  assertElementsEquals([3, 3], v2);

  v2 = goog.vec.Vec2.cloneFloat32(v0);
  goog.vec.Vec2.subtract(v2, v1, v2);
  assertElementsEquals([-3, -3], v2);

  goog.vec.Vec2.subtract(goog.vec.Vec2.subtract(v1, v0, v2), v0, v2);
  assertElementsEquals([2, 1], v2);
}

function testNegate() {
  var v0 = goog.vec.Vec2.createFloat32FromArray([1, 2]);
  var v1 = goog.vec.Vec2.createFloat32();

  goog.vec.Vec2.negate(v0, v1);
  assertElementsEquals([-1, -2], v1);
  assertElementsEquals([1, 2], v0);

  goog.vec.Vec2.negate(v0, v0);
  assertElementsEquals([-1, -2], v0);
}

function testAbs() {
  var v0 = goog.vec.Vec2.createFloat32FromValues(-1, -2);
  var v1 = goog.vec.Vec2.createFloat32();

  goog.vec.Vec2.abs(v0, v1);
  assertElementsEquals([1, 2], v1);
  assertElementsEquals([-1, -2], v0);

  goog.vec.Vec2.abs(v0, v0);
  assertElementsEquals([1, 2], v0);
}

function testScale() {
  var v0 = goog.vec.Vec2.createFloat32FromArray([1, 2]);
  var v1 = goog.vec.Vec2.createFloat32();

  goog.vec.Vec2.scale(v0, 4, v1);
  assertElementsEquals([4, 8], v1);
  assertElementsEquals([1, 2], v0);

  goog.vec.Vec2.setFromArray(v1, v0);
  goog.vec.Vec2.scale(v1, 5, v1);
  assertElementsEquals([5, 10], v1);
}

function testMagnitudeSquared() {
  var v0 = goog.vec.Vec2.createFloat32FromArray([1, 2]);
  assertEquals(5, goog.vec.Vec2.magnitudeSquared(v0));
}

function testMagnitude() {
  var v0 = goog.vec.Vec2.createFloat32FromArray([1, 2]);
  assertEquals(Math.sqrt(5), goog.vec.Vec2.magnitude(v0));
}

function testNormalize() {
  var v0 = goog.vec.Vec2.createFloat32FromArray([2, 3]);
  var v1 = goog.vec.Vec2.createFloat32();
  var v2 = goog.vec.Vec2.createFloat32();
  goog.vec.Vec2.scale(
      v0, 1 / goog.vec.Vec2.magnitude(v0), v2);

  goog.vec.Vec2.normalize(v0, v1);
  assertElementsEquals(v2, v1);
  assertElementsEquals([2, 3], v0);

  goog.vec.Vec2.setFromArray(v1, v0);
  goog.vec.Vec2.normalize(v1, v1);
  assertElementsEquals(v2, v1);
}

function testDot() {
  var v0 = goog.vec.Vec2.createFloat32FromArray([1, 2]);
  var v1 = goog.vec.Vec2.createFloat32FromArray([4, 5]);
  assertEquals(14, goog.vec.Vec2.dot(v0, v1));
  assertEquals(14, goog.vec.Vec2.dot(v1, v0));
}

function testDistanceSquared() {
  var v0 = goog.vec.Vec2.createFloat32FromValues(1, 2);
  var v1 = goog.vec.Vec2.createFloat32FromValues(1, 2);
  assertEquals(0, goog.vec.Vec2.distanceSquared(v0, v1));
  goog.vec.Vec2.setFromValues(v0, 1, 2);
  goog.vec.Vec2.setFromValues(v1, -1, -2);
  assertEquals(20, goog.vec.Vec2.distanceSquared(v0, v1));
}

function testDistance() {
  var v0 = goog.vec.Vec2.createFloat32FromValues(1, 2);
  var v1 = goog.vec.Vec2.createFloat32FromValues(1, 2);
  assertEquals(0, goog.vec.Vec2.distance(v0, v1));
  goog.vec.Vec2.setFromValues(v0, 2, 3);
  goog.vec.Vec2.setFromValues(v1, -2, 0);
  assertEquals(5, goog.vec.Vec2.distance(v0, v1));
}

function testDirection() {
  var v0 = goog.vec.Vec2.createFloat32FromValues(1, 2);
  var v1 = goog.vec.Vec2.createFloat32FromValues(1, 2);
  var dirVec = goog.vec.Vec2.createFloat32FromValues(4, 5);
  goog.vec.Vec2.direction(v0, v1, dirVec);
  assertElementsEquals([0, 0], dirVec);
  goog.vec.Vec2.setFromValues(v0, 0, 0);
  goog.vec.Vec2.setFromValues(v1, 1, 0);
  goog.vec.Vec2.direction(v0, v1, dirVec);
  assertElementsEquals([1, 0], dirVec);
  goog.vec.Vec2.setFromValues(v0, 1, 1);
  goog.vec.Vec2.setFromValues(v1, 0, 0);
  goog.vec.Vec2.direction(v0, v1, dirVec);
  assertElementsRoughlyEqual(
      [-0.707106781, -0.707106781],
      dirVec, goog.vec.EPSILON);
}

function testLerp() {
  var v0 = goog.vec.Vec2.createFloat32FromValues(1, 2);
  var v1 = goog.vec.Vec2.createFloat32FromValues(10, 20);
  var v2 = goog.vec.Vec2.cloneFloat32(v0);

  goog.vec.Vec2.lerp(v2, v1, 0, v2);
  assertElementsEquals([1, 2], v2);
  goog.vec.Vec2.lerp(v2, v1, 1, v2);
  assertElementsEquals([10, 20], v2);
  goog.vec.Vec2.lerp(v0, v1, .5, v2);
  assertElementsEquals([5.5, 11], v2);
}

function testMax() {
  var v0 = goog.vec.Vec2.createFloat32FromValues(10, 20);
  var v1 = goog.vec.Vec2.createFloat32FromValues(5, 25);
  var v2 = goog.vec.Vec2.createFloat32();

  goog.vec.Vec2.max(v0, v1, v2);
  assertElementsEquals([10, 25], v2);
  goog.vec.Vec2.max(v1, v0, v1);
  assertElementsEquals([10, 25], v1);
  goog.vec.Vec2.max(v2, 20, v2);
  assertElementsEquals([20, 25], v2);
}

function testMin() {
  var v0 = goog.vec.Vec2.createFloat32FromValues(10, 20);
  var v1 = goog.vec.Vec2.createFloat32FromValues(5, 25);
  var v2 = goog.vec.Vec2.createFloat32();

  goog.vec.Vec2.min(v0, v1, v2);
  assertElementsEquals([5, 20], v2);
  goog.vec.Vec2.min(v1, v0, v1);
  assertElementsEquals([5, 20], v1);
  goog.vec.Vec2.min(v2, 10, v2);
  assertElementsEquals([5, 10], v2);
}

function testEquals() {
  var v0 = goog.vec.Vec2.createFloat32FromValues(1, 2);
  var v1 = goog.vec.Vec2.cloneFloat32(v0);
  assertElementsEquals(v0, v1);

  v1[0] = 4;
  assertFalse(goog.vec.Vec2.equals(v0, v1));

  v1 = goog.vec.Vec2.cloneFloat32(v0);
  v1[1] = 4;
  assertFalse(goog.vec.Vec2.equals(v0, v1));
}
