# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from thread import start_new_thread
from threading import Thread
from os.path import join, isfile
from gobject import (TYPE_STRING as gString, markup_escape_text, idle_add,
                     timeout_add, threads_init)
from gtk.glade import XML as glade_XML
from gtk.gdk import (Pixbuf as gPixbuf, pixbuf_new_from_file, INTERP_BILINEAR,
                     threads_enter, threads_leave)
from gtk import (ListStore, TreeViewColumn, CellRendererPixbuf,
                 CellRendererText, STOCK_ORIENTATION_PORTRAIT,
                 ICON_SIZE_LARGE_TOOLBAR, Tooltip, VBox)

from modules.explorer.playlist import Playlist
from modules.explorer.musicdb import MusicDatabase
from modules.explorer import viewscommon
unescape = viewscommon.unescape
insensitive_sort = viewscommon.insensitive_sort

from common.functions import Functions
from common.webservices import LastFm
from common.statistics import Statistics

class FullView(object):
    ref = None
    ref2 = None

    def __new__(cls, *args, **kws):
        # Singleton
        if cls.ref is None:
            cls.ref = super(FullView, cls).__new__(cls, args, kws)
        return cls.ref

    def __init__(self, glade_file, config, userconf):
        self.playlist = Playlist(glade_file, config, userconf)
        self.glade_file = glade_file
        self.config = config
        self.userconf = userconf

        self.player_data = join(self.userconf.datadir, 'modules', 'player')
        self.functions = Functions()
        self.lastfm = LastFm()

        # Show widget
        self.glade_file.get_widget('vbox2').show()
        self.glade_file.get_widget('scrolledwindow1').show()

        if FullView.ref2 is None: 
            FullView.ref2 = 42 
            self.launch()

    def launch(self):
        threads_init()

        treestore = ListStore(gPixbuf, gString, gString)
        tview = self.glade_file.get_widget('treeview1')
        tview.set_model(treestore)
        tview.set_rules_hint(True)

        # GTK Handlers
        tview.connect('row-activated', self.load_item)
        tview.get_selection().connect('changed', self.load_artist_covers)

        column = TreeViewColumn()
        render_pixbuf = CellRendererPixbuf()
        column.pack_start(render_pixbuf, expand=False)
        column.add_attribute(render_pixbuf, 'pixbuf', 0)
        tview.append_column(column)

        column = TreeViewColumn()
        render_text = CellRendererText()
        column.pack_start(render_text, expand=True)
        column.add_attribute(render_text, 'markup', 1)
        tview.append_column(column)

        column = TreeViewColumn()
        render_text = CellRendererText()
        column.pack_start(render_text, expand=False)
        column.add_attribute(render_text, 'markup', 2)
        column.set_visible(False)
        tview.append_column(column)

        # Tooltips
        tview.connect('query-tooltip', self.query_tooltip)

        # Add artists
        self.reload_database()

    def reload_database(self, force_scan=False):
        tview = self.glade_file.get_widget('treeview1')
        treestore = tview.get_model()

        pbar = self.glade_file.get_widget('progressbar1')
        pbar.set_text(_('Opening folder…'))
        pbar.show()

        thread = Thread(group=None, target=self.insert_artists,
                        name='artists', args=(pbar, treestore, force_scan))
        self.thread_name = thread

        thread.start()

    def timeout_progress(self, pbar):
        if self.thread_name.isAlive():
            pbar.pulse()
            return True
        else:
            pbar.hide()

    def insert_artists(self, pbar, treestore, force_scan):
        """This function is called at the start-up and add artists."""
        threads_enter()

        # Call database
        musicdb = MusicDatabase(self.config, pbar)

        if self.config['scan'] or force_scan:
            musicdb.scan()

        pbar.set_text(_('Loading music…'))
        self.database = musicdb.load()

        timeout_add(100, self.timeout_progress, pbar)
        self.songs_tree = viewscommon.get_representation(self.database,
                                                        self.config['artwork'])
        self.continue_to_pulse = False

        pbar.set_text(_('Loading artists…'))
        items = self.songs_tree.items()
        items.sort(insensitive_sort)
        items_sort = [key for key, value in items]

        treestore.clear()

        artists_list = []
        for artist in items_sort:
            artists_list.append(artist)
            artist_ = markup_escape_text(artist)

            artist_picture = join(self.config['__data-dir'], 'artists',
                                  self.functions.get_hash(artist, 'picture'))

            if isfile(artist_picture):
                logo = pixbuf_new_from_file(artist_picture)
                logo = logo.scale_simple(24, 24, INTERP_BILINEAR)
            else:
                logo = None

            treestore.append((logo, artist_, 'artist'))

        if self.config['artwork']:
            start_new_thread(self.lastfm.get_pictures, (artists_list,))
        threads_leave()

    def show_albums_covers(self, artist):
        """This function shows all artist's arts."""
        artist = unescape(unicode(artist))

        def load_album_in_playlist(widget, alb):
            """This function loads an album in the playlist."""
            songs = []
            for song in self.songs_tree[str(artist)][alb]:
                songs.append(song)

            self.playlist.add_songs(songs)

        # Show the albums container
        self.glade_file.get_widget('hbox6').show()
        self.glade_file.get_widget('frame2').show()
        self.glade_file.get_widget('scrolledwindow4').hide()
        self.glade_file.get_widget('scrolledwindow5').hide()
        self.glade_file.get_widget('label7').set_markup(
                       '<b>' + markup_escape_text(unicode(artist)) + '</b>')

        # Delete all buttons
        container = self.glade_file.get_widget('hbuttonbox1')
        for wdg in container:
            container.remove(wdg)

        # Sort albums
        ordered_albums = []
        for album in self.songs_tree[str(artist)]:
            ordered_albums.append(album)

        ordered_albums.sort()

        # Add a button for each albums, based on the model
        for album in ordered_albums:
            widgets_ = glade_XML(join(self.functions.datadir, 'glade',
                       'mainwindow.glade'), 'button2', domain='bluemindo')
            album_button = widgets_.get_widget('button2')

            # Add the album name and the cover
            widgets_.get_widget('label8').set_markup(
                         '<b>' + markup_escape_text(unicode(album)) + '</b>')

            _file = join(self.player_data, 'covers',
                         self.functions.get_hash(album, artist))

            if isfile(_file):
                icon = pixbuf_new_from_file(_file)
                icon = icon.scale_simple(120, 100, INTERP_BILINEAR)
            else:
                icon = pixbuf_new_from_file(join(self.functions.datadir,
                                                 'image', 'logo_head_big.png'))
                icon = icon.scale_simple(85, 85, INTERP_BILINEAR)

            widgets_.get_widget('image5').set_from_pixbuf(icon)

            album_button.show_all()
            self.glade_file.get_widget('hbuttonbox1').add(album_button)

            # GTK handler
            album_button.connect('clicked', load_album_in_playlist, album)

    def load_item(self, tview, path, column):
        """This function handles the artists or playlists loading."""
        (mod, iter_) = tview.get_selection().get_selected()

        value = mod.get_value(iter_, 1)
        typ = mod.get_value(iter_, 2)

        if typ in 'artist':
            self.show_albums_covers(value)

            artist = unescape(unicode(value))
            songs = []

            for alb in self.songs_tree[str(artist)]:
                for sg in self.songs_tree[str(artist)][alb]:
                    songs.append(sg)

            # Finally insert songs in playlist :)
            self.playlist.add_songs(songs)

    def load_artist_covers(self, selection):
        """This function loads covers for an artist."""
        if selection:
            (mod, iter_) = selection.get_selected()
            if (iter_ and mod.get_value(iter_, 2) 
                          not in ('title', 'playlist')):
                artist = unescape(unicode(mod.get_value(iter_, 1)))
                self.show_albums_covers(artist)

    def query_tooltip(self, widget, x, y, keyboard_tip, tooltip):
        """Create a tooltip."""
        try:
            (path, col, x, y) = widget.get_path_at_pos(int(x), int(y)) 
            it = widget.get_model().get_iter(path)
            value = unescape(widget.get_model().get_value(it, 1))
            kind = widget.get_model().get_value(it, 2)

            wdg = glade_XML(join(self.functions.datadir, 'glade',
                                     'mainwindow.glade'), 'window2')

            picture_wdg = wdg.get_widget('image8')
            artist_picture = join(self.config['__data-dir'], 'artists',
                                  self.functions.get_hash(value, 'picture'))

            if isfile(artist_picture):
                logo = pixbuf_new_from_file(artist_picture)
                logo = logo.scale_simple(150, 120, INTERP_BILINEAR)
                picture_wdg.set_from_pixbuf(logo)
            else:
                picture_wdg.hide()

            wdg.get_widget('label9').set_markup('<b>%s</b>' %
                                         markup_escape_text(unicode(value)))

            # Count albums and songs
            nb_albums = 0
            nb_songs = 0

            for alb in self.songs_tree[value]:
                nb_albums += 1
                for sg in self.songs_tree[value][alb]:
                    nb_songs += 1

            wdg.get_widget('label10').set_text(_('%d albums') % nb_albums)
            wdg.get_widget('label11').set_text(_('%d songs') % nb_songs)

            # Show statistics
            stats = Statistics()
            nb_times = stats.get_stats_for_artist(value)
            wdg.get_widget('label17').set_markup(_('<b>%d</b> plays')
                                                 % nb_times)

            vbox = VBox()
            hbox = wdg.get_widget('hbox7')
            hbox.reparent(vbox)
            tooltip.set_custom(vbox)

            return True
        except (TypeError, ValueError):
            pass