/**
 * Copyright (C) 2009-2013 Paul Fretwell - aka 'Footleg' (drfootleg@gmail.com)
 * 
 * This file is part of Cave Converter.
 * 
 * Cave Converter is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Cave Converter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Cave Converter.  If not, see <http://www.gnu.org/licenses/>.
 */
package footleg.cavesurvey.data.reader;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import footleg.cavesurvey.cmdline.CaveConverter;
import footleg.cavesurvey.data.model.CaveSurvey;
import footleg.cavesurvey.data.model.SurveyLeg;
import footleg.cavesurvey.data.model.SurveySeries;
import footleg.cavesurvey.data.model.SurveyStation;
import footleg.cavesurvey.data.model.SurveyStation.FixType;
import footleg.cavesurvey.tools.UtilityFunctions;

/**
 * Parser for DXF format text data files.
 * 
 * @author      Footleg <drfootleg@gmail.com>
 * @version     2013.03.04                                (ISO 8601 YYYY.MM.DD)
 * @since       1.6                                       (The Java version used)
 */
public class DxfParser {

	private Date seriesDate;

	/**
	 * Parse Autocad DXF file into the cave data model. 
	 * 
	 * Converts polylines into series with fixed point at start.
	 * 
	 * @param surveyFileData ListArray of data lines from a DXF file
	 * @param parseMode An enumeration to indicate whether to parse file for contour polylines, spot heights or survey legs
     * @return Cave Survey object
	 */
	public CaveSurvey parseFile( List<String> surveyFileData, int parseMode ) {
		int legCount = 0;
		int iPolylineCount = 0;
		int iVertexCount = 0;
		int iPlainlineCount = 0;

		//Create new list of survey series to hold data
		CaveSurvey surveyData = new CaveSurvey();
		SurveySeries outerSeries = null;

		//Create arrays to data while processing lines
		List<List<double[]>> allChains = new ArrayList<List<double[]>>();
		List<double[]> arSurveyChain = new ArrayList<double[]>();
		List<double[]> arLines = new ArrayList<double[]>();

		//Define read state values
		final int stateParsingHeader = 0;
		final int stateFindPolyline = 1;
		final int stateFindVertex = 2;

		final int parseModeSurveyLegs = 0;
		final int parseModeSurfaceContours = 1;
		final int parseModeSpotHeights = 2;
		
		//Initialise read state
		int state = stateParsingHeader;
		
		//Local vars
		int iPointCount = 0;
		String sLineName = "";
	    double dEastValue = 0;
	    double dNorthValue = 0;
	    double dElevValue = 0;
	    double dLastEastValue = 0;
	    double dLastNorthValue = 0;
	    double dLastElevValue = 0;
		
	    //Range bounds
	    int iNorthEdge = 999999999;
	    int iSouthEdge = -999999999;
	    int iEastEdge = 999999999;
	    int iWestEdge = -999999999;
	    int iMinElev = -999999999;

		//Write output file header
		CaveConverter.logMessage("Generating survey data from lines and polylines in DXF data file");
		outerSeries = new SurveySeries( "SurveyFromDXF" );
		surveyData.add( outerSeries );

		//Loop through all data lines
		for ( int i=0; i < surveyFileData.size(); i++ ) {
			String dataLine = surveyFileData.get(i);

			//Proceed based on state
			switch( state ) {
			case stateParsingHeader :
				//First look for line:
				//    ENTITIES
				//This signifies start of data after header stuff which we ignore.
				if ( dataLine.trim().compareToIgnoreCase("ENTITIES") == 0 ) {
					//Found end of header section, so change state to searching for new polyline
					state = stateFindPolyline;
				}
				break;
			case stateFindPolyline:
				//} loop through all polylines, each indicated with header line:
				//    POLYLINE
				//and ending with:
				//    SEQEND
				if ( dataLine.trim().compareToIgnoreCase("POLYLINE") == 0 ) {
					//Found polyline
					//Check next 15 lines looking for AcDbEntity start
					int iSkipLoop = 1;
					do {
						dataLine = surveyFileData.get(++i);
					}
					while ( ( dataLine.trim().compareToIgnoreCase("AcDbEntity") != 0 ) 
					|| ( iSkipLoop > 15 ) );
					//If found it then process	
					if ( dataLine.trim().compareToIgnoreCase("AcDbEntity") == 0 ) {
						//(skip 1 line)
						dataLine = surveyFileData.get(++i);
						//Read entity type from next line:
						dataLine = surveyFileData.get(++i);
						//if Strings.Left(Trim(dataLine), 4) = "1100" {
							//Found a contour line entity (they are all 1100x)
							//Increment line counter
							iPolylineCount += 1;
							//Start new section heading, removing spaces from station names
							sLineName = dataLine.replace(" ", "_");
							if ( parseMode == parseModeSurfaceContours ) {
								sLineName = sLineName + iPolylineCount;
							}
							//Set state to finding vertices
							state = stateFindVertex;
							//Reset point counter
							iPointCount = 0;
						//}
					}
				}
				//loop through all lines, each indicated with header line:
				//    LINE
				//and ending after a fixed number of lines
				else if ( dataLine.trim().compareToIgnoreCase("LINE") == 0 ) {
					//Found line
					iPlainlineCount++;
					//Check next 15 lines looking for AcDbEntity start
					int iSkipLoop = 0;
					do {
						dataLine = surveyFileData.get(++i);
						iSkipLoop++;
					}
					while ( (dataLine.trim().compareToIgnoreCase("AcDbEntity") != 0 && dataLine.trim().compareToIgnoreCase("CentreLine") != 0 ) 
					&& ( iSkipLoop < 4 ) ); 
					//Allow anything if 4 lines read
					boolean allowAnyLineName = false;
					if ( iSkipLoop == 4 ) {
						allowAnyLineName = true;
					}
					//If found it then process	
					if  ( dataLine.trim().compareToIgnoreCase("AcDbEntity") == 0 
						 || dataLine.trim().compareToIgnoreCase("CentreLine") == 0 
					     || allowAnyLineName == true )  {
						if ( dataLine.trim().compareToIgnoreCase("CentreLine") != 0 && allowAnyLineName == false ) {
							//Check next 15 lines looking for AcDbLine start
							iSkipLoop = 1;
							do {
								dataLine = surveyFileData.get(++i);
							}
							while ( ( dataLine.trim().compareToIgnoreCase("AcDbLine") != 0 ) 
									|| ( iSkipLoop > 15 ) );
						}
						//If found it then process	
						if ( dataLine.trim().compareToIgnoreCase("AcDbLine") == 0 
							 || ( dataLine.trim().compareToIgnoreCase("CentreLine") == 0 ) 
							 || allowAnyLineName ) {
							//Check for next line:
							//10:
							dataLine = surveyFileData.get(++i);
							if ( dataLine.trim().compareToIgnoreCase("10") == 0 ) {
								//Read Easting
								dataLine = surveyFileData.get(++i);
								dEastValue = roundedDataValue(dataLine);
								//Check for next line:
								//20:
								dataLine = surveyFileData.get(++i);
								if ( dataLine.trim().compareToIgnoreCase("20") == 0 ) {
									//Read Northing
									dataLine = surveyFileData.get(++i);
									dNorthValue = roundedDataValue(dataLine);
									//Check for next line:
									//30:
									dataLine = surveyFileData.get(++i);
									if ( dataLine.trim().compareToIgnoreCase("30") == 0 ) {
										//Read Elevation
										dataLine = surveyFileData.get(++i);
										dElevValue = roundedDataValue(dataLine);
										//Check for next line:
										//11:
										dataLine = surveyFileData.get(++i);
										if ( dataLine.trim().compareToIgnoreCase("11") == 0 ) {
											//Read Easting
											dataLine = surveyFileData.get(++i);
											dLastEastValue = roundedDataValue(dataLine);
											//Check for next line:
											//21:
											dataLine = surveyFileData.get(++i);
											if ( dataLine.trim().compareToIgnoreCase("21") == 0 ) {
												//Read Northing
												dataLine = surveyFileData.get(++i);
												dLastNorthValue = roundedDataValue(dataLine);
												//Check for next line:
												//31:
												dataLine = surveyFileData.get(++i);
												if ( dataLine.trim().compareToIgnoreCase("31") == 0 ) {
													//Read Elevation
													dataLine = surveyFileData.get(++i);
													dLastElevValue = roundedDataValue(dataLine);
													//Now generate data to output
													if ( ( dEastValue >= iWestEdge )
													&& ( dEastValue <= iEastEdge )
													&& ( dNorthValue <= iNorthEdge )
													&& ( dNorthValue >= iSouthEdge )
													&& ( dElevValue >= iMinElev )
													&& ( dLastEastValue >= iWestEdge )
													&& ( dLastEastValue <= iEastEdge )
													&& ( dLastNorthValue <= iNorthEdge )
													&& ( dLastNorthValue >= iSouthEdge )
													&& ( dLastElevValue >= iMinElev )) {
														//Increment point counter
														iPointCount += 1;

														if ( parseMode == parseModeSurveyLegs ) {
															//Add points to line array
															double[] point = new double[6];
															point[0] = dEastValue;
															point[1] = dNorthValue;
															point[2] = dElevValue;
															point[3] = dLastEastValue;
															point[4] = dLastNorthValue;
															point[5] = dLastElevValue;
															arLines.add(point);
														}
													}
													else {
														//Record outside specified area
														dEastValue = dNorthValue;
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
				break;
			case stateFindVertex:
				//Within each polyline, look for vertices indicated by lines:
				//    VERTEX
				if ( dataLine.trim().compareToIgnoreCase("VERTEX") == 0 ) {
					//Reset flag
					boolean bValidRecord = false;
					//Increment count
					iVertexCount += 1;
					//Check next 15 lines looking for AcDbEntity start
					int iSkipLoop = 1;
					do {
						dataLine = surveyFileData.get(++i);
					}
					while ( ( dataLine.trim().compareToIgnoreCase("AcDbEntity") != 0 ) 
							|| ( iSkipLoop > 15 ) );
					if ( dataLine.trim().compareToIgnoreCase("AcDbEntity") == 0 ) {
						//Check next 15 lines looking for AcDbVertex start
						iSkipLoop = 1;
						do {
							dataLine = surveyFileData.get(++i);
						}
						while ( ( dataLine.trim().compareToIgnoreCase("AcDbVertex") != 0 ) 
								|| ( iSkipLoop > 15 ) );
						if ( dataLine.trim().compareToIgnoreCase("AcDbVertex") == 0 ) {
							//Check next 15 lines looking for AcDb3dPolylineVertex start
							iSkipLoop = 1;
							do {
								dataLine = surveyFileData.get(++i);
							}
							while ( ( dataLine.trim().compareToIgnoreCase("AcDb3dPolylineVertex") != 0 ) 
									|| ( iSkipLoop > 15 ) );
							if ( dataLine.trim().compareToIgnoreCase("AcDb3dPolylineVertex") == 0 ) {
								//Check for next line:
								//10:
								dataLine = surveyFileData.get(++i);
								if ( dataLine.trim().compareToIgnoreCase("10") == 0 ) {
									//Read Easting
									dataLine = surveyFileData.get(++i);
									dEastValue = roundedDataValue(dataLine);
									//Check for next line:
									//20:
									dataLine = surveyFileData.get(++i);
									if ( dataLine.trim().compareToIgnoreCase("20") == 0 ) {
										//Read Northing
										dataLine = surveyFileData.get(++i);
										dNorthValue = roundedDataValue(dataLine);
										//Check for next line:
										//30:
										dataLine = surveyFileData.get(++i);
										if ( dataLine.trim().compareToIgnoreCase("30") == 0 ) {
											//Read Elevation
											dataLine = surveyFileData.get(++i);
											dElevValue = roundedDataValue(dataLine);
											//Now generate data to output
											if ( ( dEastValue >= iWestEdge )
												&& ( dEastValue <= iEastEdge )
												&& ( dNorthValue <= iNorthEdge )
												&& ( dNorthValue >= iSouthEdge )
												&& ( dElevValue >= iMinElev ) ) {
												//Increment point counter
												iPointCount += 1;

												if ( parseMode == parseModeSurveyLegs ) {
													//Add point to chain
													double[] point = new double[3];
													point[0] = dEastValue;
													point[1] = dNorthValue;
													point[2] = dElevValue;
													arSurveyChain.add(point);
												}
												else {
													//Generate station name
													String sStnName = "S" + iPolylineCount + "P" + iPointCount;
													//Reset station position string
													String sPointPosition = "";
													if ( iPointCount == 1 ) {
														//First point, so fix it
														sPointPosition = "*fix " + sStnName + " " + 
														dEastValue + " " + 
														dNorthValue + " " + 
														dElevValue;
														if ( parseMode == parseModeSurfaceContours ) {
															//Write line header
															CaveConverter.logMessage("*begin " + sLineName);
														}
														else {
															//} point, so write differences for leg
															sPointPosition = "  " + (dEastValue - dLastEastValue) + " " + 
															(dNorthValue - dLastNorthValue) + " " + 
															(dElevValue - dLastElevValue);
														}
													}
													if ( parseMode == parseModeSpotHeights ) {
														CaveConverter.logMessage( String.valueOf(Math.round(dEastValue) - iEastEdge) );
														CaveConverter.logMessage( String.valueOf( (iNorthEdge - iSouthEdge) - ( Math.round(dNorthValue) - iSouthEdge) ) );
														CaveConverter.logMessage( String.valueOf(dElevValue) );
													}
													else {
														//Write position and station number
														CaveConverter.logMessage(sPointPosition);
														CaveConverter.logMessage(sStnName);
													}
												}
												//Update last positions
												dLastEastValue = dEastValue;
												dLastNorthValue = dNorthValue;
												dLastElevValue = dElevValue;
											}
											else {
												//Record outside specified area
												dEastValue = dNorthValue;
											}
											bValidRecord = true;
										}
									}
								}
							}
						}
					}
					//Check for invalid records
					if ( bValidRecord == false ) {
						//Invalid record
						CaveConverter.logMessage("Bad Line");
						//Set state back to searching for next vertex
						state = stateFindPolyline;
					}
				}
				//Find either next VERTEX or SEQEND
				else if ( dataLine.trim().compareToIgnoreCase("SEQEND") == 0 ) {
					//End of polyline
					//Write final station name if any points were written
					if ( parseMode == parseModeSurveyLegs ) {
						//Check that line name is unique
						String sUniqueName = sLineName;
						int k = 1;
						boolean bUnique = false;
						while ( bUnique == false ) {
							//Search existing chains for matching name
							bUnique = true;
							for ( int j = 0; j < outerSeries.innerSeriesCount(); j++ ) {
								//Check for matching series
								if ( outerSeries.getInnerSeries(j).getSeriesName().compareTo( sUniqueName ) == 0 ) {
									//Not unique, so add number to end
									sUniqueName = sLineName + k;
									k++;
									bUnique = false;
									//Exit loop
									j = outerSeries.innerSeriesCount();
								}
							}
						}

						//Add new series to list for this polyline
						outerSeries.addSeries( makeSeriesFromPolyline( arSurveyChain, sUniqueName ) );
						legCount += (arSurveyChain.size() - 1);
						
						//Store chain for linking to other chains later
						allChains.add(arSurveyChain);
						
						//Reset series data array
						arSurveyChain = new ArrayList<double[]>();

					}	
					else if ( parseMode == parseModeSurfaceContours ) {
						//End section if any points were in target area
						if ( iPointCount > 0 ) {
							CaveConverter.logMessage("*end " + sLineName);
						}
					}
					//Set state back to searching for next vertex
					state = stateFindPolyline;
				}
				break;
			default:
			}
		}

		//Attempt to join up lines into connected polylines
		int newlines = 0;
		while ( arLines.size() > 0 ) {
			//Get first line out
			double[] line = arLines.remove(0);
			//Start new polyline
			arSurveyChain = new ArrayList<double[]>();
			//Add points to chain
			double[] firstPoint = new double[3];
			firstPoint[0] = line[0];
			firstPoint[1] = line[1];
			firstPoint[2] = line[2];
			arSurveyChain.add(firstPoint);
			double[] lastPoint = new double[3];
			lastPoint[0] = line[3];
			lastPoint[1] = line[4];
			lastPoint[2] = line[5];
			arSurveyChain.add(lastPoint);

			//Search for ajoining lines
			boolean added = true;
			while ( added == true ) {
				added = false;
				for ( int linesIdx = 0; linesIdx < arLines.size(); linesIdx++ ) {
					boolean match = false;
					line = arLines.get(linesIdx);
					if ( firstPoint[0] == line[0]
					&& firstPoint[1] == line[1]
					&& firstPoint[2] == line[2] ) {
						//Add new first point to start of polyline
						firstPoint = new double[3];
						firstPoint[0] = line[3];
						firstPoint[1] = line[4];
						firstPoint[2] = line[5];
						arSurveyChain.add(0, firstPoint);
						match = true;
					}
					else if ( firstPoint[0] == line[3]
					&& firstPoint[1] == line[4]
					&& firstPoint[2] == line[5] ) {
						//Add new first point to start of polyline
						firstPoint = new double[3];
						firstPoint[0] = line[0];
						firstPoint[1] = line[1];
						firstPoint[2] = line[2];
						arSurveyChain.add(0, firstPoint);
						match = true;
					}
					else if ( lastPoint[0] == line[0]
					&& lastPoint[1] == line[1]
					&& lastPoint[2] == line[2] ) {
						//Add new last point to end of polyline
						lastPoint = new double[3];
						lastPoint[0] = line[3];
						lastPoint[1] = line[4];
						lastPoint[2] = line[5];
						arSurveyChain.add(lastPoint);
						match = true;
					}
					else if ( lastPoint[0] == line[3]
					&& lastPoint[1] == line[4]
					&& lastPoint[2] == line[5] ) {
						//Add new last point to end of polyline
						lastPoint = new double[3];
						lastPoint[0] = line[0];
						lastPoint[1] = line[1];
						lastPoint[2] = line[2];
						arSurveyChain.add(lastPoint);
						match = true;
					}
					if ( match == true ) {
						//Remove line and decrement index so next item is picked up
						arLines.remove(linesIdx);
						linesIdx--;
						added = true;
					}
				}
			}
			//Add chain to array and create series
			allChains.add(arSurveyChain);
			newlines++;
			outerSeries.addSeries( makeSeriesFromPolyline( arSurveyChain, "SeriesFromLines" + newlines ) );
			legCount += (arSurveyChain.size() - 1);
		}
		
		//Search for connected stations for any station in each polyline
		for ( int seriesIdx = 0; seriesIdx < outerSeries.innerSeriesCount(); seriesIdx++ ) {
			//Get chain corresponding to this series
			List<double[]> srcChain = allChains.get(seriesIdx);
			for ( int point1Idx = 0; point1Idx < srcChain.size(); point1Idx++ ) {
				//Get fixed position for this station in chain
				double fixX = srcChain.get(point1Idx)[0];
				double fixY = srcChain.get(point1Idx)[1];
				double fixZ = srcChain.get(point1Idx)[2];		
				//Check all chains except self for a matching point
				for ( int chainIdx = 0; chainIdx < allChains.size(); chainIdx++ ) {
					List<double[]> chain = allChains.get(chainIdx);
					for ( int pointIdx = 0; pointIdx < chain.size(); pointIdx++ ) {
						if ( ( fixX == chain.get(pointIdx)[0] ) 
						  && ( fixY == chain.get(pointIdx)[1] ) 
						  && ( fixZ == chain.get(pointIdx)[2] ) ) {
							//Found matching point
							boolean addEquate = true;
							if (chainIdx == seriesIdx ) {
								//Matching point is in same series, so only add equate 
								//when matching station occurs after test station to avoid adding
								//equate twice or equating stations onto themselves
								if ( pointIdx <= point1Idx) {
									addEquate = false;
								}
							}
							if ( addEquate == true ) {
								//Replace fixed point in leg with equate
								outerSeries.addLink(outerSeries.getInnerSeries(seriesIdx).getSeriesName(), 
									new SurveyStation(point1Idx), outerSeries.getInnerSeries(chainIdx).getSeriesName(),
									new SurveyStation(pointIdx));
							}
							//Clear fixed points apart from first one in first series
							if ( seriesIdx > 0 ) {
								SurveyStation stn = outerSeries.getInnerSeries(seriesIdx).getLegRaw(0).getFromStn();
								double easting = stn.getEasting();
								double northing = stn.getNorthing();
								double altitude = stn.getAltitude();
								
								stn.setFixed(FixType.NONE, easting, northing, altitude);
							}
						}
					}
				}
			}
		}


		//Debug dump
		UtilityFunctions.logSurveyDebugData(surveyData);

		//Completed file parsing
		CaveConverter.logMessage("Processed " + legCount + " survey legs in " + 
				outerSeries.innerSeriesCount() + " series.");
		CaveConverter.logMessage("Found:");
		CaveConverter.logMessage("Polylines: " + iPolylineCount + " containing " + iVertexCount + " line segments.");
		CaveConverter.logMessage("Lines: " + iPlainlineCount);
		CaveConverter.logMessage("Total line segments: " + (iPlainlineCount + iVertexCount));

		return surveyData;
	}

	/**
	 * Converts a series of points in a polyline with absolute coordinates into a series
	 * of connected survey legs,calculating tape, compass and clino data.
	 * @param List<double[]> List of all points in polyline as x,y,z coordinates
	 * @return SurveySeries of all the legs represented by the polyline. Fixed by position of first stn.
	 */
	private SurveySeries makeSeriesFromPolyline( List<double[]> arSurveyChain, String seriesName ) {
		//Create new series from this chain
		SurveySeries series = new SurveySeries( seriesName );
		
		//Set date
		series.setSurveyDate( seriesDate );

		//Loop through all points in chain
		for ( int i = 1; i < arSurveyChain.size(); i++ ) {
			SurveyLeg leg = new SurveyLeg();
			SurveyStation fromStn = new SurveyStation( i - 1 );
			SurveyStation toStn = new SurveyStation( i );
			double[] startPoint = arSurveyChain.get( i - 1 );
			double[] endPoint = arSurveyChain.get(i);

			if ( i == 1 ) {
				//First leg, so fix first station
				fromStn.setFixed(FixType.OTHER, startPoint[0], startPoint[1], startPoint[2]);
			}
			
			//Calculate leg data
			double x = endPoint[0] - startPoint[0];
			double y = endPoint[1] - startPoint[1];
			double z = endPoint[2] - startPoint[2];
			
			//Only add point if in a different position to previous point in polyline
			if ( x != 0 || y != 0 || z != 0 ) {
				double hori = Math.pow( Math.pow( x, 2 ) + Math.pow( y, 2 ), 0.5 );
	            double tape = Math.pow( Math.pow( hori, 2 ) + Math.pow( z, 2 ), 0.5 );
				double compass = bearing(x, y);
		        double clino = bearing(z, hori);
		        if ( clino > 90 ) {
		        	clino = clino - 360;
		        }
		        leg.setFromStn( fromStn );
		        leg.setToStn( toStn );
		        leg.setLength(tape);
		        leg.setCompass(compass);
		        leg.setClino(clino);
		        
				series.addLeg(leg);
			}
		}

		return series;
	}

	/**
	 * bearing
	 * 
	 * Returns bearing for angle described by x and y lengths
	 */
	private double bearing(double x, double y) {
		double res = 0;

		if ( x == 0 ) {
			if ( y < 0 ) {
				res = 180;
			} 
			else {
				res = 0;
			}
		} 
		else if ( x < 0 ) {
			if ( y == 0 ) {
				res = 270;
			} 
			else if ( y < 0 ) {
				//180-270
				res = 180 + (Math.atan(x / y) * 180 / Math.PI);
			} 
			else {
				//270-360
				res = 360 + (Math.atan(x / y) * 180 / Math.PI);
			}
		} 
		else {
			if ( y == 0 ) {
				res = 90;
			} 
			else if ( y < 0 ) {
				//90-180
				res = 180 + (Math.atan(x / y) * 180 / Math.PI);
			} 
			else {
				//0-90
				res = (Math.atan(x / y) * 180 / Math.PI);
			}
		}

		return res;
	}

	private double roundedDataValue( String dataLine ) {
		double roundedVal = ( (double)Math.round( Double.valueOf(dataLine) * 10000 ) ) / 10000;

		return roundedVal;
	}
}
