/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>

#include "gettext.h"
#include "cdw_config.h"
#include "cdw_xorrisorc.h"
#include "cdw_fs.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_utils.h"
#include "cdw_form.h"
#include "cdw_logging.h"

/*
  Code in this file operates on only three ISO9660 meta-data fields:
  - publisher
  - volume set id
  - system id

  These three fields are displayed in "Meta data" tab in wizard
  options ("create image" wizard and "burn from files" wizard).

  There is also a fourth ISO9660 meta-data field: "volume id", but it
  is displayed in main window of wizard, so it is handled directly by
  wizard code.

  Notice that cdw_mkisofsrc handles more ISO9660 meta-data fields than
  cdw_xorrisorc.
*/


/* the value must be larger than any of CDW_ISO9660_*_LEN values
   defined in optical_file_systems/cdw_iso9660.h */
#define BUFFER_SIZE 200


static int   cdw_xorrisorc_read_config_file(cdw_form_t *cdw_form, void *dummy);
static char *cdw_xorrisorc_top_message(int n_cols);
static void  cdw_xorrisorc_get_option_from_line(const char *line, const char *option_name, cdw_form_t *cdw_form, int fi, size_t len);
static const char *cdw_xorrisorc_get_option_value(char *buffer, char *ptr, size_t n_max);

static cdw_form_text_maker_t text_makers[] = {
	cdw_xorrisorc_top_message,
};




enum {
	f_xorrisorc_top_message_l,
	f_xorrisorc_top_message_b,
	f_xorrisorc_publisher_l,
	f_xorrisorc_publisher_i,
	f_xorrisorc_volume_set_id_l,
	f_xorrisorc_volume_set_id_i,
	f_xorrisorc_system_id_l,
	f_xorrisorc_system_id_i };




static FIELD *xorrisorc_fields[CDW_XORRISO_ISO_META_OPTIONS_N_FIELDS + 1]; /* +1 for last field == NULL */


/**
   \brief Create fields that are shown on "Meta information" page in options

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_GEN_ERROR if one of fields was created incorrectly
*/
cdw_rv_t cdw_xorrisorc_options_form(cdw_form_t *cdw_form, __attribute__((unused)) void *cdw_iso, int first_col, __attribute__((unused)) int second_col, int width_wide, __attribute__((unused)) int width_narrow)
{
	cdw_form->fields = xorrisorc_fields;

	cdw_form_descr_t descr[] = {
		/*     type                   begin_y   begin_x     n_cols           n_lines  field enum                     data1                               data2 */

		{ CDW_WIDGET_ID_TEXT,               1,  1,          width_wide + 2,        7,  f_xorrisorc_top_message_l,    text_makers,                            0 },
		/* 2TRANS: this is button label; a verb: read data from config file */
		{ CDW_WIDGET_ID_BUTTON,            10,  1,          1,                     1,  f_xorrisorc_top_message_b,    _("Read"),              CDW_COLORS_DIALOG },

		/* 2TRANS: this is a label next to input field, in which user can enter information about publisher of ISO9660 volume */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     12,  first_col,  width_wide,            1,  f_xorrisorc_publisher_l,      _("Publisher"),                         0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   13,  first_col,  width_wide,            1,  f_xorrisorc_publisher_i,      (char *) NULL,       CDW_ISO9660_PUBL_LEN },

		/* 2TRANS: this is a label next to input field, in which user can enter information about ID of ISO9660 volume set */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     14,  first_col,  width_wide,            1,  f_xorrisorc_volume_set_id_l,  _("Volume set ID"),                     0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   15,  first_col,  width_wide,            1,  f_xorrisorc_volume_set_id_i,  (char *) NULL,       CDW_ISO9660_VOLS_LEN },

		/* 2TRANS: this is a label next to input field, in which user can enter information about system on which ISO9660 volume has been created */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     16,  first_col,  width_wide,            1,  f_xorrisorc_system_id_l,      _("System ID"),                         0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   17,  first_col,  CDW_ISO9660_SYSI_LEN,  1,  f_xorrisorc_system_id_i,      (char *) NULL,       CDW_ISO9660_SYSI_LEN },

		/* guard */
		{ -1,                               0,  0,          0,                     0,  0,                            (void *) NULL,                          0 }};

	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_set_widget_callback(cdw_form, f_xorrisorc_top_message_b, cdw_xorrisorc_read_config_file);

		cdw_form_bind_input_and_label(cdw_form, f_xorrisorc_publisher_i, f_xorrisorc_publisher_l);
		cdw_form_bind_input_and_label(cdw_form, f_xorrisorc_volume_set_id_i, f_xorrisorc_volume_set_id_l);
		cdw_form_bind_input_and_label(cdw_form, f_xorrisorc_system_id_i, f_xorrisorc_system_id_l);

		return CDW_OK;
	}
}





cdw_rv_t cdw_xorrisorc_options_validate(cdw_iso9660_t *iso, int *fi)
{
	cdw_rv_t crv = cdw_string_security_parser(iso->publisher, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_xorrisorc_publisher_i;
		return CDW_NO;
	}

	crv = cdw_string_security_parser(iso->volume_set_id, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_xorrisorc_volume_set_id_i;
		return CDW_NO;
	}

	crv = cdw_string_security_parser(iso->system_id, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_xorrisorc_system_id_i;
		return CDW_NO;
	}

	return CDW_OK;
}





cdw_rv_t cdw_xorrisorc_options_save(cdw_form_t *cdw_form, cdw_iso9660_t *iso)
{
	char *s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_xorrisorc_volume_set_id_i));
	strncpy(iso->volume_set_id, s, CDW_ISO9660_VOLS_LEN);
	        iso->volume_set_id[CDW_ISO9660_VOLS_LEN] = '\0';

	s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_xorrisorc_publisher_i));
	strncpy(iso->publisher, s, CDW_ISO9660_PUBL_LEN);
	        iso->publisher[CDW_ISO9660_PUBL_LEN] = '\0';

	s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_xorrisorc_system_id_i));
	strncpy(iso->system_id, s, CDW_ISO9660_SYSI_LEN);
	        iso->system_id[CDW_ISO9660_SYSI_LEN] = '\0';

	return CDW_OK;
}





char *cdw_xorrisorc_top_message(int n_cols)
{
	/* 2TRANS: this is a message printed in configuration window;
	   don't localize "$HOME/.xorrisorc" */
	return cdw_string_wrap(_("These values won't be saved after closing cdw. Leave the fields empty (values from xorriso config file will be used) or enter custom values. You can populate the fields with values from $HOME/.xorrisorc using the button below."), (size_t) n_cols, CDW_ALIGN_CENTER);
}





static const char *file_name = ".xorrisorc";
static FILE *file = (FILE *) NULL;


/* options in the ~/.xorrisorc file should (?) be all in one line, like this:
   -system_id   mysystemid  -publisher   "my publisher string"    -volset_id my_volset
*/

int cdw_xorrisorc_read_config_file(cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	const char *home_dir = cdw_fs_get_home_dir_fullpath();
	char *path = cdw_string_concat(home_dir, file_name, (char *) NULL);
	file = fopen(path, "r");
	free(path);
	path = (char *) NULL;
	if (file == (FILE *) NULL) {
		/* 2TRANS: this is message printed to log file;
		   '%s' is configuration file name */
		cdw_logging_write(_("WARNING: can't open %s file in home directory\n"), file_name);
		cdw_vdm ("WARNING: can't open xorrisorc file\n");
		return -1;
	}

	char *line = (char *) NULL;
	while (1) {
		if (line != (char *) NULL) {
			free(line);
			line = (char *) NULL;
		}

		line = my_readline_10k(file);
		if (line == (char *) NULL) {
			break;
		}
		cdw_sdm ("INFO: processing line \"%s\"\n", line);
		cdw_xorrisorc_get_option_from_line(line, "-publisher ", cdw_form, f_xorrisorc_publisher_i, CDW_ISO9660_PUBL_LEN);
		cdw_xorrisorc_get_option_from_line(line, "-system_id ", cdw_form, f_xorrisorc_system_id_i, CDW_ISO9660_SYSI_LEN);
		cdw_xorrisorc_get_option_from_line(line, "-volset_id ", cdw_form, f_xorrisorc_volume_set_id_i, CDW_ISO9660_VOLS_LEN);
	}
	fclose(file);
	file = (FILE *) NULL;

	return 0;
}





/**
   \brief Copy option value to a buffer

   Search for a string in \p ptr that looks like an option value.
   Copy the value into \p buffer. Make sure that the option value has no
   more than \p n_max chars.

   Trim the string in buffer and remove trailing and ending quote chars.
   Return pointer to beginning of the option value.
   Because of the trimming and removing quotes, function may remove pointer
   that is inside of \p buffer, but is not at the very beginning of \p buffer.

   \return pointer to string representing option value on success
   \return NULL otherwise (this doesn't have to indicate an error)
*/
const char *cdw_xorrisorc_get_option_value(char *buffer, char *ptr, size_t n_max)
{
	buffer[0] = '\0';

	size_t len = strlen(ptr);
	for (size_t i = 0; i < len && i < n_max; i++) {
		if (ptr[i] == '-') { /* start of new option name */
			break;
		} else {
			buffer[i] = ptr[i];
			buffer[i + 1] = '\0'; /* +1 is always safe, as long as "for ( ;i < n_max; )" */
		}
	}
	buffer[BUFFER_SIZE] = '\0';
	if (!strlen(buffer)) {
		return (char *) NULL;
	}

	char *trimmed = cdw_string_ltrim(buffer);
	cdw_string_rtrim(trimmed);
	if (!strlen(trimmed)) {
		return (char *) NULL;
	}

	/* option may be quoted in file, the quotes are unwanted
	   in fields, and may be harmful when building command line
	   string for xorriso; remove the quotes */
	if (trimmed[0] == '"') {
		trimmed++;
	}
	len = strlen(trimmed);
	if (trimmed[len - 1] == '"') {
		trimmed[len - 1] = '\0';
	}

	cdw_rv_t crv = cdw_string_security_parser(trimmed, (char *) NULL);
	if (crv != CDW_OK) {
		cdw_vdm ("WARNING: option value \"%s\" from xorrisorc file rejected as insecure\n",
			 trimmed);
		return (char *) NULL;
	} else {
		return trimmed;
	}
}





/**
   \brief Copy option value into form field

   Search for given \p option_name in given \p line. If the option
   is found, get value of the option, and put it in form field specified
   by field index \p fi. Copy to the field at most \p len chars.
*/
void cdw_xorrisorc_get_option_from_line(const char *line, const char *option_name, cdw_form_t *cdw_form, int fi, size_t len)
{
	char buffer[BUFFER_SIZE + 1];
	char *ptr = strstr(line, option_name);
	if (ptr != (char *) NULL) {
		ptr += strlen(option_name);
		/* b is a pointer inside of buffer */
		const char *b = cdw_xorrisorc_get_option_value(buffer, ptr, BUFFER_SIZE);
		if (b != (char *) NULL) {
			char local_buffer[BUFFER_SIZE + 1];
			strncpy(local_buffer, b, len);
			local_buffer[len] = '\0';
			cdw_sdm ("INFO: %s = \"%s\"\n", option_name, local_buffer);
			set_field_buffer(cdw_form->fields[fi], 0, local_buffer);
		}
	}

	return;
}
