#ifndef H_CDW_CONFIG
#define H_CDW_CONFIG

#include <stdbool.h>

#include "main.h"
#include "cdw_iso9660.h"
#include "cdw_udf.h"
#include "cdw_burn_disc.h"
#include "cdw_erase_disc.h"


/** \brief Maximal length of some char tables storing long option values */
#define OPTION_FIELD_LEN_MAX 950


int cdw_config_module_init(void);
void cdw_config_module_clean(void);
cdw_rv_t cdw_config_write_to_file(void);
const char *cdw_config_get_config_dir(void);


/* volume size - maximum size of selected files, used
   in "selected files info" view in main ui window; it is tempting
   to use "disc type" from cdw_disc.h, but it's not that simple to
   unify the two sets */
enum {
	CDW_CONFIG_VOLUME_SIZE_CD74 = 0,
	CDW_CONFIG_VOLUME_SIZE_CD80,
	CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC, /* smallest common denominator for DVDs */
	CDW_CONFIG_VOLUME_SIZE_DVD_R,
	CDW_CONFIG_VOLUME_SIZE_DVD_RP,
	CDW_CONFIG_VOLUME_SIZE_DVD_RW,
	CDW_CONFIG_VOLUME_SIZE_DVD_RWP,
	CDW_CONFIG_VOLUME_SIZE_DVD_R_DL,
	CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,
	CDW_CONFIG_VOLUME_SIZE_CUSTOM,
	CDW_CONFIG_VOLUME_SIZE_AUTO };



/** \brief Pair of name-value strings that make an option */
typedef struct {
	char *name; /**< \brief Name of option, string that stands on left-hand side of '=' char in configuration file*/
	char *value; /**< \brief Value of option, string that stands on right-hand side of '=' char in configuration file */
} cdw_option_t;


bool cdw_config_split_options_line_old(cdw_option_t *option, char *line);
bool cdw_config_split_options_line(cdw_option_t *option, char *line);
void cdw_config_option_free(cdw_option_t *option);




typedef struct {
	/* This option is presented in "Create image" wizard.

	   Used when creating new image from selected files, or when
	   burning an image to optical disc.  */
	char *image_fullpath;                /**< \brief Current full path to ISO9660 or UDF image file. */

	char *log_fullpath;                  /**< \brief Full path to cdw log file */
	bool show_log;                       /**< \brief Show cdw log after finishing operation */

	cdw_id_t volume_size_id;             /**< \brief ID of preferred standard size of target media/iso file */
	long int volume_size_custom_value;   /**< \brief Custom size of target media/iso file, MB */

	bool selected_follow_symlinks;       /**< \brief Follow symlinks when calculating size of selected files. */

	/* This one doesn't appear in Configuration window
	   (yet). */
	bool display_hidden_files;           /**< Display hidden files in browsers of native file system. */

	bool fs_lynx_like_motion;            /**< Recognize left and right arrow keys as native file system navigation keys (instead of horizontal scroll keys). */

	/* Options not displayed in configuration window and not saved
	   to config file */
	bool support_dvd_rp_dl;
	bool show_dvd_rw_support_warning;
} cdw_config_general_t;



/** \brief Main cdw configuration variable, storing all major options used by cdw */
typedef struct {

	/* I need a place to globally store ID of current/last
	   executed task. This is the place. */
	cdw_id_t task_id;


	/* Some of the fields in the cdw_write_t data structure aren't
	   used by configuration module, but to have a single data
	   type storing writing-related variables, and to use single
	   code that handles them, I'm (re-)using cdw_write_t. */
	cdw_write_t burn;

	cdw_erase_t erase;

	/* Hardware configuration */

	/* device name in form of /dev/xxx - it is used as default device
	   name in cdrecord, but SCSI descriptor can be also used if needed;
	   custom_device is also used in other tools code - these tools use
	   only this entry to access reading/writing device (I don't mention
	   here cddb nor legacy cdda2wav code, because it is currently
	   unmaintained) */
	/* TODO: isn't there any better name for length of the buffer? */
	char custom_drive[OPTION_FIELD_LEN_MAX + 1]; /**< \brief Path to CD/DVD reader/burner */
	/* state of "drive" dropdown in configuration window */
	char selected_drive[OPTION_FIELD_LEN_MAX + 1];
	/* SCSI device (reader/writer) descriptor in form of X:Y:Z;
	   it should be used only in cdrecord code, and user should set this
	   string only in cases when cdrecord can't work with /dev/xxx device */
	char scsi[OPTION_FIELD_LEN_MAX + 1];        /**< \brief Variable specifying scsi device in form preferred by cdrecord: bus:target:lun */


	/* Audio configuration */
	char *audiodir;

	/* ISO9660 filesystem options. */
	cdw_iso9660_t iso9660;

	/* UDF filesystem options. */
	cdw_udf_t udf;


	/* Configuration of external tools */
	/* external tools have their own configuration variable, and there
	   are no settings (yet) that should be stored in config file */


	/* Log file settings, other setting.
	   Some of these options are not stored in config file. */
	cdw_config_general_t general;

} cdw_config_t;



cdw_rv_t cdw_config_var_copy(cdw_config_t *dest, cdw_config_t *src);
cdw_rv_t cdw_config_var_validate(cdw_config_t *config, cdw_id_t *page, int *field);


cdw_rv_t cdw_config_var_init_fields(cdw_config_t *config);
cdw_rv_t cdw_config_var_free_fields(cdw_config_t *config);

bool cdw_config_has_scsi_device(void);
const char *cdw_config_get_custom_drive(void);
const char *cdw_config_get_scsi_drive(void);

void cdw_config_debug_print_config(cdw_config_t *config);

long int cdw_config_get_volume_size_mb_by_id(cdw_id_t id);
long int cdw_config_get_current_volume_size_value_megabytes(void);
cdw_id_t cdw_config_volume_id_by_label(const char *label);

bool cdw_config_support_dvd_rp_dl(void);
bool cdw_config_follow_symlinks(void);
bool cdw_config_fs_lynx_like_motion(void);



/* this is mostly for cdw_config_window.h, but also for
   cdw_config.c/cdw_config_var_validate() */
enum c_pages_order {
	CONFIG_PAGE_ID_AUDIO = 0,
	CONFIG_PAGE_ID_HW    = 1,
	CONFIG_PAGE_ID_TOOLS = 2,
	CONFIG_PAGE_ID_OTHER = 3,
	CONFIG_PAGE_ID_UDF   = 4
};

/* unit tests */

void cdw_config_run_tests(void);

#endif /* H_CDW_CONFIG */
