/* $Id: kmo_arithmetic-test.c,v 1.7 2013-10-08 11:18:56 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:56 $
 * $Revision: 1.7 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>

#include <cpl.h>

#include "kmclipm_math.h"

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

double          scalar              = 6.7;
int             ifu_nr              = 4;

const char      *path_recipe        = "kmo_arithmetic/";

const char      *valid_files[]      = {"v_data_RAW.fits",
                                       "v_data_F2D.fits",
                                       "v_data_noise_F2D.fits",
                                       "v_data_F3I_all.fits",
                                       "v_data_noise_F3I_all.fits",
                                       "v_data_noise_F3I_1.fits",
                                       "v_data_F2I_1.fits",
                                       "v_data_noise_F2I_all.fits",
                                       "v_data_F1I_all.fits",
                                       "v_data_noise_F1I_1.fits",
                                       "v_data_RAW_zero.fits",
                                       "v_data_F3I_2_true_false.fits",
                                       "v_data_F3I_2_false_true.fits",
                                       "v_data_F3I_false.fits"};

const char      *invalid_files[]    = {"i_data_F3I_too_small.fits"};

/**
    @defgroup kmo_arithmetic_test   kmo_arithmetic unit tests

    @{
*/

/**
 * @brief   test with two frames
 */
static void test_arithmetic(const char *path_op1,
                            const char *path_op2,
                            const char *op,
                            int ret_val)
{
    // create sof-file
    const char *sof_path = "test_arithmetic.sof";
    FILE *fh = fopen( sof_path, "w" );
    fprintf (fh, "%s\n%s\n",
             path_op1, path_op2);
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_arithmetic "
            "--operator=\"%s\" %s",
            op, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_arithmetic.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
 * @brief   test with one frame and a scalar
 */
static void test_arithmetic_scalar(const char *path_op1,
                                   double scalar,
                                   const char *op)
{
    // create sof-file
    const char *sof_path = "test_arithmetic_scalar.sof";
    FILE *fh = fopen( sof_path, "w" );
    fprintf (fh, "%s\n", path_op1);
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_arithmetic "
            "--operator=\"%s\" --scalar=\"%g\" %s",
            op, scalar, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_arithmetic.txt", sof_path, TRUE);

    // execute esorex-command
    cpl_test_eq(0, system(esorex_command));
}

/**
    @brief
        Generates test data for kmo_arithmetic.

    @param path Path where the generated test date should be saved to.
*/
static int kmo_generate_test_data_arithmetic(const char *path)
{
    char        file_path[256];

    /* ----- valid test data ----- */
    /* RAW */
    cpl_test_eq(0,
    kmo_test_create_RAW_data(test_global_size_x, test_global_size_y,
                        kmo_test_cat_strings(file_path, path, valid_files[0]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0, test_global_seed_data, 0.0));

    /* F2D */
    cpl_test_eq(0,
                kmo_test_create_F2D_data(test_global_size_x, test_global_size_y, FALSE,
                        kmo_test_cat_strings(file_path, path, valid_files[1]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0));

    cpl_test_eq(0,
                kmo_test_create_F2D_data_noise(test_global_size_x, test_global_size_y,
                        kmo_test_cat_strings(file_path, path, valid_files[2]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0));

    /* F3I */
    cpl_test_eq(0,
                kmo_test_create_F3I_data(test_global_size_x, test_global_size_y, test_global_size_z,  FALSE,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[3]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    cpl_test_eq(0,
               kmo_test_create_F3I_data_noise(test_global_size_x, test_global_size_y, test_global_size_z,  FALSE,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[4]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    cpl_test_eq(0,
                kmo_test_create_F3I_data_noise(test_global_size_x, test_global_size_y, test_global_size_z,  FALSE,1,
                          kmo_test_cat_strings(file_path, path, valid_files[5]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    /* F2I */
    cpl_test_eq(0,
                kmo_test_create_F2I_data(test_global_size_x, test_global_size_y, FALSE, 1,
                          kmo_test_cat_strings(file_path, path, valid_files[6]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    cpl_test_eq(0,
                kmo_test_create_F2I_data_noise(test_global_size_x, test_global_size_y, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[7]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    /* F1I */
    cpl_test_eq(0,
                kmo_test_create_F1I_data(test_global_size_z, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[8]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    cpl_test_eq(0,
                kmo_test_create_F1I_data_noise(test_global_size_z, 1,
                          kmo_test_cat_strings(file_path, path, valid_files[9]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    /* RAW zero */
    cpl_test_eq(0,
                kmo_test_create_RAW_data_zero(test_global_size_x, test_global_size_y,
                        kmo_test_cat_strings(file_path, path, valid_files[10])));

    /* F3I (valid, invalid) & F3I (invalid, valid) */
    cpl_test_eq(0,
                kmo_test_create_F3I_data(test_global_size_x, test_global_size_y, test_global_size_z,  FALSE, 2,
                          kmo_test_cat_strings(file_path, path, valid_files[11]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    int val1[] = {FALSE, TRUE};

    cpl_test_eq(0,
                kmo_test_create_F3I_data2(test_global_size_x, test_global_size_y, test_global_size_z, 2, val1,
                          kmo_test_cat_strings(file_path, path, valid_files[12]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    int val2[] = {FALSE};

    cpl_test_eq(0,
                kmo_test_create_F3I_data2(test_global_size_x, test_global_size_y, test_global_size_z, 1, val2,
                          kmo_test_cat_strings(file_path, path, valid_files[13]),
                          NULL, NULL, NULL, 0,
                          NULL, NULL, NULL, 0));

    /* ----- invalid test data ----- */
    /* 2nd operator too small */
    cpl_test_eq(0,
                kmo_test_create_F3I_data(test_global_size_x, test_global_size_y, test_global_size_z - 1,  FALSE,
                        test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, invalid_files[0]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0));

    return 0;
}


/**
  @brief	Test of kmo_arithmetic recipe.
  @param	argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_arithmetic-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_arithmetic-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol = 0.01;

    char    test_path[256],
            op1_file[256],
            op2_file[256];

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_arithmetic(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        // just generate the data
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    FILE *fd = fopen("log_kmo_arithmetic.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    /* ----- valid tests ----- */
    cpl_msg_info(cpl_func, "Testing kmo_aithmetic with valid data.");

    //
    // RAW
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);

    test_arithmetic(op1_file, op1_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 101.7, tol);

    test_arithmetic(op1_file, op1_file, "-", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 0.0, tol);

    test_arithmetic(op1_file, op1_file, "*", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 1135.86, tol);

    test_arithmetic(op1_file, op1_file, "/", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 3.0, tol);

    // divide RAW by zero
    kmo_test_cat_strings(op2_file, test_path, valid_files[10]);
    test_arithmetic(op1_file, op2_file, "/", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 0.0, tol);

    // RAW scalar
    test_arithmetic_scalar(op1_file, scalar, "+");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 70.95, tol);

    test_arithmetic_scalar(op1_file, scalar, "-");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 30.75, tol);

    test_arithmetic_scalar(op1_file, scalar, "*");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 340.695, tol);

    test_arithmetic_scalar(op1_file, scalar, "/");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 7.58955, tol);

    test_arithmetic_scalar(op1_file, scalar, "^");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 6540364687.17815, tol);

    test_arithmetic_scalar(op1_file, 0.0, "^");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 3, tol);

    test_arithmetic_scalar(op1_file, 0.0, "/");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 0.0, tol);

    //
    // RAW & F2D and vice versa
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[1]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[0]);
    test_arithmetic(op1_file, op2_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 101.7, tol);

    test_arithmetic(op2_file, op1_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 101.7, tol);

    //
    // F2D with and without noise
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[1]);
    test_arithmetic(op1_file, op1_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 101.7, tol);

    kmo_test_cat_strings(op1_file, test_path, valid_files[2]);
    test_arithmetic(op1_file, op1_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0),112.094, tol);

    test_arithmetic(op1_file, op1_file, "-", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 10.395, tol);

    test_arithmetic(op1_file, op1_file, "*", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 1351.36, tol);

    test_arithmetic(op1_file, op1_file, "/", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 3.79259, tol);

    test_arithmetic_scalar(op1_file, scalar, "^");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 1.21047e+10, tol*1000000000);

    test_arithmetic_scalar(op1_file, 0.0, "^");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 3.0, tol);

    test_arithmetic_scalar(op1_file, 0.0, "/");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 7.35, tol);

    //
    // F3I & F3I
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[3]);
    test_arithmetic(op1_file, op1_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 155.7, tol);

    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    test_arithmetic(op1_file, op1_file, "*", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 3853.39, tol);

    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[3]);
    test_arithmetic(op1_file, op2_file, "-", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 0, tol);

    kmo_test_cat_strings(op1_file, test_path, valid_files[5]);
    test_arithmetic(op1_file, op1_file, "/", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 1.41408, tol);

    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[5]);
    test_arithmetic(op1_file, op2_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 172.3973, tol);

    //
    // F3I & F2I
    //

    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[6]);
    test_arithmetic(op1_file, op2_file, "-", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 30.0, tol);

    kmo_test_verbose_off();
    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[7]);
    test_arithmetic(op1_file, op2_file, "/", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 9.9311, tol);
    kmo_test_cat_strings(op1_file, test_path, valid_files[11]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[12]);
    test_arithmetic(op1_file, op2_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 0.0, tol);

    kmo_test_cat_strings(op1_file, test_path, valid_files[13]);
    test_arithmetic(op1_file, op1_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 0.0, tol);
    kmo_test_verbose_on();

    kmo_test_cat_strings(op1_file, test_path, valid_files[12]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[6]);
    test_arithmetic(op1_file, op2_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 34.9, tol);

    //
    // F3I & F1I
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[9]);
    test_arithmetic(op1_file, op2_file, "+", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 122.119, tol);

    kmo_test_cat_strings(op1_file, test_path, valid_files[3]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[8]);
    test_arithmetic(op1_file, op2_file, "-", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 62.95, tol);

    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[9]);
    test_arithmetic(op1_file, op2_file, "*", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 420.986, tol);

    kmo_test_cat_strings(op1_file, test_path, valid_files[3]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[8]);
    test_arithmetic(op1_file, op2_file, "/", 0);
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 18.6431, tol);

    //
    // F3I & scalar
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    test_arithmetic_scalar(op1_file, scalar, "*");
    cpl_test_abs(kmo_test_esorex_data(ARITHMETIC, 0), 555.945, tol);

    /* ----- invalid tests ----- */
    cpl_msg_info(cpl_func, "Testing kmo_aithmetic with invalid data.");
    // first: empty F3I, second: filled F2I
    kmo_test_cat_strings(op1_file, test_path, valid_files[13]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[6]);
    test_arithmetic(op1_file, op2_file, "*", -1);

    // first op has one IFU, second one has more than one IFUs
    kmo_test_cat_strings(op1_file, test_path, valid_files[5]);
    kmo_test_cat_strings(op2_file, test_path, valid_files[4]);
    test_arithmetic(op1_file, op2_file, "+", -1);

    // 2nd operator too small
    kmo_test_cat_strings(op1_file, test_path, valid_files[3]);
    kmo_test_cat_strings(op2_file, test_path, invalid_files[0]);
    test_arithmetic(op1_file, op2_file, "+", -1);

    // invalid operator
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    test_arithmetic(op1_file, op2_file, "x", -1);

    return cpl_test_end(0);
}

/** @} */
