/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "Darkroom.h"

#include <cmath>
#include <unistd.h>

#include <QDesktopWidget>
#include <QDockWidget>
#include <QProgressBar>
#include <QTimeLine>
#include <QTimer>

#include <KAction>
#include <KActionCollection>
#include <KDebug>
#include <KFileDialog>
#include <KMessageBox>
#include <KMenu>
#include <KLocale>
#include <KStandardAction>
#include <KStatusBar>
#include <KStandardDirs>


#include <threadweaver/WeaverObserver.h>
#include <threadweaver/ThreadWeaver.h>

#include "rawfiles.h"

#include "DarkroomView.h"
#include "JobExport.h"
#include "JobPreview.h"
#include "JobThumbnail.h"
#include "Icons.h"
#include "ListOptionsModel.h"
#include "ListRawFileModel.h"
#include "MessagesModel.h"
#include "RawImageInfo.h"
#include "PostProcessor.h"
#include "PreviewInfo.h"
#include "ProcessingOptions.h"
#include "ExportCodec.h"

#include "kcurve.h"
#include "ImageHistogram.h"

#include "ui_BatchProcessWidget.h"
#include "ui_ExistingFilesDialog.h"
#include "ui_HistogramDockerWidget.h"

#include "ui_WhiteBalanceOptions.h"
#include "ui_ColorOptions.h"
#include "ui_LightOptions.h"
#include "ui_NoiseReductionOptions.h"
#include "ui_QualityOptions.h"
#include "ui_LevelsOptions.h"
#include "ui_MessagesWidget.h"
#include "ui_OptionsBookmarksWidget.h"
#include "ui_ExportOptions.h"

struct Darkroom::Private {
  DarkroomView *view;
  QListView* fileListView;
  KToggleAction *toolbarAction;
  KToggleAction *statusbarAction;
  RawImageInfoSP currentRawFile;
  QProgressBar* progressBar;
  QTimeLine progressBarTimeLine;
  ListOptionsModel* listOptionsModel;
  Ui::HistogramDockerWidget histogramDockerWidget;
  Ui::WhiteBalanceOptions whiteBalanceOptions;
  Ui::ColorOptions colorOptions;
  Ui::LightOptions lightOptions;
  Ui::NoiseReductionOptions noiseReductionOptions;
  Ui::QualityOptions qualityOptions;
  Ui::OptionsBookmarksWidget optionsBookmarksDocker;
  Ui::ExportOptions exportOptions;
  KCurve* lightnessCurveWidget;
  QTimer delayedPreviewUpdate;
  ExportCodec* currentCodec;
#ifdef HAVE_OPENCTL
  KCurve* redCurveWidget;
  KCurve* greenCurveWidget;
  KCurve* blueCurveWidget;
  Ui::LevelsOptions levelsOptions;
#endif
  Ui::MessagesWidget messagesWidget;
  QDockWidget* messagesDockWidget;
};

Darkroom::Darkroom()
    : KXmlGuiWindow(), d(new Private)
{
  Icons::init();
    KGlobal::mainComponent().dirs()->addResourceType("icc_profiles", 0, "share/color/icc/");
 
    d->view = new DarkroomView(this);
    connect(&d->delayedPreviewUpdate, SIGNAL(timeout()), d->view, SLOT(updatePreview()));
    d->delayedPreviewUpdate.setSingleShot( true );
    d->delayedPreviewUpdate.setInterval(1000);
        
    d->currentRawFile = 0;
    d->listOptionsModel = new ListOptionsModel;
    d->listOptionsModel->loadFromConfigGroup( KConfigGroup(KGlobal::config(), "OptionsBookmark"));
    connect( d->listOptionsModel, SIGNAL(optionsChanged()), this, SLOT(optionsBookmarkChanged()));
    
    d->currentCodec = 0;
    // accept dnd
    setAcceptDrops(true);

    // tell the KXmlGuiWindow that this is indeed the main widget
    setCentralWidget(d->view);

    // then, setup our actions
    setupActions();

    // add a status bar
    statusBar()->show();
    
    // Create the progress bar
    d->progressBar = new QProgressBar(statusBar());
    d->progressBar->setMinimum( 0 );
    d->progressBar->setMaximum( 100 );
    d->progressBar->setVisible( false );
    d->progressBar->setTextVisible( false );
    d->progressBar->setMaximumWidth( 200 );
    d->progressBarTimeLine.setDuration( 10000 );
    d->progressBarTimeLine.setFrameRange( 0, 100 );
    d->progressBarTimeLine.start();
    d->progressBarTimeLine.setLoopCount( 0 );
    connect( &d->progressBarTimeLine, SIGNAL(frameChanged ( int ) ), d->progressBar, SLOT(setValue( int ) ) );
    statusBar()->addPermanentWidget( d->progressBar );

    // a call to KXmlGuiWindow::setupGUI() populates the GUI
    // with actions, using KXMLGUI.
    // It also applies the saved mainwindow settings, if any, and ask the
    // mainwindow to automatically save settings if changed: window size,
    // toolbar position, icon size, etc.
    setupGUI();
    
    // Initialize dockers
    setupDockers();
    setCorner(Qt::TopLeftCorner, Qt::LeftDockWidgetArea );
    setCorner(Qt::BottomLeftCorner, Qt::LeftDockWidgetArea );
    setCorner(Qt::TopRightCorner, Qt::RightDockWidgetArea );
    setCorner(Qt::BottomRightCorner, Qt::RightDockWidgetArea );
    
    // Initialize weaver observer
    observeWeaver( JobPreview::weaver() );
    observeWeaver( JobExport::weaver() );
    observeWeaver( JobThumbnail::weaver() );
    
    const int scnum = QApplication::desktop()->screenNumber(parentWidget());
    QRect desk = QApplication::desktop()->screenGeometry(scnum);

    // if the desktop is virtual then use virtual screen size
    if (QApplication::desktop()->isVirtualDesktop())
        desk = QApplication::desktop()->screenGeometry(QApplication::desktop()->screen());
    
    // Default size
    const int deskWidth = desk.width();
    if (deskWidth > 1100) // very big desktop ?
        resize( 1000, 800 );
    if (deskWidth > 850) // big desktop ?
        resize( 800, 600 );
    else // small (800x600, 640x480) desktop
        resize( 600, 400 );

    KConfigGroup config ( KGlobal::config(), "Darkroom" );
    if( not config.hasKey( "State" ) )
    {
      QString state = "AAAA/wAAAAD9AAAAAwAAAAAAAAE7AAAEI/wCAAAABPwAAABHAAAAdQAAAHUBAAAe+gAAAAABAAAAAvsAAAAeAFEAdQBhAGwAaQB0AHkAIABPAHAAdABpAG8AbgBzAQAAAAD/////AAAAiQD////7AAAALgBOAG8AaQBzAGUAIABSAGUAZAB1AGMAdABpAG8AbgAgAE8AcAB0AGkAbwBuAHMBAAAAAP////8AAAEgAP////wAAADCAAAAtwAAALcBAAAe+gAAAAABAAAABPsAAAAqAFcAaABpAHQAZQAgAEIAYQBsAGEAbgBjAGUAIABPAHAAdABpAG8AbgBzAQAAAAD/////AAABOwD////7AAAAGgBMAGkAZwBoAHQAIABPAHAAdABpAG8AbgBzAQAAAAD/////AAABIAD////7AAAAFgBSAEEAVwAgAE8AcAB0AGkAbwBuAHMBAAAAAP////8AAAAAAAAAAPsAAAAaAEMAbwBsAG8AcgAgAE8AcAB0AGkAbwBuAHMBAAAAAAAAATsAAAEcAP////wAAAF/AAABoAAAATUBAAAe+gAAAAABAAAAA/sAAAAcAEUAeABwAG8AcgB0ACAATwBwAHQAaQBvAG4AcwEAAAAA/////wAAATQA////+wAAAAwATABlAHYAZQBsAHMBAAAAAP////8AAAEIAP////sAAAASAEgAaQBzAHQAbwBnAHIAYQBtAQAAAAAAAAE7AAAAkAD////8AAADJQAAAUUAAABnAQAAHvoAAAAAAQAAAAT7AAAAHgBMAGkAZwBoAHQAbgBlAHMAcwAgAEMAdQByAHYAZQEAAAAA/////wAAAJYA////+wAAABIAUgBlAGQAIABDAHUAcgB2AGUBAAAAAP////8AAACWAP////sAAAAWAEcAcgBlAGUAbgAgAEMAdQByAHYAZQEAAAAA/////wAAAJYA////+wAAABQAQgBsAHUAZQAgAEMAdQByAHYAZQEAAAAAAAABOwAAAJYA////AAAAAQAAAQgAAAQj/AIAAAAB+wAAACAATwBwAHQAaQBvAG4AcwAgAEIAbwBvAGsAbQBhAHIAawEAAABHAAAEIwAAAHkA////AAAAAgAABTEAAACg/AEAAAAB+wAAABoAQgBhAHQAYwBoACAAUAByAG8AYwBlAHMAcwEAAAFBAAAFMQAAAEcA////AAAFMQAAA30AAAABAAAAAgAAAAEAAAAC/AAAAAEAAAACAAAAAQAAABYAbQBhAGkAbgBUAG8AbwBsAEIAYQByAQAAAAAAAAeAAAAAAAAAAAA=";
      config.writeEntry("State", state);
    }
    
    setAutoSaveSettings( "Darkroom" );
    
    // Dunno why, but  KWindowSystem::setState( winId(), state ); make Darkroom becomes smaller instead of staying big, TODO contact David Faure about this when he is around again.

    const QSize size( config.readEntry( QString::fromLatin1("Width %1").arg(desk.width()), 0 ),
                config.readEntry( QString::fromLatin1("Height %1").arg(desk.height()), 0 ) );
    resize( size );
}

Darkroom::~Darkroom()
{
  cleanUp();
  delete d;
}

void Darkroom::setupActions()
{
    KStandardAction::open(this, SLOT(fileOpen()), actionCollection());

    // custom menu and menu item - the slot is in the class DarkroomView
    KAction *custom = new KAction(KIcon("document-open-folder"), i18n("Open &directory..."), this);
    actionCollection()->addAction( QLatin1String("open_directory_action"), custom );
    connect(custom, SIGNAL(triggered(bool)), this, SLOT(fileOpenDir()));

    KStandardAction::saveAs(d->view, SLOT(fileSaveAs()), actionCollection())->setShortcut( KShortcut( Qt::CTRL + Qt::Key_S ) );
    KStandardAction::quit(qApp, SLOT(closeAllWindows()), actionCollection());

    // Zoom actions
    KStandardAction::zoomIn(d->view, SLOT(zoomIn()), actionCollection());
    KStandardAction::zoomOut(d->view, SLOT(zoomOut()), actionCollection());
    KStandardAction::fitToPage(d->view, SLOT(fitToPage()), actionCollection());
    KStandardAction::actualSize(d->view, SLOT(actualSize()), actionCollection());
    
    // Refresh preview
    KAction* refreshPreviewAction = new KAction(KIcon("view-refresh"), i18n("Refresh preview"), this );
    actionCollection()->addAction( QLatin1String("refresh_preview"), refreshPreviewAction );
    connect(refreshPreviewAction, SIGNAL(triggered(bool)), d->view, SLOT(updatePreview()));
}

void Darkroom::createCurveWidget( KCurve** curveWidget, const QString& dockTitle, const QString& objectName )
{
  QDockWidget* curveDockWidget = new QDockWidget(dockTitle, this);
  curveDockWidget->setObjectName( objectName );
  QWidget* curveDockerWidget = new QWidget( curveDockWidget );
  *curveWidget = new KCurve( curveDockerWidget );
  curveDockWidget->setWidget( *curveWidget );
  connect( *curveWidget, SIGNAL(modified()), this, SLOT(optionsChanged()) );
  addDockWidget( Qt::LeftDockWidgetArea, curveDockWidget );
}

void Darkroom::setupDockers()
{
  // Setup Raw options docker
  QDockWidget* whiteBalanceOptionsDockWidget = new QDockWidget(i18n("White Balance"), this);
  whiteBalanceOptionsDockWidget->setObjectName( "White Balance Options" );
  QWidget* whiteBalanceOptionsWidget = new QWidget;
  whiteBalanceOptionsDockWidget->setWidget( whiteBalanceOptionsWidget );
  d->whiteBalanceOptions.setupUi( whiteBalanceOptionsWidget );
  d->whiteBalanceOptions.temperatureValue->setRange( 2000, 12000, 1000 );
  d->whiteBalanceOptions.temperatureValue->setSliderEnabled( true );
  d->whiteBalanceOptions.tintValue->setRange( 1.0, 2.5, 0.1, true );
  connect( d->whiteBalanceOptions.temperatureValue, SIGNAL(valueChanged(int)), this, SLOT(optionsChanged()) );
  connect( d->whiteBalanceOptions.tintValue, SIGNAL(valueChanged(double)), this, SLOT(optionsChanged()) );
  connect( d->whiteBalanceOptions.whiteBalanceType, SIGNAL(currentIndexChanged(int)), this, SLOT(slotWhiteBalanceChanged( int)) );
  slotWhiteBalanceChanged( d->whiteBalanceOptions.whiteBalanceType->currentIndex());
  addDockWidget(Qt::LeftDockWidgetArea, whiteBalanceOptionsDockWidget);
  
  // Setup Raw options docker
  QDockWidget* qualityOptionsDockWidget = new QDockWidget(i18n("Quality"), this);
  qualityOptionsDockWidget->setObjectName( "Quality Options" );
  QWidget* qualityOptionsWidget = new QWidget;
  d->qualityOptions.setupUi( qualityOptionsWidget );
  connect( d->qualityOptions.demosaicQualityType, SIGNAL(currentIndexChanged(int)), this, SLOT(optionsChanged()) );
  qualityOptionsDockWidget->setWidget( qualityOptionsWidget );
  addDockWidget(Qt::LeftDockWidgetArea, qualityOptionsDockWidget);

  // Setup Color Options docker
  QDockWidget* colorOptionsDockWidget = new QDockWidget(i18n("Color"), this);
  colorOptionsDockWidget->setObjectName( "Color Options" );
  QWidget* colorOptionsWidget = new QWidget;
  colorOptionsDockWidget->setWidget( colorOptionsWidget );
  d->colorOptions.setupUi( colorOptionsWidget );
  connect(d->colorOptions.convertToSRGB, SIGNAL(clicked(bool)), this, SLOT(optionsChanged()) );
  connect(d->colorOptions.enableChromaticAberrationCorrection, SIGNAL(clicked(bool)), this, SLOT(optionsChanged()) );
  connect(d->colorOptions.redMultiplierValue, SIGNAL(valueChanged(double)), this, SLOT(optionsChanged()) );
  connect(d->colorOptions.blueMultiplierValue, SIGNAL(valueChanged(double)), this, SLOT(optionsChanged()) );
  d->colorOptions.redMultiplierValue->setRange( 0.0, 2.0, 0.1, true );
  d->colorOptions.blueMultiplierValue->setRange( 0.0, 2.0, 0.1, true );
  slotEnableChromaticAberration( false );
  connect( d->colorOptions.enableChromaticAberrationCorrection, SIGNAL(clicked(bool)), this, SLOT(slotEnableChromaticAberration(bool)));
  addDockWidget(Qt::LeftDockWidgetArea, colorOptionsDockWidget);
  
  // Setup Light Options docker
  QDockWidget* lightOptionsDockWidget = new QDockWidget(i18n("Light"), this);
  lightOptionsDockWidget->setObjectName( "Light Options" );
  QWidget* lightOptionsWidget = new QWidget;
  lightOptionsDockWidget->setWidget( lightOptionsWidget );
  d->lightOptions.setupUi( lightOptionsWidget );
  connect(d->lightOptions.exposureValue, SIGNAL(valueChanged(double)), this, SLOT(optionsChanged()) );
  connect(d->lightOptions.highlightsType, SIGNAL(currentIndexChanged(int)), this, SLOT(optionsChanged()) );
  connect(d->lightOptions.highlightsLevelValue, SIGNAL(valueChanged(int)), this, SLOT(optionsChanged()) );
  d->lightOptions.exposureValue->setRange( -5, 5, 1.0, true );
  d->lightOptions.highlightsLevelValue->setRange( 0, 6, 1 );
  d->lightOptions.highlightsLevelValue->setSliderEnabled( true );
  slotHighlightsChanged( d->lightOptions.highlightsType->currentIndex() );
  connect( d->lightOptions.highlightsType, SIGNAL(currentIndexChanged(int)), this, SLOT(slotHighlightsChanged( int)) );
  addDockWidget(Qt::LeftDockWidgetArea, lightOptionsDockWidget);

  // Setup Noise Reduction Options docker
  QDockWidget* noiseReductionOptionsDockWidget = new QDockWidget(i18n("Noise Reduction"), this);
  noiseReductionOptionsDockWidget->setObjectName( "Noise Reduction Options" );
  QWidget* noiseReductionOptionsWidget = new QWidget;
  noiseReductionOptionsDockWidget->setWidget( noiseReductionOptionsWidget );
  d->noiseReductionOptions.setupUi( noiseReductionOptionsWidget );
  connect(d->noiseReductionOptions.enableNoiseReduction, SIGNAL(clicked(bool)), this, SLOT(optionsChanged()) );
  connect(d->noiseReductionOptions.thresholdNoiseReductionValue, SIGNAL(valueChanged(int)), this, SLOT(optionsChanged()) );
  d->noiseReductionOptions.thresholdNoiseReductionValue->setRange( 0, 250, 10 );
  d->noiseReductionOptions.thresholdNoiseReductionValue->setSliderEnabled( true );
  connect( d->noiseReductionOptions.enableNoiseReduction, SIGNAL(clicked(bool)), this, SLOT(slotEnableNoiseReduction(bool))); 
    slotEnableNoiseReduction( false );
  addDockWidget(Qt::LeftDockWidgetArea, noiseReductionOptionsDockWidget);
  
  // Setup Histogram dockers
  QDockWidget* histogramDockWidget = new QDockWidget(i18n("Histogram"), this);
  histogramDockWidget->setObjectName( "Histogram" );
  QWidget* histogramDockerWidget = new QWidget( histogramDockWidget );
  histogramDockWidget->setWidget( histogramDockerWidget );
  d->histogramDockerWidget.setupUi( histogramDockerWidget );
  connect( d->histogramDockerWidget.histogramType, SIGNAL(currentIndexChanged(int)), this, SLOT(slotHistogramTypeChanged(int)));
  addDockWidget( Qt::LeftDockWidgetArea, histogramDockWidget );
  
  // Setup Curve dockers
  createCurveWidget( &d->lightnessCurveWidget, i18n("Lightness Curve"), "Lightness Curve" );
#ifdef HAVE_OPENCTL
  createCurveWidget( &d->redCurveWidget, i18n("Red Curve"), "Red Curve" );
  createCurveWidget( &d->greenCurveWidget, i18n("Green Curve"), "Green Curve" );
  createCurveWidget( &d->blueCurveWidget, i18n("Blue Curve"), "Blue Curve" );
  
  // Setup Level docker
  QDockWidget* levelsDockWidget = new QDockWidget(i18n("Levels"), this);
  levelsDockWidget->setObjectName( "Levels" );
  QWidget* levelsDockerWidget = new QWidget( levelsDockWidget );
  levelsDockWidget->setWidget( levelsDockerWidget );
  d->levelsOptions.setupUi( levelsDockerWidget );
  
  d->levelsOptions.ingradient->showMiddleCursor( true );
  
  connect( d->levelsOptions.blackspin, SIGNAL(valueChanged(double)), SLOT(optionsChanged()));
  connect( d->levelsOptions.whitespin, SIGNAL(valueChanged(double)), SLOT(optionsChanged()));
  connect( d->levelsOptions.ingradient, SIGNAL(middleValueChanged(double)), SLOT(optionsChanged()));

  connect( d->levelsOptions.blackspin, SIGNAL(valueChanged(double)), d->levelsOptions.ingradient, SLOT(setLeftValue(double)));
  connect( d->levelsOptions.whitespin, SIGNAL(valueChanged(double)), d->levelsOptions.ingradient, SLOT(setRightValue(double)));

  connect( d->levelsOptions.ingradient, SIGNAL(leftValueChanged(double)), d->levelsOptions.blackspin, SLOT(setValue(double)));
  connect( d->levelsOptions.ingradient, SIGNAL(rightValueChanged(double)), d->levelsOptions.whitespin, SLOT(setValue(double)));
  connect( d->levelsOptions.ingradient, SIGNAL(middleValueChanged(double)), this, SLOT(slotGammaValueChanged(double)));


  connect( d->levelsOptions.outblackspin, SIGNAL(valueChanged(double)), SLOT(optionsChanged()));
  connect( d->levelsOptions.outwhitespin, SIGNAL(valueChanged(double)), SLOT(optionsChanged()));

  connect( d->levelsOptions.outblackspin, SIGNAL(valueChanged(double)), d->levelsOptions.outgradient, SLOT(setLeftValue(double)));
  connect( d->levelsOptions.outwhitespin, SIGNAL(valueChanged(double)), d->levelsOptions.outgradient, SLOT(setRightValue(double)));

  connect( d->levelsOptions.outgradient, SIGNAL(leftValueChanged(double)), d->levelsOptions.outblackspin, SLOT(setValue(double)));
  connect( d->levelsOptions.outgradient, SIGNAL(rightValueChanged(double)), d->levelsOptions.outwhitespin, SLOT(setValue(double)));

  addDockWidget( Qt::LeftDockWidgetArea, levelsDockWidget );
#endif
  
  // Setup Export Options Docker
  QDockWidget* exportOptionsDockWidget = new QDockWidget(i18n("Export"), this);
  exportOptionsDockWidget->setObjectName( "Export Options" );
  QWidget* exportOptionsWidget = new QWidget( exportOptionsDockWidget );
  d->exportOptions.setupUi( exportOptionsWidget );
  exportOptionsDockWidget->setWidget( exportOptionsWidget );
  
  connect( d->exportOptions.fileFormat, SIGNAL(currentIndexChanged(int)), this, SLOT(exportCodecChanged()));
  connect( d->exportOptions.fileFormat, SIGNAL(currentIndexChanged(int)), this, SLOT(exportOptionsChanged()));
  
  foreach( const QString& codecId, ExportCodec::codecs() )
  {
    ExportCodec* codec = ExportCodec::codec( codecId );
    QWidget* widget = codec->configurationWidget();
    if( widget )
    {
      connect(codec, SIGNAL(optionsChanged()), this, SLOT(exportOptionsChanged()));
      widget->setParent( exportOptionsWidget );
      d->exportOptions.gridLayout->addWidget( widget, d->exportOptions.gridLayout->rowCount() - 1, 0, 1, 2);
      widget->setVisible( false );
    }
    d->exportOptions.fileFormat->addItem( codec->name(), QVariant(codec->id()));
  }
  
  exportCodecChanged();
  addDockWidget( Qt::LeftDockWidgetArea, exportOptionsDockWidget );
  
  // Setup Configuration bookmark docker
  QDockWidget* optionsBookmarksDockWidget = new QDockWidget(i18n("Options Bookmark"), this);
  optionsBookmarksDockWidget->setObjectName( "Options Bookmark" );
  QWidget* optionsBookmarksWidget = new QWidget( optionsBookmarksDockWidget );
  d->optionsBookmarksDocker.setupUi( optionsBookmarksWidget );
  optionsBookmarksDockWidget->setWidget( optionsBookmarksWidget );
  d->optionsBookmarksDocker.listOptions->setModel( d->listOptionsModel );
  connect( d->optionsBookmarksDocker.listOptions, SIGNAL(activated(const QModelIndex&)), SLOT(bookmarkedOptionsActivated(const QModelIndex& ) ) );
  connect( d->optionsBookmarksDocker.saveOptions, SIGNAL(released()), this, SLOT(bookmarkCurrentOptions()));
  connect( d->optionsBookmarksDocker.deleteOptions, SIGNAL(released()), this, SLOT(deleteCurrentBookmark()));
  addDockWidget( Qt::RightDockWidgetArea, optionsBookmarksDockWidget );

  // Setup batchProcessor widget
  QDockWidget* batchProcessDockWidget = new QDockWidget(i18n("Batch Process"), this);
  batchProcessDockWidget->setObjectName( "Batch Process" );
  QWidget* batchProcessWidget = new QWidget( batchProcessDockWidget );
  batchProcessDockWidget->setWidget( batchProcessWidget );
  Ui::BatchProcessWidget* bpw = new Ui::BatchProcessWidget;
  bpw->setupUi( batchProcessWidget );
  d->fileListView = bpw->listRawFiles;
  d->fileListView->setContextMenuPolicy( Qt::CustomContextMenu );
  connect( bpw->listRawFiles, SIGNAL(activated(const QModelIndex&)), SLOT(rawFileActivated(const QModelIndex& ) ) );
  connect( d->fileListView, SIGNAL(customContextMenuRequested( const QPoint & ) ), this, SLOT(customFileViewContextMenuRequested( const QPoint & ) ));
  addDockWidget(Qt::TopDockWidgetArea, batchProcessDockWidget);
  
  // Setup messages docker
  d->messagesDockWidget = new QDockWidget(i18n("Messages"), this);
  d->messagesDockWidget->setObjectName("MessagesDockWidget");
  QWidget* messagesWidget = new QWidget(d->messagesDockWidget);
  d->messagesDockWidget->setWidget(messagesWidget);
  d->messagesWidget.setupUi(messagesWidget);
  d->messagesWidget.tableView->setModel(MessagesModel::instance());
  connect(MessagesModel::instance(), SIGNAL(errorReceived()), SLOT(errorReceived()));
  connect(MessagesModel::instance(), SIGNAL(messageReceived()),
          d->messagesWidget.tableView, SLOT(resizeColumnsToContents()));
  addDockWidget(Qt::BottomDockWidgetArea, d->messagesDockWidget);  
}

void Darkroom::rawFileActivated ( const QModelIndex & index )
{
  d->currentRawFile = 0;
  QVariant variant = d->fileListView->model()->data( index, 100);
  d->currentRawFile = variant.value<RawImageInfoSP>();
  if(d->currentRawFile) {
    ProcessingOptions po = processingOptions();
    kDebug() << po.toXml();
    d->currentRawFile->setProcessingOptions( po );
    d->view->setRawImageInfo( d->currentRawFile );
    d->listOptionsModel->changeRawImageInfo( d->currentRawFile );
  }
}

void Darkroom::cleanUp()
{
  if( d->fileListView->model() )
  {
    QAbstractItemModel* model = d->fileListView->model();
    d->fileListView->setModel( 0 );
    delete model;
  }
}

void Darkroom::fileOpenDir()
{
  cleanUp();
  QString directoryName = KFileDialog::getExistingDirectory();
  if( not directoryName.isEmpty() and not directoryName.isNull())
  {
    openDir( directoryName );
  }
}

void Darkroom::openDir( const QString& _directoryName )
{
  d->currentRawFile = 0;
  d->fileListView->setEnabled( true );
  QDir directory( _directoryName );
  QStringList files = directory.entryList( QString(raw_file_extentions).split(" "), QDir::Readable | QDir::Files, QDir::Name | QDir::IgnoreCase );
  QList<RawImageInfoSP> entries;
  foreach(QString file, files)
  {
    entries.push_back( RawImageInfoSP( new RawImageInfo( QFileInfo( directory, file ) )  ) );
  }
  d->fileListView->setModel( new ListRawFileModel( entries) );
}

void Darkroom::fileOpen()
{
  cleanUp();
  QString file = KFileDialog::getOpenFileName( KUrl(), raw_file_extentions );
  if( not file.isEmpty() and not file.isNull() )
  {
    openFile( file );
  }
}

void Darkroom::openFile( const QString& _file )
{
  d->currentRawFile = new RawImageInfo( QFileInfo( _file ) );
  d->currentRawFile->setProcessingOptions( processingOptions() );
  d->fileListView->setEnabled( false );
  d->view->setRawImageInfo( d->currentRawFile );
  d->listOptionsModel->changeRawImageInfo( d->currentRawFile );
}

void Darkroom::openUrl(const QString& _url)
{
  QFileInfo fi(_url);
  if( fi.isDir() )
  {
    openDir( _url );
  } else if( fi.isFile() )
  {
    openFile( _url );
  } else {
    KMessageBox::error( this, i18n("Cannot open file %1.", _url) );
  }
}

QList<QVariant> Darkroom::curveToList( KCurve * _curve)
{
  QList<QVariant> lightnessCurve;
  foreach(QPointF point, _curve->getCurve())
  {
    lightnessCurve.push_back( point );
  }
  return lightnessCurve;
}

ProcessingOptions Darkroom::processingOptions()
{
  ProcessingOptions processingOptions;
  processingOptions.setOption( "", true);
  // Set RAW options
  processingOptions.setOption( "WhiteBalance", d->whiteBalanceOptions.whiteBalanceType->currentIndex());
  processingOptions.setOption( "Temperature", d->whiteBalanceOptions.temperatureValue->value());
  processingOptions.setOption( "Tint", d->whiteBalanceOptions.tintValue->value());
  
  // Set Quality options
  processingOptions.setOption( "DemosaicQuality", d->qualityOptions.demosaicQualityType->currentIndex());
  
  // Set Light options
  processingOptions.setOption( "HighlightsType", d->lightOptions.highlightsType->currentIndex());
  processingOptions.setOption( "LevelValue", d->lightOptions.highlightsLevelValue->value());
  processingOptions.setOption( "Exposure", d->lightOptions.exposureValue->value());
  
  // Set Color options
  processingOptions.setOption( "ConvertToSRGB", d->colorOptions.convertToSRGB->isChecked() );
  processingOptions.setOption( "ChromaticAberrationCorrection", d->colorOptions.enableChromaticAberrationCorrection->isChecked() );
  processingOptions.setOption( "RedMultiplier", d->colorOptions.redMultiplierValue->value() );
  processingOptions.setOption( "GreenMultiplier", d->colorOptions.blueMultiplierValue->value() );
  
  // Set Noise Reduction options
  processingOptions.setOption( "EnableNoiseReduction", d->noiseReductionOptions.enableNoiseReduction->isChecked() );
  processingOptions.setOption( "ThresholdNoiseReduction", d->noiseReductionOptions.thresholdNoiseReductionValue->value() );
  
  // Set Curve options
  processingOptions.setOption("LightnessCurve", curveToList(  d->lightnessCurveWidget ) );
#ifdef HAVE_OPENCTL
  processingOptions.setOption("RedCurve", curveToList(  d->redCurveWidget ) );
  processingOptions.setOption("GreenCurve", curveToList(  d->greenCurveWidget ) );
  processingOptions.setOption("BlueCurve", curveToList(  d->blueCurveWidget ) );
  
  // Set Levels options
  processingOptions.setOption("InputBlackValue", d->levelsOptions.ingradient->leftValue() );
  processingOptions.setOption("InputWhiteValue", d->levelsOptions.ingradient->rightValue() );
  processingOptions.setOption("InputGamma", computeGamma() );
  processingOptions.setOption("OutputBlackValue", d->levelsOptions.outgradient->leftValue() );
  processingOptions.setOption("OutputWhiteValue", d->levelsOptions.outgradient->rightValue() );
#endif
  processingOptions.setOption("CodecId", d->currentCodec->id() );
  d->currentCodec->fillProcessingOptions( &processingOptions );
  return processingOptions;
}

void Darkroom::setProcessingOptions(const ProcessingOptions& _processingOptions)
{
  d->whiteBalanceOptions.whiteBalanceType->setCurrentIndex( _processingOptions.asInteger( "WhiteBalance"));
  d->whiteBalanceOptions.temperatureValue->setValue( _processingOptions.asInteger( "Temperature"));
  d->whiteBalanceOptions.tintValue->setValue( _processingOptions.asInteger( "Tint" ));

  // Set Quality options
  d->qualityOptions.demosaicQualityType->setCurrentIndex(_processingOptions.asInteger( "DemosaicQuality"));
  
  // Set Light options
  d->lightOptions.highlightsType->setCurrentIndex( _processingOptions.asInteger( "HighlightsType" ) );
  d->lightOptions.highlightsLevelValue->setValue( _processingOptions.asInteger( "LevelValue" ) );
  d->lightOptions.exposureValue->setValue( _processingOptions.asDouble( "Exposure" ));
  
  // Set Color options
  d->colorOptions.convertToSRGB->setChecked( _processingOptions.asBool( "ConvertToSRGB" ) );
  d->colorOptions.enableChromaticAberrationCorrection->setChecked( _processingOptions.asBool( "ChromaticAberrationCorrection" ) );
  d->colorOptions.redMultiplierValue->setValue( _processingOptions.asDouble( "RedMultiplier" ) );
  d->colorOptions.blueMultiplierValue->setValue( _processingOptions.asDouble( "GreenMultiplier" ) );
  
  // Set Noise Reduction options
  d->noiseReductionOptions.enableNoiseReduction->setChecked( _processingOptions.asBool( "EnableNoiseReduction" ) );
  d->noiseReductionOptions.thresholdNoiseReductionValue->setValue( _processingOptions.asInteger( "ThresholdNoiseReduction" ) );
  
  // Set Curve options
  d->lightnessCurveWidget->setCurve( _processingOptions.asPointFList( "LightnessCurve" ) );
#ifdef HAVE_OPENCTL
  d->redCurveWidget->setCurve( _processingOptions.asPointFList( "RedCurve" ) );
  d->greenCurveWidget->setCurve( _processingOptions.asPointFList( "GreenCurve" ) );
  d->blueCurveWidget->setCurve( _processingOptions.asPointFList( "BlueCurve" ) );

  // Set Levels options
  d->levelsOptions.ingradient->setLeftValue( _processingOptions.asDouble("InputBlackValue" ) );
  d->levelsOptions.ingradient->setRightValue( _processingOptions.asDouble("InputWhiteValue" ) );
  setGamma( _processingOptions.asDouble("InputGamma"));
  d->levelsOptions.outgradient->setLeftValue( _processingOptions.asDouble("OutputBlackValue" ) );
  d->levelsOptions.outgradient->setRightValue( _processingOptions.asDouble("OutputWhiteValue" ) );

#endif
  QString codecId = _processingOptions.asString("CodecId", "PNG");
  d->exportOptions.fileFormat->setCurrentIndex( ExportCodec::codecs().indexOf( codecId ) );
  d->currentCodec->setProcessingOptions( _processingOptions );
}

HistogramWidget* Darkroom::histogramWidget()
{
  return d->histogramDockerWidget.histogramWidget;
}

void Darkroom::closeEvent(QCloseEvent *event)
{
  if( not JobExport::weaver()->isIdle() )
  {
    QMessageBox messageBox(QMessageBox::Question, i18n("Exit Darkroom"), i18n("Darkroom is still processing some images, are you sure you want to quit? All unfinished processing will be lost."), QMessageBox::Yes | QMessageBox::No, this);
    messageBox.addButton("Wait", QMessageBox::YesRole );
    messageBox.setDefaultButton( QMessageBox::No );
    int result = messageBox.exec();
    switch( result )
    {
      case QMessageBox::No:
        event->ignore();
        return;
        break;
      case QMessageBox::Yes:
        break;
      default:
      {
        setEnabled( false );
        while(not JobExport::weaver()->isIdle() )
        {
          QApplication::processEvents( );
          usleep(100);
        }
      }
    }
  }
  event->accept();
}

void Darkroom::customFileViewContextMenuRequested( const QPoint& pos )
{
  KMenu menu(d->fileListView);
  menu.addAction(i18n("Apply current settings."), this, SLOT(applyCurrentSettings()));
  menu.exec(d->fileListView->mapToGlobal(pos ));
}

void Darkroom::weaverStatusChanged()
{
  d->progressBar->setVisible( not idling() );
}

#define MAKE_FILE_NAME \
  QFileInfo info = fpo->fileInfo(); \
  QString dstFileName = info.absolutePath() + "/" + info.baseName() + "." + extension;


void Darkroom::applyCurrentSettings()
{
  kDebug() << "Darkroom";
  QList<RawImageInfoSP> riis;
  QStringList existingFiles;
  ProcessingOptions processingOptions_ = processingOptions();
  foreach( QModelIndex index, d->fileListView->selectionModel()->selectedIndexes () )
  {
    QVariant variant = d->fileListView->model()->data( index, 100);
    RawImageInfoSP fpo = variant.value<RawImageInfoSP>();
    QString extension = ExportCodec::codec( processingOptions_.asString("CodecId", "PNG") )->extension();
    MAKE_FILE_NAME
    if(fpo) {
      riis.push_back( fpo );
      if(QFileInfo( dstFileName).exists())
      {
        existingFiles.push_back( dstFileName );
      }
    }
  }
  if( not existingFiles.empty() )
  {
    QDialog dialog(this);
    Ui::ExistingFilesDialog efd;
    efd.setupUi( &dialog );
    foreach( QString existingFile, existingFiles)
    {
      efd.listExistingFiles->addItem( existingFile );
    }
    if( dialog.exec() == QDialog::Rejected )
    {
      return;
    }
  }
  foreach( RawImageInfoSP fpo, riis )
  {
    QString extension = ExportCodec::codec( processingOptions_.asString("CodecId", "PNG") )->extension();
    MAKE_FILE_NAME
    fpo->setProcessingOptions( processingOptions_ );
    JobExport::weaver()->enqueue( new JobExport( fpo, dstFileName) );
  }
}

bool Darkroom::idling()
{
  return  JobPreview::weaver()->isIdle() and
          JobExport::weaver()->isIdle() and
          JobThumbnail::weaver()->isIdle();
}

void Darkroom::observeWeaver( ThreadWeaver::Weaver* weaver )
{
  ThreadWeaver::WeaverObserver* observer = new ThreadWeaver::WeaverObserver( this );
  weaver->registerObserver( observer );
  connect( observer, SIGNAL(threadBusy (ThreadWeaver::Thread *, ThreadWeaver::Job * )), SLOT(weaverStatusChanged()));
  connect( observer, SIGNAL(threadExited (ThreadWeaver::Thread *)), SLOT(weaverStatusChanged()));
  connect( observer, SIGNAL(threadStarted (ThreadWeaver::Thread *)), SLOT(weaverStatusChanged()));
  connect( observer, SIGNAL(weaverStateChanged (ThreadWeaver::State *)), SLOT(weaverStatusChanged()));
  connect( observer, SIGNAL(threadSuspended (ThreadWeaver::Thread *)), SLOT(weaverStatusChanged()));
}

void Darkroom::slotWhiteBalanceChanged(int _wb)
{
  if( _wb == 3 )
  {
    d->whiteBalanceOptions.temperatureValue->setEnabled( true );
    d->whiteBalanceOptions.labelTemperature->setEnabled( true );
    d->whiteBalanceOptions.tintValue->setEnabled( true );
    d->whiteBalanceOptions.labelTint->setEnabled( true );
  } else {
    d->whiteBalanceOptions.temperatureValue->setEnabled( false );
    d->whiteBalanceOptions.labelTemperature->setEnabled( false );
    d->whiteBalanceOptions.tintValue->setEnabled( false );
    d->whiteBalanceOptions.labelTint->setEnabled( false );
  }
}

void Darkroom::slotEnableChromaticAberration( bool  v)
{
  d->colorOptions.labelRedMultiplier->setEnabled( v );
  d->colorOptions.labelGreenMultiplier->setEnabled( v );
  d->colorOptions.redMultiplierValue->setEnabled( v );
  d->colorOptions.blueMultiplierValue->setEnabled( v );
}

void Darkroom::slotHighlightsChanged(int _hg)
{
  if( _hg == 3 )
  {
    d->lightOptions.highlightsLevelLabel->setEnabled( true );
    d->lightOptions.highlightsLevelValue->setEnabled( true );
  } else {
    d->lightOptions.highlightsLevelLabel->setEnabled( false );
    d->lightOptions.highlightsLevelValue->setEnabled( false );
  }
}

void Darkroom::slotEnableNoiseReduction( bool v )
{
  d->noiseReductionOptions.thresholdNoiseReductionLabel->setEnabled( v );
  d->noiseReductionOptions.thresholdNoiseReductionValue->setEnabled( v );
}

void Darkroom::slotHistogramTypeChanged(int _value)
{
  if( _value == 4 )
  {
    d->histogramDockerWidget.histogramWidget->m_channelType = HistogramWidget::ColorChannelsHistogram;
  } else {
    d->histogramDockerWidget.histogramWidget->m_channelType = _value;
  }
  d->histogramDockerWidget.histogramWidget->update();
}

#ifdef HAVE_OPENCTL
void Darkroom::slotGammaValueChanged(double)
{
  d->levelsOptions.gammaspin->setNum( computeGamma() );
}

double Darkroom::computeGamma() const
{
  double delta = (double) ( d->levelsOptions.ingradient->rightValue() - d->levelsOptions.ingradient->leftValue() ) / 2.0;
  double mid = (double)d->levelsOptions.ingradient->leftValue() + delta;
  double tmp = (d->levelsOptions.ingradient->middleValue() - mid) / delta;
  return 1.0 / pow (10, tmp);
}

void Darkroom::setGamma( double g)
{
  double delta = (double) ( d->levelsOptions.ingradient->rightValue() - d->levelsOptions.ingradient->leftValue() ) / 2.0;
  double mid = (double)d->levelsOptions.ingradient->leftValue() + delta;
  double tmp = log10( 1.0 / g);
  d->levelsOptions.ingradient->setMiddleValue( mid + tmp * delta );
}
#endif

void Darkroom::optionsChanged()
{
  if( d->currentRawFile )
  {
    d->currentRawFile->setProcessingOptions( processingOptions() );
    d->delayedPreviewUpdate.start();
  }
}

void Darkroom::exportOptionsChanged()
{
  if( d->currentRawFile )
  {
    d->currentRawFile->setProcessingOptions( processingOptions() );
  }
}

void Darkroom::bookmarkCurrentOptions()
{
  if( d->currentRawFile and d->currentRawFile->previewInfo() )
  {
    d->listOptionsModel->saveOption( "", processingOptions(), d->currentRawFile->previewInfo()->asQImage() );
  } else {
    d->listOptionsModel->saveOption( "", processingOptions(), KIcon("image-x-dcraw").pixmap(100).toImage() );
  }
}

void Darkroom::bookmarkedOptionsActivated ( const QModelIndex & index )
{
  QVariant variant = d->optionsBookmarksDocker.listOptions->model()->data( index, 100);
  setProcessingOptions( variant.value<ProcessingOptions>() );
}

void Darkroom::deleteCurrentBookmark()
{
  d->listOptionsModel->removeBookmarkAt( d->optionsBookmarksDocker.listOptions->currentIndex() );
}

void Darkroom::optionsBookmarkChanged()
{
  KConfigGroup group( KGlobal::config(), "OptionsBookmark");
  d->listOptionsModel->saveToConfigGroup( group );
}

void Darkroom::exportCodecChanged()
{
  if(d->currentCodec and d->currentCodec->configurationWidget() )
  {
    d->currentCodec->configurationWidget()->setVisible( false );
  }
  d->currentCodec = ExportCodec::codec( d->exportOptions.fileFormat->itemData( d->exportOptions.fileFormat->currentIndex() ).toString() );
  Q_ASSERT( d->currentCodec );
  if(d->currentCodec->configurationWidget() )
  {
    d->currentCodec->configurationWidget()->setVisible( true );
  }
}

#ifdef HAVE_OPENCTL
void Darkroom::setIntermediateData( uchar *data, uint w, uint h )
{
  kDebug() << "========== Setting interemdiate data";
  d->levelsOptions.histogramWidget->updateData( data, w, h, true, false );
  d->levelsOptions.histogramWidget->m_channelType = HistogramWidget::ValueHistogram;
  while( d->levelsOptions.histogramWidget->m_imageHistogram->isCalculating() )
  {
    usleep(10);
  }
  d->levelsOptions.histogramWidget->update();
}
#endif

void Darkroom::errorReceived()
{
  d->messagesDockWidget->raise();
}

#include "Darkroom.moc"
