/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "ExportCodec.h"

#include <QMap>
#include <QMutex>
#include <QWidget>

struct ExportCodec::Private {
  mutable QWidget* widget;
  QString id;
  QString name;
  QString extension;
  static bool codecsInited;
  static QMap<QString, ExportCodec*> codecs;
  static void initCodecs();
  static QMutex* exiv2Mutex;
};

bool ExportCodec::Private::codecsInited = false;
QMap<QString, ExportCodec*> ExportCodec::Private::codecs;
QMutex* ExportCodec::Private::exiv2Mutex = 0;

ExportCodec::ExportCodec(const QString& _id, const QString& _name, const QString& _extension) : d( new Private)
{
  d->widget = 0;
  d->id = _id;
  d->name = _name;
  d->extension = _extension;
}
ExportCodec::~ExportCodec()
{
  delete d->widget;
  delete d;
}

const QString& ExportCodec::id() const
{
  return d->id;
}

const QString& ExportCodec::name() const
{
  return d->name;
}

const QString& ExportCodec::extension() const
{
  return d->extension;
}

QWidget* ExportCodec::configurationWidget() const
{
  return d->widget;
}

void ExportCodec::setConfigurationWidget(QWidget* _widget) const
{
  d->widget = _widget;
}

ExportCodec* ExportCodec::codec( const QString& _id )
{
  if( not Private::codecsInited) Private::initCodecs();
  return Private::codecs[ _id ];
}

QList<QString> ExportCodec::codecs()
{
  if( not Private::codecsInited) Private::initCodecs();
  return Private::codecs.keys();
}

QMutex* ExportCodec::exiv2Mutex()
{
  if(not Private::exiv2Mutex)
  {
    Private::exiv2Mutex = new QMutex;
  }
  return Private::exiv2Mutex;
}

#include "codecs/PngExportCodec.h"
#include "codecs/JpegExportCodec.h"

#define ADD_CODEC( _NAME_ ) \
    _NAME_* _NAME_##codec = new _NAME_; codecs[ _NAME_##codec->id() ] = _NAME_##codec;

void ExportCodec::Private::initCodecs()
{
  ADD_CODEC( PngExportCodec );
  ADD_CODEC( JpegExportCodec );
  Private::codecsInited = true;
}