/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file rl.c
 * \brief Reverse lookup wrapper function
 */

#include <ffgtk.h>

/** Global lookup list */
static GList *psLookupList = NULL;
/** Current lookup version */
static gchar *pnCurrentVersion = NULL;
/** Lookup country code hash table */
static GHashTable *psLookupTable = NULL;

/**
 * \brief Get lookup list
 * \param pnCountryCode country code
 * \return lookup list
 */
GList *getLookupList( const gchar *pnCountryCode ) {
	GList *psList = NULL;

	/* If country code is NULL, return current lookup list */
	if ( pnCountryCode == NULL ) {
		return psLookupList;
	}

	if ( psLookupTable == NULL ) {
		return NULL;
	}

	/* Extract country code list from hashtable */
	psList = g_hash_table_lookup( psLookupTable, ( gpointer ) atol( pnCountryCode ) );

	return psList;
}

/**
 * \brief Extract country code from full number
 * \param pnFullNumber full international number
 * \return country code or NULL on error
 */
gchar *getCountryCode( const gchar *pnFullNumber ) {
	gchar *pnSubString = NULL;
	int nIndex;
	int nLen = strlen( pnFullNumber );

	for ( nIndex = 6; nIndex > 0; nIndex-- ) {
		if ( nLen > nIndex ) {
			pnSubString = getSubString( ( char * ) pnFullNumber, 0, nIndex );
			if ( pnSubString != NULL ) {
				if ( g_hash_table_lookup( psLookupTable, ( gpointer ) atol( pnSubString ) ) ) {
					return pnSubString;
				}
				g_free( pnSubString );
			}
		}
	}

	return NULL;
}

/**
 * \brief Reverse lookup wrapper
 * \param pnNumber telephone number
 * \param ppnFirstName where to store first name to
 * \param ppnLastName where to store last name to
 * \param ppnStreet where to store street to
 * \param ppnZipCode where to store zip code to
 * \param ppnCity where to store city to
 * \return error code
 */
int ReverseLookup( gchar *pnNumber, gchar **ppnFirstName, gchar **ppnLastName, gchar **ppnStreet, gchar **ppnZipCode, gchar **ppnCity ) {
	struct sLookup *psLookupDefault = NULL;
	struct sLookup *psLookup = NULL;
	struct sUrlHandler *psHandler = NULL;
	GList *psList = NULL;
	gchar *pnFullNumber = NULL;
	gchar **ppnLines = NULL;
	gchar *pnUrl = NULL;
	gchar *pnNewData = NULL;
	gchar *pnCountryCode = NULL;
	gint nError = -1;
	const gint nInternationalPrefixLen = strlen( routerGetInternationalPrefix( getActiveProfile() ) );
	gchar *pnName = NULL;
	gchar *pnZipCity = NULL;
	gchar *pnTmp = NULL;

	*ppnFirstName = NULL;
	*ppnLastName = NULL;
	*ppnStreet = NULL;
	*ppnZipCode = NULL;
	*ppnCity = NULL;

	/* Get full number and extract country code if possible */
	pnFullNumber = fullNumber( pnNumber, TRUE );
	if ( pnFullNumber != NULL ) {
		Debug( KERN_DEBUG, "Input number '%s'\n", pnNumber );
		Debug( KERN_DEBUG, "full number '%s'\n", pnFullNumber );
		pnCountryCode = getCountryCode( pnFullNumber );
		if ( pnCountryCode != NULL ) {
			Debug( KERN_DEBUG, "Country code: %s\n", pnCountryCode + nInternationalPrefixLen  );
		} else {
			Debug( KERN_DEBUG, "Warning: Could not get country code!!\n" );
		}
		g_free( pnFullNumber );
	}

	if ( pnCountryCode != NULL && strcmp( pnCountryCode + nInternationalPrefixLen, routerGetCountryCode( getActiveProfile() ) ) != 0 ) {
		/* if country code is not the same as the router country code, loop through country list */
		psLookupDefault = NULL;
		psList = getLookupList( pnCountryCode + nInternationalPrefixLen );
	} else {
		/* if country code is the same as the router country code, use default plugin */
		psLookupDefault = getActiveProfile() -> psDefaultLookup;
		psList = getLookupList( routerGetCountryCode( getActiveProfile() ) );
	}

	if ( pnCountryCode != NULL ) {
		g_free( pnCountryCode );
	}

	for ( ; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psLookup = psList -> data;

		/* if we have a default lookup service, skip everything else */
		if ( psLookupDefault != NULL && psLookup != psLookupDefault ) {
			continue;
		}

		/* Make sure we do not have any leftovers */
		if ( pnName != NULL ) {
			g_free( pnName );
			pnName = NULL;
		}
		if ( *ppnFirstName != NULL ) {
			g_free( *ppnFirstName );
			*ppnFirstName = NULL;
		}
		if ( *ppnLastName != NULL ) {
			g_free( *ppnLastName );
			*ppnLastName = NULL;
		}
		if ( *ppnStreet != NULL ) {
			g_free( *ppnStreet );
			*ppnStreet = NULL;
		}
		if ( pnZipCity != NULL ) {
			g_free( pnZipCity );
			pnZipCity = NULL;
		}
		if ( *ppnZipCode != NULL ) {
			g_free( *ppnZipCode );
			*ppnZipCode = NULL;
		}
		if ( *ppnCity != NULL ) {
			g_free( *ppnCity );
			*ppnCity = NULL;
		}

		Debug( KERN_DEBUG, "Using service '%s'\n", psLookup -> pnService );

		/* get full number according to service preferences */
		pnFullNumber = fullNumber( pnNumber, psLookup -> bPrefix );
		if ( pnFullNumber == NULL ) {
			goto error1;
		}

		pnUrl = replaceNumber( psLookup -> pnUrl, "", pnFullNumber, NULL );

		Debug( KERN_DEBUG, "URL: %s\n", pnUrl );

		/* create url handler */	
		psHandler = urlHandler( pnUrl, 80 );
		/* set timeout to 5sec */
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_TIMEOUT, 5 );
		/* try to get data */
		nError = readUrl( psHandler, NULL );
		if ( nError != 0 ) {
			Debug( KERN_DEBUG, "Error: %s\n", curl_easy_strerror( nError ) );
			goto error1;
		}

		/* convert charset */
		pnTmp = g_convert( psHandler -> pnData, psHandler -> nSize, "UTF-8", "iso-8859-1", NULL, NULL, NULL );
		if ( pnTmp == NULL ) {
			Debug( KERN_WARNING, "pnTmp is NULL!!\n" );
			goto error1;
		}
		pnNewData = convertEntities( pnTmp );
		if ( pnNewData == NULL ) {
			Debug( KERN_WARNING, "pnNewData is NULL!!\n" );
			g_free( pnTmp );
			goto error1;
		}

		if ( psLookup -> eLookupType == perLine ) {
			/* use separate regexps on array of lines */
			/* split data in lines */

			ppnLines = g_strsplit( pnNewData, "\n", -1 );
			if ( ppnLines == NULL ) {
				Debug( KERN_WARNING, "ppnLines is NULL!!\n" );
				goto error1;
			}

			/* extract data */
			if ( psLookup -> pnName != NULL ) {
				pnName = extractInformation( ppnLines, psLookup -> pnName );
				if ( pnName != NULL && strlen( pnName ) != 0 ) {

					/* split name into first/last name */
					int nPos = findString( ( gchar * ) pnName, 0, " " );
					if ( nPos < 0 ) {
						nPos = 0;
					}
					
					if ( ppnFirstName != NULL ) {
						*ppnFirstName = getSubString( (gchar *) pnName, 0, nPos );
					}
					if ( ppnLastName != NULL ) {
						*ppnLastName = g_strdup( pnName + nPos + 1 );
					}
					g_free(pnName);
					pnName = NULL;
				}		
			}

			if ( ( ppnStreet != NULL ) && ( psLookup -> pnStreet != NULL ) ) {
				*ppnStreet = extractInformation( ppnLines, psLookup -> pnStreet );
			}

			if ( psLookup -> pnCity != NULL ) {
				pnZipCity = extractInformation( ppnLines, psLookup -> pnCity );
				/* split into Zip and city */
				if ( ( pnZipCity != NULL ) && ( strlen( pnZipCity ) != 0 ) ) {
					int nPos = findString( ( gchar * ) pnZipCity, 0, " " );
					if ( nPos < 0 ) {
						nPos = 0;
					}
					if ( ppnZipCode != NULL ) {
						*ppnZipCode = getSubString( (gchar *) pnZipCity, 0, nPos );
					}
					if ( ppnCity != NULL) {
						if ( nPos != 0 ) {
							nPos++;
						}

						*ppnCity = g_strdup( pnZipCity + nPos );
					}
				}
			}

			g_strfreev( ppnLines );
		} else {
			/* perform lookup using named regexp pattern */
			extractInformationNamed( pnNewData, ppnFirstName, ppnLastName, ppnStreet, ppnZipCode, ppnCity, psLookup -> pnPattern );
		}

error1:
		if ( psHandler != NULL ) {
			freeHandler( psHandler );
			psHandler = NULL;
		}

		if ( pnNewData != NULL ) {
			g_free( pnNewData );
			pnNewData = NULL;
		}
		if ( pnUrl != NULL ) {
			g_free( pnUrl );
			pnUrl = NULL;
		}
		if ( pnFullNumber != NULL ) {
			g_free( pnFullNumber );
			pnFullNumber = NULL;
		}

		/* in case we found some valid data, break loop */
		if ( *ppnLastName != NULL ) {

			/* but first set empty strings when no data is found) */
			if (ppnFirstName != NULL && *ppnFirstName == NULL ) {
				*ppnFirstName = g_strdup( "" );
			}
			if (ppnStreet != NULL && *ppnStreet == NULL) {
				*ppnStreet = g_strdup( "" );
			}
			if (ppnZipCode != NULL && *ppnZipCode == NULL) {
				*ppnZipCode = g_strdup( "" );
			}
			if (ppnCity != NULL && *ppnCity == NULL) {
				*ppnCity = g_strdup( "" );
			}
			break;
		} else {
			nError = -1;
		}
	}

	return nError;
}

/**
 * \brief Add lookup from xmlnode
 * \param psNode xml node structure
 */
static void addLookup( xmlnode *psNode ) {
	struct sLookup *psLookup = NULL;
	xmlnode *psChild = NULL;
	gchar *pnService = NULL;
	gchar *pnPrefix = NULL;
	gchar *pnUrl = NULL;
	gchar *pnName = NULL;
	gchar *pnStreet = NULL;
	gchar *pnCity = NULL;
	gchar *pnPattern = NULL;

	psChild = xmlnode_get_child( psNode, "service" );
	if ( psChild != NULL ) {
		pnService = xmlnode_get_data( psChild );

		psChild = xmlnode_get_child( psNode, "prefix" );
		if ( psChild != NULL ) {
			pnPrefix = xmlnode_get_data( psChild );

			psChild = xmlnode_get_child( psNode, "url" );
			if ( psChild != NULL ) {
				pnUrl = xmlnode_get_data( psChild );
				psChild = xmlnode_get_child(psNode, "pattern");

				if ( psChild != NULL ) {
					pnPattern = xmlnode_get_data( psChild );
					while ( ( psChild = xmlnode_get_next_twin( psChild ) ) != NULL ) {
						gchar *pnEntry = xmlnode_get_data( psChild );
						gchar *pnTmp = g_strconcat( pnPattern, "\\R", pnEntry, NULL );
						g_free( pnEntry );
						g_free( pnPattern );
						pnPattern = pnTmp;
					}

					psLookup = g_malloc0( sizeof( struct sLookup ) );

					if ( psLookup != NULL ) {
						Debug( KERN_DEBUG, "Service: '%s', prefix: %s\n", pnService, pnPrefix );
						psLookup -> pnService = g_strdup( pnService );
						psLookup -> bPrefix = pnPrefix[ 0 ] == '1';
						psLookup -> pnUrl = g_strdup( pnUrl );
						psLookup -> eLookupType = named;
						psLookup -> pnName = NULL;
						psLookup -> pnStreet = NULL;
						psLookup -> pnCity = NULL;
						psLookup -> pnPattern = g_strdup( pnPattern );
						psLookupList = g_list_prepend( psLookupList, psLookup );
					}

					g_free ( pnPattern );
			} else	
				psChild = xmlnode_get_child( psNode, "name" );
				if ( psChild != NULL ) {
					pnName = xmlnode_get_data( psChild );

					psChild = xmlnode_get_child( psNode, "street" );
					if ( psChild != NULL ) {
						pnStreet = xmlnode_get_data( psChild );

						psChild = xmlnode_get_child( psNode, "city" );
						if ( psChild != NULL ) {
							pnCity = xmlnode_get_data( psChild );

							psLookup = g_malloc0( sizeof( struct sLookup ) );

							if ( psLookup != NULL ) {
								Debug( KERN_DEBUG, "Service: '%s', prefix: %s\n", pnService, pnPrefix );
								psLookup -> pnService = g_strdup( pnService );
								psLookup -> bPrefix = pnPrefix[ 0 ] == '1';
								psLookup -> pnUrl = g_strdup( pnUrl );
								psLookup -> eLookupType = perLine;
								psLookup -> pnName = g_strdup( pnName );
								psLookup -> pnStreet = g_strdup( pnStreet );
								psLookup -> pnCity = g_strdup( pnCity );
								psLookup -> pnPattern = NULL;
								psLookupList = g_list_prepend( psLookupList, psLookup );
							}

							g_free( pnCity );
						}

						g_free( pnStreet );
					}

					g_free( pnName );
				}

				g_free( pnUrl );
			}
			
			g_free( pnPrefix );
		}

		g_free( pnService );
	}
}

/**
 * \brief Add country code from xmlnode
 * \param psNode xml node structure
 */
static void addCountryCode( xmlnode *psNode ) {
	xmlnode *psChild = NULL;
	const gchar *pnCode = NULL;

	pnCode = xmlnode_get_attrib( psNode, "code" );
	Debug( KERN_DEBUG, "Country Code: %s\n", pnCode );

	psLookupList = NULL;

	for ( psChild = xmlnode_get_child( psNode, "lookup" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
		addLookup( psChild );
	}

	g_hash_table_insert( psLookupTable, ( gpointer ) atol( pnCode ), psLookupList );
}

/**
 * \brief Check for lookup updates
 * \return TRUE if update is available, else FALSE
 */
static gboolean CheckForUpdate( void ) {
	struct sUrlHandler *psHandler = NULL;
	gchar *pnUrl = NULL;
	gchar *pnPath = NULL;
	gchar **ppnLines = NULL;
	gint nError = -1;
	gint nVersion = 0;
	gboolean bReturn = FALSE;

	/* Read latest-lookup file for version information */
	pnUrl = g_strdup( "http://www.tabos.org/ffgtk/download/latest-lookup" );

	/* create url handler */	
	psHandler = urlHandler( pnUrl, 80 );
	/* set timeout to 3sec */
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_TIMEOUT, 3 );
	/* try to get data */
	nError = readUrl( psHandler, NULL );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "Error: %s\n", curl_easy_strerror( nError ) );
		goto error1;
	}

	if ( strstr( psHandler -> pnData, "404" ) != NULL ) {
		Debug( KERN_WARNING, "404!\n" );
		goto error1;
	}

	/* split data in lines */
	ppnLines = g_strsplit( psHandler -> pnData, "\n", -1 );
	if ( ppnLines == NULL ) {
		Debug( KERN_WARNING, "ppnLines is NULL!!\n" );
		goto error1;
	}

    Debug( KERN_DEBUG, "Content: '%s'\n", ppnLines[ 0 ] );
    nVersion = atoi( ppnLines[ 0 ] );

	g_strfreev( ppnLines );

	Debug( KERN_DEBUG, "Version-Check: %d <-> %d\n", nVersion, atoi( pnCurrentVersion ) );

	if ( nVersion > atoi( pnCurrentVersion ) ) {
		freeHandler( psHandler );
		g_free( pnUrl );

		/* read lookup file */
		pnUrl = g_strdup( "http://www.tabos.org/ffgtk/download/lookup.xml" );

		/* create url handler */	
		psHandler = urlHandler( pnUrl, 80 );
		/* try to get data */
		nError = readUrl( psHandler, NULL );
		if ( nError != 0 ) {
			Debug( KERN_DEBUG, "Error: %s\n", curl_easy_strerror( nError ) );
			goto error1;
		}

		if ( strstr( psHandler -> pnData, "404" ) == NULL ) {
			pnPath = g_strdup_printf( "%s/lookup.xml", getUserDir() );

			saveData( pnPath, psHandler -> pnData, psHandler -> nSize );

			g_free( pnPath );

			bReturn = TRUE;
		}
	}

error1:
	freeHandler( psHandler );
	g_free( pnUrl );

	return bReturn;
}

/**
 * \brief Load lookups from disk
 */
void LookupLoad( void ) {
	xmlnode *psNode = NULL, *psChild;
	DIR *psDir = NULL;
	gchar *pnPath = NULL;

	/* Check for user directory */
	psDir = opendir( getUserDir() );
	if ( psDir == NULL ) {
		g_mkdir( getUserDir(), 0755 );
	} else {
		closedir( psDir );
	}

again:
	/* First, try to open user specific lookup file
	 * Second, try to to open general lookup file
	 */
	pnPath = g_strdup_printf( "%s/lookup.xml", getUserDir() );

	psNode = readXmlFromFile( pnPath, _( "lookup" ) );
	if ( psNode == NULL ) {
		gchar *pnLookupFile = g_build_filename( getDirectory( PKGDATADIR ), "lookup.xml", NULL );
		psNode = readXmlFromFile( pnLookupFile, _( "lookup" ) );
		if ( psNode == NULL ) {
			g_free( pnPath );
			Debug( KERN_DEBUG, "Could not read %s\n", pnLookupFile );
			g_free( pnLookupFile );
			return;
		}
		g_free( pnLookupFile );
	}
	g_free( pnPath );

	/* Extract version information */
	if ( xmlnode_get_attrib( psNode, "version" ) != NULL ) {
		pnCurrentVersion = g_strdup( xmlnode_get_attrib( psNode, "version" ) );
	} else {
		pnCurrentVersion = g_strdup( "0" );
	}

	Debug( KERN_DEBUG, "Version: %s\n", pnCurrentVersion );

	/* Check for updates and if needed start this function again */
	if ( CheckForUpdate() == TRUE ) {
		Debug( KERN_DEBUG, "Update available!\n" );
		g_free( pnCurrentVersion );
		xmlnode_free( psNode );
		goto again;
	}

	g_free( pnCurrentVersion );

	/* Create new lookup hash table */
	psLookupTable = g_hash_table_new( NULL, NULL );

	for ( psChild = xmlnode_get_child( psNode, "country" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
		/* Add new country code lists to hash table */
		addCountryCode( psChild );
	}

	xmlnode_free( psNode );
}
