/* $Id: food.c,v 1.20 2003/06/16 11:24:05 paul Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "global.h"
#include "model.h"
#include "object.h"
#include "food.h"
#include "player.h"
#include "collision.h"

#define MDEBUG(...)      DEBUG(DMODEL, "Model", __VA_ARGS__)


/*****************************************************************************/
/* Simpel helper functions                                                   */
/*****************************************************************************/

int
food_get_short_string(void *type, void *str) {
  Object_type *foodtype = (Object_type *)type;
  char *s = (char *)str;

  s[strlen(s)] = food_type_short_string[*foodtype];

  return TRUE;
}


/*****************************************************************************/
/* Food object powerup helper functions                                      */
/*****************************************************************************/

static void
food_powerup_banana(Model_object *o, Model_object *p) {
  /* Increase initial speed and set reverse accel for boomrang effect. */
  o->speed->x *= BAN_X_SPEED_MUL;
  o->accel->x = -SGN(o->speed->x) * BAN_X_ACCEL_BOOMR; 
  /* No gravity needed until banana hits something. */
  o->speed->y = 0;
  o->accel->y = 0;
}

static void
food_powerup_pumpkin(Model_object *o, Model_object *p) {
  /* Leave initial speed, but decrease ground friction drastically and
   * make a pumpkin bounce more. */
  o->speed->x *= PMK_X_SPEED_MUL;
  o->bounciness->x = PMK_X_BOUNCINESS;
  o->friction_ground = PMK_X_FRICT_GROUND;
}

static void
food_powerup_pepper(Model *m, Model_object *o) {
  Model_object *o1, *o2;
  Vector *pos, *size;

  /* Remove pepper that hit the ground. */
  o->gfx_state = GFX_DELETED;

  /* Explode! */
  /* Generate first pepper (if possible). */
  size = m->reg_types[OT_PEPPER];
  pos = new_vector(o->pos->x - size->x, o->pos->y);
  
  if ((o1 = food_generate(m, OT_PEPPER, o->owner, pos, size)) != NULL) {
    o1->speed->y = -3 * FD_Y_SPEED_THROW;
    o1->speed->x = -FD_X_SPEED_THROW;
  }

  /* Generate second pepper (if possible) at the other side. */
  pos->x = o->pos->x + o->size->x;
  
  if ((o2 = food_generate(m, OT_PEPPER, o->owner, pos, size)) != NULL) {
    o2->speed->y = -3 * FD_Y_SPEED_THROW;
    o2->speed->x = +FD_X_SPEED_THROW;
  }
  
  del_vector(pos);
}

static void
food_powerup_cucumber(Model_object *o, Model_object *p) {
  /* Boost speed of cucumber. */
  o->speed->x *= CCB_X_SPEED_MUL;
}

static void
food_powerup_orange(Model_object *o, Model_object *p) {
  /* Give orange great increase in initial speed, no gravity needed. */
  o->speed->x *= ORA_X_SPEED_MUL;
  o->speed->y = 0;
  o->accel->y = 0;
}

static void
food_powerup_cherries(Model *m, Model_object *o, Model_object *p) {
  Vector *pos, *size;
  Model_object *o2;

  /* Create new cherrie at other side of player if possible to
   * spawn it there. */
  size = m->reg_types[OT_CHERRIES];
  pos = new_vector(p->pos->x + (LOOKS_LEFT(p) ? p->size->x : -size->x), 
                   o->pos->y);

  if ((o2 = food_generate(m, OT_CHERRIES, p, pos, size)) != NULL) {
    /* Give same speed but in opposite X direction. */
    o2->speed->y = o->speed->y;
    o2->speed->x = -o->speed->x;
    o2->state.fs = FS_POWERUP;
  }
  
  del_vector(pos);
}

void
food_powerup(Model *m, Model_object *o, Model_object *p) {
  o->state.fs = FS_POWERUP;

  /* Select fitting powerup function. */
  switch(o->type) {
    case OT_BANANA:   food_powerup_banana(o, p);      break;
    case OT_PEPPER:   break; /* food_friction_lies will handle this. */
    case OT_PUMPKIN:  food_powerup_pumpkin(o, p);     break;
    case OT_CUCUMBER: food_powerup_cucumber(o, p);    break;
    case OT_ORANGE:   food_powerup_orange(o, p);      break;
    case OT_CHERRIES: food_powerup_cherries(m, o, p); break;
    default:          break;
  }
}


/*****************************************************************************/
/* Food object generation function                                           */
/*****************************************************************************/

Model_object *
food_generate(Model *m, Object_type t, Model_object *owner, 
              Vector *pos, Vector *size) {

if (collides_with_objects(m->act_food, pos, size) ||
    collides_with_objects(m->pas_food, pos, size) ||
    collides_with_objects(m->boxes,    pos, size))
  return NULL;
else
  return new_object(m, t, owner, pos->x, pos->y, size->x, size->y);
}

/*****************************************************************************/
/* Food object friction functions                                            */
/*****************************************************************************/

static void
food_friction_lies(Model *m, Model_object *o) {
  if (o->speed->x == 0) {
    if (IS_POWEREDUP(o) && o->type == OT_PEPPER)
      /* Powered up pepper has landed, remove pepper and create
         two new ones. */
      food_powerup_pepper(m, o);

    o->accel->x = 0;
  }
  else 
    /* Food is still moving, apply friction. */
    o->accel->x = -SGN(o->speed->x) * o->friction_ground;
  
  /* Mark food (almost) disowned because it is (barely) moving. */
  if (o->accel->x < FD_X_SPEED_DISOWN) {
    if (o->accel->x == 0) {
      /* Food has halted, set owner to nobody. */
      o->owner = NULL;
      o->state.fs = FS_NORMAL;
    }
    else
      /* Food is barely moving (below treshold, set virtual disonwed. */
      o->disown = TRUE;
  }
}

void
food_friction_fly(Model *m, Model_object *o) {
  /* No friction for powered up bananas, reverse acceleration will regulate
   * it (boomrang effect). */
  if (!IS_POWEREDUP(o) || o->type != OT_BANANA)
    o->accel->x = -SGN(o->speed->x) * o->friction_air;
}


/*****************************************************************************/
/* Food function for collision detection/response                            */
/*****************************************************************************/

int
food_deliver(Model *m, Model_object *h, Model_object *f) {
  /* No delivery if home is not owned by owner of food. */
  if (h->owner != f->owner || IS_DEAD(f->owner))
    return FALSE;
    
  /* Deliver food, request new food spawned and increase food and life. */
  if (m->settings->spawn_deliver)
    m->spawn_food++;
  f->gfx_state = GFX_DELETED;
  h->owner->life = MIN(h->owner->life + m->settings->food_deliver_life, 
                       m->settings->player_max_life);
  h->owner->score += SC_DELIV;
  h->owner->events |= EVF_DELIV;
  MDEBUG("Player %p delivers food, life is now %d.", h->owner,
         h->owner->life);
  
  return TRUE;
}


/*****************************************************************************/
/* Food object tick function                                                 */
/*****************************************************************************/

void
food_tick(Model *m, Model_object *o) {
  /* Friction for food objects either on the ground or in the air. */
  if (IS_STANDING(o))
    food_friction_lies(m, o);
  else
    food_friction_fly(m, o);
}
