/* -*- mode: c; c-basic-offset: 2; indent-tabs-mode: nil; -*- */
/* geoclue-stumbler-preferences-dialog.c
 *
 * Copyright 2024 Chris Talbot <chris@talbothome.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author(s):
 *   Chris Talbot <chris@talbothome.com>
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <glib/gi18n.h>

#include "geoclue-stumbler-preferences-dialog.h"
#include "geoclue-stumbler-settings.h"

struct _GeoclueStumblerPreferencesDialog
{
  AdwPreferencesDialog  parent_instance;

  GeoclueStumblerSettings *settings;
  GtkEntryBuffer          *submission_url_buffer;
  AdwComboRow             *preset_submission_url_row;
  GtkSwitch               *submission_location_switch;
};

G_DEFINE_TYPE (GeoclueStumblerPreferencesDialog, geoclue_stumbler_preferences_dialog, ADW_TYPE_PREFERENCES_DIALOG)

static void
cancel_button_clicked_cb (GeoclueStumblerPreferencesDialog *self)
{
  g_assert (GEOCLUE_STUMBLER_IS_PREFERENCES_DIALOG (self));

  g_debug("Cancel Button Clicked");
  adw_dialog_close (ADW_DIALOG (self));
}

static void
save_button_clicked_cb (GeoclueStumblerPreferencesDialog *self)
{
  g_assert (GEOCLUE_STUMBLER_IS_PREFERENCES_DIALOG (self));

  g_debug("Save Button Clicked");
  geoclue_stumbler_settings_set_submission_url (self->settings,
                                                gtk_entry_buffer_get_text (self->submission_url_buffer));
  geoclue_stumbler_settings_set_show_submission_location (self->settings,
                                                          gtk_switch_get_active (self->submission_location_switch));

  adw_dialog_close (ADW_DIALOG (self));

}

static void
preset_submission_url_button_clicked_cb (GeoclueStumblerPreferencesDialog *self)
{
  unsigned int selection = 0;
  const char *submission_url = NULL;

  g_assert (GEOCLUE_STUMBLER_IS_PREFERENCES_DIALOG (self));

  selection = adw_combo_row_get_selected (self->preset_submission_url_row);
  g_debug("Preset Submission URL Button Clicked: %d", selection);
  switch (selection) {
    case 0: //Unset
      submission_url = "";
      break;
    case 1: // beaconDB
      submission_url = "https://api.beacondb.net/v2/geosubmit";
      break;
    default:
      submission_url = "https://example.com/v2/geosubmit";
      break;
    }

  gtk_entry_buffer_set_text (self->submission_url_buffer, submission_url, -1);

}

static void
geoclue_stumbler_preferences_dialog_dispose (GObject *object)
{
//  GeoclueStumblerPreferencesDialog *self = (GeoclueStumblerPreferencesDialog *)object;

  G_OBJECT_CLASS (geoclue_stumbler_preferences_dialog_parent_class)->dispose (object);
}

static void
geoclue_stumbler_preferences_dialog_class_init (GeoclueStumblerPreferencesDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

  object_class->dispose = geoclue_stumbler_preferences_dialog_dispose;

  gtk_widget_class_set_template_from_resource (widget_class,
                                               "/org/kop316/stumbler/"
                                               "geoclue-stumbler-preferences-dialog.ui");

  gtk_widget_class_bind_template_child (widget_class, GeoclueStumblerPreferencesDialog, preset_submission_url_row);
  gtk_widget_class_bind_template_child (widget_class, GeoclueStumblerPreferencesDialog, submission_url_buffer);
  gtk_widget_class_bind_template_child (widget_class, GeoclueStumblerPreferencesDialog, submission_location_switch);

  gtk_widget_class_bind_template_callback (widget_class, cancel_button_clicked_cb);
  gtk_widget_class_bind_template_callback (widget_class, save_button_clicked_cb);
  gtk_widget_class_bind_template_callback (widget_class, preset_submission_url_button_clicked_cb);
}

static void
geoclue_stumbler_preferences_dialog_init (GeoclueStumblerPreferencesDialog *self)
{
  g_autofree char *submission_url = NULL;

  gtk_widget_init_template (GTK_WIDGET (self));

  self->settings = geoclue_stumbler_settings_get_default ();

  gtk_switch_set_active (self->submission_location_switch, geoclue_stumbler_settings_get_show_submission_location(self->settings));

  submission_url = geoclue_stumbler_settings_get_submission_url (self->settings);
  gtk_entry_buffer_set_text (self->submission_url_buffer, submission_url, -1);
}

GeoclueStumblerPreferencesDialog *
geoclue_stumbler_preferences_new (void)
{
  GeoclueStumblerPreferencesDialog *self;

  self = g_object_new (GEOCLUE_STUMBLER_TYPE_PREFERENCES_DIALOG, NULL);
  return self;
}
