-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-connection.adb,v $
--  Description     : Ada Database Objects - Connection object               --
--  Author          : Michael Erdmann                                        --
--  Created         : 18.1.2002                                              --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/02/21 20:30:55 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2002 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  This package implements the connection object features.                  --
--                                                                           --
--                                                                           --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  Only Linux                                                               --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
with Unchecked_Deallocation;
with Ada.Strings.Unbounded;                    use Ada.Strings.Unbounded;
with Ada.Exceptions;                           use Ada.Exceptions;

with GNU.DB.ADBC.Statement;                    use GNU.DB.ADBC.Statement;
with GNU.DB.ADBC.Resultset;                    use GNU.DB.ADBC.Resultset;
with GNU.DB.ADBC.Driver;                       use GNU.DB.ADBC.Driver;
with GNU.DB.ADBC;                              use GNU.DB.ADBC;
use  GNU.DB;

package body GNU.DB.ADBC.Connection is

   Version : constant String :=
      "$Id: gnu-db-adbc-connection.adb,v 1.3 2002/02/21 20:30:55 merdmann Exp $";

   subtype Connection_Index is Connection_ID range Null_ID+1 .. Connection_ID'Last;

   CT : array( Connection_Index ) of Handle := (others => null );

   ---====================================================================---
   ---===                O B J  E C T     D A T A                      ===---
   ---===                                                              ===---
   ---=== This section contains all declarations of data structures    ===---
   ---=== to implement one instance of the connection object           ===---
   ---===                                                              ===---
   ---====================================================================---
   type Statement_Table is array( Statement_ID'Range ) of Boolean;

   -----------------
   -- Object_Data --
   -----------------
   type Object_Data is record
         Self         : Handle := null;
         Stmt         : Statement_Table := (others => False);

         Is_Connected : Boolean := False;
         Connection   : Connection_ID;

         Name         : Unbounded_String := Null_Unbounded_String;
         Password     : Unbounded_String := Null_Unbounded_String;
         Database     : Unbounded_String := Null_Unbounded_String;
      end record;

   ---=====================================================================---
   ---===         L O C A L   S U P P O R T   P R O C E D U R E S       ===---
   ---===                                                               ===---
   ---=====================================================================---

   --------------------
   -- Drop_Resources --
   --------------------
   procedure Drop_Resources(
      This : in out Object ) is
      -- drop all resources from the connection
      Data : Object_Data_Access renames This.Data;
   begin
      for I in Data.Stmt'Range loop
         if Data.Stmt(I) then
            Statement.Deallocate( I );
            Data.Stmt(I) := False;
         end if;
      end loop;
   end Drop_Resources;

   ----------------------
   -- Create_Exception --
   ----------------------
   procedure Create_Exception(
      This : in Object;
      Id   : in Exception_Id;
      Info : in String )  is
      Data : Object_Data_Access renames This.Data;
      Q    : Unbounded_String := Null_Unbounded_String;
   begin
      Q := Data.Name & "/" & Data.Database ;
      Raise_Exception( Id, Info & ":" & To_String(Q) );
   end Create_Exception;

   ----------------------
   -- Create_Exception --
   ----------------------
   procedure Create_Exception(
      this     : in Object;
      theError : in Exception_Occurrence;
      info     : in String ) is
   begin
      Create_Exception( This, Exception_Identity(TheError), Info );
   end Create_Exception;

   ---======================================================================---
   ---===             C O M P O  N E N T    I N T E R F A C E            ===---
   ---======================================================================---

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize(
      This : in out Object ) is
   begin
      This.Data := new Object_Data;
      This.Data.Self := This'Unchecked_Access;

      for I in CT'Range loop
         if CT(I) = null then
            CT(I) := This.Data.Self;
            This.Data.Connection := I;
            exit;
         end if;
      end loop;
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize(
      This : in out Object ) is
      Data : Object_Data_Access renames This.Data;

      procedure Free is
            new Unchecked_Deallocation( Object_Data, Object_Data_Access);
   begin
      Drop_Resources( This );
      Free( This.Data );
   end Finalize;

   ---=====================================================================---
   ---===           A T T R I B U T E    F U N C T I O N S              ===---
   ---=====================================================================---

   ----------
   -- Self --
   ----------
   function Self(
      This : in Object ) return Handle is
   begin
      return This.Data.Self;
   end Self;

   -------------------
   -- Driver_Handle --
   -------------------
   function Driver_Handle(
      This : in Object ) return Driver.Handle is
   begin
      return This.Db;
   end Driver_Handle;

   -------------------
   -- Driver_Handle --
   -------------------
   function Driver_Handle(
      Id : in Connection_ID  ) return Driver.Handle is
   begin
      return CT(Id).Db;
   end Driver_Handle;

   ---=====================================================================---
   ---===                        M E T H O D S                          ===---
   ---=====================================================================---

   -------------
   -- Connect --
   -------------
   procedure Connect(
      This       : in out Object;
      Name       : in String;
      Password   : in String := Default_Password;
      Database   : in String := Default_Database;
      Parameters : in String := "" ) is
      Data       : Object_Data_Access renames This.Data;
   begin
      Data.Name := To_Unbounded_String(Name);

      if Password /= Default_Password then
         Data.Password := To_Unbounded_String(Password);
      end if;

      if Database /= Default_Database then
         Data.Database := To_Unbounded_String(Database);
      end if;

      Connect(
         This.Db.all,
         Name,
         Password,
         Database);

      Data.Is_Connected := True;
   end Connect;

   ----------------
   -- Disconnect --
   ----------------
   procedure Disconnect(
      This : in out Object ) is
      Data : Object_Data_Access renames This.Data;
   begin
      Drop_Resources( This );
      -- driver disconnect
      Disconnect( This.Db.all );

      Data.Is_Connected := False;
   end Disconnect;

   -------------------
   -- Add_Statement --
   -------------------
   procedure Add_Statement(
      Con  : in Connection_ID;
      Stmt : in Statement_ID ) is
      Data : Object_Data_Access renames CT(Con).Data;
   begin
      Data.Stmt(Stmt) := True;
   end Add_Statement;

   ----------------------
   -- Delete_Statement --
   ----------------------
   procedure Delete_Statement(
      Con  : in Connection_ID;
      Stmt : in Statement_ID ) is
      Data : Object_Data_Access renames CT(Con).Data;
   begin
      Data.Stmt(Stmt) := False;
   end Delete_Statement;

   -------------
   -- Prepare --
   -------------
   function Prepare(
      This      : in Object;
      Statement : in String ) return Statement_ID is
      -- prepare a statement for processing. This is split into two
      -- parts.
      --
      -- 1. Parse the statement according to ADO rules
      -- 2. Pass the statement to the data base driver as a statement
      --    object for further processing
      --
      Data      : Object_Data_Access renames This.Data;
      S         : Statement_ID;
   begin
      if not Data.Is_Connected then
         Create_Exception( This, Not_Connected'Identity, "prepare statement");
      end if;

      S := Create_Statement(This.Db.all, Data.Connection);  -- create a statement
      ADBC.Statement.Prepare( S, Statement );               -- parse the statement
      Prepare(This.Db.all, S );

      return S;
   end Prepare;

   -------------
   -- Execute --
   -------------
   function Execute(
      This      : in Object;
      Stmt      : in Statement_ID ) return Resultset_ID is
      -- execute a prepared statement
      Data      : Object_Data_Access renames This.Data;
      Result    : Resultset_ID;
   begin
      if not Data.Is_Connected then
         Create_Exception( This, Not_Connected'Identity, "execute statement");
      end if;

      Result := Create_Resultset(This.Db.all, Stmt);
      Bind_Host_Variables( Stmt );
      Execute_Statement( This.Db.all, Result, Stmt );
      Retrive_Host_Variables( Stmt, null );

      return Result;

   exception
      when The_Error: others =>
         Create_Exception( This, The_Error, "execute statement" );
         raise;
   end Execute;

   -------------
   -- Execute --
   -------------
   function Execute(
      This      : in Object;
      Statement : in String ) return Resultset_ID is
      -- execute a direct SQL command
      Data      : Object_Data_Access renames This.Data;
      Result    : Resultset_ID;
      Stmt      : Statement_ID;
   begin
      if not Data.Is_Connected then
         Create_Exception( This, Not_Connected'Identity, "execute statement");
      end if;

      Stmt := Prepare( This, Statement );

      Result := Create_Resultset(This.Db.all, Stmt);
      Execute_Statement( This.Db.all, Result, Stmt );

      Delete_Resultset( Stmt, Result );
      Deallocate( Stmt );

      return Result;

   exception
      when The_Error : others  =>
         Create_Exception( This, The_Error, Statement );
         raise;
   end Execute;

end GNU.DB.ADBC.Connection;

