/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#define PLUGIN_NAME "Jabber IMSpector protocol plugin"
#define PROTOCOL_NAME "Jabber"
#define PROTOCOL_PORT 5222
#ifdef HAVE_SSL
#define PROTOCOL_PORT_SSL 5223
#endif

extern "C"
{
	bool initprotocolplugin(struct protocolplugininfo &pprotocolplugininfo,
		class Options &options, bool debugmode);
	void closeprotocolplugin(void);
	int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer,
		int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress);
	int generatemessagepacket(struct response &response, char *replybuffer, int *replybufferlength);
};

int recvuntil(Socket &sock, char *string, int length, char end);

int packetcount = 0;
bool tracing = false;
bool localdebugmode = false;
bool starttls = false;

#ifdef HAVE_SSL
bool monitortls = false;
#endif

std::string localdomain = "Unknown";

std::string localid = "Unknown";
std::string remoteid = "Unknown";

bool initprotocolplugin(struct protocolplugininfo &protocolplugininfo,
	class Options &options, bool debugmode)
{
	if (options["jabber_protocol"] != "on") return false;

#ifdef HAVE_SSL
	if (options["ssl"] == "on")
	{
		syslog(LOG_INFO, PROTOCOL_NAME ": Monitoring SSL/TLS");
		
		monitortls = true;
	}
#endif
	
	localdebugmode = debugmode;
	
	protocolplugininfo.pluginname = PLUGIN_NAME;
	protocolplugininfo.protocolname = PROTOCOL_NAME;
	protocolplugininfo.port = htons(PROTOCOL_PORT);
#ifdef HAVE_SSL
	protocolplugininfo.sslport = htons(PROTOCOL_PORT_SSL);
#endif

	if (options["jabber_trace"] == "on") tracing = true;
	
	return true;
}

void closeprotocolplugin(void)
{
	return;
}

/* The main plugin function. See protocolplugin.cpp. */
int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer, 
	int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress)
{
	int len;
	int replybufferoffset = 0;
	int ret = 0;

	/* If we have switched to tls then simply shuvel data each way. */
#ifdef HAVE_SSL
	if (starttls && !monitortls)
#else
	if (starttls)
#endif
	{
		if ((len = incomingsock.recvdata(replybuffer, BUFFER_SIZE)) < 1) 
			return 1;

		debugprint(localdebugmode, PROTOCOL_NAME ": Got %d bytes of TLS data", len);
		
		replybufferoffset = len;
	}
	else
	{
		/* State variables for processing the document. */
		std::string openingtag; std::string closingtag; int level = 0;
		bool needmore = false; /* The main looping flag for sucking down tags. */
		bool inhtml = false;
	
		int type = TYPE_NULL;
	
		/* Two sets, one for normal html messages and a reserve for when a html message isn't
		 * included in the message. */
		std::string eventdata = "";
		struct messageextent messageextent = { 0, 0 };
		std::string backupeventdata = "";
		struct messageextent backupmessageextent = { 0, 0 };
	
		/* Loop getting the document tags. */
		do
		{
			/* Get upto the closing >, or a loan \n (for when this char is sitting in the socket
			 * buffer after the previous tag has been processed by the previous call to this plugin. */
			if ((len = recvuntil(incomingsock, replybuffer + replybufferoffset, BUFFER_SIZE, '>')) < 1) 
				return 1;
	
			/* All we got was a CR, most likely trailing rubbish from the previous tag. */
			if (replybuffer[replybufferoffset] == '\n' || replybuffer[replybufferoffset] == '\r' ||
				replybuffer[replybufferoffset] == '\t' || replybuffer[replybufferoffset] == ' ')
			{
				replybufferoffset += len;
				continue;
			}
			
			debugprint(localdebugmode, PROTOCOL_NAME ": XML received: %s", replybuffer + replybufferoffset);
	
			/* Parse the tag: payload<tag key1='value1' key2="value2" /> */
			std::string payload; int payloadlength; std::string tag; bool closing;
			std::map<std::string, std::string> params;
			
			parsexmltag(localdebugmode, replybuffer + replybufferoffset, payload, payloadlength, tag, closing, params);
			
			/* See if this is the first tag in the document. stream:stream and the xml marker
			 * are special. */
			if (!tag.empty() && tag != "?xml" && tag != "stream:stream" && tag != "/stream:stream" && 
				closingtag.empty() && !closing)
			{
				level = 0; /* So we can support nesting of openingtag. */
				openingtag = tag;
				closingtag = "/" + tag;
				if (!closing) needmore = true; /* if tag isn't <bob ... /> */
				
				debugprint(localdebugmode, PROTOCOL_NAME ": Start of document. Closing tag will be: %s", closingtag.c_str());
			}
	
			/* See if its an open tag, including the first one in the document. */		
			if (!tag.empty() && tag == openingtag)
			{
				level++;
			}
			
			/* Or a closing one. */
			if (tag == closingtag)
			{
				level--;
				/* Top level closing? */
				if (!level)
				{
					debugprint(localdebugmode, PROTOCOL_NAME ": End of document");
					needmore = false;
				}
			}
			
			/* Determine localid via login */
			if (tag == "stream:stream")
				if (!params["to"].empty()) localdomain = params["to"];
			if (tag == "/username")
				localid = payload + "@" + localdomain;
			
			/* Determine the localid via jid. */
			if (tag == "/jid")
				localid = payload;
			
			/* Get the non html body, as a fall back. */
			if (tag == "/body" && !inhtml)
			{
				backupeventdata = payload;
				backupmessageextent.start = replybufferoffset;
				backupmessageextent.length = payloadlength;
				type = TYPE_MSG;
			}
			
			/* Handle the starting and ending html tag. */
			if (tag == "html")
			{
				inhtml = true;
				messageextent.start = replybufferoffset + len;
				messageextent.length = 0;
			}
			if (tag == "/html")
			{
				inhtml = false;
				type = TYPE_MSG;
			}
			
			/* Typing event? */		
			if (tag == "composing")
			{
				eventdata = "";
				type = TYPE_TYPING;
			}
			
			/* File transfers... */
			if (tag == "file")
			{
				eventdata = params["name"] + " " + params["size"] + " bytes";
				type = TYPE_FILE;
			}
	
			/* The remoteid is available in the message tag (TYPE_MSG/TYPE_TYPING) and 
			 * iq tag (TYPE_FILE). */
			if (tag == "message" || tag == "iq")
			{
				if (outgoing)
				{
					if (!params["to"].empty()) remoteid = params["to"];
				}
				else
				{
					if (!params["from"].empty()) remoteid = params["from"];
				}
			}
	
			/* At the closing message, or iq, we have enough info to generate an event! */		
			if ((tag == "/message" || tag == "/iq") && type != TYPE_NULL)
			{
				struct imevent imevent;
				
				imevent.timestamp = time(NULL);
				imevent.clientaddress = clientaddress;
				imevent.protocolname = PROTOCOL_NAME;
				imevent.outgoing = outgoing;
				imevent.type = type;
				imevent.filtered = false;
				imevent.localid = localid;
				imevent.remoteid = remoteid;
				
				/* Choose the regular (html) event data in preference. */
				if (!eventdata.empty())
				{
					imevent.eventdata = eventdata;
					imevent.messageextent = messageextent;
				}
				else
				{
					imevent.eventdata = backupeventdata;
					imevent.messageextent = backupmessageextent;
				}
				
				/* Jabber names are like: user@host/something. This removes the /something, which
				 * would bugger up the file logging plugin and probably isn't useful anyway. */
				stripslash(imevent.localid);
				stripslash(imevent.remoteid);
				
				imevents.push_back(imevent);
	
				/* So another closing message or iq wont generate a second event. */			
				type = TYPE_NULL;
			}
			
			/* Big switch to tell plugin to simply parse packets, if tls is asked for. */
			if (tag == "starttls")
			{
				debugprint(localdebugmode, PROTOCOL_NAME ": Got start TLS request");
				starttls = true; /* See top of this function for TLS procesing. */
			}

#ifdef HAVE_SSL			
			if (tag == "proceed" && starttls && monitortls)
			{
				debugprint(localdebugmode, PROTOCOL_NAME ": Will switch to TLS at end of this packet");
				ret = -1;
			}
#endif
	
			/* If we are in a html block, shuvel the payload (non tag) bits into the event text.
			 * This will remove all the styling tags etc. */		
			if (inhtml)
			{
				if (payloadlength)
					eventdata += payload;
				
				/* Might as well convert html breaks. */
				if (tag == "br/")
					eventdata += "\n";
				
				/* Unfortunately the message extent includes the whole span of the html, tags
				 * and all. This means the tags will be exposed to the content filtering. :( The
				 * long-term fix is to allow multiple message extents in an event, but this is a
				 * big change to the filtering mechanism. */
				messageextent.length += len;
			}
	
			/* Advance the offset to the end of this tag, so the next tag will be read into the 
			 * buffer after this one. */
			replybufferoffset += len;
		}
		while (needmore); /* Loop getting more tags into the document, if needed. */
	}
		
	*replybufferlength = replybufferoffset;
	
	/* Write out trace packets if enabled. */
	if (tracing) tracepacket("jabber", packetcount, replybuffer, *replybufferlength);
	
	packetcount++;
	
	return ret;
}

int generatemessagepacket(struct response &response, char *replybuffer, int *replybufferlength)
{
	if (localid.empty() || remoteid.empty()) return 1;

	const char *from = remoteid.c_str();
	const char *to = localid.c_str();

	if (response.outgoing)
	{
		from = localid.c_str();
		to = remoteid.c_str();
	}

	snprintf(replybuffer, BUFFER_SIZE - 1, "<message type='chat' " \
		"from='%s' to='%s'><body>%s</body></message>", from, to, response.text.c_str());
	
	*replybufferlength = strlen(replybuffer);
	
	if (tracing) tracepacket("jabber-out", packetcount, replybuffer, *replybufferlength);
	
	packetcount++;
	
	return 0;
}	

/* This is stolen out of socket.cpp. */
int recvuntil(Socket &sock, char *string, int length, char end)
{
	int totalreceved = 0;
	int receved = 0;
		
	while (totalreceved < length)
	{
		if (!(receved = sock.recvdata(&string[totalreceved], 1)) > 0) return -1;
		if (string[totalreceved] == end) return totalreceved + 1;
		/* This oddity is because there may be extra data at the end of the closing
		 * tag marker which the next call will need to suck down but ignore. */
		if (!totalreceved &&
			(string[0] == '\n' || string[0] == '\r' || string[0] == '\t' || string[0] == ' '))
		{
			return 1;
		}	
		totalreceved += receved;
	}
	
	/* It was too long, but nevermind. */
	return totalreceved;
}
