/***************************************************************************
 *   Copyright (C) 2005-2013 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QDateTime>

#include <KGlobal>
#include <KLocale>

#include "kmdcodec.h"
#include "kfileitemext.h"

#include "klistviewitemsingle.h"

#include <version.h>

KListViewItemSingle::KListViewItemSingle( QTreeWidget *parent, KFileItemExt *_item, KListViewItemSingle *_brother )
	: QTreeWidgetItem( (QTreeWidget*)parent ), item( _item ), brother( _brother ), m_filter( NONE ), m_kind( NONE )
{
	m_color = Qt::black;
	m_use_color = true;

	QString date = item->getKFileItemExtModificationTimeString();

	setTextAlignment( 2, Qt::AlignRight );

	// Fill the item with data
	setText( 0, KIO::decodeFileName( item->url().fileName() ) );
	setText( 1, "" ); // the directory is set individually, depending on the directory
		// we're in (it is stripped, and only subdirectories are displayed)
	setText( 2, /*KGlobal::locale()->formatNumber( item_tmp.size(), 0 )*/KIO::convertSize( item->size() ) ); // Beautiful size instead of bytes
	setText( 3, date );
	setText( 4, item->permissionsString() );
	setText( 5, item->user() );
	setText( 6, item->group() );

	// put the icon into the leftmost column
	item->determineMimeType();
	setIcon( 0, item->pixmap( 16 ) );
}

KListViewItemSingle::~KListViewItemSingle()
{
}

bool KListViewItemSingle::operator< ( const QTreeWidgetItem &other ) const
{
	int column = treeWidget()->sortColumn();

	switch ( column )
	{
		case 2:  // Returns size in bytes. Used for sorting
			return (uint)item->size() < (uint)((KListViewItemSingle*)&other)->item->size();
		case 3:  // Returns time in secs from 1/1/1970. Used for sorting
			return item->getKFileItemExtModificationTimeT() < ((KListViewItemSingle*)&other)->item->getKFileItemExtModificationTimeT();
		case 7: // Reference directory. Don't sort by the name of it, but by the dir number.
			return item->dir < ((KListViewItemSingle*)&other)->item->dir;
	}
	return text( column ) < other.text( column );

}

void KListViewItemSingle::setKind( FILTER kind, bool use_color, const QColor &color )
{
	m_kind = kind;
	m_color = color;
	m_use_color = use_color;

	if ( m_color != Qt::black && m_use_color )
	{
		QBrush textbrush( m_color );
		QBrush backgroundbrush( QColor( 255-m_color.red(), 255-m_color.green(), 255-m_color.blue() ) );

		int i;
		for ( i=0; i<columnCount(); i++ )
		{
			setForeground( i, textbrush );
		}
	}
	else
	{
		int i;
		for ( i=0; i<columnCount(); i++ )
		{
			setForeground( i, QBrush() );
		}
	}
}

int KListViewItemSingle::kind()
{
	return m_kind;
}

QColor KListViewItemSingle::color()
{
	return m_color;
}

void KListViewItemSingle::addFilter( FILTER filter )
{
	m_filter |= filter;
}

void KListViewItemSingle::removeFilter( FILTER filter )
{
	if ( m_filter & filter ) m_filter ^= filter;
}

void KListViewItemSingle::setVisibility()
{
	setHidden( ( m_filter == NONE ) ? false : true );
}
