/*
 * examples/dot-product.C
 *
 * Copyright (C) 2002, 2010 Bradford Hovinen <hovinen@cis.udel.edu>
 * ========LICENCE========
 * This file is part of the library LinBox.
 *
 * LinBox is free software: you can redistribute it and/or modify
 * it under the terms of the  GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * ========LICENCE========
 */

/** \file examples/dot-product.C
 * @example  examples/dot-product.C
 * \brief Timings on dot products of random vectors.
 * \ingroup examples
 *
 * \author Bradford Hovinen <hovinen@cis.udel.edu>
 *
 * Use of vectors meeting the LinBox dense and sparse vector archetypes
 * is illustrated and their dot-product times are benchmarked.
 *
 * Constructs random vectors and computes their dot product, giving the
 * required time.
 */

#include <linbox/linbox-config.h>

#include <iostream>

#include <linbox/ring/modular.h>
#include <linbox/vector/vector-domain.h>
#include <linbox/vector/stream.h>
#include <linbox/util/commentator.h>

using namespace LinBox;

typedef Givaro::Modular<uint32_t> Field;

/// no command line args
int main (int argc, char **argv)
{
	// Constants: we are working with an n x n matrix over GF(q)
	const int    n = 10000000;
	const double p = .001;
	const int    q = 32749;


	commentator().setMaxDepth (1);
	commentator().setReportStream (std::cout);

	Field F (q); Field::RandIter gen(F);

	RandomDenseStream<Field, Vector<Field>::Dense> factory1 (F, gen, n);
	RandomSparseStream<Field, Vector<Field>::SparseSeq> factory2 (F, gen, p, n);
	RandomSparseStream<Field, Vector<Field>::SparsePar> factory3 (F, gen, p, n);

	Vector<Field>::Dense v1 (n), v2 (n);
	Vector<Field>::SparseSeq v3;
	Vector<Field>::SparsePar v4;

	factory1 >> v1 >> v2;
	factory2 >> v3;
	factory3 >> v4;

	VectorDomain<Field> VD (F);
	Field::Element res;

	commentator().start ("dense/dense dot product (1)");
	for (int i = 0; i < 1; i++)
		VD.dot (res, v1, v2);
	commentator().stop ("done");

	commentator().start ("dense/sparse sequence dot product (1000)");
	for (int i = 0; i < 1000; i++)
		VD.dot (res, v1, v3);
	commentator().stop ("done");

	commentator().start ("dense/sparse parallel dot product (1000)");
	for (int i = 0; i < 1000; i++)
		VD.dot (res, v1, v4);
	commentator().stop ("done");

	return 0;
}

// vim:sts=8:sw=8:ts=8:noet:sr:cino=>s,f0,{0,g0,(0,:0,t0,+0,=s
// Local Variables:
// mode: C++
// tab-width: 8
// indent-tabs-mode: nil
// c-basic-offset: 8
// End:

