#include <cassert>
#include <iostream>
#include <cmath>
#include <X11/extensions/XTest.h>
#include <X11/extensions/Xrandr.h>
#include "display.h"
#include "logger.h"
#include "image_cache.h"
#include <ft2build.h>
#include FT_FREETYPE_H

using namespace std;

bool MyDisplay::Open(){
    disp = XOpenDisplay(0);

    if (!disp){
        LOG(INFO) << "Failed to open X display";
	return false;
    }

    screen = DefaultScreen(disp);

    width = DisplayWidth(disp, screen);
    height = DisplayHeight(disp, screen);

    // Has shape extions?
    if (!XShapeQueryExtension(disp, &shape_event_base, &shape_error_base)){
	LOG(INFO) << "NO shape extension in your system !";
	return false;
    }
    if (!XRRQueryExtension(disp, &randr_event_base, &randr_error_base)){
	LOG(INFO) << "No xrandr extension!";
	return false;
    }
    fk = fakekey_init(disp);
    return true;
}

void MyDisplay::UpdateWidthHeight(){
    width = DisplayWidth(disp, screen);
    height = DisplayHeight(disp, screen);
}

int MyDisplay::Width() const {
    return width;
}

int MyDisplay::Height() const {
    return height;
}

int MyDisplay::Depth() const {
    return DefaultDepth(disp, screen);
}

void MyDisplay::Flush(){
    XFlush(disp);
}

void MyDisplay::GetEvent(XEvent* xev){
    XNextEvent(disp, xev);
}

void MyDisplay::GenKeyEvent(string action){
    unsigned int ksym;
    bool is_alt = false;
    bool is_ctrl = false;


    if (action.substr(0,5) == "ctrl:"){
	action = action.substr(5);
	is_ctrl = true;
    }
    if (action.substr(0,4) == "alt:"){
	action = action.substr(4);
	is_alt = true;
    }

    if (action[0] != '$' || (action == "$")) {
	int mod = 0;
	if (is_alt)
	    mod |= FAKEKEYMOD_ALT;
	if (is_ctrl)
	    mod |= FAKEKEYMOD_CONTROL;

	fakekey_press(fk, (unsigned char*)action.c_str(), action.size(), mod);
	fakekey_release(fk);
	return;
    } else
	action = action.substr(1);


    if (action == "backspace")
	ksym = XK_BackSpace;
    else if (action == "return")
	ksym = XK_Return;
    else if (action == "key_up")
	ksym = XK_Up;
    else if (action == "key_down")
	ksym = XK_Down;
    else if (action == "escape")
	ksym = XK_Escape;
    else if (action == "delete")
	ksym = XK_Delete;
    else if (action == "home")
	ksym = XK_Home;
    else if (action == "key_left")
	ksym = XK_Left;
    else if (action == "key_right")
	ksym = XK_Right;
    else if (action == "end")
	ksym = XK_End;
    else if (action == "page_up")
	ksym = XK_Page_Up;
    else if (action == "page_down")
	ksym = XK_Page_Down;
    else if (action == "tab")
	ksym = XK_Tab;
    else if (action == "space")
	ksym = ' ';
    else if (action == "win_ctrl1"){
	ksym = XK_Escape;
	is_alt = true;
	is_ctrl = false;
	LOG(INFO) << "Win ctrl1";
    }
    else if (action == "win_ctrl2"){
	ksym = XK_Escape;
	is_ctrl = true;
	is_alt = false;
	LOG(INFO) << "Win ctrl2";
    }
    else if (action == "win_ctrl_close"){
	ksym = XK_F4;
	is_ctrl = false;
	is_alt = true;
	LOG(INFO) << "Win close";
    }
    else if (action.substr(0, 14) == "desktop_switch"){
	LOG(INFO) << "Desktop switch" << string(1,action[14]);
	switch (action[14]){
	    case '0': ksym = XK_1;
		    break;
	    case '1': ksym = XK_2;
		    break;
	    case '2': 
	    default:ksym = XK_3;
		    break;
	}
	is_ctrl = true;
	is_alt = true;
    }

    else 
	return;



    unsigned int keycode = XKeysymToKeycode(disp, ksym);

    bool is_shift = false;
    for (int i = 0; i<4; ++i)
	if (XKeycodeToKeysym(disp, keycode, i) == ksym ){
	    is_shift = i>0;
	    break;
	}

    static unsigned int Shift_code = XKeysymToKeycode(disp, XK_Shift_L);
    static unsigned int Ctrl_code = XKeysymToKeycode(disp, XK_Control_L);
    static unsigned int Alt_code = XKeysymToKeycode(disp, XK_Alt_L);

    if (is_shift)
	XTestFakeKeyEvent(disp, Shift_code, True, 0);
    if (is_ctrl)
	XTestFakeKeyEvent(disp, Ctrl_code, True, 0);
    if (is_alt)
	XTestFakeKeyEvent(disp, Alt_code, True, 0);
    XTestFakeKeyEvent(disp, keycode, True, 0);
    XTestFakeKeyEvent(disp, keycode, False, 0);
    if (is_shift)
	XTestFakeKeyEvent(disp, Shift_code, False, 0);
    if (is_ctrl)
	XTestFakeKeyEvent(disp, Ctrl_code, False, 0);
    if (is_alt)
	XTestFakeKeyEvent(disp, Alt_code, False, 0);
}
    
void MyDisplay::GenMouseClickEvent(int x, int y, int button){
    LOG(INFO) << "click " << x << " " << y << " " << button;
    if (x>-1)
	XTestFakeMotionEvent(disp, -1, x, y,0);
    XTestFakeButtonEvent(disp, button, true,0);
    XTestFakeButtonEvent(disp, button, false,0);
}

void MyDisplay::GenMouseMoveEvent(int x, int y){
    XTestFakeMotionEvent(disp, -1, x, y,0);
}


void MyDisplay::SetWidth(int w) {
    width = w;
}

void MyDisplay::SetHeight(int h) {
    height = h;
}

XColor MyDisplay::MakeXColor(const short rgb[]){
    return MakeXColor(rgb[0], rgb[1], rgb[2]);
}

XColor MyDisplay::MakeXColor(short red, short green, short blue)
{
    XColor color;

    // m_color.red = red * 65535 / 255;
    color.red = (red * 0xFFFF) / 0xFF;
    color.green = (green * 0xFFFF) / 0xFF;
    color.blue = (blue * 0xFFFF) / 0xFF;
    color.flags = DoRed | DoGreen | DoBlue;

    if (!XAllocColor(disp, DefaultColormap(disp, screen), &color))
    {
        LOG(INFO) << "createXColorFromRGB: Cannot create color";
    }
    return color;
}


MyWindow::MyWindow(int w, int h, MyDisplay& d):width(w),height(h),display(d),transparent(false){
    
}

MyWindow::~MyWindow(){
    XFreePixmap(display.disp, buf);
    XDestroyWindow(display.disp, win);
    XDestroyWindow(display.disp, input);
}

Window& MyWindow::GetInputWindow(){
    return input;
}

Window& MyWindow::GetWindow(){
    return win;
}


MyDisplay& MyWindow::GetDisplay(){
    return display;
}

void MyWindow::SetBackground(XColor col){
    XSetBackground(display.disp, gc, col.pixel);
}
void MyWindow::SetForeground(XColor col){
    XSetForeground(display.disp, gc, col.pixel);
}

int MyWindow::Width() const {
    return width;
}

int MyWindow::Height() const {
    return height;
}

void MyWindow::SetWidth(int w) {
    width = w;
}

void MyWindow::SetHeight(int h) {
    height = h;
}


void MyWindow::Create(){
    Create(0,0);
}
void MyWindow::Create(int posx, int posy){
    XSetWindowAttributes wattr;
    XSetWindowAttributes wattr2;
    XColor black, black_e;

    Window root = DefaultRootWindow(display.disp);
    // Visual *visual = DefaultVisual(display.disp, display.screen);

    // Events for normal windows
    #define BASIC_EVENT_MASK (StructureNotifyMask|ExposureMask| \
        PropertyChangeMask|EnterWindowMask|LeaveWindowMask|KeyPressMask|KeyReleaseMask|KeymapStateMask)

    #define NOT_PROPAGATE_MASK (KeyPressMask|KeyReleaseMask|ButtonPressMask|ButtonReleaseMask|PointerMotionMask|ButtonMotionMask)

    unsigned long mask = CWBackPixmap|CWBackPixel|CWBorderPixel|CWWinGravity|CWBitGravity|
                         CWSaveUnder|CWEventMask|CWDontPropagate|CWOverrideRedirect|CWColormap;

    unsigned long mask2 = CWDontPropagate;
    wattr.background_pixmap = None;
    wattr.background_pixel = black.pixel;

    wattr.border_pixel=0;

    wattr.win_gravity=NorthWestGravity;
    wattr.bit_gravity=ForgetGravity;

    wattr.save_under=1;

    wattr.event_mask=BASIC_EVENT_MASK;

    wattr.do_not_propagate_mask=NOT_PROPAGATE_MASK;
    wattr2.do_not_propagate_mask=NOT_PROPAGATE_MASK;

    wattr.override_redirect=0;

    wattr.colormap= DefaultColormap(display.disp, display.screen);

    XAllocNamedColor(display.disp, DefaultColormap(display.disp, display.screen), "black", &black, &black_e);
    
    win = XCreateWindow(display.disp, root, posx, posy, width, height, 1, CopyFromParent, InputOutput, CopyFromParent, mask , &wattr);
    input = XCreateWindow(display.disp, root, posx, posy, width, height, 0, 0, InputOnly, NULL, mask2 , &wattr2);

    // Tell the Window Manager not to draw window borders (frame) or title.
    wattr.override_redirect = 1;
    XChangeWindowAttributes(display.disp, input, CWOverrideRedirect, &wattr);
    XChangeWindowAttributes(display.disp, win, CWOverrideRedirect, &wattr);
    XSelectInput(display.disp, input, ExposureMask |
	    ButtonPressMask |
	    ButtonReleaseMask |
	    ButtonMotionMask |
	    StructureNotifyMask);
    XRRSelectInput(display.disp, input, RRScreenChangeNotifyMask);

    // Show the window
    XMapWindow(display.disp, win);
    XMapWindow(display.disp, input);
    display.Flush();

    buf = XCreatePixmap(display.disp, win, width, height, display.Depth());
    
    XGCValues gcv;
    gcv.foreground = display.MakeXColor(255,255,255).pixel;
    gcv.background = display.MakeXColor(0,0,0).pixel;
    unsigned long gcmask = GCForeground | GCBackground;
    gc = XCreateGC(display.disp, buf, gcmask, &gcv);

    Clear();
}

void MyWindow::UnmapInput(){
    XUnmapWindow(display.disp, input);
}

void MyWindow::MapInput(){
    XMapWindow(display.disp, input);
}
void MyWindow::UnmapWin(){
    XUnmapWindow(display.disp, win);
}

void MyWindow::MapWin(){
    XMapWindow(display.disp, win);
}



int MyWindow::PosX(){
    XWindowAttributes attr;
    XGetWindowAttributes(display.disp, win, &attr);
    return attr.x;
}

int MyWindow::PosY(){
    XWindowAttributes attr;
    XGetWindowAttributes(display.disp, win, &attr);
    return attr.y;
}
 
void MyWindow::Move(int px, int py){
    Move(px, py, -1, -1);
}
void MyWindow::Move(int px, int py, int w, int h){
    if (w == Width() && h == Height())
	w = -1;
    if (w > -1){
	SetWidth(w);
	SetHeight(h);
    }
    XWindowChanges changes;
    changes.x = px;
    changes.y = py;
    if (w > -1){
	changes.width = w;
	changes.height = h;
    }
    XConfigureWindow(display.disp, win, CWY | CWX | ((w > -1) ? (CWWidth | CWHeight) : 0), &changes);
    XConfigureWindow(display.disp, input, CWY | CWX | ((w > -1) ? (CWWidth | CWHeight) : 0), &changes);
    if (w > -1 && buf_name == ""){
	XFreePixmap(display.disp, buf);
	buf = XCreatePixmap(display.disp, win, width, height, display.Depth());
    }
    XSync(display.disp, False);
}

void MyWindow::SetTransparent(bool trans){
    unsigned char fbitmap[Height()*(Width())/8+1];
    memset(fbitmap, 255, sizeof(fbitmap));
    transparent = trans;
    Visual *vis = DefaultVisual(display.disp, display.screen);
    double rRatio = vis->red_mask / 255.0,
	   gRatio = vis->green_mask / 255.0,
	   bRatio = vis->blue_mask / 255.0;

    if (transparent){
	XImage* image = XGetImage(display.disp, buf, 0, 0, Width(), Height(), AllPlanes, ZPixmap);
	unsigned int* buf32 = (unsigned int*) image->data;
	unsigned short* buf16 = (unsigned short*) image->data;
	for (int x=0; x<Width(); ++x)
	    for (int y=0; y<Height(); ++y){
		int pix = y * Width() + x;
		int r,g,b;
		if (display.Depth() >= 24){
		    r = (buf32[pix] & vis->red_mask) / rRatio,
		    g = (buf32[pix] & vis->green_mask) / gRatio,
		    b = (buf32[pix] & vis->blue_mask) / bRatio;
		} else {
		    r = (buf16[pix] & vis->red_mask) / rRatio,
		    g = (buf16[pix] & vis->green_mask) / gRatio,
		    b = (buf16[pix] & vis->blue_mask) / bRatio;
		}
		if (!r && !g && !b)
		    fbitmap[pix/8] &= ~(1 << (pix%8));
	    }
	XDestroyImage(image);
    }
    Pixmap g_bitmap = XCreateBitmapFromData (display.disp, 
	    RootWindow(display.disp, display.screen),
	    (char*)fbitmap,
	    Width(),
	    Height());

    XShapeCombineMask(display.disp, win, ShapeBounding, 0, 0, g_bitmap, ShapeSet);
    XFreePixmap(display.disp, g_bitmap);

    display.Flush();
}


void MyWindow::Refresh(){
    Refresh(0, 0, Width(), Height());
}

void MyWindow::Refresh(int px, int py, int w, int h){
    if (buf_name != "")
	LOG(INFO) << "Refreshing with buffer " << buf_name;
    XCopyArea(display.disp, buf, win, gc, px, py, w, h, px, py);
    display.Flush();
}

void MyWindow::AddImage(int px, int py, const string& name, int* ret_width, int* ret_height, ImageBegin image_begin, int fr, int fg, int fb){
    Visual *vis = DefaultVisual(display.disp, display.screen);
    PngImage& png = image_cache.GetImage(name);;
    if (ret_width && ret_height){
	*ret_width = png.get_width();
	*ret_height = png.get_height();
    }
    if (image_begin == CENTER){
	px -= png.get_width()/2;
	py -= png.get_height()/2;
    }
    unsigned int *buf32;
    unsigned short *buf16;
    if (display.Depth() >= 24)
	buf32 = new unsigned int[png.get_width() * png.get_height()];
    else 
	buf16 = new unsigned short[png.get_width() * png.get_height()];
    double rRatio = vis->red_mask / 255.0,
	   gRatio = vis->green_mask / 255.0,
	   bRatio = vis->blue_mask / 255.0;

 //   XImage* wimage = XGetImage(display.disp, buf, 0, 0, Width(), Height(), AllPlanes, ZPixmap);

    for (int x=0; x<png.get_width(); ++x)
	for (int y=0; y<png.get_height(); ++y){

	    png::rgb_pixel p = png.get_pixel(x,y);

	    unsigned int r = p.red,
			 g = p.green,
			 b = p.blue;
	    
	    bool inside = true;
	    if (x == 0 || x == png.get_width()-1 || y == 0 || y == png.get_height()-1){
		inside = false;
	    } else {
		png::rgb_pixel p1 = png.get_pixel(x-1,y);
		png::rgb_pixel p2 = png.get_pixel(x+1,y);
		png::rgb_pixel p3 = png.get_pixel(x,y-1);
		png::rgb_pixel p4 = png.get_pixel(x,y+1);

		if (!((p1.red || p1.green || p1.blue) &&
		    (p2.red || p2.green || p2.blue) &&
		    (p3.red || p3.green || p3.blue) &&
		    (p4.red || p4.green || p4.blue) ))
		    inside = false;
	    }

	    if (fr > -1 && !inside && (r | g | b) != 0){
		r = 255 - fr;
		g = 255 - fg;
		b = 255 - fb;
	    }
	    else if (fr > -1 && (r | g | b) != 0){
		r = fr;
		g = fg;
		b = fb;
	    }

	    r*=rRatio;
	    g*=gRatio;
	    b*=bRatio;

	    r&=vis->red_mask;
	    g&=vis->green_mask;
	    b&=vis->blue_mask;
	    unsigned int pix = r | g | b;
	    if (pix == 0){
//		pix = XGetPixel(wimage, x, y);
	    }


	    if (display.Depth() >= 24)
		buf32[y * png.get_width() + x] = pix;
	    else
		buf16[y * png.get_width() + x] = pix;
	}
    XImage* image;
    if (display.Depth() >= 24){
	image = XCreateImage(display.disp,
					   CopyFromParent,
					   display.Depth(),
					   ZPixmap, 0, (char*) buf32, png.get_width(), png.get_height(), 32, 0);
    } else {
	image = XCreateImage(display.disp,
					   CopyFromParent,
					   display.Depth(),
					   ZPixmap, 0, (char*) buf16, png.get_width(), png.get_height(), 16, 0);
    }
    XInitImage(image);
    XPutImage(display.disp, buf, gc, image, 0, 0, px, py, png.get_width(), png.get_height());
    XDestroyImage(image);
}

void MyWindow::Clear(){
    GC tmp_gc;
    XGCValues gcv;
    XGetGCValues(display.disp, gc, GCBackground | GCForeground, &gcv);
    swap(gcv.foreground, gcv.background);
    tmp_gc = XCreateGC(display.disp, buf, GCForeground, &gcv);
    if (buf_name != ""){
	buf_name = "";
	buf = XCreatePixmap(display.disp, win, width, height, display.Depth());
    }
    XFillRectangle(display.disp, buf, tmp_gc, 0, 0, Width(), Height());
    XFreeGC(display.disp, tmp_gc);
}

void MyWindow::AddPixel(int x, int y){
    XDrawPoint(display.disp, buf, gc, x, y);
}

void MyWindow::AddPixel(int x, int y, XColor col){
    GC tmp_gc;
    XGCValues gcv;
    gcv.foreground = col.pixel;
    tmp_gc = XCreateGC(display.disp, buf, GCForeground, &gcv);
    XDrawPoint(display.disp, buf, tmp_gc, x, y);
    XFreeGC(display.disp, tmp_gc);
}


bool MyWindow::IsTransparent(){
    return transparent;
}

bool MyWindow::SetFont(const string& path){
    font_path = path;
    return true;
}

void MyWindow::SetFontSize(int size){
    font_size = size;
}

class BufferCache {
  public:
    void SaveBuffer(Pixmap p, int w, int h, const string& name){
	for (int i=0; i<buffers.size(); ++i)
	    if (buffers[i].name == name && buffers[i].width == w && buffers[i].height == h)
		return;
	buffers.push_back(Buffer(name, p, w, h));
    }
    bool LoadBuffer(const string& name, Pixmap* p, int w, int h){
	LOG(INFO) << "buffers:";
	for (int i=0; i<buffers.size(); ++i)
	    LOG(INFO) << "    " << (int)buffers[i].pixmap;
	for (int i=0; i<buffers.size(); ++i)
	    if (buffers[i].name == name && buffers[i].width == w && buffers[i].height == h){
		*p = buffers[i].pixmap;
		return true;
	    }
	return false;
    }

  private:
    struct Buffer {
	string name;
	Pixmap pixmap;
	int width, height;
	Buffer(string n, Pixmap p, int w, int h):name(n), pixmap(p), width(w), height(h){}
    };

    vector<Buffer> buffers;
} buffer_cache;

void MyWindow::SaveBuffer(const string& name){
    buf_name = name;
    buffer_cache.SaveBuffer(buf, Width(), Height(), name);
    LOG(INFO) << "Saving " << name << " " << (int)buf;
}

bool MyWindow::LoadBuffer(const string& name){
    Pixmap p;
    if (buffer_cache.LoadBuffer(name, &p, Width(), Height())){
	if (buf_name == ""){
	    XFreePixmap(display.disp, buf);
	}
	buf_name = name;
	buf = p;
	LOG(INFO) << "Loading " << name << "  " << (int)buf;
	return true;
    }
    return false;
}

void MyWindow::AddText(int px, int py, const string& text, ImageBegin image_begin, int fr, int fg, int fb){
   FT_Library  library;
   FT_Face     face;
   FT_Matrix   matrix;      // transformation matrix
   FT_Vector   pen;

   FT_UInt glyph_index;
   FT_Error error;

   FT_Bool use_kerning;
   FT_UInt previous = 0;

   double angle = 0;

   /* Set up transformation Matrix */
   matrix.xx = (FT_Fixed)( cos(angle)*0x10000);   /* It would make more sense to do this (below), but, bizzarely, */
   matrix.xy = (FT_Fixed)(-sin(angle)*0x10000);   /* if one does, FT_Load_Glyph fails consistently.               */
   matrix.yx = (FT_Fixed)( sin(angle)*0x10000);  //   matrix.yx = - matrix.xy;
   matrix.yy = (FT_Fixed)( cos(angle)*0x10000);  //   matrix.yy = matrix.xx;

   /* Place starting coordinates in adequate form. */
   pen.x = px*64 ;
   pen.y = (int)(py/64.0);

   /*Count the length of the string */
   int num_bytes = text.size();

   //The array of ucs4 glyph indexes, which will by at most the number of bytes in the utf-8 file.
   long ucs4text[num_bytes+1];

   unsigned char u,v,w,x,y,z;

   int num_chars=0;

   long iii=0;

   while(iii<num_bytes){
       z = text[iii];

       if(z<=127)
	   ucs4text[num_chars] = z;

       if((192<=z)&&(z<=223)){
	   iii++; y = text[iii];
	   ucs4text[num_chars] = (z-192)*64 + (y -128);
       }

       if((224<=z)&&(z<=239)){
	   iii++; y = text[iii];
	   iii++; x = text[iii];
	   ucs4text[num_chars] = (z-224)*4096 + (y -128)*64 + (x-128);
       }

       if((240<=z)&&(z<=247)){
	   iii++; y = text[iii];
	   iii++; x = text[iii];
	   iii++; w = text[iii];
	   ucs4text[num_chars] = (z-240)*262144 + (y -128)*4096 + (x-128)*64 + (w-128);
       }

       if((248<=z)&&(z<=251)){
	   iii++; y = text[iii];
	   iii++; x = text[iii];
	   iii++; w = text[iii];
	   iii++; v = text[iii];
	   ucs4text[num_chars] = (z-248)*16777216 + (y -128)*262144 + (x-128)*4096 + (w-128)*64 +(v-128);
       }

       if((252==z)||(z==253)){
	   iii++; y = text[iii];
	   iii++; x = text[iii];
	   iii++; w = text[iii];
	   iii++; v = text[iii];
	   u = text[iii];
	   ucs4text[num_chars] = (z-252)*1073741824 + (y -128)*16777216   + (x-128)*262144 + (w-128)*4096 +(v-128)*64 + (u-128);
       }

       if((z==254)||(z==255))
	   LOG(INFO) << "Invalid UTF-8 data";

       iii++;
       num_chars++;
   }

   // num_chars now contains the number of characters in the string.
   /*
   std::cout << "Num chars is: "<< num_chars << std::endl;
   */

   /* Initialize FT Library object */
   error = FT_Init_FreeType( &library );
   if (error) 
       LOG(FATAL) << "Error initializing Freetype library";
   /* Initialize FT face object */
   error = FT_New_Face( library,font_path.c_str(),0,&face );
   if (error == FT_Err_Unknown_File_Format)
       LOG(FATAL) << "FreeType: Font was opened, but type not supported";
   else if (error)
       LOG(FATAL) << "FreeType: Could not find or load font file";

   /* Set the Char size */
   error = FT_Set_Char_Size( face,          /* handle to face object           */
			     0,             /* char_width in 1/64th of points  */
			     font_size*64,   /* char_height in 1/64th of points */
			     100,           /* horizontal device resolution    */
			     100 );         /* vertical device resolution      */

   /* A way of accesing the glyph directly */
   FT_GlyphSlot  slot = face->glyph;  // a small shortcut

   /* Does the font file support kerning? */
   use_kerning = FT_HAS_KERNING( face );

   for (int n = 0; n < num_chars; n++){
       /* Convert character code to glyph index */
       glyph_index = FT_Get_Char_Index( face, ucs4text[n] );

       /* Retrieve kerning distance and move pen position */
       if ( use_kerning && previous&& glyph_index ){
	   FT_Vector  delta;
	   FT_Get_Kerning( face,
		   previous,
		   glyph_index,
		   ft_kerning_default, //FT_KERNING_DEFAULT,
		   &delta );

	   /* Transform this kerning distance into rotated space */
	   pen.x += (int) (((double) delta.x)*cos(angle));
	   pen.y +=  (int) (((double) delta.x)*( sin(angle)));
       }

       /* Set transform */
       FT_Set_Transform( face, &matrix, &pen );

       /*set char size*/

       if (error) 
	   LOG(FATAL) << "FreeType: Set char size error.";

       /* Retrieve glyph index from character code */
       glyph_index = FT_Get_Char_Index( face, ucs4text[n] );

       /* Load glyph image into the slot (erase previous one) */
       error = FT_Load_Glyph( face, glyph_index, FT_LOAD_DEFAULT );
       if (error) 
	   LOG(FATAL) << "FreeType: Could not load glyph (in loop). (FreeType error "<< error <<").";

       /* Convert to an anti-aliased bitmap */
       error = FT_Render_Glyph( face->glyph, ft_render_mode_normal );
       if (error) 
	   LOG(FATAL) << "FreeType: Render glyph error";

       FT_Bitmap& bitmap = slot->bitmap;
       unsigned int *buf32;
       unsigned short *buf16;
       if (display.Depth() >= 24)
	   buf32 = new unsigned int[bitmap.width * bitmap.rows];
       else 
	   buf16 = new unsigned short[bitmap.width * bitmap.rows];
       Visual *vis = DefaultVisual(display.disp, display.screen);
       double rRatio = vis->red_mask / 255.0,
	      gRatio = vis->green_mask / 255.0,
	      bRatio = vis->blue_mask / 255.0;

       /* Now, draw to our target surface */
       for (int x = 0; x<bitmap.width; ++x)
	   for (int y = 0; y<bitmap.rows; ++y){
	       int ind = x + y*bitmap.width;
	       int val = bitmap.buffer[ind];
	       bool inside = true;
	       if (x == 0 || x == bitmap.width-1 || y == 0 || y == bitmap.rows -1){
		   inside= false;
	       } else{
		   if (bitmap.buffer[ind -1] == 0)
		       inside = false;
		   if (bitmap.buffer[ind +1] == 0)
		       inside = false;
		   if (bitmap.buffer[ind -bitmap.width] == 0)
		       inside = false;
		   if (bitmap.buffer[ind +bitmap.width] == 0)
		       inside = false;
	       }

	       unsigned int r = 0,g = 0,b = 0;
	       
	       if (val > 0){
		   r = fr;
		   g = fg;
		   b = fb;
		   if (!inside){
		       r = 255 - r;
		       g = 255 - g;
		       b = 255 - b;
		   }
	       }
	       r*=rRatio;
	       g*=gRatio;
	       b*=bRatio;

	       r&=vis->red_mask;
	       g&=vis->green_mask;
	       b&=vis->blue_mask;
	       unsigned int pix = r | g | b;
	       if (pix == 0){
		   //		pix = XGetPixel(wimage, x, y);
	       }


	       if (display.Depth() >= 24)
		   buf32[ind] = pix;
	       else
		   buf16[ind] = pix;
	   }
       XImage* image;
       if (display.Depth() >= 24){
	   image = XCreateImage(display.disp,
		   CopyFromParent,
		   display.Depth(),
		   ZPixmap, 0, (char*) buf32, bitmap.width, bitmap.rows, 32, 0);
       } else {
	   image = XCreateImage(display.disp,
		   CopyFromParent,
		   display.Depth(),
		   ZPixmap, 0, (char*) buf16, bitmap.width, bitmap.rows, 16, 0);
       }
       int posx = slot->bitmap_left;
       int posy = -slot->bitmap_top;
       if (image_begin == CENTER){
	   posx -= slot->bitmap.width/2;
	   posy += slot->bitmap.rows/2;
       }

       XInitImage(image);
       XPutImage(display.disp, buf, gc, image, 0, 0, posx, py + posy, bitmap.width, bitmap.rows);
       XDestroyImage(image);


/*       my_draw_bitmap( &slot->bitmap,
	       slot->bitmap_left,
	       y_start + slot->bitmap_top,
	       red,
	       green,
	       blue );*/

       /* Advance to the next position */
       pen.x += slot->advance.x;
       pen.y += slot->advance.y;

       /* record current glyph index */
       previous = glyph_index;
   }

   /* Free the face and the library objects */
   FT_Done_Face    ( face );
   FT_Done_FreeType( library );

}

