/*
 * Copyright (C) 2016 Canonical, Ltd.
 *
 * Authors:
 *  Renato Araujo Oliveira Filho <renato@canonical.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


import QtQuick 2.4
import QtTest 1.0
import Lomiri.Test 0.1

import "../../src/qml/player/"


Item {
    id: root

    width: 800
    height: 600

    visible: true

    // Mock Application object
    QtObject {
        id: mpApplication

        property bool desktopMode: true
        property string playUri: ""
        function getStoredPosition() { return -1 }
        function storePosition(position) {}
    }

    Component {
        id: playerComp

        VideoPlayer {
            id: player
            width: 800
            height: 600
        }
    }

    LomiriTestCase {
        name: "Date tests"

        property var player: null
        readonly property string videUri: Qt.resolvedUrl("../videos/small.ogg")

        when: windowShown

        function init()
        {
            player = playerComp.createObject(root, {})
            waitForRendering(player)
        }

        function cleanup()
        {
            player.destroy()
            player = null
        }

        function test_player_init()
        {
            tryCompare(player, 'paused', false)
            tryCompare(player, 'playing', false)
            tryCompare(player, 'wasPlaying', false)
            tryCompare(player, 'isEmpty', true)
        }

        function test_empty_state()
        {
            tryCompare(player, 'isEmpty', true)
            tryCompare(player, 'controlsActive', true)

            // test if controls stay active after click on empty state
            mouseClick(player)
            tryCompare(player, 'controlsActive', true)

            // play any uri should leave empty state
            player.playUri(videUri)
            tryCompare(player, 'isEmpty', false)
        }

        function test_open_video()
        {
            player.playUri(videUri)
            tryCompare(player, 'paused', false)
            tryCompare(player, 'playing', true)
        }

        function test_play_pause_video()
        {
            player.playUri(videUri)

            tryCompare(player, 'isEmpty', false)
            tryCompare(player, 'playing', true)
            player.playPause()
            tryCompare(player, 'paused', true)
            player.playPause()
            tryCompare(player, 'paused', false)
        }

        function test_click_to_show_and_hide_controls()
        {
            player.playUri(videUri)
            tryCompare(player, 'playing', true)
            tryCompare(player, 'controlsActive', false)

            mouseClick(player)
            tryCompare(player, 'controlsActive', true)

            mouseClick(player)
            tryCompare(player, 'controlsActive', false)
        }

        function test_pause_button()
        {
            // play video
            player.playUri(videUri)
            tryCompare(player, 'playing', true)

            var toolbar = findChild(player, "toolbar")
            tryCompare(toolbar, 'fullVisible', false)

            // show controls
            mouseClick(player)
            tryCompare(player, 'controlsActive', true)
            tryCompare(toolbar, 'fullVisible', true)
            tryCompare(player, 'paused', false)

            // click to pause
            player.playPause()
            tryCompare(player, 'paused', true)
        }

        function test_play_after_pause()
        {
            // play video
            var toolbar = findChild(player, "toolbar")
            player.playUri(videUri)
            tryCompare(player, 'playing', true)

            // show controls
            mouseClick(player)
            tryCompare(player, 'controlsActive', true)
            tryCompare(toolbar, 'fullVisible', true)

            // click to pause
            player.playPause()
            tryCompare(player, 'paused', true)

            player.playPause()
            tryCompare(player, 'paused', false)
        }
    }
}
