#ifndef OSL_PTYPE_H
#define OSL_PTYPE_H

#include "osl/config.h"
#include "osl/player.h"
#include <cassert>
#include <iosfwd>

namespace osl
{

  /** 駒の種類を4ビットでコード化する */
  enum Ptype
  {
    PTYPE_EMPTY=0,
    PTYPE_EDGE=1,
    PPAWN=2,
    PLANCE=3,
    PKNIGHT=4,
    PSILVER=5,
    PBISHOP=6,
    PROOK=7,
    KING=8,
    GOLD=9,
    PAWN=10,
    LANCE=11,
    KNIGHT=12,
    SILVER=13,
    BISHOP=14,
    ROOK=15,

    PTYPE_MIN=0,
    PTYPE_BASIC_MIN=KING,
    PTYPE_PIECE_MIN=2,
    PTYPE_MAX=15,
  };
  const int PTYPE_SIZE=PTYPE_MAX-PTYPE_MIN+1;
  
  std::istream& operator>>(std::istream& is, Ptype& ptype);
  std::ostream& operator<<(std::ostream& os,const Ptype ptype);
  
  /**
   * int等からcastして作ったptypeが，正しい範囲に入っているかどうかのチェック
   */
  bool isValid(Ptype ptype);

  /**
   * ptypeが空白やEDGEでないかのチェック
   */
  inline bool isPiece(Ptype ptype)
  {
    assert(isValid(ptype));
    return static_cast<int>(ptype)>=PTYPE_PIECE_MIN;
  }
  /**
   * ptypeが基本型(promoteしていない)かのチェック
   */
  inline bool isBasic(Ptype ptype)
  {
    assert(isValid(ptype));
    return static_cast<int>(ptype)>PROOK;
  }
  
  /**
   * ptypeがpromote後の型かどうかのチェック
   */
  inline bool isPromoted(Ptype ptype)
  {
    assert(isPiece(ptype));
    return static_cast<int>(ptype)<KING; 
  }

  /**
   * ptypeがpromote可能な型かどうかのチェック
   * promote済みの場合はfalseを返す
   */
  inline bool canPromote(Ptype ptype)
  {
    assert(isPiece(ptype));
    return static_cast<int>(ptype)>GOLD; 
  }
  
  /** 
   * ptypeがpromote後の型の時に，promote前の型を返す．
   * promoteしていない型の時はそのまま返す
   */
  inline Ptype unpromote(Ptype ptype)
  {
    assert(isPiece(ptype));
    Ptype ret=static_cast<Ptype>(static_cast<int>(ptype)|8); 
    assert(isPiece(ret));
    return ret;
  }
  inline Ptype unpromoteSafe(Ptype ptype)
  {
    if (! isPiece(ptype))
      return ptype;
    return unpromote(ptype);
  }
  
  /** 
   * promote可能なptypeに対して，promote後の型を返す
   * promote不可のptypeを与えてはいけない．
   */
  inline Ptype promote(Ptype ptype)
  {
    assert(canPromote(ptype));
    Ptype ret=static_cast<Ptype>(static_cast<int>(ptype)-8); 
    assert(isPiece(ret));
    return ret;
  }

  inline bool isMajorBasic(Ptype ptype)
  {
    return ptype >= 14;
  }
  inline bool isMajor(Ptype ptype)
  {
    assert(isPiece(ptype));
    return isMajorBasic(unpromote(ptype));
  }
  inline bool isMajorNonPieceOK(Ptype ptype)
  {
    return (static_cast<int>(ptype)|8)>=14;
  }
  
  /**
   * Player + Ptype [-15, 15] 
   * PtypeO の O は Owner の O
   */
  enum PtypeO {
    PTYPEO_MIN= PTYPE_EMPTY-16,
    PTYPEO_MAX= 15,
  };
  
#define NEW_PTYPEO(player,ptype) static_cast<PtypeO>(static_cast<int>(ptype)-(16&static_cast<int>(player)))
  inline unsigned int ptypeOIndex(PtypeO ptypeo)
  {
    const int result = ptypeo - PTYPEO_MIN;
    assert(result >= 0);
    return result;
  }
  inline PtypeO newPtypeO(Player player,Ptype ptype)
  {
    return static_cast<PtypeO>(static_cast<int>(ptype)-(16&static_cast<int>(player)));
  }
  
  
  inline Ptype getPtype(PtypeO ptypeO)
  {
    return static_cast<Ptype>(static_cast<int>(ptypeO)& 15);
  }
  
  /** pieceをpromoteさせる. promote不可のptypeを与えてはいけない．*/
  inline PtypeO promote(PtypeO ptypeO)
  {
    assert(canPromote(getPtype(ptypeO)));
    PtypeO ret=static_cast<PtypeO>(static_cast<int>(ptypeO)-8); 
    assert(isPiece(getPtype(ret)));
    return ret;
  }
  
  /** pieceを引数次第でpromoteさせる */
  inline PtypeO promoteWithMask(PtypeO ptypeO,int promoteMask)
  {
    assert(promoteMask==0 || promoteMask==0x800000);
    PtypeO ret=static_cast<PtypeO>(static_cast<int>(ptypeO)-(promoteMask>>20)); 
    return ret;
  }
  
  /** pieceをunpromoteさせる.  promoteしていないptypeを与えてもよい */
  inline PtypeO unpromote(PtypeO ptypeO)
  {
    return static_cast<PtypeO>(static_cast<int>(ptypeO)|8); 
  }

  bool isValidPtypeO(int ptypeO);
  
  /**
   * EMPTY, EDGEではない
   */
  inline bool isPiece(PtypeO ptypeO)
  {
    assert(isValidPtypeO(ptypeO));
    return isPiece(getPtype(ptypeO));
  }

  inline Player getOwner(PtypeO ptypeO)
  {
    assert(isPiece(ptypeO));
    return static_cast<Player>(static_cast<int>(ptypeO)>>31);
  }
  

  /** unpromoteすると共に，ownerを反転する． */
  inline PtypeO captured(PtypeO ptypeO)
  {
    assert(isPiece(ptypeO));
    return static_cast<PtypeO>((static_cast<int>(ptypeO)|8)^(~15));
  }
  
  /** owner を反転する */
  inline PtypeO alt(PtypeO ptypeO)
  {
    assert(isPiece(ptypeO));
    return static_cast<PtypeO>(static_cast<int>(ptypeO)^(~15));
  }

  /** 
   * Pieceの時にはowner を反転する 
   * 
   */
  inline PtypeO altIfPiece(PtypeO ptypeO)
  {
    int v=static_cast<int>(ptypeO);
    return static_cast<PtypeO>(v^((1-(v&15))&~15));
  }

  inline bool canPromote(PtypeO ptypeO)
  {
    return canPromote(getPtype(ptypeO));
  }


  /**
   * ptypeOが promote済みかどうか
   */
  inline bool isPromoted(PtypeO ptypeO)
  {
    assert(isValidPtypeO(ptypeO));
    return isPromoted(getPtype(ptypeO));
  }


  const PtypeO PTYPEO_EMPTY=newPtypeO(BLACK,PTYPE_EMPTY);
  const PtypeO PTYPEO_EDGE=newPtypeO(WHITE,PTYPE_EDGE);
  
  std::ostream& operator<<(std::ostream& os,const PtypeO ptypeO);
  
  const int PTYPEO_SIZE=PTYPEO_MAX-PTYPEO_MIN+1;
  
} // namespace osl

#endif /* OSL_PTYPE_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
