/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once


/////////////////////// Qt includes
#include <QString>


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/export-import-config.h"
#include "MsXpS/libXpertMassCore/PolChemDef.hpp"
#include "MsXpS/libXpertMassCore/Formula.hpp"
#include "MsXpS/libXpertMassCore/Monomer.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{


class DECLSPEC FragmentationRule
{
  public:
  FragmentationRule(PolChemDefCstSPtr pol_chem_def_csp = nullptr,
                    const QString &name                = QString(),
                    const QString &prev_code            = QString(),
                    const QString &current_code        = QString(),
                    const QString &next_code           = QString(),
                    const QString &comment             = QString(),
                    const QString &formula_string      = QString());

  FragmentationRule(PolChemDefCstSPtr pol_chem_def_csp,
                    const QDomElement &element,
                    int version);

  FragmentationRule(const FragmentationRule &other);

  virtual ~FragmentationRule();

  //////////////// THE POLCHEMDEF /////////////////////
  void setPolchemDefCstSPtr(PolChemDefCstSPtr pol_chem_def_csp);
  PolChemDefCstSPtr getPolchemDefCstSPtr() const;

  //////////////// THE NAME /////////////////////
  void setName(const QString &name);
  const QString &getName();

  //////////////// THE PREVIOUS MONOMER CODE /////////////////////
  void setPrevCode(const QString &code);
  const QString &getPrevCode() const;

  //////////////// THE CURRENT MONOMER CODE /////////////////////
  void setCurrCode(const QString &code);
  const QString &getCurrCode() const;

  //////////////// THE NEXT MONOMER CODE /////////////////////
  void setNextCode(const QString &code);
  const QString &getNextCode() const;

  //////////////// THE COMMENT /////////////////////
  void setComment(const QString &comment);
  const QString &getComment() const;

  //////////////// THE FORMULA /////////////////////
  void setFormula(const Formula &formula);
  void setFormula(const QString &formula_string);
  const Formula &getFormulaCstRef() const;
  Formula &getFormulaRef();

  //////////////// OPERATORS /////////////////////
  FragmentationRule &operator=(const FragmentationRule &other);
  bool operator==(const FragmentationRule &other) const;
  bool operator!=(const FragmentationRule &other) const;

  //////////////// VALIDATIONS /////////////////////
  bool validate(ErrorList *error_list_p) const;
  bool isValid() const;

  //////////////// UTILITIES /////////////////////
  QString toString() const;

  //////////////// XML DATA LOADING WRITING /////////////////////
  bool renderXmlFgrElement(const QDomElement &element);
  QString
  formatXmlFgrElement(int offset,
                      const QString &indent = Utils::xmlIndentationToken);

  protected:
  PolChemDefCstSPtr mcsp_polChemDef = nullptr;
  QString m_name;
  QString m_prevCode;
  QString m_currCode;
  QString m_nextCode;
  QString m_comment;
  Formula m_formula;
  mutable bool m_isValid = false;
};

typedef std::shared_ptr<FragmentationRule> FragmentationRuleSPtr;
typedef std::shared_ptr<const FragmentationRule> FragmentationRuleCstSPtr;

} // namespace libXpertMassCore
} // namespace MsXpS
