/*
  Copyright (C) 2008 Jan Friederich

        This file is part of mmpong.

        mmpong is free software: you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation, either version 3 of the License, or
        (at your option) any later version.

        mmpong is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "glhelper.h"

void create_box(GLuint *list) {
        *list = glGenLists(1);
        glNewList(*list, GL_COMPILE);

                glBegin(GL_QUADS);
                        // front face
                        glNormal3f(0.0f, 0.0f, 1.0f);
                        glVertex3f(-0.5f, -0.5f, 0.5f);
                        glVertex3f(0.5f, -0.5f, 0.5f);
                        glVertex3f(0.5f, 0.5f, 0.5f);
                        glVertex3f(-0.5f, 0.5f, 0.5f);

                        // back face
                        glNormal3f(0.0f, 0.0f, -1.0f);
                        glVertex3f(0.5f, -0.5f, -0.5f);
                        glVertex3f(-0.5f, -0.5f, -0.5f);
                        glVertex3f(-0.5f, 0.5f, -0.5f);
                        glVertex3f(0.5f, 0.5f, -0.5f);

                        // right face
                        glNormal3f(1.0f, 0.0f, 0.0f);
                        glVertex3f(0.5f, -0.5f, 0.5f);
                        glVertex3f(0.5f, -0.5f, -0.5f);
                        glVertex3f(0.5f, 0.5f, -0.5f);
                        glVertex3f(0.5f, 0.5f, 0.5f);

                        // left face
                        glNormal3f(-1.0f, 0.0f, 0.0f);
                        glVertex3f(-0.5f, -0.5f, -0.5f);
                        glVertex3f(-0.5f, -0.5f, 0.5f);
                        glVertex3f(-0.5f, 0.5f, 0.5f);
                        glVertex3f(-0.5f, 0.5f, -0.5f);

                        // top face
                        glNormal3f(0.0f, 1.0f, 0.0f);
                        glVertex3f(-0.5f, 0.5f, 0.5f);
                        glVertex3f(0.5f, 0.5f, 0.5f);
                        glVertex3f(0.5f, 0.5f, -0.5f);
                        glVertex3f(-0.5f, 0.5f, -0.5f);

                        // bottom face
                        glNormal3f(0.0f, -1.0f, 0.0f);
                        glVertex3f(-0.5f, -0.5f, -0.5f);
                        glVertex3f(0.5f, -0.5f, -0.5f);
                        glVertex3f(0.5f, -0.5f, 0.5f);
                        glVertex3f(-0.5f, -0.5f, 0.5f);
                glEnd();

        glEndList();
}

void create_cylinder(GLuint *list, const GLuint n) {
        const GLfloat step = 2.0f * M_PI / n;

        *list = glGenLists(1);
        glNewList(*list, GL_COMPILE);
                // top
                glBegin(GL_TRIANGLE_FAN);
                        glNormal3f(0.0f, 0.0f, 1.0f);
                        glVertex3f(0.0f, 0.0f, 0.5f);
                        for (GLuint k = 0; k < n+1; ++k) {
                                glVertex3f(0.5f*cosf(step*k), 0.5f*sinf(step*k), 0.5f);
                        }
                glEnd();
                // bottom
                glBegin(GL_TRIANGLE_FAN);
                        glNormal3f(0.0f, 0.0f, -1.0f);
                        glVertex3f(0.0f, 0.0f, -0.5f);
                        for (GLuint k = 0; k < n+1; ++k) {
                                glVertex3f(0.5f*cosf(step*k), -0.5f*sinf(step*k), -0.5f);
                        }
                glEnd();
                // side
                glBegin(GL_QUAD_STRIP);
                        for (GLuint k = 0; k < n+1; ++k) {
                                glNormal3f(cosf(step*k), sinf(step*k), 0.0f);
                                glVertex3f(0.5f*cosf(step*k), 0.5f*sinf(step*k), 0.5f);
                                glVertex3f(0.5f*cosf(step*k), 0.5f*sinf(step*k), -0.5f);
                        }
                glEnd();

        glEndList();
}

void create_ball(GLuint *list, const GLuint m, const GLuint n) {
        const GLfloat slice = 0.5f * M_PI / m;
        const GLfloat step = 2.0f * M_PI / n;
        const GLfloat r = 0.5f;
        GLfloat h1 = cosf(slice*(m-1)), h2;

        *list = glGenLists(1);
        glNewList(*list, GL_COMPILE);
                // top
                glBegin(GL_TRIANGLE_FAN);
                        glNormal3f(0.0f, 0.0f, 1.0f);
                        glVertex3f(0.0f, 0.0f, r);
                        for (GLuint k = 0; k < n+1; ++k) {
                                glNormal3f(h1*cosf(step*k), h1*sinf(step*k), sinf(slice*(m-1)));
                                glVertex3f(r*h1*cosf(step*k), r*h1*sinf(step*k), r*sinf(slice*(m-1)));
                        }
                glEnd();

                // bottom
                glBegin(GL_TRIANGLE_FAN);
                        glNormal3f(0.0f, 0.0f, -1.0f);
                        glVertex3f(0.0f, 0.0f, -r);
                        for (GLuint k = 0; k < n+1; ++k) {
                                glNormal3f(h1*cosf(step*k), -h1*sinf(step*k), -sinf(slice*(m-1)));
                                glVertex3f(r*h1*cosf(step*k), -r*h1*sinf(step*k), -r*sinf(slice*(m-1)));
                        }
                glEnd();

                // side
                glBegin(GL_QUAD_STRIP);
                        for (GLint j = -m+1; j < (GLint)m-1; ++j) {
                                h1 = cosf(slice*(j+1));
                                h2 = cosf(slice*j);
                                for (GLuint k = 0; k < n+1; ++k) {
                                        glNormal3f(h1*cosf(step*k), h1*sinf(step*k), sinf(slice*(j+1)));
                                        glVertex3f(r*h1*cosf(step*k), r*h1*sinf(step*k), r*sinf(slice*(j+1)));
                                        glNormal3f(h2*cosf(step*k), h2*sinf(step*k), sinf(slice*j));
                                        glVertex3f(r*h2*cosf(step*k), r*h2*sinf(step*k), r*sinf(slice*j));
                                }
                        }
                glEnd();

        glEndList();
}

int load_png(GLuint *texture, const char *filename) {
        SDL_Surface *surface = NULL;
        GLuint channels = 0, format = 0;
        GLubyte *data = NULL;

        surface = IMG_Load(filename);
        if (surface == NULL) {
                fprintf(stderr, "load_png: Unable to load bitmap: %s\n", filename);
                return -1;
        }
        if ((surface = SDL_DisplayFormatAlpha(surface)) == NULL) {
                fprintf(stderr, "load_png: Unable to format bitmap: %s\n", filename);
                return -1;
        }

        channels = surface->format->BytesPerPixel;
        if (channels == 4) {
                //format = (surface->format->Rmask == 0xff) ? GL_RGBA : GL_BGRA;
                format = GL_RGBA;
        }
        else if (channels == 3) {
                //format = (surface->format->Rmask == 0xff) ? GL_RGB : GL_BGR;
                format = GL_RGB;
        }

        data = (GLubyte *)malloc(surface->w * surface->h * channels);
        if (data == NULL) {
                fprintf(stderr, "load_png: Unable to allocate memory: %s\n", filename);
                return -1;
        }

        for(int row = 0; row < surface->h; ++row) {
                memcpy(data + channels * surface->w * (surface->h - 1 - row), (char *)surface->pixels + row * surface->w * channels, surface->w * channels);
        }

        glGenTextures(1, texture);
        glBindTexture(GL_TEXTURE_2D, *texture);

        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);

        glTexImage2D(GL_TEXTURE_2D, 0, channels, surface->w, surface->h, 0, format, GL_UNSIGNED_BYTE, data);

        free(data);
        SDL_FreeSurface(surface);

        return 0;
}

