"""
Unit Test Case
"""
#  Copyright (C) 2004  Henning Jacobs <henning@srcco.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  $Id: testCoCuMa_Server.py 82 2004-07-11 13:01:44Z henning $

import unittest
from CoCuMa_Server import CoCuMa_Server
from vcard import vCard
from vcalendar import vEvent
from testvcard import fillDemoCard
from testvcalendar import fillDemoJournal

class CoCuMa_ServerTestCase(unittest.TestCase):
    TESTFN = '*CoCuMa_Server TestCase Card*'
    TESTCATEGORIES = 'TestCase, CoCuMa_Server, Private'
    TESTFN2 = 'Will be deleted soon..'
    TESTCATEGORIES2 = u'Delete me, TestCase, Umlaut\xe4' 
    def setUp(self):
        import tempfile
        self.tmpabkfilename = tempfile.mktemp()
        self.tmpcalfilename = tempfile.mktemp()
        self.democard = vCard()
        fillDemoCard(self.democard)
        self.demojour = vEvent()
        fillDemoJournal(self.demojour)
        self.server = CoCuMa_Server(self.tmpabkfilename, self.tmpcalfilename)
        self.server.SessionInit()
 
    def tearDown(self):
        "delete our temporary addressbook and calendar files"
        self.server.SessionQuit()
        self.server._shutdown()
        import os
        try:
            os.unlink(self.tmpabkfilename)
            os.unlink(self.tmpcalfilename)
        except OSError:
            # Ignore non-existing files
            pass

    def testEmptyDB(self):
        "operations on empty database"
        handles = self.server.ListHandles()
        self.assertEqual(handles, [])
        jourhdl = self.server.NewJournal()
        self.server.DelJournal(jourhdl)
        handle = self.server.NewContact()
        self.server.PutContact(handle, self.democard.VCF_repr())
        self.server.DelContact(handle)
        handles = self.server.ListHandles()
        self.assertEqual(handles, [])
        
    def testLoadSave(self):
        "saving and loading database"
        # Create new card with some inital values set:
        handle = self.server.NewContact({
            'fn'        : self.TESTFN,
            'categories': self.TESTCATEGORIES})
        # Now Add a Calendar Entry:    
        jourhdl = self.server.NewJournal()
        self.server.PutJournal(jourhdl, self.demojour.VCF_repr())
        # Create another contact:    
        temphandle = self.server.NewContact({
            'fn'        : self.TESTFN2,
            'categories': self.TESTCATEGORIES2})
        # handles are sorted (by DisplayName):
        handles = self.server.ListHandles()
        self.assertEqual([handle, temphandle], handles)
        # Query single field:
        attrs = self.server.QueryAttributes(handles, "FormattedName")
        self.assertEqual(attrs[0], self.TESTFN)
        # Query multiple fields:
        attrs = self.server.QueryAttributes(handles, ("FormattedName", "Categories"))
        self.assertEqual(attrs[0][0], self.TESTFN)
        self.assertEqual(attrs[0][1], self.TESTCATEGORIES)
        # Push our Demo Card to the self.server:
        self.server.PutContact(handle, self.democard.VCF_repr())
        # Save to file:
        self.server.SessionQuit()
        self.server._shutdown()
        self.server = None

        # Now try again (this time the file has to be loaded from disk):
        self.server = CoCuMa_Server(self.tmpabkfilename, self.tmpcalfilename)
        self.server.SessionInit()
        handles = self.server.ListHandles()
        # Query multiple fields:
        attrs = self.server.QueryAttributes(handles, ("FormattedName", "Categories"))
        self.assertEqual(attrs[1][0], self.TESTFN2)
        self.assertEqual(attrs[1][1], self.TESTCATEGORIES2)
        # Delete our second card:
        self.server.DelContact(handles[1])
        # now try our first one:
        data = self.server.GetContact(handles[0])
        self.assertEqual(data, self.democard.VCF_repr())
        # Check if Journal is still there:
        data = self.server.GetJournal(jourhdl)
        self.assertEqual(data, self.demojour.VCF_repr())

if __name__ == "__main__":
    unittest.main()
