///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// input space_constitution files:
//   idiststream& operator >> (idiststream&, space_constitution&);
//
// author: Pierre.Saramito@imag.fr
//
// date: 19 dec 2011
//
// Note:
//  - some technical stuff with bison & flex
//  - some others for distributed issues:
//        * lecture du fichier sur proc=0 -> arbre sur proc=0
//        * lecture ecriture arbre sur stringstream sur proc=0
//        * broadcast string sur ts procs
//        * lecture du stringstream sur ts les proc -> arbre
//        * conversion arbre en space_constitution sur ts les procs
//

// 
#include <sstream> // flex include it, but in namespace rheolef
#include <cstring>
#include "rheolef/space.h"

namespace rheolef {

// ================================================================================
// part 1 : temporary data structure: tree
// ================================================================================

typedef std::pair<std::string,std::string>       base_geo_pair;

struct tree_type {
// data:
  bool                                 is_hier;
  base_geo_pair                        base_geo;
  std::list<tree_type>                 hier;
  std::string                          spec;
// allocators:
  tree_type(const std::string& base, const std::string& geo)
	: is_hier(false), base_geo(base,geo), hier(), spec("undef") {}
  tree_type() : is_hier(true), base_geo(), hier(), spec("undef") {}
  tree_type(const std::string& s, const std::list<tree_type>& h) : is_hier(true), base_geo(), hier(h), spec(s) {}
};
typedef std::list<tree_type> list_type;

static tree_type* result_ptr;

} // namespace rheolef

// ================================================================================
// part 2 : read from idiststeam and build as tree_type* result_ptr
// ================================================================================
/* AIX requires this to be the first thing in the file.  */
#ifndef __GNUC__
# if _RHEOLEF_HAVE_ALLOCA_H
#  include <alloca.h>
# else
#  ifdef _AIX
#pragma alloca
#  else
#   ifndef alloca /* predefined by HP cc +Olibcalls */
char *alloca ();
#   endif
#  endif
# endif
#endif

namespace rheolef {

using namespace std;

typedef size_t size_type;

static size_type space_constitution_line_no = 1;
static size_type space_constitution_n_error = 0;

extern int space_constitution_lex();
void space_constitution_error (const char* msg) {
  std::string near;
  error_macro("space constitution input:" << space_constitution_line_no << ": " << msg);
  space_constitution_n_error++;
}
int space_constitution_wrap () { return 1; }

static std::vector<std::string> symbol_table;
static const std::string& symbol (size_t i) { return symbol_table[i]; }
static size_t insert (const std::string& str) {
        size_t i = symbol_table.size();
        symbol_table.push_back (str);
        return i;
}
#define YYMALLOC ::malloc
#define YYFREE   ::free

#include "space_constitution_yacc.cc"

// avoid re-definition of YY_NULL within flex
#ifdef YY_NULL
#undef YY_NULL
#endif
#include "space_constitution_lex.cc"

static yyFlexLexer input_space_constitution;

int space_constitution_lex() { return input_space_constitution.yylex(); }

void
tree_dump (const tree_type& x, std::ostream& out, size_t level = 0)
{
  if (!x.is_hier) {
    out << x.base_geo.first << "(" << x.base_geo.second << ")";
    return;
  }
  if (x.spec != "mixed") { out << x.spec; }
  if (x.spec != "mixed" || level != 0) { out << "("; }
  std::list<tree_type>::const_iterator iter = x.hier.begin(), last = x.hier.end();
  while (iter != last) {
    tree_dump (*iter, out, level+1);
    ++iter;
    if (iter != last) { out << "*"; }
  }
  if (x.spec != "mixed" || level != 0) { out << ")"; }
}
static
void
space_constitution_get_pass_1_2 (idiststream& ids)
{
  // --------------------------------------------
  // 1) read from file on io_proc into a string
  // --------------------------------------------
  typedef distributor::size_type size_type;
  std::istream& is = ids.is();
  communicator comm = ids.comm();
  size_type io_proc = odiststream::io_proc();
  size_type my_proc = comm.rank();
  std::string str;
  if (my_proc == io_proc) {
    std::getline (ids.is(), str);
  }
  // then broadcast the string
#ifdef _RHEOLEF_HAVE_MPI
  mpi::broadcast (mpi::communicator(), str, io_proc); 
#endif // _RHEOLEF_HAVE_MPI
  // ------------------------------------------------------------------------------------
  // 2) parse from string on all procs and re-build result_ptr available on all procs
  // ------------------------------------------------------------------------------------
  std::istringstream istrstr;
  istrstr.str (str);
  input_space_constitution = yyFlexLexer(&istrstr);
  symbol_table.clear();
  space_constitution_line_no = 1;
  space_constitution_n_error = 0;
  if (space_constitution_parse() != 0 || space_constitution_n_error != 0) {
    is.setstate (std::ios::badbit);
    delete_macro (result_ptr); result_ptr = 0;
    error_macro ("internal error in space_constitution io (pass 2)");
  }
  symbol_table.clear();
  // then result_ptr is available on all procs
}
// ================================================================================
// part 3 : space_constitution allocator from tree_type
// ================================================================================
template<class T, class M>
static
space_constitution<T,M>
build_from_tree (const tree_type& x) {
  if (! x.is_hier) {
    geo_basic<T,M> omega (x.base_geo.second); // TODO: geo hash_table
    return space_constitution<T,M>(omega, x.base_geo.first);
  }
  if (x.spec == "vector" || x.spec == "tensor") {
    assert_macro (x.hier.size() == 1, "unexpected tree");
    const tree_type& node = *(x.hier.begin());
    assert_macro (! node.is_hier, "unexpected tree");
    geo_basic<T,M> omega (node.base_geo.second); // TODO: geo hash_table
    space_constitution<T,M> base (omega, node.base_geo.first);
    space_constitution<T,M> constit;
    constit.set_hierarchy(true);
    size_t d = omega.dimension();
    space_constant::valued_type valued_tag = space_constant::valued_tag(x.spec);
    size_type n_comp = space_constant::n_component (valued_tag, d, omega.coordinate_system());
    constit.get_hierarchy().resize (n_comp);
    for (size_t i_comp = 0; i_comp < n_comp; i_comp++) {
      constit.get_hierarchy().operator[] (i_comp) = base;
    }
    constit.set_valued_tag (valued_tag);
    constit.data().initialize();
    return constit;
  }
  space_constitution<T,M> constit;
  constit.set_hierarchy(true);
  size_t n_comp = x.hier.size();
  constit.get_hierarchy().resize(n_comp);
  std::list<tree_type>::const_iterator iter = x.hier.begin();
  for (size_t i_comp = 0; i_comp < n_comp; i_comp++, ++iter) {
    const tree_type& xi = *iter;
    constit.get_hierarchy().operator[] (i_comp) = build_from_tree<T,M> (xi); // recursive call
  }
  constit.data().initialize();
  return constit;
}
// ================================================================================
// part 4 : main call
// ================================================================================
template<class T, class M>
idiststream&
operator>> (idiststream& ids, space_constitution<T,M>& constit)
{
  space_constitution_get_pass_1_2 (ids);
  // convert tree_type result_ptr to space_constitution
  const tree_type* ptr = result_ptr;
  constit = build_from_tree<T,M> (*ptr);
  delete_macro (result_ptr); result_ptr = 0;
  return ids;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template idiststream& operator>> (idiststream&, space_constitution<Float,sequential>&);

#ifdef _RHEOLEF_HAVE_MPI
template idiststream& operator>> (idiststream&, space_constitution<Float,distributed>&);
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
